﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using App.Command;
using App.Controls;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public class TextureMerger
    {
        /// <summary>
        /// テクスチャファイルのマージ。
        /// </summary>
        public void MergeFile(Texture dstTexture, MergerUtility.MergeMode mergeMode)
        {
            string[] fileNames;
            if (DialogUtility.ExecuteOpenFileDialog(out fileNames, dstTexture.ObjectID, false))
            {
                Texture mergedTexture = null;
                try
                {
                    mergedTexture = MergeFile(dstTexture, fileNames[0], mergeMode);
                }
                catch (Exception e)
                {
                    UIMessageBox.Error(Strings.Merge_Error + "\n" + e.Message);
                    mergedTexture = null;
                }

                // 編集コマンドで結果を戻す
                if (mergedTexture != null)
                {
                    var commandSet	= new EditCommandSet();
                    {
                        commandSet.Add(DocumentManager.CreateDocumentReplaceCommand(dstTexture, mergedTexture));

                        commandSet.Add(
                            new GeneralGroupReferenceEditCommand<object>(
                                new GuiObjectGroup(mergedTexture),
                                GuiObjectID.Texture,
                                Enumerable.Repeat<object>(null, 1),
                                delegate(ref GuiObject target, ref object data, ref object swap){},
                                createEventArgsDelegate: (target, command) => new FileView.FileTreeView.TextureImageChangedArgs(target, command)
                            )
                        );
                    }
                    TheApp.CommandManager.Execute(commandSet);
                }
            }
        }

        public Texture MergeFile(Texture dstTexture, string srcTexturePath, MergerUtility.MergeMode mergeMode)
        {
            // テクスチャは mergeMode を無視する。常にオートモードとして動く。

            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            using (var vdsb = new Viewer.ViewerDrawSuppressBlock())
            {
                using (var convertedFileName = TemporaryFileUtility.MakeDisposableFileName(".current" + G3dPath.TextureBinaryExtension))
                {
                    byte[] convertedDataArray = null;
                    {
                        // 一旦ファイルにする
                        using (var dstTempFileName = TemporaryFileUtility.MakeDisposableFileName(".current" + G3dPath.TextureBinaryExtension))
                        {
                            (new DocumentSaver()).WriteDocument(dstTexture, dstTempFileName.Path, false);

                            // ファイルとフォーマットの指定
                            {
                                TexcvtrManager.Clear();

                                bool isSuccess = TexcvtrManager.ReadInputFile(
                                    new[] { dstTempFileName.Path, null },
                                    new[]{
                                string.Format("--output=\"{0}\"", convertedFileName.Path),
                                string.Format("--merge=\"{0}\"", srcTexturePath),
                                null
                            }
                                );

                                if (!isSuccess)
                                {
                                    var error = TexcvtrManager.GetError();
                                    TexcvtrManager.Clear();
                                    throw new Exception(error);
                                }
                            }

                            // コンバートする
                            var convertedData = IntPtr.Zero;
                            int convertedDataSize = 0;
                            {
                                bool isSuccess = TexcvtrManager.ConvertToData(ref convertedData, ref convertedDataSize);
                                if (!isSuccess)
                                {
                                    var error = TexcvtrManager.GetError();
                                    TexcvtrManager.Clear();
                                    throw new Exception(error);
                                }
                            }

                            // マネージドなところに持ってくる
                            convertedDataArray = new byte[convertedDataSize];
                            Marshal.Copy(convertedData, convertedDataArray, 0, convertedDataSize);

                            TexcvtrManager.Clear();
                        }
                    }

                    using (var fs = new FileStream(convertedFileName.Path, FileMode.Create, FileAccess.Write))
                    {
                        fs.Write(convertedDataArray, 0, convertedDataArray.Length);
                    }

                    var clonedDstTexture = MergerUtility.LoadDocument<Texture>(convertedFileName.Path);

                    // パス情報を作る
                    clonedDstTexture.FileLocation = dstTexture.FileLocation;
                    clonedDstTexture.Name = dstTexture.Name;
                    clonedDstTexture.FileDotExt = dstTexture.FileDotExt;
                    clonedDstTexture.BaseName = dstTexture.BaseName;
                    clonedDstTexture.BaseName = dstTexture.BasePath;

                    // 最適化ログの追加
                    MergerUtility.AddOptimizeLog(clonedDstTexture, clonedDstTexture.Data, srcTexturePath);

                    return clonedDstTexture;
                }
            }
        }
    }
}
