﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;

namespace App.Data
{
    public class DocumentContainer<DocumentType>
        where DocumentType : Document
    {
        private readonly List<DocumentType> documents_ = new List<DocumentType>();

        public int Count{ get{ return documents_.Count; } }

        public IEnumerable<DocumentType> Documents { get { return documents_; } }

        public void Add(DocumentType document)
        {
            document.OnDocumentAdd();
            documents_.Add(document);
        }

        public void Remove(DocumentType document)
        {
            document.OnDocumentRemove();
            documents_.Remove(document);
        }

        public void Add(IEnumerable<DocumentType> docs)
        {
            foreach (var doc in docs)
            {
                doc.OnDocumentAdd();
            }
            documents_.AddRange(docs);
        }

        public void Remove(IEnumerable<DocumentType> docs)
        {
            foreach(var doc in docs.ToArray())
            {
                doc.OnDocumentRemove();
                documents_.Remove(doc);
            }
        }

        // ドキュメント入れ替えコマンドを作る
        // dstDoc(旧) を srcDoc(新) に入れ替える
        public EditCommand CreateDocumentReplaceCommand(DocumentType dstDoc, DocumentType srcDoc, bool disconnect = false, Action postEditAction = null)
        {
            Debug.Assert(dstDoc != null);
            Debug.Assert(srcDoc != null);
            Debug.Assert(documents_.Contains(dstDoc));
            Debug.Assert(documents_.Contains(srcDoc) == false);

            DocumentType addedDocument = dstDoc;
            DocumentType removedDocument = srcDoc;

            return
                new GeneralGroupReferenceEditCommand<object>(
                    new GuiObjectGroup(srcDoc),
                    srcDoc.ObjectID,
                    Enumerable.Repeat(new object(), 1),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        // 入れ替え
                        var index = documents_.IndexOf(addedDocument);
                        documents_[index] = removedDocument;
                        removedDocument = addedDocument;
                        addedDocument = documents_[index];

                        // スワップ時のアクション
                        removedDocument.OnDocumentSwapRemove();
                        addedDocument.OnDocumentSwapAdd(removedDocument);
                        addedDocument.SetContentsMaybeModified();

                        // ビューアへ転送
                        //Viewer.ViewerUtility.ReplaceDocument(addedDocument, removedDocument);
                    },
                    createEventArgsDelegate: delegate(GuiObject target, EditCommand command)
                    {
                        return
                            new DocumentAddedOrRemovedArg(target, command)
                            {
                                AddedDocuments = new List<Document>() { addedDocument },
                                RemovedDocuments = new List<Document>() { removedDocument }
                            };
                    },
                    preEditDelegate: (x, y) =>
                    {
                        if (disconnect)
                        {
                            // 必要?
                            Debug.Assert(!Viewer.Manager.Instance.IsConnected);
                            Viewer.ViewerUtility.Disconnect(string.Format(res.Strings.Viewer_ModifiedDisconnect, srcDoc.FileName), false, false);
                        }
                    },
                    postEditDelegate: (postEditAction == null ?
                        (GeneralGroupReferenceEditCommand<object>.PostEditDelegate)null:
                        (x, y) => postEditAction())
                );
        }
    }

    public class AnimationContainer			: DocumentContainer<AnimationDocument>{}
    public class ModelContainer				: DocumentContainer<Model>{}
    public class SeparateMaterialContainer  : DocumentContainer<SeparateMaterial>{}
    public class ShaderDefinitionContainer	: DocumentContainer<ShaderDefinition>{}
    public class TextureContainer			: DocumentContainer<Texture>{}
}
