﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

namespace App.Data
{
    [Serializable]
    public class UserDataArray
    {
        [Serializable]
        public class UserData
        {
            string name_ = string.Empty;
            public string Name { get { return name_; } set { name_ = value; } }
            public UserData(string name)
            {
                Name = name;
            }

            public virtual bool IsSame(UserData userData)
            {
                return Name == userData.Name;
            }
        }

        [Serializable]
        public class UserDataInt : UserData
        {
            List<int> intItem_ = null;
            public List<int> Item { get { return intItem_; } set { intItem_ = value; } }
            public UserDataInt(string name, List<int> intItem)
                :base(name)
            {
                Item = intItem;
            }

            public override bool IsSame(UserData userData)
            {
                return base.IsSame(userData) &&
                    userData is UserDataInt &&
                    Item.SequenceEqual(((UserDataInt)userData).Item);
            }
        }

        [Serializable]
        public class UserDataFloat : UserData
        {
            List<float> floatItem_ = null;
            public List<float> Item { get { return floatItem_; } set { floatItem_ = value; } }
            public UserDataFloat(string name, List<float> floatItem)
                : base(name)
            {
                Item = floatItem;
            }

            public override bool IsSame(UserData userData)
            {
                return base.IsSame(userData) &&
                    userData is UserDataFloat &&
                    Item.SequenceEqual(((UserDataFloat)userData).Item);
            }
        }

        [Serializable]
        public class UserDataString : UserData
        {
            List<string> stringItem_ = null;
            public List<string> Item { get { return stringItem_; } set { stringItem_ = value; } }
            public UserDataString(string name, List<string> stringItem)
                : base(name)
            {
                Item = stringItem;
            }

            public override bool IsSame(UserData userData)
            {
                return base.IsSame(userData) &&
                    userData is UserDataString &&
                    Item.SequenceEqual(((UserDataString)userData).Item);
            }
        }

        [Serializable]
        public class UserDataWString : UserData
        {
            List<string> wstringItem_ = null;
            public List<string> Item { get { return wstringItem_; } set { wstringItem_ = value; } }
            public UserDataWString(string name, List<string> wstringItem)
                : base(name)
            {
                Item = wstringItem;
            }

            public override bool IsSame(UserData userData)
            {
                return base.IsSame(userData) &&
                    userData is UserDataWString &&
                    Item.SequenceEqual(((UserDataWString)userData).Item);
            }
        }

        [Serializable]
        public class UserDataStream : UserData
        {
            List<byte> byteItem_ = null;

            [NonSerialized]
            int stream_index_ = -1;

            public List<byte> Item { get { return byteItem_; } set { byteItem_ = value; } }
            public int StreamIndex { get { return stream_index_; } set { stream_index_ = value; } }
            public UserDataStream(string name, List<byte> byteItem)
                : base(name)
            {
                Item = byteItem;
            }

            [System.Runtime.Serialization.OnDeserialized]
            private void OnSerializedMethod(System.Runtime.Serialization.StreamingContext context)
            {
                stream_index_ = -1;
            }

            public override bool IsSame(UserData userData)
            {
                return base.IsSame(userData) &&
                    userData is UserDataStream &&
                    Item.SequenceEqual(((UserDataStream)userData).Item);
            }
        }

        List<UserData> userData_ = new List<UserData>();
        public List<UserData> Data { get { return userData_; } set { userData_ = value; } }

        public UserDataArray()
        {
        }

        public void Add(string name, List<int> intItem)
        {
            UserDataInt item = new UserDataInt(name, intItem);
            userData_.Add(item);
        }
        public void Add(string name, List<float> floatItem)
        {
            UserDataFloat item = new UserDataFloat(name, floatItem);
            userData_.Add(item);
        }
        public void Add(string name, List<string> stringItem, bool utf8 = false)
        {
            UserData item = null;
            if (utf8)
            {
                item = new UserDataWString(name, stringItem);
            }
            else
            {
                item = new UserDataString(name, stringItem);
            }
            userData_.Add(item);
        }

        public void Add(string name, List<byte> byteItem, int streamIndex)
        {
            UserDataStream item = new UserDataStream(name, byteItem);
            item.StreamIndex = streamIndex;
            userData_.Add(item);
        }

        public void InsertUserData(UserData userData, int index)
        {
            Data.Insert(index, userData);
        }

        public void RemoveUserData(int index)
        {
            Data.RemoveAt(index);
        }

        public void ClearAllStreamIndex()
        {
            foreach (var usrData in Data)
            {
                if (usrData is UserDataStream)
                {
                    UserDataStream usrDataStream = usrData as UserDataStream;
                    usrDataStream.StreamIndex = -1;
                }
            }
        }

        public bool IsSame(UserDataArray userDataArray)
        {
            return Data.Count == userDataArray.Data.Count &&
                Data.Zip(userDataArray.Data, (x, y) => x.IsSame(y)).All(x => x);
        }
    }
}
