﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;

using nw.g3d.nw4f_3dif;

using TexUtils = nw.g3d.iftexutil;

namespace App.Data
{
    static public class TexUtilsProxy
    {
        private static TexUtils.Converter converter = null;

        private static void SafeInitialize()
        {
            if (converter == null)
            {
                // CafeのDLLを読み込む
                // 環境変数(NW4F_3DEDITOR_ROOT)は、3DEditorのConfig読み込み時に設定されているはず。
                // それは、NW4F_ROOT/Tool/3DEditorに準ずるものになっているはずなので、
                // そこからの相対パスで、フルパスを作成する。
                var basePath = new List<string>(3);
                var editorPath = Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT");

                if (string.IsNullOrEmpty(editorPath) == false)
                {
                    if (Environment.Is64BitProcess)
                    {
                        basePath.Add(Path.Combine(editorPath, @"..\3dTools\")); // Nintendo SDK パッケージ
                        basePath.Add(Path.Combine(editorPath, @"..\GraphicsTools\")); // Nintendo SDK パッケージ(TextureConverter)
                        basePath.Add(Path.Combine(editorPath, @"..\G3dTool\win64\")); // Nintendoware パッケージ
                    }
                    else
                    {
                        basePath.Add(Path.Combine(editorPath, @"..\3dTools\x86\")); // Nintendo SDK パッケージ
                        basePath.Add(Path.Combine(editorPath, @"..\GraphicsTools\x86\")); // Nintendo SDK パッケージ(TextureConverter)
                        basePath.Add(Path.Combine(editorPath, @"..\G3dTool\win32\")); // Nintendoware パッケージ
                    }
                }
                var cafeConverter = new TexUtils.Converter();
                cafeConverter.Initialize(basePath.ToArray());
                converter = cafeConverter;
                DebugConsole.WriteLine(
                    "TexUtilsProxy.SafeInitialize(): Load Converter Finish (Platform Converter Loaded[{0}]).",
                    converter.IsPlatformConverterLoaded);
            }
        }


        public static void Initialize()
        {
        }

        public static void Destroy()
        {
            if (converter != null)
            {
                converter.Destroy();
                converter = null;
            }
        }

        public static string GetConverterName(textureType texture)
        {
            return converter != null ? converter.GetPlatformName(texture) : "";
        }

        public static string ConverterName
        {
            get { return converter != null ? converter.PlatformName : ""; }
        }

        public static Bitmap[] ConvertTo1d2dBitmap(textureType texture, List<G3dStream> streams)
        {
            SafeInitialize();
            return converter.ConvertTo1d2dBitmap(texture, streams);
        }

        public static Bitmap[][] ConvertTo3dBitmap(textureType texture, List<G3dStream> streams)
        {
            SafeInitialize();
            return converter.ConvertTo3dBitmap(texture, streams);
        }

        public static Bitmap[][] ConvertToCubeBitmap(textureType texture, List<G3dStream> streams)
        {
            SafeInitialize();
            return converter.ConvertToCubeBitmap(texture, streams);
        }

        public static Bitmap[][] ConvertTo1dArrayBitmap(textureType texture, List<G3dStream> streams)
        {
            SafeInitialize();
            return converter.ConvertTo1dArrayBitmap(texture, streams);
        }

        public static Bitmap[][] ConvertTo2dArrayBitmap(textureType texture, List<G3dStream> streams)
        {
            SafeInitialize();
            return converter.ConvertTo2dArrayBitmap(texture, streams);
        }

        public static Bitmap[][] ConvertToCubeArrayBitmap(textureType texture, List<G3dStream> streams)
        {
            SafeInitialize();
            return converter.ConvertToCubeArrayBitmap(texture, streams);
        }
    }
}
