﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;
using ConfigCommon;

namespace App.Data
{
    /// <summary>
    /// ドキュメントプロパティ編集イベントハンドラデリゲート。
    /// </summary>
    public delegate void DocumentPropertyChangedEventHandler(object sender, IEnumerable<DocumentPropertyChangedArgs> args);
    public delegate void DocumentAddOrRemovedEventHandler(object sender, IEnumerable<Document> added, IEnumerable<Document> removed, Dictionary<GuiObject, GuiObject> swaped, IEnumerable<Document> reloaded);

    /// <summary>
    /// ドキュメントプロパティ編集ハンドラデリゲート。
    /// </summary>
    public delegate void DocumentPropertyChangedHandler(object sender, DocumentPropertyChangedArgs args);

    /// <summary>
    /// ドキュメントプロパティ編集イベント引数クラス。
    /// </summary>
    public sealed class DocumentPropertyChangedEventArgs : EventArgs
    {
        // 引数リスト
        private readonly List<DocumentPropertyChangedArgs> _argsList = new List<DocumentPropertyChangedArgs>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentPropertyChangedEventArgs() {}

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentPropertyChangedEventArgs(DocumentPropertyChangedArgs args)
        {
            AddArgs(args);
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentPropertyChangedEventArgs(DocumentPropertyChangedEventArgs eventArgs)
        {
            Set(eventArgs);
        }

        /// <summary>
        /// クリア
        /// </summary>
        public void Clear()
        {
            _argsList.Clear();
        }

        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(DocumentPropertyChangedEventArgs eventArgs)
        {
            Clear();
            Merge(eventArgs);
        }

        /// <summary>
        /// マージ。
        /// </summary>
        public void Merge(DocumentPropertyChangedEventArgs eventArgs)
        {
            foreach (DocumentPropertyChangedArgs args in eventArgs.GetArgs())
            {
                AddArgs(args);
            }
        }

        //---------------------------------------------------------------------
        #region 引数
        /// <summary>
        /// 引数の追加。
        /// </summary>
        public void AddArgs(DocumentPropertyChangedArgs additionalArgs)
        {
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                // 同値の引数は無視する
                //if (args.Equals(additionalArgs)) { return; }
                if (args.Equivalence(additionalArgs)) { return; }
            }
            _argsList.Add(additionalArgs);
        }

        /// <summary>
        /// 引数の取得。
        /// </summary>
        public List<DocumentPropertyChangedArgs> GetArgs()
        {
            return new List<DocumentPropertyChangedArgs>(_argsList);
        }

        /// <summary>
        /// 引数の取得。
        /// </summary>
        public List<DocumentPropertyChangedArgs> GetArgs(Type argsType)
        {
            Debug.Assert(argsType.IsSubclassOf(typeof(DocumentPropertyChangedArgs)));
            List<DocumentPropertyChangedArgs> result = new List<DocumentPropertyChangedArgs>();
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (args.GetType().Equals(argsType)) { result.Add(args); }
            }
            return result;
        }

        /// <summary>
        /// 引数を持っているか。
        /// </summary>
        public bool HasArgs() { return (_argsList.Count > 0); }

        /// <summary>
        /// 引数を持っているか。
        /// </summary>
        public bool HasArgs(Type argsType)
        {
            Debug.Assert(argsType.IsSubclassOf(typeof(DocumentPropertyChangedArgs)));
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (args.GetType().Equals(argsType)) { return true; }
            }
            return false;
        }
        #endregion

        //---------------------------------------------------------------------
        #region オブジェクト
        /// <summary>
        /// オブジェクトの取得。
        /// </summary>
        public List<GuiObject> GetObjects()
        {
            List<GuiObject> result = new List<GuiObject>();
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (result.Contains(args.Target)) { continue; }
                result.Add(args.Target);
            }
            return result;
        }

        /// <summary>
        /// オブジェクトの取得。
        /// </summary>
        public List<GuiObject> GetObjects(Type argsType)
        {
            Debug.Assert(argsType.IsSubclassOf(typeof(DocumentPropertyChangedArgs)));
            List<GuiObject> result = new List<GuiObject>();
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (result.Contains(args.Target)) { continue; }
                if (args.GetType().Equals(argsType) || args.GetType().IsSubclassOf(argsType))
                {
                    result.Add(args.Target);
                }
            }
            return result;
        }

        /// <summary>
        /// オブジェクトの取得。
        /// </summary>
        public List<GuiObject> GetObjects(GuiObjectID objectID)
        {
            List<GuiObject> result = new List<GuiObject>();
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (result.Contains(args.Target)) { continue; }
                if (args.Target.ObjectID == objectID) { result.Add(args.Target); }
            }
            return result;
        }

        /// <summary>
        /// オブジェクトの取得。
        /// </summary>
        public List<GuiObject> GetObjects(Type argsType, GuiObjectID objectID)
        {
            Debug.Assert(argsType.IsSubclassOf(typeof(DocumentPropertyChangedArgs)));
            List<GuiObject> result = new List<GuiObject>();
            foreach (DocumentPropertyChangedArgs args in _argsList)
            {
                if (result.Contains(args.Target)) { continue; }
                if (args.GetType().Equals(argsType) && (args.Target.ObjectID == objectID))
                {
                    result.Add(args.Target);
                }
            }
            return result;
        }
        #endregion
    }

    //-------------------------------------------------------------------------
    /// <summary>
    /// ドキュメントプロパティ編集引数クラス。
    /// </summary>
    public abstract class DocumentPropertyChangedArgs
    {
        // ターゲット
        private readonly GuiObject _target;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected DocumentPropertyChangedArgs(GuiObject target, EditCommand command)
        {
            Debug.Assert(target != null);
            _target = target;
            Command = command;
        }

        /// <summary>
        /// ターゲット。
        /// </summary>
        public GuiObject Target
        {
            get { return _target; }
        }

        /// <summary>
        /// 実行したコマンド
        /// </summary>
        public EditCommand Command;

        #region 比較
        /// <summary>
        /// 等値比較。
        /// </summary>
        public virtual bool Equivalence(DocumentPropertyChangedArgs src)
        {
            // 必ずサブクラスからas演算子を使って呼び出す為にprotected
            if (src == null) return false;
            if (src == this) return true;
            if (src.GetType() != GetType()) return false;
            return Target == src.Target && Command == src.Command;
        }
        #endregion
    }
    //-------------------------------------------------------------------------
    /// <summary>
    /// ドキュメント内容編集引数クラス。
    /// </summary>
    public class DocumentContentsChangedArgs : DocumentPropertyChangedArgs
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentContentsChangedArgs(GuiObject target, EditCommand command) : base(target, command) {}
    }

    /// <summary>
    /// ドキュメントの追加
    /// </summary>
    public sealed class DocumentAddedOrRemovedArg : DocumentPropertyChangedArgs
    {
        public DocumentAddedOrRemovedArg(GuiObject target, EditCommand command, IEnumerable<Document> added, IEnumerable<Document> removed) : base(target, command) {
            AddedDocuments = added.ToList();
            RemovedDocuments = removed.ToList();
        }
        public DocumentAddedOrRemovedArg(GuiObject target, EditCommand command) : base(target, command) {
        }

        public List<Document> AddedDocuments { get; set; }
        public List<Document> RemovedDocuments { get; set; }
        #region 比較
        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equivalence(DocumentPropertyChangedArgs obj)
        {
            return this == obj;
        }
        #endregion
    }

    /// <summary>
    /// アニメーションのバインド変更
    /// </summary>
    public sealed class DocumentAnimationBindArg : DocumentPropertyChangedArgs
    {
        public DocumentAnimationBindArg(GuiObject target, EditCommand command, bool autoBound) : base(target, command)
        {
            AutoBound = autoBound;
        }

        /// <summary>
        /// 3DEditor が自動でバインドの解決したか
        /// </summary>
        public bool AutoBound { get; set; }
    }

    /// <summary>
    /// オブジェクトの名前変更
    /// </summary>
    public sealed class ObjectRenamedArg : DocumentPropertyChangedArgs
    {
        public ObjectRenamedArg(GuiObject target, EditCommand command, GuiObject renamedObject) : base(target, command)
        {
            RenamedObject = renamedObject;
        }

        /// <summary>
        /// 3DEditor が自動でバインドの解決したか
        /// </summary>
        public GuiObject RenamedObject { get; set; }
    }

    /// <summary>
    /// アニメーションのバインド変更
    /// </summary>
    public sealed class SceneAnimationContentArg : DocumentPropertyChangedArgs
    {
        public SceneAnimationContentArg(GuiObject target, EditCommand command)
            : base(target, command)
        {
        }
    }

    public sealed class PreviewAnimSetUpdatedArg : DocumentPropertyChangedArgs
    {
        public PreviewAnimSetUpdatedArg() : base(DummyObject.TheDummyObject, null)
        {
        }
    }
}
