﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Linq;
using App.Command;
using App.Controls;
using App.PropertyEdit;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public partial class ShaderParameterAnimation
    {
        public nw.g3d.nw4f_3dif.G3dArrayElement<nw.g3d.nw4f_3dif.original_material_animType> OriginalMaterialAnimArray {
            get
            {
                return Data.original_material_anim_array;
            }
        }

        private string getUniformHintFromParamId(string materialName, string paramAnimId)
        {
            string targetHint = string.Empty;
            //if (this.Data.shader_param_anim_info.shader_assigned)
            {
                IEnumerable<Material> mats = ParentModelMaterialNames;
                foreach (Material mat in mats)
                {
                    // 対象マテリアル
                    if (mat.Name != materialName)
                    {
                        continue;
                    }

                    IEnumerable<uniform_varType> uniforms = ShaderTypeUtility.MaterialUniforms(mat.MaterialShaderAssign.ShadingModel);
                    foreach (var uniform in uniforms)
                    {
                        if (uniform.id == paramAnimId)
                        {
                            targetHint = uniform.hint;
                            return targetHint;
                        }
                    }
                }
            }

            return targetHint;
        }

        //
        public bool IsExistOriginalParamAnim(string materialName, string paramAnimId)
        {
            if (Data.original_material_anim_array == null)
            {
                return false;
            }
            if (Data.original_material_anim_array.original_material_anim == null)
            {
                return false;
            }
            if (!Data.original_material_anim_array.original_material_anim.Any())
            {
                return false;
            }

            string targetHint = getUniformHintFromParamId(materialName, paramAnimId);

            foreach (original_material_animType MatAnim in Data.original_material_anim_array.original_material_anim)
            {
                if (MatAnim.mat_name != materialName)
                {
                    continue;
                }

                if (MatAnim.original_color_anim_array != null)
                {
                    foreach (original_color_animType paramAnim in MatAnim.original_color_anim_array.original_color_anim)
                    {

                        if (!string.IsNullOrEmpty(targetHint) &&
                            paramAnim.hint == targetHint)
                        {
                            return true;
                        }
                        else
                        {
                            string paramId = IfAnimationAssignUtility.GetIdFromOriginalColorAnim(paramAnim.hint);
                            if (!string.IsNullOrEmpty(paramId) &&
                                (paramId == paramAnimId))
                            {
                                return true;
                            }
                        }
                    }
                }
                if (MatAnim.original_texsrt_anim_array != null)
                {
                    foreach (original_texsrt_animType paramAnim in MatAnim.original_texsrt_anim_array.original_texsrt_anim)
                    {
                        if (!string.IsNullOrEmpty(targetHint) &&
                            paramAnim.hint == targetHint)
                        {
                            return true;
                        }
                        else
                        {
                            string paramId = IfAnimationAssignUtility.GetIdFromOriginalTexSRTAnim(paramAnim.hint);
                            if (!string.IsNullOrEmpty(paramId) &&
                                (paramId == paramAnimId))
                            {
                                return true;
                            }
                        }
                    }
                }
            }

            return false;
        }

        //
        public bool IsExistOriginalColorAnim()
        {
            foreach (original_material_animType MatAnim in Data.original_material_anim_array.original_material_anim)
            {
                if (MatAnim.original_color_anim_array != null)
                {
                    if (MatAnim.original_color_anim_array.original_color_anim != null &&
                        MatAnim.original_color_anim_array.original_color_anim.Any())
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        //
        public bool IsExistOriginalTexSRTAnim()
        {
            foreach (original_material_animType MatAnim in Data.original_material_anim_array.original_material_anim)
            {
                if (MatAnim.original_texsrt_anim_array != null)
                {
                    if (MatAnim.original_texsrt_anim_array.original_texsrt_anim != null &&
                        MatAnim.original_texsrt_anim_array.original_texsrt_anim.Any())
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        // ParamAnimに対応するOriginalAnimが存在するか？
        public bool IsExistOriginalParamAnim(string materialName, IEnumerable<App.PropertyEdit.CurveTreeNode> nodes)
        {
            foreach (var node in nodes)
            {
                ParamAnim paramAnim = GetParamAnimFromId(materialName, node.id);
                if (paramAnim == null)
                {
                    continue;
                }

                //
                if (IsExistOriginalParamAnim(materialName, paramAnim.id))
                {
                    // 1つでも存在すればtrue
                    return true;
                }
            }

            return false;
        }

        // ParamAnimに対応するOriginalAnimが存在するか？
        public bool IsExistOriginalParamAnim(string materialName, CurveTreeNode node)
        {
            ParamAnim paramAnim = GetParamAnimFromId(materialName, node.id);
            if (paramAnim == null)
            {
                return false;
            }

            switch (paramAnim.type)
            {
                case shader_param_typeType.float4:
                case shader_param_typeType.float3:
                    return IsExistOriginalColorAnim();
                case shader_param_typeType.srt2d:
                case shader_param_typeType.texsrt:
                    return IsExistOriginalTexSRTAnim();
                case shader_param_typeType.@float:
                    return IsExistOriginalColorAnim() || IsExistOriginalTexSRTAnim();
            }

            return false;
        }

        // オリジナルデータ自体があるかどうか？
        public bool IsExistOriginalAnim()
        {
            if (Data.original_material_anim_array == null)
            {
                return false;
            }

            foreach (var anim in Data.original_material_anim_array.original_material_anim)
            {
                if (anim.original_color_anim_array != null)
                {
                    return true;
                }

                if(anim.original_texsrt_anim_array != null)
                {
                    return true;
                }
            }

            return false;
        }

        // ParamAnimに対応するOriginalAnimから、ParamAnimのAnimTargetを更新する。
        public void UpdateParamAnimFromOriginalAnim(string materialName, IEnumerable<App.PropertyEdit.CurveTreeNode> nodes, EditCommandSet commandSet)
        {
            GuiObjectGroup group = new GuiObjectGroup();
            group.Add(this);

            if (Data.original_material_anim_array == null)
            {
                return;
            }

            original_material_animType original = Data.original_material_anim_array.original_material_anim.FirstOrDefault(x => x.mat_name == materialName);
            if (original == null)
            {
                return;
            }

            foreach (var node in nodes)
            {
                ParamAnim paramAnim = GetParamAnimFromId(materialName, node.id);
                if (paramAnim == null)
                {
                    continue;
                }

                string hint = getUniformHintFromParamId(materialName, paramAnim.id);
                param_animType param_anim = IfAnimationAssignUtility.CreateParamAnimFromOriginal(original, paramAnim.id, paramAnim.type, hint, BinaryStreams, Data.shader_param_anim_info, false);
                if (param_anim != null)
                {
                    ParamAnim newParamAnim = CreateParamAnim(param_anim, BinaryStreams);
                    commandSet.Add(CreateAnimTargetFromOriginalTargetEditCommand(group, ObjectID, materialName, newParamAnim).Execute());
                }
            }
            Viewer.LoadOrReloadAnimation.Send(this);
            commandSet.OnPostEdit += (o, args) => Viewer.LoadOrReloadAnimation.Send(this);
        }

        // 指定されたOriginalAnimから、ParamAnimのAnimTargetを更新する。
        public void UpdateParamAnimFromSpecificOriginalAnim(string materialName, App.PropertyEdit.CurveTreeNode node, original_color_animType colorAnim, original_texsrt_animType texSRTAnim, EditCommandSet commandSet)
        {
            GuiObjectGroup group = new GuiObjectGroup();
            group.Add(this);

            ParamAnim paramAnim = GetParamAnimFromId(materialName, node.id);
            if (paramAnim == null)
            {
                // こんなことはまずないはず！
                return;
            }

            param_animType param_anim = null;
            if (colorAnim != null)
            {
                param_anim = IfAnimationAssignUtility.CreateParamAnimFromOriginalColorAnim(colorAnim, paramAnim.id, paramAnim.type, BinaryStreams, Data.shader_param_anim_info, false);

            }
            else if (texSRTAnim != null)
            {
                param_anim = IfAnimationAssignUtility.CreateParamAnimFromOriginalSrtAnim(texSRTAnim, paramAnim.id, paramAnim.type, BinaryStreams, Data.shader_param_anim_info, false);
            }

            if (param_anim != null)
            {
                ParamAnim newParamAnim = CreateParamAnim(param_anim, BinaryStreams);
                commandSet.Add(CreateAnimTargetFromOriginalTargetEditCommand(group, ObjectID, materialName, newParamAnim).Execute());
                Viewer.LoadOrReloadAnimation.Send(this);
                commandSet.OnPostEdit += (o, args) => Viewer.LoadOrReloadAnimation.Send(this);
            }
        }

        // 指定されたOriginalAnimTargetから、ParamAnimのAnimTargetを更新する。
        public void UpdateParamAnimFromOriginalAnimTarget(string materialName, App.PropertyEdit.CurveTreeNode node, int componentIndex,
                                                          original_color_animType colorAnim, original_color_anim_targetType colorAnimTarget,
                                                          original_texsrt_animType texSRTAnim, original_texsrt_anim_targetType texSRTAnimTarget,
                                                          EditCommandSet commandSet)
        {
            GuiObjectGroup group = new GuiObjectGroup();
            group.Add(this);

            ParamAnim paramAnim = GetParamAnimFromId(materialName, node.id);
            if (paramAnim == null)
            {
                // こんなことはまずないはず！
                return;
            }

            // texsrt, texsrtex の場合はcomponentIndexが一つずれる
            if (paramAnim.type == shader_param_typeType.texsrt)
            {
                componentIndex++;
            }

            IG3dCurveContainer originalCurve = null;
            if (colorAnim != null)
            {
                originalCurve = colorAnimTarget;
            }
            else if (texSRTAnim != null)
            {
                originalCurve = texSRTAnimTarget;
            }

            param_anim_targetType param_anim_target = IfAnimationAssignUtility.CreateParamAnimTargetFromOriginalAnim(
                originalCurve,
                componentIndex,
                paramAnim.type,
                Data.shader_param_anim_info,
                BinaryStreams);

            if (param_anim_target != null)
            {
                var oldParamAnimTarget = paramAnim.ParamAnimTargets[componentIndex];
                ParamAnimTarget newParamAnimTarget = CreateParamAnimTarget(param_anim_target, BinaryStreams, oldParamAnimTarget.IsTexSrt, oldParamAnimTarget.CurveInterpolationType);
                commandSet.Add(CreateAnimTargetEditCommand(group, ObjectID, materialName, paramAnim.id, componentIndex, newParamAnimTarget).Execute());
                Viewer.LoadOrReloadAnimation.Send(this);
                commandSet.OnPostEdit += (o, args) => Viewer.LoadOrReloadAnimation.Send(this);
            }
        }

        // ShaderParameterAnimation全体に対して、オリジナル適応を行う。
        public void ApplyOriginalAnimation(EditCommandSet commandSet, bool isLoadAnimation)
        {
            bool updateFlg = false;
            foreach (var shaderParamAnim in ShaderParamAnims)
            {
                string materialName = shaderParamAnim.mat_name;
                GuiObjectGroup group = new GuiObjectGroup();
                group.Add(this);
                if (Data.original_material_anim_array == null)
                {
                    continue;
                }

                original_material_animType original = Data.original_material_anim_array.original_material_anim.FirstOrDefault(x => x.mat_name == materialName);

                foreach (var paramAnim in shaderParamAnim.ParamAnims)
                {
                    string hint = getUniformHintFromParamId(materialName, paramAnim.id);
                    param_animType param_anim = IfAnimationAssignUtility.CreateParamAnimFromOriginal(original, paramAnim.id, paramAnim.type, hint, BinaryStreams, Data.shader_param_anim_info, false);
                    if (param_anim != null)
                    {
                        ParamAnim newParamAnim = CreateParamAnim(param_anim, BinaryStreams);
                        commandSet.Add(CreateAnimTargetFromOriginalTargetEditCommand(group, ObjectID, materialName, newParamAnim).Execute());
                        updateFlg = true;
                    }
                }
            }

            if (updateFlg)
            {
                Viewer.LoadOrReloadAnimation.Send(this);
                commandSet.OnPostEdit += (o, args) => Viewer.LoadOrReloadAnimation.Send(this);

                // アップデートした旨をダイアログ表示する。
                var logMessage = string.Format(Strings.IO_Apply_OriginalAnimation, UIText.EnumValue(ObjectID), Name);
                MessageLog.Write(MessageLog.LogType.Information, logMessage);
            }
        }

        /// <summary>
        /// AnimTarget を上書き（OriginalAnimTargetをAnimTargetに反映用に使用します。)
        /// </summary>
        public static ICommand CreateAnimTargetFromOriginalTargetEditCommand(GuiObjectGroup targetGroup, GuiObjectID targetGuiObjectID, string materialName, ParamAnim newParamAnim)
        {
            List<ParamAnim> list = new List<ParamAnim>();
            list.Add(newParamAnim);
            return new GeneralGroupReferenceEditCommand<ParamAnim>(
                targetGroup,
                targetGuiObjectID,
                list,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    ShaderParameterAnimation shaderParamAnim = target as ShaderParameterAnimation;
                    ParamAnim paramAnim = data as ParamAnim;
                    ParamAnim oldParamAnim = shaderParamAnim.GetParamAnimFromId(materialName, paramAnim.id);
                    swap = oldParamAnim;
                    shaderParamAnim.SwapParamAnim(materialName, oldParamAnim, paramAnim);

                    // IsModified を更新
                    var savedShaderParamAnim = shaderParamAnim.savedShaderParamAnims[materialName];
                    var savedParamAnim = savedShaderParamAnim != null ? savedShaderParamAnim.ParamAnims[paramAnim.id] : null;
                    foreach (var paramAnimTarget in paramAnim.ParamAnimTargets)
                    {
                        var savedAnimTarget = savedParamAnim != null ?
                            savedParamAnim.ParamAnimTargets.FirstOrDefault(x => x.component_index == paramAnimTarget.component_index) :
                            null;

                        if (savedAnimTarget == null && paramAnimTarget.IsTexSrt && paramAnimTarget.component_index == 0)
                        {
                            var key = shaderParamAnim.CreateInitialSrtModesKey(materialName, paramAnim.id);
                            shaderParamAnim.initialSrtModes.TryGetValue(key, out savedAnimTarget);
                        }

                        shaderParamAnim.UpdateIsModifiedAnimTarget(paramAnimTarget, savedAnimTarget);
                    }

                    shaderParamAnim.UpdateIsModifiedParamAnim(paramAnim, savedParamAnim);
                }
            );
        }

        public static ICommand CreateAnimTargetEditCommand(
            GuiObjectGroup targetGroup,
            GuiObjectID targetGuiObjectID,
            string materialName,
            string paramId,
            int componentIndex,
            ParamAnimTarget newParamAnimTarget)
        {
            return new GeneralGroupReferenceEditCommand<ParamAnimTarget>(
                targetGroup,
                targetGuiObjectID,
                ObjectUtility.MultipleClone(newParamAnimTarget, targetGroup.GetObjects(targetGuiObjectID).Count),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    ShaderParameterAnimation shaderParamAnim = target as ShaderParameterAnimation;
                    ParamAnim paramAnim = shaderParamAnim.GetParamAnimFromId(materialName, paramId);
                    ParamAnimTarget paramAnimTarget = (ParamAnimTarget)data;
                    swap = paramAnim.ParamAnimTargets[componentIndex];
                    paramAnim.ParamAnimTargets[componentIndex] = paramAnimTarget;

                    // IsModified を更新
                    var savedShaderParamAnim = shaderParamAnim.savedShaderParamAnims[materialName];
                    var savedParamAnim = savedShaderParamAnim != null ? savedShaderParamAnim.ParamAnims[paramAnim.id] : null;

                    var savedAnimTarget = savedParamAnim != null ?
                        savedParamAnim.ParamAnimTargets.FirstOrDefault(x => x.component_index == componentIndex) :
                        null;

                    if (savedAnimTarget == null && paramAnimTarget.IsTexSrt && paramAnimTarget.component_index == 0)
                    {
                        var key = shaderParamAnim.CreateInitialSrtModesKey(materialName, paramAnim.id);
                        shaderParamAnim.initialSrtModes.TryGetValue(key, out savedAnimTarget);
                    }

                    shaderParamAnim.UpdateIsModifiedAnimTarget(paramAnimTarget, savedAnimTarget);
                });
        }

    }
}
