﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class SceneAnimation :
        AnimationDocument,
        IHasUserData,
        NintendoWare.G3d.Edit.IEditSceneAnimTarget
    {
        public scene_animType Data{ get; private set; }
        public override object nw4f_3difItem
        {
            get { return Data; }
        }

        public override process_log_arrayType process_log_array
        {
            get { return Data.process_log_array; }
        }

        public override int FrameCount
        {
            get
            {
                int maxFrame = 0;
                {
                    foreach(var camera in CameraAnims)
                    {
                        if (camera.Data.frame_count > maxFrame)
                        {
                            maxFrame = camera.Data.frame_count;
                        }
                    }

                    foreach(var light in LightAnims)
                    {
                        if (light.Data.frame_count > maxFrame)
                        {
                            maxFrame = light.Data.frame_count;
                        }
                    }

                    foreach(var fog in FogAnims)
                    {
                        if (fog.Data.frame_count > maxFrame)
                        {
                            maxFrame = fog.Data.frame_count;
                        }
                    }
                }
                return maxFrame;
            }
            set { ; }
        }

        public override bool Loop
        {
            get
            {
                foreach(var camera in CameraAnims)
                {
                    if (camera.Data.loop)
                    {
                        return true;
                    }
                }

                foreach(var light in LightAnims)
                {
                    if (light.Data.loop)
                    {
                        return true;
                    }
                }

                foreach(var fog in FogAnims)
                {
                    if (fog.Data.loop)
                    {
                        return true;
                    }
                }

                return false;
            }
            set { ; }
        }

        public override bool IsFrameCountModified
        {
            //get { return savedData.frame_count != Data.skeletal_anim_info.frame_count; }
            get { return false; }
        }

        public override bool IsLoopModified
        {
            //get { return savedData.loop != Data.skeletal_anim_info.loop; }
            get { return false; }
        }

        // ファイル出力できるか？
        public override bool IsFileOutputable
        {
            get
            {
                return FileNotOutputErrorMessage == null;
            }
        }

        public override string FileNotOutputErrorMessage
        {
            get
            {
                return CameraAnims.Select(x => x.FileNotOutputMessage)
                    .Concat(LightAnims.Select(x => x.FileNotOutputMessage))
                    .Concat(FogAnims.Select(x => x.FileNotOutputMessage))
                    .Where(x => x != null).FirstOrDefault();
            }
        }

        public override IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse()
        {
            return CameraAnims.SelectMany(x => x.QuantizeAnalyse(null))
                .Concat(LightAnims.SelectMany(x => x.QuantizeAnalyse(null)))
                .Concat(FogAnims.SelectMany(x => x.QuantizeAnalyse(null)));
        }

        public override bool HasAnimation
        {
            get
            {
                return
                    CameraAnims.Any(x => x.HasAnimation) ||
                    LightAnims.Any(x => x.HasAnimation) ||
                    FogAnims.Any(x => x.HasAnimation);
            }
        }

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        // 子要素
        public override IEnumerable<GuiObject> ContentObjects
        {
            get
            {
                return base.ContentObjects.Concat(ChildAnims);
            }
        }
        public List<CameraAnimation> CameraAnims { get; set; }
        public List<LightAnimation> LightAnims { get; set; }
        public List<FogAnimation> FogAnims { get; set; }

        public IEnumerable<GuiObject> ChildAnims
        {
            get
            {
                return CameraAnims.Concat<GuiObject>(LightAnims).Concat(FogAnims);
            }
        }

        #region プレビュー情報
        // ランタイム連携用に表示するか？
        public HashSet<AnimationSet> InvisibleBinds { get; } = new HashSet<AnimationSet>();
        #endregion

        public SceneAnimation(scene_animType animation, List<G3dStream> streams)
            : base(GuiObjectID.SceneAnimation, streams)
        {
            Data = animation;
            Initialize();
        }

        public void Initialize()
        {
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, BinaryStreams);

            MakeSceneAnims();
        }

        #region Bridge
        public class SceneSwapData : SwapData
        {
            public ToolData ToolData;
            public List<G3dStream> streams;
            public string Comment;
            public RgbaColor? EditColor;
            public scene_animType Data;
            public List<CameraAnimation> cameraAnims;
            public List<LightAnimation> lightAnims;
            public List<FogAnimation> fogAnims;
            public UserDataArray userData;
        }

        public override SwapData CreateSwapData(object data, List<G3dStream> streams)
        {
            var current = GetSwapData();
            Data = (scene_animType)data;
            BinaryStreams = streams;
            Initialize();

            // 保存済みデータの設定
            foreach (var content in CameraAnims.Concat<GuiObject>(LightAnims).Concat(FogAnims))
            {
                content.UpdateSavedData();
            }

            return Swap(current, false);
        }

        public override SwapData GetSwapData()
        {
            return new SceneSwapData()
            {
                Data = Data,
                streams = BinaryStreams,
                Comment = Comment,
                EditColor = EditColor,
                cameraAnims = CameraAnims.ToList(),
                lightAnims = LightAnims.ToList(),
                fogAnims = FogAnims.ToList(),
                userData = UserDataArray,
            };
        }

        public override SwapData Swap(SwapData swapData, bool copyHistory)
        {
            var current = (SceneSwapData)GetSwapData();
            var data = (SceneSwapData)swapData;
            Data = data.Data;
            BinaryStreams = data.streams;
            Comment = data.Comment;
            EditColor = data.EditColor;
            CameraAnims = data.cameraAnims;
            LightAnims = data.lightAnims;
            FogAnims = data.fogAnims;
            UserDataArray = data.userData;

            if (copyHistory)
            {
                UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);

                foreach (var old in current.cameraAnims)
                {
                    var anim = CameraAnims.FirstOrDefault(x => x.Name == old.Name);
                    if (anim != null)
                    {
                        SavedContents.Remove(old);
                        SavedContents.Add(anim);
                        anim.CopySavedData(old);
                    }
                }

                foreach (var old in current.lightAnims)
                {
                    var anim = LightAnims.FirstOrDefault(x => x.Name == old.Name);
                    if (anim != null)
                    {
                        SavedContents.Remove(old);
                        SavedContents.Add(anim);
                        anim.CopySavedData(old);
                    }
                }

                foreach (var old in current.fogAnims)
                {
                    var anim = FogAnims.FirstOrDefault(x => x.Name == old.Name);
                    if (anim != null)
                    {
                        SavedContents.Remove(old);
                        SavedContents.Add(anim);
                        anim.CopySavedData(old);
                    }
                }
            }

            return current;
        }
        #endregion

        /// <summary>
        /// アニメーションのラッパーを生成
        /// </summary>
        public void MakeSceneAnims()
        {
            CameraAnims = new List<CameraAnimation>();
            if (Data.camera_anim_array != null)
            {
                foreach (var camera_anim in Data.camera_anim_array.camera_anim)
                {
                    CameraAnims.Add(new CameraAnimation(camera_anim, this));
                }
            }

            LightAnims = new List<LightAnimation>();
            if (Data.light_anim_array != null)
            {
                foreach (var light_anim in Data.light_anim_array.light_anim)
                {
                    LightAnims.Add(new LightAnimation(light_anim, this));
                }
            }

            FogAnims = new List<FogAnimation>();
            if (Data.fog_anim_array != null)
            {
                foreach (var fog_anim in Data.fog_anim_array.fog_anim)
                {
                    FogAnims.Add(new FogAnimation(fog_anim, this));
                }
            }
        }

        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            lock (this)
            {
                UpdateData(viewer);

                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        public void Optimize()
        {
            // ストリームを更新する。
            UpdateData();

            // <hermite_curve>、<linear_curve>、<step_curve> の frame_type と key_type を
            // none に設定しないと、再分析が行われません。
            if (Data.camera_anim_array != null)
            {
                foreach (var anim in Data.camera_anim_array.camera_anim)
                {
                    if (anim.camera_anim_target != null)
                    {
                        foreach (var target in anim.camera_anim_target.Where(x => x.Curve != null))
                        {
                            target.Curve.frame_type = curve_frame_typeType.none;
                            target.Curve.key_type = curve_key_typeType.none;
                        }
                    }
                }
            }
            if (Data.light_anim_array != null)
            {
                foreach (var anim in Data.light_anim_array.light_anim)
                {
                    if (anim.light_anim_target != null)
                    {
                        foreach (var target in anim.light_anim_target.Where(x => x.Curve != null))
                        {
                            target.Curve.frame_type = curve_frame_typeType.none;
                            target.Curve.key_type = curve_key_typeType.none;
                        }
                    }
                }
            }
            if (Data.fog_anim_array != null)
            {
                foreach (var anim in Data.fog_anim_array.fog_anim)
                {
                    if (anim.fog_anim_target != null)
                    {
                        foreach (var target in anim.fog_anim_target.Where(x => x.Curve != null))
                        {
                            target.Curve.frame_type = curve_frame_typeType.none;
                            target.Curve.key_type = curve_key_typeType.none;
                        }
                    }
                }
            }

            try
            {
                // アニメーションに合わせたAnimQuantizationAnalysisOptimizerを呼び出す。
                IfSceneAnimQuantizationAnalysisOptimizer optimizer = new IfSceneAnimQuantizationAnalysisOptimizer();
                optimizer.Optimize(Data, BinaryStreams);
            }
            catch(Exception e)
            {
                DebugConsole.WriteLine("Exception Catch : IfSceneAnimQuantizationAnalysisOptimizer.Optimize() : {0}", e.Message);
            }

            // 最適化後に、データを再作成する。
            this.MakeUserData(Data.user_data_array, BinaryStreams);
            MakeSceneAnims();
        }

        public void UpdateData(bool viewer = false)
        {
            ToolData.Load(Data.tool_data);
            Data.comment = GetComment();

            List<G3dStream> streams = new List<G3dStream>();

            // カメラアニメーション
            var camera_anims = new List<camera_animType>();
            foreach (var cameraAnim in CameraAnims)
            {
                var camera_anim = cameraAnim.Data;
                camera_anim.comment = cameraAnim.GetComment();
                camera_anim.camera_name = cameraAnim.Name;

                List<camera_anim_targetType> targets = new List<camera_anim_targetType>();
                foreach (var cameraAnimTarget in cameraAnim.CameraAnimTargets)
                {
                    G3dStream stream;
                    var target = cameraAnimTarget.ConvertTo_camera_animType(out stream, viewer);
                    targets.Add(target);

                    if (stream != null)
                    {
                        target.Curve.stream_index = streams.Count;
                        streams.Add(stream);
                    }
                }

                camera_anim.camera_anim_target = targets.ToArray();
                camera_anims.Add(camera_anim);

                if (cameraAnim.UserDataArray.Data == null ||
                    cameraAnim.UserDataArray.Data.Count == 0)
                {
                    camera_anim.user_data_array = null;
                }
                else
                {
                    camera_anim.user_data_array = new user_data_arrayType();
                    cameraAnim.MakeSerializeData(camera_anim.user_data_array, cameraAnim.UserDataArray, streams);
                }
            }

            if (camera_anims.Any())
            {
                Data.camera_anim_array = new camera_anim_arrayType()
                {
                    camera_anim = camera_anims.ToArray(),
                };
            }
            else
            {
                Data.camera_anim_array = null;
            }

            // ライトアニメーション
            var light_anims = new List<light_animType>();
            foreach (var lightAnim in LightAnims)
            {
                var light_anim = lightAnim.Data;
                light_anim.comment = lightAnim.GetComment();
                light_anim.light_name = lightAnim.Name;

                var groups = from lightAnimTarget in lightAnim.LightAnimTargets
                             let target = lightAnimTarget.targetType.ToString()
                             let range = target.LastIndexOf('_')
                             let category = range >= 0 ? target.Substring(0, range) : target
                             group lightAnimTarget by category;

                var presetTargets = LightAnimation.GetTargetTypes(lightAnim.Data.type);
                List<light_anim_targetType> targets = new List<light_anim_targetType>();
                foreach (var group in groups)
                {
                    foreach (var lightAnimTarget in group)
                    {
                        if (presetTargets.Contains(lightAnimTarget.targetType) == false)
                        {
                            continue;
                        }

                        G3dStream stream;
                        var target = lightAnimTarget.ConvertTo_light_animType(out stream, lightAnim.Data.type, viewer);
                        targets.Add(target);

                        if (stream != null)
                        {
                            target.Curve.stream_index = streams.Count;
                            streams.Add(stream);
                        }
                    }
                }

                if (lightAnim.UserDataArray.Data == null ||
                    lightAnim.UserDataArray.Data.Count == 0)
                {
                    light_anim.user_data_array = null;
                }
                else
                {
                    light_anim.user_data_array = new user_data_arrayType();
                    lightAnim.MakeSerializeData(light_anim.user_data_array, lightAnim.UserDataArray, streams);
                }

                light_anim.light_anim_target = targets.Any() ? targets.ToArray() : null;

                light_anims.Add(light_anim);
            }

            if (light_anims.Any())
            {
                Data.light_anim_array = new light_anim_arrayType()
                {
                    light_anim = light_anims.ToArray(),
                };
            }
            else
            {
                Data.light_anim_array = null;
            }

            // フォグアニメーション
            var fog_anims = new List<fog_animType>();
            foreach (var fogAnim in FogAnims)
            {
                var fog_anim = fogAnim.Data;
                fog_anim.comment = fogAnim.GetComment();
                fog_anim.fog_name = fogAnim.Name;

                var groups = from fogAnimTarget in fogAnim.FogAnimTargets
                             let target = fogAnimTarget.targetType.ToString()
                             let range = target.LastIndexOf('_')
                             let category = range >= 0 ? target.Substring(0, range) : target
                             group fogAnimTarget by category;

                List<fog_anim_targetType> targets = new List<fog_anim_targetType>();
                foreach (var group in groups)
                {
                    // グループ毎に出力するかしないかを決める。
                    if (group.All(x => x.KeyFrames.Count == 0))
                    {
                        continue;
                    }

                    foreach (var fogAnimTarget in group)
                    {
                        G3dStream stream;
                        var target = fogAnimTarget.ConvertTo_fog_animType(out stream, viewer);
                        targets.Add(target);

                        if (stream != null)
                        {
                            target.Curve.stream_index = streams.Count;
                            streams.Add(stream);
                        }
                    }
                }

                if (fogAnim.UserDataArray.Data == null ||
                    fogAnim.UserDataArray.Data.Count == 0)
                {
                    fog_anim.user_data_array = null;
                }
                else
                {
                    fog_anim.user_data_array = new user_data_arrayType();
                    fogAnim.MakeSerializeData(fog_anim.user_data_array, fogAnim.UserDataArray, streams);
                }

                fog_anim.fog_anim_target = targets.Any() ? targets.ToArray() : null;

                fog_anims.Add(fog_anim);
            }

            if (fog_anims.Any())
            {
                Data.fog_anim_array = new fog_anim_arrayType()
                {
                    fog_anim = fog_anims.ToArray(),
                };
            }
            else
            {
                Data.fog_anim_array = null;
            }


            // ユーザーデータのシリアライズ
            if (UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
            }
            else
            {
                Data.user_data_array = new user_data_arrayType();
                this.MakeSerializeData(Data.user_data_array, UserDataArray, streams);
            }

            // ストリームの設定
            {
                BinaryStreams = streams;

                // ストリームのソート(や削除等)を行う。
                nw.g3d.iflib.StreamUtility.SortStream(Data, BinaryStreams);

                Data.stream_array = null;
            }
        }

        // 編集コマンド用
        [Serializable]
        public class QuantizeToleranceData
        {
            public float QuantizeToleranceTextureScale { get; set; }
            public float QuantizeToleranceTextureRotate { get; set; }
            public float QuantizeToleranceTextureTranslate { get; set; }
        }

        public static string Group<T>(T type)
        {
            string name = type.ToString();
            int index = name.LastIndexOf('_');
            if (index >= 0)
            {
                return name.Substring(0, index);
            }
            else
            {
                return name;
            }
        }

        public static string Subscript(string group, string target)
        {
            if (group.Length >= target.Length)
            {
                return target;
            }

            var sub = target.Substring(group.Length + 1);
            if (sub.Length == 1)
            {
                return sub.ToUpper();
            }
            else
            {
                return sub;
            }
        }

        #region savedData
        private scene_anim_infoType savedData;
        string[] savedCameraAnimNames;
        string[] savedLightAnimNames;
        string[] savedFogAnimNames;

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data.scene_anim_info);

            savedCameraAnimNames = CameraAnims.Select(x => x.Name).ToArray();
            savedFogAnimNames = FogAnims.Select(x => x.Name).ToArray();
            savedLightAnimNames = LightAnims.Select(x => x.Name).ToArray();

            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            foreach (var content in ContentObjects.Where(x => x != this))
            {
                content.UpdateSavedData();
            }
        }

        public void CopySavedData(SceneAnimation source)
        {
            CopyDocumentSavedData(source);
            savedData = source.savedData;

            savedCameraAnimNames = source.savedCameraAnimNames;
            savedLightAnimNames = source.savedLightAnimNames;
            savedFogAnimNames = source.savedFogAnimNames;

            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !source.SavedUserDataArray.IsSame(UserDataArray);

            var cameraPairs = from t in CameraAnims
                              from s in source.CameraAnims
                              where t.Name == s.Name
                              select new { s, t };
            foreach (var p in cameraPairs)
            {
                p.t.CopySavedData(p.s);
            }

            var lightPairs = from t in LightAnims
                              from s in source.LightAnims
                              where t.Name == s.Name
                              select new { s, t };
            foreach (var p in lightPairs)
            {
                p.t.CopySavedData(p.s);
            }

            var fogPairs = from t in FogAnims
                           from s in source.FogAnims
                           where t.Name == s.Name
                           select new { s, t };
            foreach (var p in fogPairs)
            {
                p.t.CopySavedData(p.s);
            }
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !IsContentsNameChanged() &&
                !SceneAnimationGeneralPage.IsModified(this) &&
                !UserDataPage.IsModified(this);
        }

        public bool IsContentsNameChanged()
        {
            return !CameraAnims.Select(x => x.Name).SequenceEqual(savedCameraAnimNames) ||
                !FogAnims.Select(x => x.Name).SequenceEqual(savedFogAnimNames) ||
                !LightAnims.Select(x => x.Name).SequenceEqual(savedLightAnimNames);
        }

        public bool IsValueChanged<T>(Func<scene_anim_infoType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data.scene_anim_info));
        }

        public bool IsStringChanged(Func<scene_anim_infoType, string> select)
        {
            return select(savedData) != select(Data.scene_anim_info);
        }

        #endregion

        public override void OnDocumentSwapAdd(Document old)
        {
            CopySavedData((SceneAnimation)old);
            base.OnDocumentSwapAdd(old);
        }
    }
}
