﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using App.ConfigData;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using TeamConfig;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class LightAnimation : GuiObject, IHasUserData, ISceneAnimationObject, IQuantizeAnalysis
    {
        public light_animType Data { get; private set; }
        public SceneAnimation Owner { get; private set; }
        public override Document OwnerDocument
        {
            get { return Owner; }
        }

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public bool HasAnimation
        {
            get
            {
                return LightAnimTargets != null && LightAnimTargets.Any();
            }
        }

        // ライトアニメーションが矛盾してないか？
        public bool IsValidLightAnimations
        {
            get
            {
                var hasPosAnim = LightAnimTargets
                    .Where(x => x.targetType >= light_anim_target_targetType.position_x &&
                                x.targetType <= light_anim_target_targetType.position_z)
                    .Any(x => x.KeyFrames.Count > 0);
                var hasDirAnim = LightAnimTargets
                    .Where(x => x.targetType >= light_anim_target_targetType.direction_x &&
                                x.targetType <= light_anim_target_targetType.direction_z)
                    .Any(x => x.KeyFrames.Count > 0);
                var hasAimAnim = LightAnimTargets
                    .Where(x => x.targetType >= light_anim_target_targetType.aim_x &&
                                x.targetType <= light_anim_target_targetType.aim_z)
                    .Any(x => x.KeyFrames.Count > 0);

                if (hasPosAnim && hasDirAnim)
                {
                    // こちらのケースも不正なデータとした方がいいのか？
                    //return false;
                }
                if (hasAimAnim && hasDirAnim)
                {
                    //return false;
                }
                return true;
            }
        }

        // ファイル出力できるか？
        public string FileNotOutputMessage
        {
            get
            {
                // anim と target の両方がある場合
                if (!IsValidLightAnimations)
                {
                    return string.Format(res.Strings.FileNotOutputError_LightAnimation_LightInvalid, Name);
                }

                // 量子化エラー
                string error = null;
                Action<LightAnimationTarget> errorHandler = delegate(LightAnimationTarget animTarget)
                {
                    error = string.Format(res.Strings.FileNotOutputError_LightAnimation, Name, animTarget.targetType.ToString());
                };

                foreach (var tuple in QuantizeAnalyse(errorHandler))
                {
                    if (error != null)
                    {
                        return error;
                    }
                }

                return null;
            }
        }

        public IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse()
        {
            return QuantizeAnalyse(null);
        }

        public IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse(Action<LightAnimationTarget> errorHandler)
        {
            foreach (var animTarget in LightAnimTargets)
            {
                var curve = new LightAnimationCurveTreeNodeInfo(this, animTarget.targetType);

                IfQuantizationAnalysisResult result = null;
                try
                {
                    result = AnimationCurveEditCommand.MakeQuantizationAnalysis(this, ObjectID, curve, false);
                }
                catch (Exception)
                {
                    if (errorHandler != null)
                    {
                        errorHandler(animTarget);
                    }
                }
                yield return new Tuple<AnimTarget, IfQuantizationAnalysisResult>(animTarget, result);
            }
        }

        public LightAnimation(light_animType light_anim, SceneAnimation owner)
            : base(GuiObjectID.LightAnimation)
        {
            Name = light_anim.light_name;
            Data = light_anim;
            Owner = owner;
            MakeComment(Data.comment);

            LightAnimTargets = new List<LightAnimationTarget>();
            OldLightAnimTargets = new Dictionary<light_anim_target_targetType, List<KeyFrame>>();

            var allTargetTypes = Enum.GetValues(typeof(light_anim_target_targetType)).OfType<light_anim_target_targetType>().ToArray();
            var srcLightAnimTargets = light_anim.light_anim_target ?? Enumerable.Empty<light_anim_targetType>();
            var srcLightTargets = srcLightAnimTargets.Select(x => x.target);

            // 全てのtargettypeに対してデータにtargetTypeがない場合は作成する
            var allLightTargets =
                allTargetTypes.Select(
                    targetType =>
                    srcLightAnimTargets.FirstOrDefault(anim => anim.target == targetType)
                    ?? new light_anim_targetType() { target = targetType });

            var targetTypes = GetTargetTypes(Data.type);

            foreach (var lightAnimTarget in allLightTargets.OrderBy(x => x.target))
            {
                var target = new LightAnimationTarget()
                {
                    targetType = lightAnimTarget.target,
                };
                var defaultInterpolationType = (lightAnimTarget.target == light_anim_target_targetType.enable)
                                                   ? InterpolationType.Step
                                                   : InterpolationType.Hermite;

                if (lightAnimTarget.Curve != null)
                {
                    target.QuantizationInfo = new AnimationDocument.QuantizationInfo()
                                                  {
                                                      frame_type = lightAnimTarget.Curve.frame_type,
                                                      key_type = lightAnimTarget.Curve.key_type,
                                                      offset = lightAnimTarget.Curve.offset,
                                                      scale = lightAnimTarget.Curve.scale,
                                                  };
                }

                if (lightAnimTarget.Curve != null || srcLightTargets.Contains(target.targetType))
                {
                    target.Initialize(lightAnimTarget.Curve, lightAnimTarget.base_value, owner.BinaryStreams, defaultInterpolationType);
                }

                LightAnimTargets.Add(target);
            }

            foreach (var lightAnimTarget in allTargetTypes)
            {
                OldLightAnimTargets[lightAnimTarget] = null;
            }

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, Owner.BinaryStreams);

            UpdateSavedData();
        }

        public static IEnumerable<light_anim_target_targetType> GetTargetTypes(string type, bool allTypes = false)
        {
            var enable = new[] {
                light_anim_target_targetType.enable
            };

            var position = new[] {
                light_anim_target_targetType.position_x,
                light_anim_target_targetType.position_y,
                light_anim_target_targetType.position_z
            };

            var direction = new[] {
                light_anim_target_targetType.direction_x,
                light_anim_target_targetType.direction_y,
                light_anim_target_targetType.direction_z
            };

            var aim = new[] {
                light_anim_target_targetType.aim_x,
                light_anim_target_targetType.aim_y,
                light_anim_target_targetType.aim_z
            };

            var color0 = new[] {
                light_anim_target_targetType.color0_r,
                light_anim_target_targetType.color0_g,
                light_anim_target_targetType.color0_b
            };

            var color1 = new[] {
                light_anim_target_targetType.color1_r,
                light_anim_target_targetType.color1_g,
                light_anim_target_targetType.color1_b
            };

            var dist = new[] {
                light_anim_target_targetType.dist_attn_start,
                light_anim_target_targetType.dist_attn_end
            };

            var ang = new[] {
                light_anim_target_targetType.angle_attn_start,
                light_anim_target_targetType.angle_attn_end
            };

            IEnumerable<light_anim_target_targetType> types = new List<light_anim_target_targetType>();

            var lightPreset = ApplicationConfig.Preset.LightAnimPresets.FirstOrDefault(x => x.Type.ToLower() == type.ToLower());

            if (allTypes)
            {
                types = types.Concat(enable);
                types = types.Concat(color0);
                types = types.Concat(color1);
                types = types.Concat(position);
                types = types.Concat(direction);
                types = types.Concat(aim);
                types = types.Concat(dist);
                types = types.Concat(ang);
            }
            else if (lightPreset == null)
            {
                types = types.Concat(enable);
                types = types.Concat(color0);
                types = types.Concat(color1);
                types = types.Concat(position);
                //types = types.Concat(direction);
                types = types.Concat(aim);
                types = types.Concat(dist);
                types = types.Concat(ang);
            }
            else
            {
                foreach (var anim in lightPreset.LightAnimTargets)
                {
                    switch (anim.Target)
                    {
                        case Preset.AnimTargetEnum.enable:
                            types = types.Concat(enable);
                            break;
                        case Preset.AnimTargetEnum.position:
                            types = types.Concat(position);
                            break;
                        case Preset.AnimTargetEnum.direction:
                            types = types.Concat(direction);
                            break;
                        case Preset.AnimTargetEnum.aim:
                            types = types.Concat(aim);
                            break;
                        case Preset.AnimTargetEnum.color0:
                            types = types.Concat(color0);
                            break;
                        case Preset.AnimTargetEnum.color1:
                            types = types.Concat(color1);
                            break;
                        case Preset.AnimTargetEnum.dist_attn:
                            types = types.Concat(dist);
                            break;
                        case Preset.AnimTargetEnum.angle_attn:
                            types = types.Concat(ang);
                            break;
                    }
                }
            }

            return types.OrderBy(x => x);
        }

        /// <summary>
        /// TargetBasesを初期化します。
        /// </summary>
        public static void InitializeTargetBases()
        {
            foreach (var preset in ApplicationConfig.Preset.LightAnimPresets)
            {
                var targetBases = new Dictionary<light_anim_target_targetType, float>(DefaultBases);

                foreach (var anim in preset.LightAnimTargets)
                {
                    switch (anim.Target)
                    {
                        case Preset.AnimTargetEnum.enable:
                            {
                                var val = (anim.HasDefault() ? anim.GetDefaultBool() : null);
                                if (val != null)
                                {
                                    targetBases[light_anim_target_targetType.enable] = (val == true ? 1 : 0);
                                }
                            }
                            break;
                        case Preset.AnimTargetEnum.dist_attn:
                            {
                                var val = (anim.HasDefault() ? anim.GetDefaultFloat2() : null);
                                if (val != null)
                                {
                                    targetBases[light_anim_target_targetType.dist_attn_start] = val[0];
                                    targetBases[light_anim_target_targetType.dist_attn_end] = val[1];
                                }
                            }
                            break;
                        case Preset.AnimTargetEnum.angle_attn:
                            {
                                var val = (anim.HasDefault() ? anim.GetDefaultFloat2() : null);
                                if (val != null)
                                {
                                    targetBases[light_anim_target_targetType.angle_attn_start] = val[0];
                                    targetBases[light_anim_target_targetType.angle_attn_end] = val[1];
                                }
                            }
                            break;
                        case Preset.AnimTargetEnum.color0:
                            {
                                var val = (anim.HasDefault() ? anim.GetDefaultFloat3() : null);
                                if (val != null)
                                {
                                    targetBases[light_anim_target_targetType.color0_r] = val[0];
                                    targetBases[light_anim_target_targetType.color0_g] = val[1];
                                    targetBases[light_anim_target_targetType.color0_b] = val[2];
                                }
                            }
                            break;
                        case Preset.AnimTargetEnum.color1:
                            {
                                var val = (anim.HasDefault() ? anim.GetDefaultFloat3() : null);
                                if (val != null)
                                {
                                    targetBases[light_anim_target_targetType.color1_r] = val[0];
                                    targetBases[light_anim_target_targetType.color1_g] = val[1];
                                    targetBases[light_anim_target_targetType.color1_b] = val[2];
                                }
                            }
                            break;
                        case Preset.AnimTargetEnum.position:
                            {
                                var val = (anim.HasDefault() ? anim.GetDefaultFloat3() : null);
                                if (val != null)
                                {
                                    targetBases[light_anim_target_targetType.position_x] = val[0];
                                    targetBases[light_anim_target_targetType.position_y] = val[1];
                                    targetBases[light_anim_target_targetType.position_z] = val[2];
                                }
                            }
                            break;
                        case Preset.AnimTargetEnum.direction:
                            {
                                var val = (anim.HasDefault() ? anim.GetDefaultFloat3() : null);
                                if (val != null)
                                {
                                    targetBases[light_anim_target_targetType.direction_x] = val[0];
                                    targetBases[light_anim_target_targetType.direction_y] = val[1];
                                    targetBases[light_anim_target_targetType.direction_z] = val[2];
                                }
                            }
                            break;
                        case Preset.AnimTargetEnum.aim:
                            {
                                var val = (anim.HasDefault() ? anim.GetDefaultFloat3() : null);
                                if (val != null)
                                {
                                    targetBases[light_anim_target_targetType.aim_x] = val[0];
                                    targetBases[light_anim_target_targetType.aim_y] = val[1];
                                    targetBases[light_anim_target_targetType.aim_z] = val[2];
                                }
                            }
                            break;
                    }
                }

                TargetBases.Add(preset.Type, targetBases);
            }
        }

        /// <summary>
        /// デフォルト値を取得します。
        /// </summary>
        /// <param name="type">ライトタイプ</param>
        /// <returns>デフォルト値</returns>
        public static Dictionary<light_anim_target_targetType, float> GetDefaultBases(string type)
        {
            Dictionary<light_anim_target_targetType, float> defaultBases;
            bool find = TargetBases.TryGetValue(type, out defaultBases);

            if (find)
            {
                return defaultBases;
            }
            else
            {
                return DefaultBases;
            }
        }

        /// <summary>
        /// ライトアニメーションターゲットの既定のデフォルト値。
        /// </summary>
        public static Dictionary<light_anim_target_targetType, float> DefaultBases = new Dictionary<light_anim_target_targetType, float>()
        {
            { light_anim_target_targetType.enable, 1 },
            { light_anim_target_targetType.position_x, 0 },
            { light_anim_target_targetType.position_y, 0 },
            { light_anim_target_targetType.position_z, 0 },
            { light_anim_target_targetType.direction_x, 0 },
            { light_anim_target_targetType.direction_y, 0 },
            { light_anim_target_targetType.direction_z, 0 },
            { light_anim_target_targetType.aim_x, 0 },
            { light_anim_target_targetType.aim_y, 0 },
            { light_anim_target_targetType.aim_z, 0 },
            { light_anim_target_targetType.dist_attn_start, 0 },
            { light_anim_target_targetType.dist_attn_end, 0 },
            { light_anim_target_targetType.angle_attn_start, 0 },
            { light_anim_target_targetType.angle_attn_end, 0 },
            { light_anim_target_targetType.color0_r, 0 },
            { light_anim_target_targetType.color0_g, 0 },
            { light_anim_target_targetType.color0_b, 0 },
            { light_anim_target_targetType.color1_r, 0 },
            { light_anim_target_targetType.color1_g, 0 },
            { light_anim_target_targetType.color1_b, 0 },
        };

        /// <summary>
        /// ライトアニメーションのタイプごとのデフォルト値。
        /// </summary>
        public static Dictionary<string, Dictionary<light_anim_target_targetType, float>> TargetBases =
            new Dictionary<string, Dictionary<light_anim_target_targetType, float>>();

        public LightAnimationTarget GetTarget(light_anim_target_targetType targetType)
        {
            return LightAnimTargets.FirstOrDefault(x => x.targetType == targetType);
        }

        public LightAnimationTarget GetAnimTarget(LightAnimationCurveTreeNodeInfo info)
        {
            return GetTarget((light_anim_target_targetType)info.ComponentIndex);
        }
        public List<LightAnimationTarget> LightAnimTargets;

        public Dictionary<light_anim_target_targetType, List<KeyFrame>> OldLightAnimTargets;

        #region savedData
        private light_animType savedData;
        public List<LightAnimationTarget> savedLightAnimTargets;

        /// <summary>
        /// 空以外の保存されたカーブ
        /// </summary>
        public int SavedCurveCount { get; private set; }

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data);
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            SavedCurveCount = 0;
            foreach (var animTarget in LightAnimTargets)
            {
                animTarget.IsModified = false;
                if (NotEmpty(animTarget))
                {
                    animTarget.IsSaved = true;
                    SavedCurveCount++;
                }
                else
                {
                    animTarget.IsSaved = false;
                }
            }
            savedLightAnimTargets = ObjectUtility.Clone(LightAnimTargets);
        }

        public void CopySavedData(LightAnimation source)
        {
            CopyGuiObjectSavedData(source);
            savedData = source.savedData;
            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !source.SavedUserDataArray.IsSame(UserDataArray);

            SavedCurveCount = source.SavedCurveCount;
            savedLightAnimTargets = source.savedLightAnimTargets;
            UpdateIsModifiedAnimTargetAll();
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !LightAnimationGeneralPage.IsModified(this) &&
                !IsCurvesModified() &&
                !UserDataPage.IsModified(this);
        }

        public void UpdateIsModifiedAnimTargetAll()
        {
            foreach (var animTarget in LightAnimTargets)
            {
                var savedAnimTarget = savedLightAnimTargets.FirstOrDefault(x => x.targetType == animTarget.targetType);
                UpdateIsModifiedAnimTarget(animTarget, savedAnimTarget);
            }
        }

        /// <summary>
        /// カーブの変更フラグを更新する
        /// </summary>
        public void UpdateIsModifiedAnimTarget(LightAnimationTarget current, LightAnimationTarget saved)
        {
            if (saved == null)
            {
                current.IsModified = NotEmpty(current);
                current.IsSaved = false;
            }
            else
            {
                current.IsModified = !current.IsSame(saved);
                current.IsSaved = saved.IsSaved;
            }
        }

        public bool NotEmpty(LightAnimationTarget current)
        {
            return current.KeyFrames.Count > 0;
        }

        public bool IsValueChanged<T>(Func<light_animType, T> select)
        {
            return !select(savedData).Equals(select(Data));
        }

        public bool IsCurvesModified()
        {
            int savedCount = 0;
            foreach (var animTarget in LightAnimTargets)
            {
                if (animTarget.IsModified)
                {
                    return true;
                }
                if (animTarget.IsSaved)
                {
                    savedCount++;
                }
            }

            return savedCount != SavedCurveCount;
        }
        #endregion
    }


    [Serializable]
    public class LightAnimationTarget : AnimTarget
    {
        public light_anim_target_targetType targetType;
        public light_anim_targetType ConvertTo_light_animType(out G3dStream stream, string lightType, bool viewer = false)
        {
            IG3dQuantizedCurve curve;
            float base_value;

            if (KeyFrames.Count != 0)
            {
                Convert(out curve, out stream, viewer);
                base_value = GetBaseValue();
            }
            else
            {
                curve = null;
                stream = null;
                base_value = LightAnimation.GetDefaultBases(lightType)[targetType];
            }

            var target = new light_anim_targetType()
            {
                base_value = GetBaseValue(),
                Curve = curve,
                target = targetType,
            };
            return target;
        }
    }


    enum StandardLightType
    {
        ambient,
        directional,
        point,
        spot
    }
}
