﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class BoneVisibilityAnimation : AnimationDocument, IHasUserData
    {
        public bone_visibility_animType Data{ get; private set; }
        public override object nw4f_3difItem
        {
            get { return Data; }
        }

        public override process_log_arrayType process_log_array
        {
            get { return Data.process_log_array; }
        }

        public override int FrameCount
        {
            get{ return Data.bone_visibility_anim_info.frame_count; }
            set{ Data.bone_visibility_anim_info.frame_count = value; }
        }

        public override bool Loop
        {
            get { return Data.bone_visibility_anim_info.loop; }
            set { Data.bone_visibility_anim_info.loop = value; }
        }

        public override bool IsFrameCountModified
        {
            get { return savedData.frame_count != Data.bone_visibility_anim_info.frame_count; }
        }

        public override bool IsLoopModified
        {
            get { return savedData.loop != Data.bone_visibility_anim_info.loop; }
        }

        // ファイル出力できるか？
        public override bool IsFileOutputable
        {
            get
            {
                return FileNotOutputErrorMessage == null;
            }
        }

        public override string FileNotOutputErrorMessage
        {
            get
            {
                string error = null;
                Action<BoneVisibilityBoneAnim> errorHandler = delegate(BoneVisibilityBoneAnim boneVisibilityBoneAnim)
                {
                    error = string.Format(res.Strings.FileNotOutputError_CurveAnimationDocument, boneVisibilityBoneAnim.bone_name);
                };
                foreach (var tuple in QuantizeAnalyse(errorHandler))
                {
                    if (error != null)
                    {
                        return error;
                    }
                }

                return null;
            }
        }

        public override bool HasAnimation
        {
            get { return BoneVisibilityBoneAnims != null && BoneVisibilityBoneAnims.Any(); }
        }

        public override IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse()
        {
            return QuantizeAnalyse(null);
        }

        private IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse(Action<BoneVisibilityBoneAnim> errorHandler)
        {
            foreach (var boneVisibilityBoneAnim in BoneVisibilityBoneAnims)
            {
                var curve = new BoneVisibilityAnimationCurveTreeNodeInfo(this, boneVisibilityBoneAnim.bone_name);

                IfQuantizationAnalysisResult result = null;
                try
                {
                    result = AnimationCurveEditCommand.MakeQuantizationAnalysis(this, ObjectID, curve, false);
                }
                catch (Exception)
                {
                    if (errorHandler != null)
                    {
                        errorHandler(boneVisibilityBoneAnim);
                    }
                }
                yield return new Tuple<AnimTarget, IfQuantizationAnalysisResult>(boneVisibilityBoneAnim, result);
            }
        }

        public override bool IsEditable
        {
            get
            {
                // 注意：
                // BoneVisibilityAnimationCurveEditPage の UpdateTreeView() をもとに実装しました
                // 方針：
                // なるべく元のソースをいじらないように実装

                bool IsBound = true;
                var ActiveTarget = this;

                var bones = new HashSet<string>((from model in DocumentManager.Models
                                                 where model.AllAnimations.Contains(ActiveTarget)
                                                 from bone in model.Bones
                                                 select bone.Name).Distinct());

                foreach (var boneVisibilityBoneAnim in ActiveTarget.BoneVisibilityBoneAnims)
                {
                    var boneName = boneVisibilityBoneAnim.bone_name;

                    IsBound = bones.Contains(boneName);
                    if (IsBound == false)
                    {
                        return IsBound;
                    }
                }

                Debug.Assert(IsBound);
                return IsBound;
            }
        }

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        [Serializable]
        public sealed class BoneVisibilityBoneAnim : AnimTarget
        {
            public BoneVisibilityBoneAnim()
            {
                KeyFrames = new List<KeyFrame>();
                QuantizationInfo = new QuantizationInfo();
                matrix_index = new[]{-1, -1};
                binarize_visibility = true;
                CurveInterpolationType = InterpolationType.Step;
            }

            public string			bone_name{			get; set; }				// bone_vis_bone_anim.bone_name
            public string			parent_name{		get; set; }				// bone_vis_bone_anim.parent_name
            public bool				rigid_body{			get; set; }				// bone_vis_bone_anim.rigid_body
            public int[]			matrix_index{		get; set; }				// bone_vis_bone_anim.matrix_index
            public bool             binarize_visibility { get; set; }			// bone_vis_bone_anim.binarize_visibility
            public bool             compress_enable { get; set; }				// bone_vis_bone_anim.compress_enable

            public bool				render_matrix { get { return (rigid_body || (matrix_index[0] != -1) || (matrix_index[1] != -1)); } }

            public override bool IsSame(AnimTarget animTarget)
            {
                var boneAnim = animTarget as BoneVisibilityBoneAnim;
                return base.IsSame(animTarget) &&
                    boneAnim.binarize_visibility == binarize_visibility &&
                    boneAnim.compress_enable == compress_enable;
            }
        }

        public List<BoneVisibilityBoneAnim>	BoneVisibilityBoneAnims{ get; set; }

        #region 圧縮率
        public bool	IsAllFrameCurveConstant { get { return BoneVisibilityBoneAnims.Any(x => x.KeyFrames.Any()) == false; } }
        public int	FrameCompressDataSize   { get { return GetFrameDataSize(false); } }
        public int	FrameUncompressDataSize { get { return GetFrameDataSize(true);  } }
        #endregion

        // 編集コマンド用
        [Serializable]
        public class QuantizeToleranceData
        {
            public List<BoneVisibilityBoneAnim>		BoneVisibilityBoneAnims{ get; set; }

            public float	QuantizeToleranceFrame{ get; set; }
        }

        public BoneVisibilityAnimation(bone_visibility_animType animation, List<G3dStream> streams) : base(GuiObjectID.BoneVisibilityAnimation, streams)
        {
            Data = animation;
            Initialize();
        }

        public void Initialize()
        {
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, BinaryStreams);

            MakeBoneVisibilityBoneAnims();
        }

        #region Bridge
        public class BoneVisibilitySwapData : SwapData
        {
            public ToolData ToolData;
            public List<G3dStream> streams;
            public string Comment;
            public RgbaColor? EditColor;
            public bone_visibility_animType Data;
            public List<BoneVisibilityBoneAnim> BoneVisibilityBoneAnims;
            public UserDataArray userData;
        }

        public override SwapData CreateSwapData(object data, List<G3dStream> streams)
        {
            var current = GetSwapData();
            Data = (bone_visibility_animType)data;
            BinaryStreams = streams;
            Initialize();

            return Swap(current, false);
        }

        public override SwapData GetSwapData()
        {
            return new BoneVisibilitySwapData()
            {
                Data = Data,
                streams = BinaryStreams,
                Comment = Comment,
                EditColor = EditColor,
                BoneVisibilityBoneAnims = BoneVisibilityBoneAnims,
                userData = UserDataArray,
            };
        }

        public override SwapData Swap(SwapData swapData, bool copyHistory)
        {
            var current = (BoneVisibilitySwapData)GetSwapData();
            var data = (BoneVisibilitySwapData)swapData;
            Data = data.Data;
            BinaryStreams = data.streams;
            Comment = data.Comment;
            EditColor = data.EditColor;
            BoneVisibilityBoneAnims = data.BoneVisibilityBoneAnims;
            UserDataArray = data.userData;

            if (copyHistory)
            {
                UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);
                UpdateIsModifiedAnimTargetAll();
            }

            return current;
        }
        #endregion

        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            lock (this)
            {
                UpdateData(viewer);

                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        public void Optimize()
        {
            // ストリームを更新する。
            UpdateData();

            // <hermite_curve>、<linear_curve>、<step_curve> の frame_type と key_type を
            // none に設定しないと、再分析が行われません。
            if (Data.bone_vis_bone_anim_array != null)
            {
                foreach (var anim in Data.bone_vis_bone_anim_array.bone_vis_bone_anim)
                {
                    if (anim.step_curve != null)
                    {
                        anim.step_curve.frame_type = curve_frame_typeType.none;
                        anim.step_curve.key_type = curve_key_typeType.none;
                    }
                }
            }

            try
            {
                // アニメーションに合わせたAnimQuantizationAnalysisOptimizerを呼び出す。
                IfBoneVisibilityAnimQuantizationAnalysisOptimizer optimizer = new IfBoneVisibilityAnimQuantizationAnalysisOptimizer();
                optimizer.Optimize(Data, BinaryStreams);
            }
            catch(Exception e)
            {
                DebugConsole.WriteLine("Exception Catch : IfBoneVisibilityAnimQuantizationAnalysisOptimizer.Optimize() : {0}", e.Message);
            }

            // 最適化後に、データを再作成する。
            this.MakeUserData(Data.user_data_array, BinaryStreams);
            MakeBoneVisibilityBoneAnims();
        }

        private int GetFrameDataSize(bool isBase)
        {
            int size = 0;
            {
                foreach(var boneVisibilityBoneAnim in BoneVisibilityBoneAnims)
                {
                    size += boneVisibilityBoneAnim.KeyFrames.Count * (isBase ? 4 : IfUtility.GetByteSizeFromCurveFrameType(boneVisibilityBoneAnim.QuantizationInfo.frame_type));
                }
            }
            return size;
        }

        public BoneVisibilityBoneAnim GetParamAnimFromId(string boneName)
        {
            return BoneVisibilityBoneAnims.FirstOrDefault(x => x.bone_name == boneName);
        }

        private void MakeBoneVisibilityBoneAnims()
        {
            BoneVisibilityBoneAnims = new List<BoneVisibilityBoneAnim>();

            if ((Data.bone_vis_bone_anim_array == null) ||
                (Data.bone_vis_bone_anim_array.bone_vis_bone_anim == null))
            {
                return;
            }

            foreach (var boneVisBoneAnim in Data.bone_vis_bone_anim_array.bone_vis_bone_anim)
            {
                var curve = boneVisBoneAnim.step_curve;

                bool exist = true;
                if (curve == null)
                {
                    curve = new step_curveType();
                    exist = false;
                }

                var newBoneVisibilityBoneAnim = new BoneVisibilityBoneAnim()
                {
                    bone_name		= boneVisBoneAnim.bone_name,
                    parent_name		= boneVisBoneAnim.parent_name,
                    rigid_body		= boneVisBoneAnim.rigid_body,
                    matrix_index	= ObjectUtility.Clone(boneVisBoneAnim.matrix_index),
                    binarize_visibility = boneVisBoneAnim.binarize_visibility,
                    compress_enable	= boneVisBoneAnim.compress_enable,
                    //
                    QuantizationInfo = new QuantizationInfo()
                    {
                        frame_type	= curve.frame_type,
                        key_type	= curve.key_type,
                        scale		= curve.scale,
                        offset		= curve.offset,
                    },
                    pre_wrap	= curve.pre_wrap,
                    post_wrap	= curve.post_wrap,
                    baked       = curve.baked
                };

                if (exist)
                {
                    const int dataElementCount = 2;		// frame, value
                    var stream = BinaryStreams[curve.stream_index];
                    var data = stream.FloatData;
                    Debug.Assert(data != null);
                    Debug.Assert(data.Count == curve.count * dataElementCount);

                    for (int i = 0; i != curve.count; ++i)
                    {
                        newBoneVisibilityBoneAnim.KeyFrames.Add(
                            new KeyFrame()
                            {
                                Frame = data[i * dataElementCount + 0],
                                Value = data[i * dataElementCount + 1],
                            }
                        );
                    }
                }
                else
                {
                    // コンスタントカーブの場合、中間ファイルにはカーブは存在しないので、
                    // base_value を使用して、0フレームにキーフレームを生成する。
                    newBoneVisibilityBoneAnim.KeyFrames.Add(
                        new KeyFrame()
                        {
                            Frame = 0.0f,
                            Value = boneVisBoneAnim.base_value
                        }
                    );
                }

                BoneVisibilityBoneAnims.Add(newBoneVisibilityBoneAnim);
            }
        }

        public void UpdateData(bool viewer = false)
        {
            ToolData.Load(Data.tool_data);
            Data.comment = GetComment();

            if (Data.bone_vis_bone_anim_array == null)
            {
                Data.bone_vis_bone_anim_array = new bone_vis_bone_anim_arrayType();
            }

            var	tempBoneVisBoneAnims	= new List<bone_vis_bone_animType>();
            var	tempStreams				= new List<G3dStream>();

            int targetIndex = 0;    // stream_indexとして使用します。
            foreach (var boneVisibilityBoneAnim in BoneVisibilityBoneAnims)
            {
                // データ
                {
                    if (boneVisibilityBoneAnim.KeyFrames.Any() == false)
                    {
                        // 保存しない
                        continue;
                    }

                    var boneVisBoneAnim = new bone_vis_bone_animType()
                    {
                        bone_name		= boneVisibilityBoneAnim.bone_name,
                        parent_name		= boneVisibilityBoneAnim.parent_name,
                        rigid_body		= boneVisibilityBoneAnim.rigid_body,
                        matrix_index	= ObjectUtility.Clone(boneVisibilityBoneAnim.matrix_index),
                        binarize_visibility = boneVisibilityBoneAnim.binarize_visibility,
                        compress_enable = boneVisibilityBoneAnim.compress_enable,
                        base_value		= boneVisibilityBoneAnim.GetBaseValue(),
                    };

                    if (boneVisibilityBoneAnim.KeyFrames.Any())
                    {
                        boneVisBoneAnim.step_curve = new step_curveType()
                        {
                            count			= boneVisibilityBoneAnim.KeyFrames.Count,
                            frame_type		= boneVisibilityBoneAnim.QuantizationInfo.frame_type,
                            key_type		= boneVisibilityBoneAnim.QuantizationInfo.key_type,
                            scale			= boneVisibilityBoneAnim.QuantizationInfo.scale,
                            offset			= boneVisibilityBoneAnim.QuantizationInfo.offset,
                            pre_wrap		= boneVisibilityBoneAnim.pre_wrap,
                            post_wrap		= boneVisibilityBoneAnim.post_wrap,
                            baked           = boneVisibilityBoneAnim.baked,
                            stream_index = targetIndex
                        };
                    }

                    tempBoneVisBoneAnims.Add(boneVisBoneAnim);
                }

                // ストリーム
                if (boneVisibilityBoneAnim.KeyFrames.Any())
                {
                    // コンスタントカーブかどうかをチェックする
                    {
                        float base_value = 0.0f;
                        if (boneVisibilityBoneAnim.KeyFrames.IsConstantCurve(out base_value))
                        {
                            if (viewer)
                            {
                                // ランタイム連携用に水平なカーブを出力する
                                tempBoneVisBoneAnims.Last().step_curve = new step_curveType()
                                {
                                    count = 2,
                                    frame_type = curve_frame_typeType.none,
                                    key_type = curve_key_typeType.none,
                                    scale = 1,
                                    offset = 0,
                                    pre_wrap = curve_wrapType.clamp,
                                    post_wrap = curve_wrapType.clamp,
                                    baked = false,
                                    stream_index = targetIndex,
                                };
                                var stream = new G3dStream()
                                {
                                    type = stream_typeType.@float,
                                    column = 2,
                                };

                                stream.FloatData.Add(0); // フレーム
                                stream.FloatData.Add(base_value);

                                stream.FloatData.Add(1); // フレーム
                                stream.FloatData.Add(base_value);
                                tempStreams.Add(stream);
                                ++targetIndex;
                            }
                            else
                            {
                                // コンスタントカーブなのでベース値を保存し、カーブを削除する
                                tempBoneVisBoneAnims.Last().step_curve = null;
                                tempBoneVisBoneAnims.Last().base_value = base_value;
                            }

                            continue;
                        }
                    }

                    var tempStream = new G3dStream()
                    {
                        type	= stream_typeType.@float,
                        column	= 2
                    };

                    foreach(var keyFrame in boneVisibilityBoneAnim.KeyFrames)
                    {
                        tempStream.FloatData.Add(keyFrame.Frame);
                        tempStream.FloatData.Add(keyFrame.Value);
                    }

                    tempStreams.Add(tempStream);
                    ++targetIndex;
                }
            }

            //  リストを配列に変換して設定
            {
                Data.bone_vis_bone_anim_array.length = tempBoneVisBoneAnims.Count;
                Data.bone_vis_bone_anim_array.bone_vis_bone_anim = tempBoneVisBoneAnims.ToArray();

                if (Data.bone_vis_bone_anim_array.length == 0)
                {
                    Data.bone_vis_bone_anim_array = null;
                }
            }

            // ユーザーデータのシリアライズ
            if (UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
            }
            else
            {
                Data.user_data_array = new user_data_arrayType();
                this.MakeSerializeData(Data.user_data_array, UserDataArray, tempStreams);
            }

            // ストリームの設定
            {
                BinaryStreams = tempStreams;

                // ストリームのソート(や削除等)を行う。
                nw.g3d.iflib.StreamUtility.SortStream(Data, BinaryStreams);

                Data.stream_array = null;
            }
        }

        public override EditCommand CreateUpdateBindCommand()
        {
            bool update = false;
            var newBoneVisibilityBoneAnims = new List<BoneVisibilityBoneAnim>();

            // バインドされたモデルのボーン
            var bones = (from model in DocumentManager.Models
                             where model.AllAnimations.Contains(this)
                             from bone in model.Bones
                             select bone).ToArray();

            var names = new HashSet<string>(bones.Select(x => x.Name).Distinct());

            var oldBoneAnims = new List<BoneVisibilityBoneAnim>();
            var oldBoneAnimDictionary = new Dictionary<string, BoneVisibilityBoneAnim>();

            // 既存のもの
            foreach (var anim in BoneVisibilityBoneAnims)
            {
                if (names.Contains(anim.bone_name) || anim.ExportType != CurveExportType.Ignored)
                {
                    oldBoneAnims.Add(anim);
                    oldBoneAnimDictionary.Add(anim.bone_name, anim);
                }
                else
                {
                    update = true;
                }
            }

            // ボーン
            foreach (var bone in bones)
            {
                if (names.Contains(bone.Name))
                {
                    if (oldBoneAnimDictionary.ContainsKey(bone.Name))
                    {
                        newBoneVisibilityBoneAnims.Add(oldBoneAnimDictionary[bone.Name]);
                        oldBoneAnimDictionary.Remove(bone.Name);
                    }
                    else
                    {
                        newBoneVisibilityBoneAnims.Add(new BoneVisibilityBoneAnim()
                        {
                            bone_name = bone.Name,
                            parent_name = bone.Data.parent_name,
                            rigid_body = bone.Data.rigid_body,
                            matrix_index = bone.Data.matrix_index.ToArray(),
                            binarize_visibility = true,
                            compress_enable = bone.Data.compress_enable,
                        });
                        update = true;
                    }
                    names.Remove(bone.Name);
                }
            }

            // 既存のもので追加されていないもの
            foreach (var anim in oldBoneAnims)
            {
                if (oldBoneAnimDictionary.ContainsKey(anim.bone_name))
                {
                    newBoneVisibilityBoneAnims.Add(anim);
                }
            }

            if (update)
            {
                // 編集フラグを更新しない
                return new GeneralGroupReferenceEditCommand<DummyObject>(
                    DummyObject.TheDummyObjectGroup,
                    GuiObjectID.DummyObject,
                    Enumerable.Repeat<DummyObject>(DummyObject.TheDummyObject, 1),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var temp = BoneVisibilityBoneAnims;
                        BoneVisibilityBoneAnims = newBoneVisibilityBoneAnims;
                        newBoneVisibilityBoneAnims = temp;

                        // 変更マークを更新する
                        UpdateIsModifiedAnimTargetAll();

                        // シリアライズされるデータに変更はないので Viewer に再転送する必要はない
                    });
            }

            return null;
        }

        #region savedData
        private bone_visibility_anim_infoType savedData;
        public List<BoneVisibilityBoneAnim> savedBoneVisiblityBoneAnims { get; private set; }

        /// <summary>
        /// 空以外の保存されたカーブ
        /// </summary>
        public int SavedCurveCount { get; private set; }

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data.bone_visibility_anim_info);
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            SavedCurveCount = 0;
            foreach (var animTarget in BoneVisibilityBoneAnims)
            {
                animTarget.IsModified = false;
                if (NotEmpty(animTarget))
                {
                    animTarget.IsSaved = true;
                    SavedCurveCount++;
                }
                else
                {
                    animTarget.IsSaved = false;
                }
            }
            savedBoneVisiblityBoneAnims = ObjectUtility.Clone(BoneVisibilityBoneAnims);
        }

        public void CopySavedData(BoneVisibilityAnimation source)
        {
            CopyDocumentSavedData(source);
            savedData = source.savedData;

            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);

            SavedCurveCount = source.SavedCurveCount;
            savedBoneVisiblityBoneAnims = source.savedBoneVisiblityBoneAnims;

            UpdateIsModifiedAnimTargetAll();
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !BoneVisibilityAnimationGeneralPage.IsModified(this) &&
                !IsCurvesModified() &&
                !UserDataPage.IsModified(this);
        }

        public void UpdateIsModifiedAnimTargetAll()
        {
            foreach (var boneVisibilityBoneAnim in BoneVisibilityBoneAnims)
            {
                var saved = savedBoneVisiblityBoneAnims.FirstOrDefault(x => x.bone_name == boneVisibilityBoneAnim.bone_name);
                UpdateIsModifiedAnimTarget(boneVisibilityBoneAnim, saved);
            }
        }

        /// <summary>
        /// カーブの変更フラグを更新する
        /// </summary>
        public void UpdateIsModifiedAnimTarget(BoneVisibilityBoneAnim current, BoneVisibilityBoneAnim saved)
        {
            if (saved == null)
            {
                current.IsModified = NotEmpty(current);
                current.IsSaved = false;
            }
            else
            {
                current.IsModified = !current.IsSame(saved);
                current.IsSaved = saved.IsSaved;
            }
        }

        public bool NotEmpty(BoneVisibilityBoneAnim current)
        {
            return current.KeyFrames.Count > 0;
        }

        public bool IsValueChanged<T>(Func<bone_visibility_anim_infoType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data.bone_visibility_anim_info));
        }

        public bool IsStringChanged(Func<bone_visibility_anim_infoType, string> select)
        {
            return select(savedData) != select(Data.bone_visibility_anim_info);
        }

        public bool IsCurvesModified()
        {
            int savedCount = 0;
            foreach (var animTarget in BoneVisibilityBoneAnims)
            {
                if (animTarget.IsModified)
                {
                    return true;
                }
                if (animTarget.IsSaved)
                {
                    savedCount++;
                }
            }

            return savedCount != SavedCurveCount;
        }
        #endregion

        public override void OnDocumentSwapAdd(Document old)
        {
            CopySavedData((BoneVisibilityAnimation)old);
            base.OnDocumentSwapAdd(old);
        }
    }
}
