﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System;
using App.Command;
using App.Properties;
using App.Utility;
using ConfigCommon;

namespace App.Data
{
    /// <summary>
    /// アニメーションセット
    /// </summary>
    public class AnimationSet : GuiObject
    {
        public bool IsSceneAnimationSet { get; private set; }
        public bool IsDefaultAnimationSet { get; set; }

        public override bool Editable  { get { return false; } }
        public override bool Renamable { get { return IsDefaultAnimationSet == false; } }

        public bool SuppressOpenTreeNode { get; set; }

        private List<AnimationSetItem> animations = new List<AnimationSetItem>();
        virtual public List<AnimationSetItem> Animations
        {
            get { return animations; }
            set { animations = value; }
        }

        private Image objectIcon_;
        public override Image ObjectIcon
        {
            get
            {
                if (IsDefaultAnimationSet)
                {
                    if (objectIcon_ == null)
                    {
                        objectIcon_ = Resources.GuiObject_AnimationSet_DefaultAnimationSet;
                        Debug.Assert(objectIcon_ != null);
                    }

                    return objectIcon_;
                }
                else
                {
                    return base.ObjectIcon;
                }

            }
        }

        public override Document OwnerDocument
        {
            get { return DocumentManager.ProjectDocument; }
        }

        public AnimationSet(bool sceneAnimationSet) : base(GuiObjectID.AnimationSet)
        {
            IsSceneAnimationSet = sceneAnimationSet;
            SuppressOpenTreeNode = false;
            SetMaybeModified();
        }

        public override ICommand CreateReferenceObjectRenameCommand(Document srcObject, string oldName, string oldDirectory, string newName, string newExt, string newDirectory)
        {
            var commandSet = new EditCommandSet();
            {
                var newAnims = new List<AnimationSetItem>();
                {
                    foreach(var anim in Animations)
                    {
                        if (anim == new AnimationSetItem(oldName + srcObject.FileDotExt, oldDirectory))
                        {
                            newAnims.Add(new AnimationSetItem(newName + newExt, newDirectory));
                        }
                        else
                        {
                            newAnims.Add(anim);
                        }
                    }
                }

                commandSet.Add(
                    new GeneralGroupReferenceEditCommand<List<AnimationSetItem>>(
                        new GuiObjectGroup(this),
                        ObjectID,
                        Enumerable.Repeat(newAnims, 1),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            var animSet = target as AnimationSet;

                            swap				= ObjectUtility.Clone(animSet.Animations);
                            animSet.Animations	= (List<AnimationSetItem>)data;
                            if (animSet == DocumentManager.PreviewSceneAnimSet)
                            {
                                DocumentManager.PreviewSceneAnimSetUpdated();
                            }
                        }
                    )
                );
            }
            return (commandSet.CommandCount > 0) ? commandSet : null;
        }

        #region savedData
        public override bool CheckOwnerDocumentSavedContents()
        {
            return false;
        }

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
        }

        public override bool EqualsToSavedData()
        {
            var animations = DocumentManager.ProjectDocument.GetSavedAnimationSet(this);
            if (animations == null)
            {
                return IsDefaultAnimationSet && Animations.Count == 0;
            }

            return Animations.SequenceEqual(animations);
        }
        #endregion


    }

    // string のみを使っていた時からの互換性を確保するために struct にした
    [Serializable]
    public struct AnimationSetItem
    {
        public AnimationSetItem(string name, string directory)
        {
            Name = name;
            Directory = directory;
        }

        public readonly string Name;
        public readonly string Directory;

        public static bool operator ==(AnimationSetItem lhs, AnimationSetItem rhs)
        {
            return lhs.Name == rhs.Name && lhs.Directory.Equals(rhs.Directory, StringComparison.OrdinalIgnoreCase);
        }

        public static bool operator !=(AnimationSetItem lhs, AnimationSetItem rhs)
        {
            return !(lhs == rhs);
        }

        public override bool Equals(object obj)
        {
            if (!(obj is AnimationSetItem))
            {
                return false;
            }

            var item = (AnimationSetItem)obj;

            return this == item;
        }

        public override int GetHashCode()
        {
            unchecked
            {
                return Name.GetHashCode() * 31 + Directory.ToLower().GetHashCode();
            }
        }
    }
}
