﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// ビュータイトルバークラス。
    /// </summary>
    [ToolboxItem(true)]
    public sealed class ViewTitleBar : UnselectableControl
    {
        // テキスト配置方向
        private HorizontalAlignment _textAlign = HorizontalAlignment.Left;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ViewTitleBar()
        {
        }

        private bool useColor;
        public bool UseColor
        {
            get
            {
                return useColor;
            }
            set
            {
                useColor = value;
                Invalidate();
            }
        }

        /// <summary>
        /// テキスト配置方向。
        /// </summary>
        [DefaultValue(HorizontalAlignment.Left)]
        [Description("テキストの配置方向を示します。")]
        public HorizontalAlignment TextAlign
        {
            get { return _textAlign; }
            set
            {
                _textAlign = value;
                Invalidate();
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string Text
        {
            get { return base.Text; }
            set
            {
                base.Text = value;
                Invalidate();
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(120, 18); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            Rectangle rcClient = ClientRectangle;

            // 背景
            using (Brush brush = CreateBackBrush(rcClient))
            {
                e.Graphics.FillRectangle(brush, rcClient);
            }

            // 枠
            {
                GraphicsUtility.DrawRectangle(e.Graphics, SystemPens.Control, rcClient);
                if (UseColor)
                {
                    using (Pen pen = new Pen(BackColor))
                    {
//						GraphicsUtility.DrawRoundishRectangle(e.Graphics, pen, rcClient);
                        GraphicsUtility.DrawRectangle(e.Graphics, pen, rcClient);
                    }
                }
                else
                {
//					GraphicsUtility.DrawRoundishRectangle(e.Graphics, SystemPens.ControlDark, rcClient);
                    GraphicsUtility.DrawRectangle(e.Graphics, SystemPens.ControlDark, rcClient);
                }
            }

            // テキスト
            using (StringFormat sf = new StringFormat())
            {
                sf.LineAlignment = StringAlignment.Center;
                sf.Trimming      = StringTrimming.EllipsisCharacter;
                sf.FormatFlags  |= StringFormatFlags.NoWrap;

                switch (_textAlign)
                {
                    case HorizontalAlignment.Left:   sf.Alignment = StringAlignment.Near;   break;
                    case HorizontalAlignment.Center: sf.Alignment = StringAlignment.Center; break;
                    case HorizontalAlignment.Right:  sf.Alignment = StringAlignment.Far;    break;
                    default: break;
                }

                Rectangle rcText = rcClient;
                rcText.Inflate(-1, -1);
                RectangleUtility.OffsetLeft(ref rcText, 2);

                if (UseColor)
                {
                    using (Brush brush = new SolidBrush(ForeColor))
                    {
                        e.Graphics.DrawString(Text, Font, brush, rcText, sf);
                    }
                }
                else
                {
                    e.Graphics.DrawString(Text, Font, SystemBrushes.WindowText, rcText, sf);
                }
            }
        }

        private Brush CreateBackBrush(Rectangle rcClient)
        {
            return
                UseColor ?
                    (Brush)(new SolidBrush(BackColor)) :
                    (Brush)(new LinearGradientBrush(rcClient, ControlPaint.Light(SystemColors.Control), SystemColors.Control, LinearGradientMode.Vertical));
        }
    }
}
