﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;

namespace App.Controls
{
    /// <summary>
    /// 値編集パネルクラス。
    /// </summary>
    [ToolboxItem(false)]
    [DefaultEvent("SequentialValueChanged")]
    public abstract partial class ValueEditPanel : UIUserControl
    {
        // スライダーボタン表示フラグ
        private bool _showSliderButton = true;
        // スライダー範囲調節フラグ
        private bool _sliderRangeAdjustable = false;
        // スライダー表示中フラグ
        private bool _sliderShowing = false;

        /// <summary>アップダウン。</summary>
        protected readonly UpDownBase _upDown;
        /// <summary>スライダー。</summary>
        protected ValueEditPopupSlider _slider = null;
        /// <summary>同調コントロールリスト。</summary>
        protected readonly List<ValueEditPanel> _buddyControls = new List<ValueEditPanel>();

        private static readonly Bitmap imgValueEditPanelSlider_ = App.Properties.Resources.Control_ValueEditPanelSlider;
        private readonly UIButton btnSlider;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ValueEditPanel(UpDownBase upDown)
        {
            InitializeComponent();

            // アップダウン
            _upDown = upDown;
            _upDown.Parent = this;
            _upDown.Location = new Point(0, 1);
            _upDown.Width = Width - (120 - 92);
            _upDown.Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right;
            if (_upDown is UINumericUpDown)
            {
                ((UINumericUpDown)_upDown).SequentialValueChanged += new SequentialValueChangedEventHandler(upDown_SequentialValueChanged);
            }
            if (_upDown is UINumericUpDown2)
            {
                ((UINumericUpDown2)_upDown).SequentialValueChanged += new SequentialValueChangedEventHandler(upDown_SequentialValueChanged);
            }

            // スライダーボタン
            btnSlider = new App.Controls.UIButton();

            btnSlider.Anchor = AnchorStyles.Top | AnchorStyles.Right;
            btnSlider.Location = new Point(96, 0);
            btnSlider.Size = new Size(24, 23);
            btnSlider.TabIndex = 1;
            btnSlider.Image = imgValueEditPanelSlider_;
            btnSlider.MouseDown += new MouseEventHandler(btnSlider_MouseDown);
            btnSlider.Click += new EventHandler(btnSlider_Click);

            Controls.Add(btnSlider);
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            _upDown.Width = Width - (120 - 92);
        }

        #region プロパティ
        /// <summary>
        /// スライダーボタン表示フラグ。
        /// </summary>
        [DefaultValue(true)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("スライダーボタンを表示するかどうかを示します。")]
        public bool ShowSliderButton
        {
            get { return _showSliderButton; }
            set
            {
                _showSliderButton = value;

                // 表示切り替え
                btnSlider.Visible = value;
            }
        }

        /// <summary>
        /// スライダー範囲調節フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("スライダーの範囲が調節可能かどうかを示します。")]
        public bool SliderRangeAdjustable
        {
            get { return _sliderRangeAdjustable; }
            set
            {
                _sliderRangeAdjustable = value;
                if (_upDown is UINumericUpDown2)
                {
                    ((UINumericUpDown2)_upDown).EnableInputMinMax = value;
                }
            }
        }

        protected bool SuspendRangeAdjust { get; set; }

        public FormClosedEventHandler SliderCloseHandler;
        #endregion

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_SEQUENTIALVALUECHANGED = new object();

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("値が変更された時に発生します。")]
        public event SequentialValueChangedEventHandler SequentialValueChanged
        {
            add { base.Events.AddHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        protected virtual void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            SequentialValueChangedEventHandler handler = (SequentialValueChangedEventHandler)base.Events[EVENT_SEQUENTIALVALUECHANGED];
            if (handler != null) { handler(this, e); }
        }

        public event EventHandler SliderSettingChanged;
        #endregion

        /// <summary>
        /// 同調コントロールをクリア。
        /// </summary>
        public void ClearBuddyControls()
        {
            _buddyControls.Clear();
        }

        /// <summary>
        /// 同調コントロールを設定。
        /// </summary>
        public void SetBuddyControls(params ValueEditPanel[] controls)
        {
            _buddyControls.Clear();
            foreach (ValueEditPanel control in controls)
            {
                Debug.Assert(control != this);
                _buddyControls.Add(control);
            }
        }

        // スライダのヒントメッセージを設定する
        public void SetSliderToolTipHintMessage(string message)
        {
            if (ttpHint == null)
            {
                ttpHint = new App.Controls.HintToolTip();
            }

            ttpHint.SetToolTip(btnSlider, message);
        }

        public void SetContextMenuStrip(UIContextMenuStrip menu)
        {
            ContextMenuStrip = menu;
            _upDown.ContextMenuStrip = menu;
        }

        #region 派生クラス実装用
        /// <summary>
        /// スライダー作成。
        /// </summary>
        protected abstract ValueEditPopupSlider CreateSlider();

        /// <summary>
        /// スライダー設定更新。
        /// </summary>
        protected abstract void UpdateSliderSetting();

        /// <summary>
        /// スライダー値変更ハンドラ。
        /// </summary>
        protected abstract void OnSliderSequentialValueChanged();

        /// <summary>
        /// スライダー設定変更ハンドラ。
        /// </summary>
        protected virtual void OnSliderSettingChanged()
        {
            if (SliderSettingChanged != null)
            {
                SliderSettingChanged(this, new EventArgs());
            }
        }

        /// <summary>
        /// アップダウン値変更ハンドラ。
        /// </summary>
        protected abstract void OnUpDownValueEdit();
        #endregion

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(120, 24); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnVisibleChanged(EventArgs e)
        {
            // 自分が無効になったらスライダーを消す
            if (!Visible && _slider != null)
            {
                _slider.Hide();
                _slider = null;
            }
            base.OnVisibleChanged(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnEnabledChanged(EventArgs e)
        {
            // 自分が無効になったらスライダーを消す
            if (!Enabled && _slider != null)
            {
                _slider.Hide();
                _slider = null;
            }
            base.OnEnabledChanged(e);
        }

        public override Color ForeColor
        {
            get
            {
                return base.ForeColor;
            }
            set
            {
                base.ForeColor = value;
                _upDown.ForeColor = value;
            }
        }

        public bool SliderShowing
        {
            get { return _sliderShowing; }
        }

        public bool SliderFocused
        {
            get { return (_slider != null) && _slider.Focused; }
        }

        #endregion

        #region イベントハンドラ
        //---------------------------------------------------------------------------
        // アップダウン
        private void upDown_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            // 同調コントロール更新
            OnUpDownValueEdit();
            // 値変更イベント発行
            OnSequentialValueChanged(e);
        }

        //---------------------------------------------------------------------------
        // ボタン
        private void btnSlider_MouseDown(object sender, MouseEventArgs e)
        {
            // スライダー表示中フラグ設定
            if (e.Button == MouseButtons.Left)
            {
                _sliderShowing = (_slider != null);
            }
        }


        private void btnSlider_Click(object sender, EventArgs e)
        {
            // スライダー表示中は連続して表示させない
            if (SliderShowing) { return; }
            _sliderShowing = true;

            // スライダー作成＆初期設定
            _slider = CreateSlider();
            _slider.FormClosed += new FormClosedEventHandler(slider_FormClosed);
            if (SliderCloseHandler != null) _slider.FormClosed += SliderCloseHandler;
            _slider.SequentialValueChanged += new SequentialValueChangedEventHandler(slider_SequentialValueChanged);
            _slider.SettingChanged += new EventHandler(slider_SettingChanged);
            UpdateSliderSetting();

            // 表示位置計算
            Rectangle rcButton = ControlUtility.GetScreenBounds(btnSlider);
            Rectangle rcScreen = SystemInformation.VirtualScreen;
            Rectangle rcSlider = _slider.SliderBounds;

            // 既定の表示位置（ボタンの右）
            Point location = new Point(rcButton.Right + 1, rcButton.Top - rcSlider.Top - 2);

            // 画面外補正
            if (location.X + _slider.Width > rcScreen.Right)
            {
                // ボタンの右下を既定位置にする
                location = new Point(rcButton.Left, rcButton.Bottom + 1);

                // 右端補正
                if (location.X + _slider.Width > rcScreen.Right)
                {
                    location.X -= (location.X + _slider.Width - rcScreen.Right);
                }
                // 下端補正
                if (location.Y + _slider.Height > rcScreen.Bottom)
                {
                    location.Y = rcButton.Top - _slider.Height;
                }
            }

            // 表示
            _slider.Location = location;
            _slider.ShowSlider(btnSlider);
        }

        //---------------------------------------------------------------------------
        // スライダー
        private void slider_FormClosed(object sender, FormClosedEventArgs e)
        {
            _slider = null;
            _sliderShowing = false;
            OnSliderColsed();
        }

        protected abstract void OnSliderColsed();

        private void slider_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            SuspendRangeAdjust = true;

            // アップダウン更新
            OnSliderSequentialValueChanged();
            // 値変更イベント発行
            OnSequentialValueChanged(e);

            SuspendRangeAdjust = false;
        }

        protected abstract void slider_SettingChanged(object sender, EventArgs e);
        #endregion
    }

    /// <summary>
    /// 整数値編集パネルクラス。
    /// </summary>
    [ToolboxItem(true)]
    [DefaultProperty("Value")]
    public sealed class IntEditPanel : ValueEditPanel
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntEditPanel() : base(new IntUpDown())
        {
            MinDefault = -1;
            MaxDefault = 1;
            MinBound = int.MinValue;
            MaxBound = int.MaxValue;
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("値。")]
        public int Value
        {
            get { return UpDown.Value; }
            set
            {
                if (SliderRangeAdjustable && !SuspendRangeAdjust)
                {
                    FitRange(value);
                }
                UpDown.Value = value;
                UpdateSliderSetting();
            }
        }

        public void FitRange(int value)
        {
            IntRangeProperty range = new IntRangeProperty(Minimum, Maximum, Increment, LargeIncrement, MinBound, MaxBound, MinDefault, MaxDefault);
            while (!range.Contains(value) && range.Scale(true))
            {
                ;
            }
            if (range.Contains(value))
            {
                Minimum = range.Minimum;
                Maximum = range.Maximum;
                Increment = range.ChangeS;
                LargeIncrement = range.ChangeL;
            }
        }
        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最小値。")]
        public int Minimum
        {
            get { return UpDown.Minimum; }
            set
            {
                UpDown.Minimum = value;
                UpdateSliderSetting();
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最大値。")]
        public int Maximum
        {
            get { return UpDown.Maximum; }
            set
            {
                UpDown.Maximum = value;
                UpdateSliderSetting();
            }
        }

        /// <summary>
        /// 増減値。
        /// </summary>
        [DefaultValue(1)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("増減値。")]
        public int Increment
        {
            get { return UpDown.Increment; }
            set
            {
                UpDown.Increment = value;
                UpdateSliderSetting();
            }
        }

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(10)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("大増減値。PageUp、PageDown入力時に使用します。")]
        public int LargeIncrement
        {
            get { return UpDown.LargeIncrement; }
            set
            {
                UpDown.LargeIncrement = value;
                UpdateSliderSetting();
            }
        }

        [DefaultValue(-1)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最小値の初期値。")]
        public int MinDefault { get; set; }


        [DefaultValue(1)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最大値の初期値。")]
        public int MaxDefault { get; set; }

        private int minBound;

        [DefaultValue(int.MinValue)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最小値の下限。")]
        public int MinBound
        {
            get
            {
                return minBound;
            }
            set
            {
                minBound = value;
                UpDown.InputMin = value;
                if (Minimum < minBound)
                {
                    Minimum = minBound;
                }
            }
        }

        private int maxBound;

        [DefaultValue(int.MaxValue)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最大値の上限。")]
        public int MaxBound
        {
            get
            {
                return maxBound;
            }
            set
            {
                maxBound = value;
                UpDown.InputMax = value;
                if (maxBound < Maximum)
                {
                    Maximum = maxBound;
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override ValueEditPopupSlider CreateSlider()
        {
            return new IntEditPopupSlider();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateSliderSetting()
        {
            // スライダー表示中の場合のみ
            IntEditPopupSlider slider = Slider;
            if (slider != null)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    slider.Minimum         = UpDown.Minimum;
                    slider.Maximum         = UpDown.Maximum;
                    slider.SmallChange     = UpDown.Increment;
                    slider.LargeChange     = UpDown.LargeIncrement;
                    slider.MinDefault = MinDefault;
                    slider.MaxDefault = MaxDefault;
                    slider.MinBound = MinBound;
                    slider.MaxBound = MaxBound;
                    slider.Value = UpDown.Value;
                    slider.RangeAdjustable = SliderRangeAdjustable;
                    slider.UpdateScaleEnabled();
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSliderSequentialValueChanged()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                UpDown.Value = Slider.Value;

                // 同調コントロールの設定
                foreach (IntEditPanel buddy in _buddyControls)
                {
                    buddy.UpDown.Value = Slider.Value;
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSliderSettingChanged()
        {
            IntEditPopupSlider slider = Slider;
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                UpDown.Minimum        = slider.Minimum;
                UpDown.Maximum        = slider.Maximum;
                UpDown.Increment      = slider.SmallChange;
                UpDown.LargeIncrement = slider.LargeChange;
                UpDown.Value          = slider.Value;

                // 同調コントロールの設定
                foreach (IntEditPanel buddy in _buddyControls)
                {
                    buddy.UpDown.Minimum        = slider.Minimum;
                    buddy.UpDown.Maximum        = slider.Maximum;
                    buddy.UpDown.Increment      = slider.SmallChange;
                    buddy.UpDown.LargeIncrement = slider.LargeChange;
                    buddy.UpDown.Value          = slider.Value;
                }
            }

            base.OnSliderSettingChanged();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnUpDownValueEdit()
        {
            // 同調コントロールの設定
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                foreach (IntEditPanel buddy in _buddyControls)
                {
                    buddy.UpDown.Minimum        = UpDown.Minimum;
                    buddy.UpDown.Maximum        = UpDown.Maximum;
                    buddy.UpDown.Increment      = UpDown.Increment;
                    buddy.UpDown.LargeIncrement = UpDown.LargeIncrement;
                    buddy.UpDown.Value          = UpDown.Value;
                }
            }
        }

        #region コントロールアクセス
        /// <summary>
        /// アップダウン。
        /// </summary>
        private IntUpDown UpDown
        {
            get { return (IntUpDown)_upDown; }
        }

        /// <summary>
        /// スライダー。
        /// </summary>
        private IntEditPopupSlider Slider
        {
            get { return (IntEditPopupSlider)_slider; }
        }
        #endregion

        protected override void slider_SettingChanged(object sender, EventArgs e)
        {
            // 変更前の値
            var value = UpDown.Value;
            // アップダウン更新
            OnSliderSettingChanged();
            // 値変更イベント発行
            if (value != UpDown.Value)
            {
                OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
            }
        }

        protected override void OnSliderColsed()
        {
            if (SliderRangeAdjustable)
            {
                FitRange(Value);
            }
        }

        //[Category(UIControlHelper.OriginalEventCategoryName)]
        //[Description("値が変更された時に発生します。")]
        public event EventHandler MinMaxValueChanged
        {
            add
            {
                //UpDown.Va
            }
            remove
            {
            }
        }
    }

    /// <summary>
    /// 整数値編集パネルクラス。
    /// </summary>
    [ToolboxItem(true)]
    [DefaultProperty("Value")]
    public sealed class UintEditPanel : ValueEditPanel
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UintEditPanel()
            : base(new UintUpDown())
        {
            MinDefault = 0;
            MaxDefault = 1;
            MinBound = uint.MinValue;
            MaxBound = uint.MaxValue;
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("値。")]
        public uint Value
        {
            get { return UpDown.Value; }
            set
            {
                if (SliderRangeAdjustable && !SuspendRangeAdjust)
                {
                    FitRange(value);
                }
                UpDown.Value = value;
                UpdateSliderSetting();
            }
        }

        public void FitRange(uint value)
        {
            UintRangeProperty range = new UintRangeProperty(Minimum, Maximum, Increment, LargeIncrement, MinBound, MaxBound, MinDefault, MaxDefault);
            while (!range.Contains(value) && range.Scale(true))
            {
                ;
            }
            if (range.Contains(value))
            {
                Minimum = range.Minimum;
                Maximum = range.Maximum;
                Increment = range.ChangeS;
                LargeIncrement = range.ChangeL;
            }
        }
        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最小値。")]
        public uint Minimum
        {
            get { return UpDown.Minimum; }
            set
            {
                UpDown.Minimum = value;
                UpdateSliderSetting();
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最大値。")]
        public uint Maximum
        {
            get { return UpDown.Maximum; }
            set
            {
                UpDown.Maximum = value;
                UpdateSliderSetting();
            }
        }

        /// <summary>
        /// 増減値。
        /// </summary>
        [DefaultValue(1)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("増減値。")]
        public uint Increment
        {
            get { return UpDown.Increment; }
            set
            {
                UpDown.Increment = value;
                UpdateSliderSetting();
            }
        }

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(10)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("大増減値。PageUp、PageDown入力時に使用します。")]
        public uint LargeIncrement
        {
            get { return UpDown.LargeIncrement; }
            set
            {
                UpDown.LargeIncrement = value;
                UpdateSliderSetting();
            }
        }

        [DefaultValue(0)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最小値の初期値。")]
        public uint MinDefault { get; set; }


        [DefaultValue(1)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最大値の初期値。")]
        public uint MaxDefault { get; set; }

        private uint minBound;

        [DefaultValue(uint.MinValue)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最小値の下限。")]
        public uint MinBound
        {
            get
            {
                return minBound;
            }
            set
            {
                minBound = value;
                UpDown.InputMin = value;
                if (Minimum < minBound)
                {
                    Minimum = minBound;
                }
            }
        }

        private uint maxBound;

        [DefaultValue(uint.MaxValue)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最大値の上限。")]
        public uint MaxBound
        {
            get
            {
                return maxBound;
            }
            set
            {
                maxBound = value;
                UpDown.InputMax = value;
                if (maxBound < Maximum)
                {
                    Maximum = maxBound;
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override ValueEditPopupSlider CreateSlider()
        {
            return new UintEditPopupSlider();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateSliderSetting()
        {
            // スライダー表示中の場合のみ
            UintEditPopupSlider slider = Slider;
            if (slider != null)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    slider.Minimum = UpDown.Minimum;
                    slider.Maximum = UpDown.Maximum;
                    slider.SmallChange = UpDown.Increment;
                    slider.LargeChange = UpDown.LargeIncrement;
                    slider.MinDefault = MinDefault;
                    slider.MaxDefault = MaxDefault;
                    slider.MinBound = MinBound;
                    slider.MaxBound = MaxBound;
                    slider.Value = UpDown.Value;
                    slider.RangeAdjustable = SliderRangeAdjustable;
                    slider.UpdateScaleEnabled();
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSliderSequentialValueChanged()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                UpDown.Value = Slider.Value;

                // 同調コントロールの設定
                foreach (UintEditPanel buddy in _buddyControls)
                {
                    buddy.UpDown.Value = Slider.Value;
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSliderSettingChanged()
        {
            UintEditPopupSlider slider = Slider;
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                UpDown.Minimum = slider.Minimum;
                UpDown.Maximum = slider.Maximum;
                UpDown.Increment = slider.SmallChange;
                UpDown.LargeIncrement = slider.LargeChange;
                UpDown.Value = slider.Value;

                // 同調コントロールの設定
                foreach (UintEditPanel buddy in _buddyControls)
                {
                    buddy.UpDown.Minimum = slider.Minimum;
                    buddy.UpDown.Maximum = slider.Maximum;
                    buddy.UpDown.Increment = slider.SmallChange;
                    buddy.UpDown.LargeIncrement = slider.LargeChange;
                    buddy.UpDown.Value = slider.Value;
                }
            }

            base.OnSliderSettingChanged();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnUpDownValueEdit()
        {
            // 同調コントロールの設定
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                foreach (UintEditPanel buddy in _buddyControls)
                {
                    buddy.UpDown.Minimum = UpDown.Minimum;
                    buddy.UpDown.Maximum = UpDown.Maximum;
                    buddy.UpDown.Increment = UpDown.Increment;
                    buddy.UpDown.LargeIncrement = UpDown.LargeIncrement;
                    buddy.UpDown.Value = UpDown.Value;
                }
            }
        }

        #region コントロールアクセス
        /// <summary>
        /// アップダウン。
        /// </summary>
        private UintUpDown UpDown
        {
            get { return (UintUpDown)_upDown; }
        }

        /// <summary>
        /// スライダー。
        /// </summary>
        private UintEditPopupSlider Slider
        {
            get { return (UintEditPopupSlider)_slider; }
        }
        #endregion

        protected override void slider_SettingChanged(object sender, EventArgs e)
        {
            // 変更前の値
            var value = UpDown.Value;
            // アップダウン更新
            OnSliderSettingChanged();
            // 値変更イベント発行
            if (value != UpDown.Value)
            {
                OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
            }
        }

        protected override void OnSliderColsed()
        {
            if (SliderRangeAdjustable)
            {
                FitRange(Value);
            }
        }
    }

    /// <summary>
    /// 実数値編集パネルクラス。
    /// </summary>
    [ToolboxItem(true)]
    [DefaultProperty("Value")]
    public sealed class FloatEditPanel : ValueEditPanel
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatEditPanel() : base(new FloatUpDown())
        {
            MinDefault = -1.0f;
            MaxDefault = 1.0f;
            MinBound = float.MinValue;
            MaxBound = float.MaxValue;
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("値。")]
        public float Value
        {
            get { return UpDown.Value; }
            set
            {
                if (SliderRangeAdjustable && !SuspendRangeAdjust)
                {
                    FitRange(value);
                }
                UpDown.Value = value;
                UpdateSliderSetting();
            }
        }

        public void FitRange(float value)
        {
            FloatRangeProperty range = new FloatRangeProperty(Minimum, Maximum, Increment, LargeIncrement, MinBound, MaxBound, MinDefault, MaxDefault);
            while (!range.Contains(value) && range.Scale(true))
            {
                ;
            }
            if (range.Contains(value))
            {
                Minimum = range.Minimum;
                Maximum = range.Maximum;
                Increment = range.ChangeS;
                LargeIncrement = range.ChangeL;
            }
        }

        public void UpdateIncrement()
        {
            Increment = RangeHelper.CalcIncrementValue(Minimum, Maximum);
            LargeIncrement = Increment * 10;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最小値。")]
        public float Minimum
        {
            get { return UpDown.Minimum; }
            set
            {
                UpDown.Minimum = value;
                UpdateSliderSetting();
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100.0f)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最大値。")]
        public float Maximum
        {
            get { return UpDown.Maximum; }
            set
            {
                UpDown.Maximum = value;
                UpdateSliderSetting();
            }
        }

        /// <summary>
        /// 増減値。
        /// </summary>
        [DefaultValue(1.0f)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("増減値。")]
        public double Increment
        {
            get { return UpDown.Increment; }
            set
            {
                UpDown.Increment = value;
                UpdateSliderSetting();
            }
        }

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(10.0f)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("大増減値。PageUp、PageDown入力時に使用します。")]
        public double LargeIncrement
        {
            get { return UpDown.LargeIncrement; }
            set
            {
                UpDown.LargeIncrement = value;
                UpdateSliderSetting();
            }
        }

        [DefaultValue(-1.0f)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最小値の初期値。指定しないときは0を使用")]
        public float MinDefault { get; set; }


        [DefaultValue(1.0f)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最大値の初期値。指定しないときは0を使用")]
        public float MaxDefault { get; set; }

        private float minBound;

        [DefaultValue(float.MinValue)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最小値の下限。")]
        public float MinBound {
            get
            {
                return minBound;
            }
            set
            {
                minBound = value;
                UpDown.InputMin = value;
                if (Minimum < minBound)
                {
                    Minimum = minBound;
                }
            }
        }

        private float maxBound;

        [DefaultValue(float.MaxValue)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最大値の上限。")]
        public float MaxBound {
            get
            {
                return maxBound;
            }
            set
            {
                maxBound = value;
                UpDown.InputMax = value;
                if (maxBound < Maximum)
                {
                    Maximum = maxBound;
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override ValueEditPopupSlider CreateSlider()
        {
            return new FloatEditPopupSlider();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateSliderSetting()
        {
            FloatEditPopupSlider slider = Slider;
            if (slider != null)
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    slider.Minimum         = UpDown.Minimum;
                    slider.Maximum         = UpDown.Maximum;
                    slider.SmallChange     = UpDown.Increment;
                    slider.LargeChange     = UpDown.LargeIncrement;
                    slider.MinDefault = MinDefault;
                    slider.MaxDefault = MaxDefault;
                    slider.MinBound = MinBound;
                    slider.MaxBound = MaxBound;
                    slider.Value = UpDown.Value;
                    slider.RangeAdjustable = SliderRangeAdjustable;
                    slider.UpdateScaleEnabled();
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSliderSequentialValueChanged()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                UpDown.Value = Slider.Value;

                // 同調コントロールの設定
                foreach (FloatEditPanel buddy in _buddyControls)
                {
                    buddy.UpDown.Value = Slider.Value;
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSliderSettingChanged()
        {
            FloatEditPopupSlider slider = Slider;
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                UpDown.Minimum        = slider.Minimum;
                UpDown.Maximum        = slider.Maximum;
                UpDown.Increment      = slider.SmallChange;
                UpDown.LargeIncrement = slider.LargeChange;
                UpDown.Value          = slider.Value;

                // 同調コントロールの設定
                foreach (FloatEditPanel buddy in _buddyControls)
                {
                    buddy.UpDown.Minimum        = slider.Minimum;
                    buddy.UpDown.Maximum        = slider.Maximum;
                    buddy.UpDown.Increment      = slider.SmallChange;
                    buddy.UpDown.LargeIncrement = slider.LargeChange;
                    buddy.UpDown.Value          = slider.Value;
                }
            }

            base.OnSliderSettingChanged();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnUpDownValueEdit()
        {
            // 同調コントロールの設定
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                foreach (FloatEditPanel buddy in _buddyControls)
                {
                    buddy.UpDown.Minimum        = UpDown.Minimum;
                    buddy.UpDown.Maximum        = UpDown.Maximum;
                    buddy.UpDown.Increment      = UpDown.Increment;
                    buddy.UpDown.LargeIncrement = UpDown.LargeIncrement;
                    buddy.UpDown.Value          = UpDown.Value;
                }
            }
        }

        #region コントロールアクセス
        /// <summary>
        /// アップダウン。
        /// </summary>
        private FloatUpDown UpDown
        {
            get { return (FloatUpDown)_upDown; }
        }

        /// <summary>
        /// スライダー。
        /// </summary>
        private FloatEditPopupSlider Slider
        {
            get { return (FloatEditPopupSlider)_slider; }
        }
        #endregion

        protected override void slider_SettingChanged(object sender, EventArgs e)
        {
            // 変更前の値
            var value = UpDown.Value;
            // アップダウン更新
            OnSliderSettingChanged();
            // 値変更イベント発行
            if (value != UpDown.Value)
            {
                OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
            }
        }

        protected override void OnSliderColsed()
        {
            if (SliderRangeAdjustable)
            {
                FitRange(Value);
            }
        }
    }
}
