﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;

namespace App.Controls
{
    /// <summary>
    /// ＵＩツールストリップコンテナクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStripContainer))]
    public class UIToolStripContainer : ToolStripContainer
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIToolStripContainer()
        {
        }
    }

    /// <summary>
    /// ＵＩツールストリップクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ToolStrip))]
    public class UIToolStrip : ToolStrip
    {
        /// <summary>MouseMove発行前イベント。</summary>
        public event MouseEventHandler BeforeMouseMove = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIToolStrip()
        {
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs mea)
        {
            // 発行前イベント呼び出し
            if (BeforeMouseMove != null)
            {
                BeforeMouseMove(this, mea);
            }
            base.OnMouseMove(mea);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // マウスアクティベート
                // 既定では戻り値が MA_ACTIVATEANDEAT なので以降のマウス処理がされない
                case Win32.WM.WM_MOUSEACTIVATE:
                    m.Result = (IntPtr)Win32.MA.MA_ACTIVATE;
                    return;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// 全ての子アイテムを有効・無効化
        /// </summary>
        public void EnableToolStripItems(bool isEnable)
        {
            foreach (var item in Items.OfType<ToolStripItem>())
            {
                item.Enabled = isEnable;
            }
        }

    }

    public class UIToolStripRenderer : ToolStripProfessionalRenderer
    {
        protected override void OnRenderButtonBackground(ToolStripItemRenderEventArgs e)
        {
            var btn = e.Item as ToolStripButton;
            if (btn != null && !btn.Enabled)
            {
                return;
            }
            if (btn != null && btn.Checked)
            {
                using (var brush = new SolidBrush(ProfessionalColors.ButtonSelectedHighlight))
                using (var pen = new Pen(ProfessionalColors.ButtonSelectedHighlightBorder))
                {
                    Rectangle bounds = new Rectangle(1, 1, e.Item.Size.Width - 2, e.Item.Size.Height - 2);
                    e.Graphics.FillRectangle(brush, bounds);
                    if (btn.Selected)
                    {
                        bounds = new Rectangle(0, 0, e.Item.Size.Width - 1, e.Item.Size.Height - 1);
                        e.Graphics.DrawRectangle(pen, bounds);
                    }
                }

                return;
            }

            base.OnRenderButtonBackground(e);
        }
    }

    // TODO: UIToolStripRenderer と統一
    public class UIToolStripRenderer2 : ToolStripProfessionalRenderer
    {
        protected override void OnRenderButtonBackground(ToolStripItemRenderEventArgs e)
        {
            var btn = e.Item as UIToolStripButton;
            if (btn != null && btn.Checked && btn.Style == UIToolStripButton.RenderStyle.Custom)
            {
                using (var brush = new SolidBrush(btn.Enabled? ProfessionalColors.ButtonSelectedHighlight: SystemColors.ControlLight))
                using (var pen = new Pen(ProfessionalColors.ButtonSelectedHighlightBorder))
                {
                    Rectangle bounds = new Rectangle(1, 1, e.Item.Size.Width - 2, e.Item.Size.Height - 2);
                    e.Graphics.FillRectangle(brush, bounds);
                    if (btn.Selected)
                    {
                        bounds = new Rectangle(0, 0, e.Item.Size.Width - 1, e.Item.Size.Height - 1);
                        e.Graphics.DrawRectangle(pen, bounds);
                    }
                }

                return;
            }

            base.OnRenderButtonBackground(e);
        }
    }

    /// <summary>
    /// ＵＩメニューストリップクラス。
    /// </summary>
    [ToolboxBitmap(typeof(MenuStrip))]
    public class UIMenuStrip : MenuStrip
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIMenuStrip()
        {
            // オーバーフロー可能にする
            base.CanOverflow = true;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // マウスアクティベート
                // 既定では戻り値が MA_ACTIVATEANDEAT なので以降のマウス処理がされない
                case Win32.WM.WM_MOUSEACTIVATE:
                    m.Result = (IntPtr)Win32.MA.MA_ACTIVATE;
                    return;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(true)]
        public new bool CanOverflow
        {
            get { return base.CanOverflow; }
            set { base.CanOverflow = value; }
        }
    }

    /// <summary>
    /// ＵＩコンテキストメニューストリップクラス。
    /// </summary>
    [ToolboxBitmap(typeof(ContextMenuStrip))]
    public class UIContextMenuStrip : ContextMenuStrip
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIContextMenuStrip()
        {
        }

        public bool IsObjectViewContextMenu;

        private ContextMenuStrip proxy = null;

        /// <summary>
        /// コンテキストメニューをマウスポインタで開くための代理
        /// </summary>
        public ContextMenuStrip ProxyContextMenuStrip
        {
            get
            {
                if (proxy == null)
                {
                    proxy = new ContextMenuStrip();
                    proxy.Opening += (x, y) =>
                        {
                            Show(Cursor.Position);
                        };
                }

                return proxy;
            }
        }

        /// <summary>
        /// 代理で開こうとした最後のコントロール
        /// </summary>
        public Control ProxySourceControl
        {
            get
            {
                return proxy == null ? null : proxy.SourceControl;
            }
        }
    }

    /// <summary>
    /// ＵＩステータスストリップクラス。
    /// </summary>
    [ToolboxBitmap(typeof(StatusStrip))]
    public class UIStatusStrip : StatusStrip
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIStatusStrip()
        {
            // ToolStrip、MenuStripと同じ描画モードにする
            base.RenderMode = ToolStripRenderMode.ManagerRenderMode;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool DefaultShowItemToolTips
        {
            // ツールチップを表示するようにする
            get { return true; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(ToolStripRenderMode.ManagerRenderMode)]
        public new ToolStripRenderMode RenderMode
        {
            get { return base.RenderMode; }
            set { base.RenderMode = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(true)]
        public new bool ShowItemToolTips
        {
            get { return base.ShowItemToolTips; }
            set { base.ShowItemToolTips = value; }
        }
    }
}
