﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Linq;
using System.Windows.Forms;
using App.res;

namespace App.Controls
{
    /// <summary>
    /// ＵＩプロパティグリッドクラス。
    /// </summary>
    public class UIPropertyGrid : PropertyGrid
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIPropertyGrid()
        {
        }

        /// <summary>
        /// 表示されているプロパティを更新。
        /// </summary>
        public void RefreshProperty()
        {
            // 同じオブジェクトを設定し直すだけ
            object[] selectedObjects = SelectedObjects;
            SelectedObjects = selectedObjects;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseCompatibleTextRendering
        {
            get { return base.UseCompatibleTextRendering;  }
            set { base.UseCompatibleTextRendering = value; }
        }
        #endregion
    }

    #region PropertyOrderAttribute
    /// <summary>
    /// プロパティ順序属性クラス。
    /// </summary>
    [AttributeUsage(AttributeTargets.Property)]
    public sealed class PropertyOrderAttribute : Attribute
    {
        // 順序
        private readonly int _order;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PropertyOrderAttribute(int order)
        {
            _order = order;
        }

        /// <summary>
        /// 順序。
        /// </summary>
        public int Order
        {
            get { return _order; }
        }
    }
    #endregion

    // TODO: DisplayNameAttribute の派生クラスにできないか？
    #region LocalizedPropertyNameAttribute
    /// <summary>
    /// ローカライズ対応プロパティ名属性クラス。
    /// </summary>
    [AttributeUsage(AttributeTargets.Property)]
    public sealed class LocalizedPropertyNameAttribute : Attribute
    {
        // プロパティ名
        private readonly string _propertyName;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public LocalizedPropertyNameAttribute(string s)
        {
            _propertyName = Strings.ResourceManager.GetString(s, Strings.Culture);
        }

        /// <summary>
        /// プロパティ名。
        /// </summary>
        public string PropertyName
        {
            get { return _propertyName; }
        }
    }
    #endregion

    #region LocalizedCategoryAttribute
    /// <summary>
    /// ローカライズ対応カテゴリ属性クラス。
    /// </summary>
    [AttributeUsage(AttributeTargets.Property)]
    public sealed class LocalizedCategoryAttribute : CategoryAttribute
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public LocalizedCategoryAttribute(string s) : base(Strings.ResourceManager.GetString(s, Strings.Culture))
        {
        }
    }
    #endregion

    #region LocalizedDescriptionAttribute
    /// <summary>
    /// ローカライズ対応説明属性クラス。
    /// </summary>
    [AttributeUsage(AttributeTargets.Property)]
    public sealed class LocalizedDescriptionAttribute : DescriptionAttribute
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public LocalizedDescriptionAttribute(string s) : base(Strings.ResourceManager.GetString(s, Strings.Culture))
        {
        }
    }
    #endregion

    #region LocalizedPropertyGridConverter
    /// <summary>
    /// ローカライズ対応プロパティグリッド用型コンバータクラス。
    /// </summary>
    public sealed class LocalizedPropertyGridConverter : TypeConverter
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public LocalizedPropertyGridConverter() {}

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override PropertyDescriptorCollection GetProperties(ITypeDescriptorContext context, object component, Attribute[] attributes)
        {
            // コレクションを準備
            PropertyDescriptorCollection collection = new PropertyDescriptorCollection(null);
            PropertyDescriptorCollection properties = TypeDescriptor.GetProperties(component, attributes, false);

            // Stable Sort
            var sortedProperties = properties.Cast<PropertyDescriptor>().OrderBy(x => x, new PropertyDescriptorComparer());

            // 専用ディスクリプタを使用するように変更する
            foreach (PropertyDescriptor descriptor in sortedProperties)
            {
                collection.Add(new LocalizedPropertyDescriptor(descriptor));
            }

            return collection;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool GetPropertiesSupported(ITypeDescriptorContext context)
        {
            return true;
        }

        // プロパティディスクリプタのコンペアラ
        private class PropertyDescriptorComparer : System.Collections.Generic.IComparer<PropertyDescriptor>
        {
            /// <summary>
            /// 比較。
            /// </summary>
            public int Compare(PropertyDescriptor x, PropertyDescriptor y)
            {
                Type key = typeof(PropertyOrderAttribute);
                PropertyOrderAttribute lhs =
                    (PropertyOrderAttribute)((PropertyDescriptor)x).Attributes[key];
                PropertyOrderAttribute rhs =
                    (PropertyOrderAttribute)((PropertyDescriptor)y).Attributes[key];

                if (lhs == null)
                {
                    if (rhs == null)
                    {
                        // 両方ともnullなら何もしない
                        return 0;
                    }
                    else
                    {
                        // lhsのみnull
                        return 1;
                    }
                }
                else
                {
                    if (rhs == null)
                    {
                        // rhsのみnull
                        return -1;
                    }
                    else
                    {
                        // 両方共にOrderAttributeが付いていれば比較する
                        return lhs.Order - rhs.Order;
                    }
                }
            }
        }

    }
    #endregion

    #region LocalizedPropertyDescriptor
    /// <summary>
    /// ローカライズ対応プロパティディスクリプタクラス。
    /// </summary>
    public sealed class LocalizedPropertyDescriptor : PropertyDescriptor
    {
        // ディスクリプタ
        private readonly PropertyDescriptor _descriptor;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public LocalizedPropertyDescriptor(PropertyDescriptor descriptor) : base(descriptor)
        {
            _descriptor = descriptor;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string DisplayName
        {
            get
            {
                // プロパティ名指定属性が付いていればそこから取得
                LocalizedPropertyNameAttribute attribute = (LocalizedPropertyNameAttribute)_descriptor.Attributes[typeof(LocalizedPropertyNameAttribute)];
                if (attribute != null)
                {
                    return attribute.PropertyName;
                }
                return _descriptor.DisplayName;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override Type ComponentType
        {
            get { return _descriptor.ComponentType; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override Type PropertyType
        {
            get { return _descriptor.PropertyType; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool IsReadOnly
        {
            get { return _descriptor.IsReadOnly; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void SetValue(object component, object value)
        {
            _descriptor.SetValue(component, value);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override object GetValue(object component)
        {
            return _descriptor.GetValue(component);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool CanResetValue(object component)
        {
            return _descriptor.CanResetValue(component);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void ResetValue(object component)
        {
            _descriptor.ResetValue(component);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool ShouldSerializeValue(object component)
        {
            return _descriptor.ShouldSerializeValue(component);
        }
    }
    #endregion
}
