﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Globalization;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Diagnostics;
using App.Utility;

namespace App.Controls
{
    public abstract class UINumericUpDown2 : UpDownBase, KeyEditableControl
    {
        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_SEQUENTIALVALUECHANGED = new object();

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("値が変更された時に発生します。")]
        public event SequentialValueChangedEventHandler SequentialValueChanged
        {
            add { base.Events.AddHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        protected virtual void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            SequentialValueChangedEventHandler handler = (SequentialValueChangedEventHandler)base.Events[EVENT_SEQUENTIALVALUECHANGED];
            if (handler != null) { handler(this, e); }
        }

        /// <summary>
        /// テキスト入力時の専用の範囲を有効にするか
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool EnableInputMinMax
        {
            get;
            set;
        }
        #endregion

        public bool CanKeyEdit		// interface KeyEditableControl
        {
            get
            {
                return ReadOnly == false;
            }
        }
    }

    public class UINumericUpDown2ValueChangeFromTextEventArgs : EventArgs{}

    /// <summary>
    /// ＵＩ数値アップダウンクラス。
    /// </summary>
    [ToolboxBitmap(typeof(NumericUpDown))]
    [DefaultEvent("SequentialValueChanged")]
    public abstract class UINumericUpDown2<T, S> : UINumericUpDown2, ISupportInitialize where T : IComparable<T>, IEquatable<T> where S : IComparable<S>
    {
        // 内部ボタンコントロール
        private readonly Control _internalButtons;
        // 内部テキストボックス
        private readonly TextBox _internalTextBox;
        // 大増減値。
        private S _largeIncrement;// = new decimal(10);
        // リターン入力有効フラグ
        private bool _wantReturn = true;
        // 自動範囲拡張フラグ
        private bool _autoRangeExpansion = false;
        // 値変更中フラグ
        private bool _valueChanging = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UINumericUpDown2()
        {
            ImeMode = ImeMode.Disable;
            base.TextAlign = HorizontalAlignment.Right;

            // 内部コントロール取得
            _internalButtons = base.Controls[0];
            _internalTextBox = (TextBox)base.Controls[1];

            UpdateEditText();
        }


        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new ImeMode ImeMode
        {
            get
            {
                return base.ImeMode;
            }
            set
            {
                base.ImeMode = value;
            }
        }
        void ISupportInitialize.BeginInit() { }
        void ISupportInitialize.EndInit()
        {
            UpdateEditText();
        }
        #region プロパティ
        private T _minimum;
        /// <summary>
        /// 最小値
        /// </summary>
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最小値。")]
        public T Minimum {
            get { return _minimum; }
            set
            {
                _minimum = value;
                UpdateEditText();
            }
        }

        private T _maximum;
        /// <summary>
        /// 最大値
        /// </summary>
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("最大値。")]
        public T Maximum
        {
            get { return _maximum; }
            set
            {
                _maximum = value;
                UpdateEditText();
            }
        }

        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("増減値。")]
        public S Increment { get; set; }

        /// <summary>
        /// 大増減値。
        /// </summary>
        //[DefaultValue(typeof(T), "10")]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("大増減値。PageUp、PageDown入力時に使用します。")]
        public S LargeIncrement
        {
            get { return _largeIncrement; }
            set { _largeIncrement = value; }
        }

        /// <summary>
        /// リターン入力有効フラグ。
        /// </summary>
        [DefaultValue(true)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("リターン入力を有効にするかどうかを示します。")]
        public bool WantReturn
        {
            get { return _wantReturn; }
            set { _wantReturn = value; }
        }

        // TODO: 使われていないので削除
        /// <summary>
        /// 自動範囲拡張フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("廃止予定。")]
        public bool AutoRangeExpantion
        {
            get { return _autoRangeExpansion; }
            set { _autoRangeExpansion = value; }
        }

        /// <summary>
        /// テキスト入力時の下限
        /// EnableInputMinMax == true のときのみ機能する
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public T InputMin { get; set; }

        /// <summary>
        /// テキスト入力時の上限
        /// EnableInputMinMax == true のときのみ機能する
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public T InputMax { get; set; }
        #endregion

        /// <summary>
        /// 全選択。
        /// </summary>
        public void SelectAll()
        {
            _internalTextBox.SelectAll();
        }

        /// <summary>
        /// 内部ボタンコントロール。
        /// </summary>
        protected Control InternalButtons
        {
            get { return _internalButtons; }
        }

        /// <summary>
        /// 内部テキストボックス。
        /// </summary>
        protected TextBox InternalTextBox
        {
            get { return _internalTextBox; }
        }

        private T _value;
        public T Value {
            get {
                // ユーザーが編集中に Value を取得しに来た場合
                if (UserEdit)
                {
                    ValidateEditText();
                }
                return _value;
            }
            set {
                bool changed = !_value.Equals(value);
                // 範囲チェック
                _value = value;
                UserEdit = false;
                UpdateEditText();
                if (changed)
                {
                    OnValueChanged(
                        isValueChangeFromText_ ?
                            new UINumericUpDown2ValueChangeFromTextEventArgs() :
                            new EventArgs()
                    );
                }
            }
        }

        private bool isValueChangeFromText_ = false;

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateEditText()
        {
            if (base.UserEdit)
            {
                ValidateEditText();
            }

            // 条件要調査
            {
                ChangingText = true;
                Text = Format(_value);
            }

            if (EnableInputMinMax)
            {
                if (_value.CompareTo(InputMin) < 0 || 0 < _value.CompareTo(InputMax))
                {
                    _internalTextBox.ForeColor = Color.Red;
                }
                else
                {
                    _internalTextBox.ForeColor = SystemColors.ControlText;
                }
            }
            else
            {
                if (_value.CompareTo(Minimum) < 0 || 0 < _value.CompareTo(Maximum))
                {
                    _internalTextBox.ForeColor = Color.Red;
                }
                else
                {
                    _internalTextBox.ForeColor = SystemColors.ControlText;
                }
            }

            // 全選択
            if (Focused)
            {
                _internalTextBox.SelectAll();
            }
        }

        protected override void ValidateEditText()
        {
            T value;
            if (TryParseText(Text, out value))
            {
                if (EnableInputMinMax)
                {
                    if (value.CompareTo(InputMin) < 0)
                    {
                        value = InputMin;
                    }
                    else if (value.CompareTo(InputMax) > 0)
                    {
                        value = InputMax;
                    }
                }
                else
                {
                    if (value.CompareTo(Minimum) < 0)
                    {
                        value = Minimum;
                    }
                    else if (value.CompareTo(Maximum) > 0)
                    {
                        value = Maximum;
                    }
                }

                isValueChangeFromText_ = true;

                Value = value;

                isValueChangeFromText_ = false;
            }
        }

        protected abstract bool TryParseText(string text, out T value);
        protected abstract string Format(T value);

        /// <summary>
        /// 値を増減させる。
        /// </summary>
        protected void IncrementValue(bool up, S increment, bool startValueChanging)
        {
            T result = _value;
            if (increment.CompareTo(default(S)) > 0)
            {
                // 増加
                if (up)
                {
                    result = this.increment(_value, increment);

                }
                // 減少
                else
                {
                    result = decrement(_value, increment);

                }

                // 丸める
                if (result.CompareTo(Maximum) > 0)
                {
                    result = Maximum;
                }
                else if (result.CompareTo(Minimum) < 0)
                {
                    result = Minimum;
                }

                if (!result.Equals(_value))
                {
                    if (startValueChanging)
                    {
                        _valueChanging = true;
                    }
                    Value = result;
                }
            }
        }

        // 範囲チェックは無
        protected abstract T increment(T original, S delta);
        protected abstract T decrement(T original, S delta);

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(80, base.DefaultSize.Height); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            // リターン入力処理
            if (_wantReturn)
            {
                if (keyData == Keys.Return)
                {
                    // OnLostFocus() と同じ処理をする
                    if (base.UserEdit)
                    {
                        UpdateEditText();
                    }
                    return true;
                }
            }
            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // マウスホイール
                case Win32.WM.WM_MOUSEWHEEL:
                    {
                        // フォーカスあり
                        if (Focused)
                        {
                            // コントロール領域内なら増減させる
                            Point point = PointToClient(Win32.Utility.LParamToPoint(m.LParam));
                            if (ClientRectangle.Contains(point))
                            {
                                int delta = Win32.Utility.SignedHIWORD(m.WParam);
                                IncrementValue(delta > 0, Increment, false);
                                return;
                            }
                        }

                        // 親に委譲する
                        Win32.NativeMethods.SendMessage(Parent.Handle, m.Msg, m.WParam, m.LParam);
                    }
                    return;

                default:
                    break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void UpButton()
        {
            IncrementValue(true, Increment, true);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void DownButton()
        {
            IncrementValue(false, Increment, true);
        }

        protected override void OnLostFocus(EventArgs e)
        {
            if (base.UserEdit)
            {
                UpdateEditText();
            }
            base.OnLostFocus(e);
        }
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            // 全選択（TABキー移動時のみ）
            if (Control.MouseButtons == MouseButtons.None)
            {
                SelectAll();
            }
            base.OnGotFocus(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected virtual void OnValueChanged(EventArgs e)
        {
            // 値変更イベント発行
            OnSequentialValueChanged(
                new SequentialValueChangedEventArgs(_valueChanging)
                {
                    SourceEventArgs = e
                }
            );
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            // 最小値
            if (e.KeyData == Keys.Home)
            {
                Value = Minimum;
                e.Handled = true;
            }
            // 最大値
            else if (e.KeyData == Keys.End)
            {
                Value = Maximum;
                e.Handled = true;
            }
            // 大増加
            else if (e.KeyData == Keys.PageUp)
            {
                IncrementValue(true, _largeIncrement, true);
                e.Handled = true;
            }
            // 大減少
            else if (e.KeyData == Keys.PageDown)
            {
                IncrementValue(false, _largeIncrement, true);
                e.Handled = true;
            }
            base.OnKeyDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyUp(KeyEventArgs e)
        {
            // 値変更中解除
            if (e.KeyData == Keys.Up || e.KeyData == Keys.Down || e.KeyData == Keys.PageUp || e.KeyData == Keys.PageDown)
            {
                if (_valueChanging)
                {
                    _valueChanging = false;
                    OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
                }
            }
            base.OnKeyUp(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            // 値変更中解除
            if (e.Button == MouseButtons.Left)
            {
                if (_valueChanging)
                {
                    _valueChanging = false;
                    OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
                }
            }
            base.OnMouseUp(e);
        }

        protected override void OnTextBoxKeyPress(object source, KeyPressEventArgs e)
        {
            // 数字以外入力できないようにします。
            if (!char.IsControl(e.KeyChar)
                && !char.IsDigit(e.KeyChar))
            {
                switch (e.KeyChar)
                {
                    default:
                        e.Handled = true;
                        break;
                    case '+':
                    case '-':
                        break;
                    case '.':
                    case 'e':
                    case 'E':
                        // 本来は派生先でやるべきかも
                        e.Handled = typeof(T) != typeof(float);
                        break;
                }
            }

            base.OnTextBoxKeyPress(source, e);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override bool AutoSize
        {
            get { return base.AutoSize; }
            set { base.AutoSize = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool InterceptArrowKeys
        {
            get { return base.InterceptArrowKeys; }
            set { base.InterceptArrowKeys = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool ReadOnly
        {
            get { return base.ReadOnly; }
            set
            {
                base.ReadOnly = value;

                // 0個目がボタンのはず
                Controls[0].Enabled = ! value;
            }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft; }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(HorizontalAlignment.Right)]
        public new HorizontalAlignment TextAlign
        {
            get { return base.TextAlign; }
            set { base.TextAlign = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new LeftRightAlignment UpDownAlign
        {
            get { return base.UpDownAlign; }
            set { base.UpDownAlign = value; }
        }
        #endregion
    }

    /// <summary>
    /// ＵＩ数値アップダウンクラス。
    /// </summary>
    [ToolboxBitmap(typeof(NumericUpDown))]
    [DefaultEvent("SequentialValueChanged")]
    public class UINumericUpDown : NumericUpDown
    {
        // 内部ボタンコントロール
        private readonly Control _internalButtons;
        // 内部テキストボックス
        private readonly TextBox _internalTextBox;
        // 大増減値。
        private decimal _largeIncrement = new decimal(10);
        // リターン入力有効フラグ
        private bool _wantReturn = true;
        // 自動範囲拡張フラグ
        private bool _autoRangeExpansion = false;
        // 値変更中フラグ
        private bool _valueChanging = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UINumericUpDown()
        {
            base.TextAlign = HorizontalAlignment.Right;

            // 内部コントロール取得
            _internalButtons = base.Controls[0];
            _internalTextBox = (TextBox)base.Controls[1];
        }

        #region プロパティ
        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(typeof(decimal), "10")]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("大増減値。PageUp、PageDown入力時に使用します。")]
        public decimal LargeIncrement
        {
            get { return _largeIncrement; }
            set { _largeIncrement = value; }
        }

        /// <summary>
        /// リターン入力有効フラグ。
        /// </summary>
        [DefaultValue(true)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("リターン入力を有効にするかどうかを示します。")]
        public bool WantReturn
        {
            get { return _wantReturn; }
            set { _wantReturn = value; }
        }

        /// <summary>
        /// 自動範囲拡張フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("範囲外の値が設定された時に範囲を自動で拡張するかどうかを示します。")]
        public bool AutoRangeExpantion
        {
            get { return _autoRangeExpansion; }
            set { _autoRangeExpansion = value; }
        }
        #endregion

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_SEQUENTIALVALUECHANGED = new object();

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("値が変更された時に発生します。")]
        public event SequentialValueChangedEventHandler SequentialValueChanged
        {
            add { base.Events.AddHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        protected virtual void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            SequentialValueChangedEventHandler handler = (SequentialValueChangedEventHandler)base.Events[EVENT_SEQUENTIALVALUECHANGED];
            if (handler != null) { handler(this, e); }
        }
        #endregion

        /// <summary>
        /// 全選択。
        /// </summary>
        public void SelectAll()
        {
            _internalTextBox.SelectAll();
        }

        /// <summary>
        /// 内部ボタンコントロール。
        /// </summary>
        protected Control InternalButtons
        {
            get { return _internalButtons; }
        }

        /// <summary>
        /// 内部テキストボックス。
        /// </summary>
        protected TextBox InternalTextBox
        {
            get { return _internalTextBox; }
        }

        /// <summary>
        /// 現在の値を取得。
        /// </summary>
        protected decimal GetCurrentValue()
        {
            decimal value = decimal.Zero;

            // UserEdit == true 時は base.Value で
            // ValidateEditText が呼ばれてしまう
            bool userEdit = base.UserEdit;
            base.UserEdit = false;
            {
                value = base.Value;
            }
            base.UserEdit = userEdit;

            return value;
        }

        /// <summary>
        /// 数値テキストを取得。
        /// </summary>
        protected string GetNumberText(decimal value)
        {
            // NumericUpDown の private GetNumberText() とまったく同じ
            // 公開されていないので、そのまま持ってきてます
            if (base.Hexadecimal)
            {
                long number = (long)value;
                return number.ToString("X", CultureInfo.InvariantCulture);
            }
            return value.ToString((base.ThousandsSeparator ? "N" : "F") + base.DecimalPlaces.ToString(CultureInfo.CurrentCulture), CultureInfo.CurrentCulture);
        }

        /// <summary>
        /// 小数部の有効桁数を取得。
        /// </summary>
        public static int GetDecimalPlaces(decimal value)
        {
            // 一旦文字列にする
            string valueString = value.ToString();

            // 小数点位置
            int periodIndex = valueString.LastIndexOf('.');
            if (periodIndex == -1)
            {
                return 0;
            }

            // 小数部文字列
            string decimalString = valueString.Substring(periodIndex + 1);

            // 右側から「０」の部分を除いた文字数を計測
            int result = decimalString.Length;
            for (int i = decimalString.Length - 1; i >= 0; i--)
            {
                if (decimalString[i] == '0')
                {
                    result--;
                }
                else
                {
                    break;
                }
            }
            return result;
        }

        /// <summary>
        /// 値を増減させる。
        /// </summary>
        private void IncrementValue(bool up, decimal increment, bool startValueChanging)
        {
            decimal value  = GetCurrentValue();
            decimal result = value;
            if (increment > decimal.Zero)
            {
                // 増加
                if (up)
                {
                    result = Math.Floor((value + increment) / increment) * increment;
                    if (result > base.Maximum)
                    {
                        result = base.Maximum;
                    }
                }
                // 減少
                else
                {
                    result = Math.Ceiling((value - increment) / increment) * increment;
                    if (result < base.Minimum)
                    {
                        result = base.Minimum;
                    }
                }
                if (result != value)
                {
                    if (startValueChanging)
                    {
                        _valueChanging = true;
                    }
                    base.Value = result;
                }
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(80, base.DefaultSize.Height); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            // リターン入力処理
            if (_wantReturn)
            {
                if (keyData == Keys.Return)
                {
                    // OnLostFocus() と同じ処理をする
                    if (base.UserEdit)
                    {
                        UpdateEditText();
                    }
                    return true;
                }
            }
            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // マウスホイール
                case Win32.WM.WM_MOUSEWHEEL:
                    {
                        // フォーカスあり
                        if (Focused)
                        {
                            // コントロール領域内なら増減させる
                            Point point = PointToClient(Win32.Utility.LParamToPoint(m.LParam));
                            if (ClientRectangle.Contains(point))
                            {
                                int delta = Win32.Utility.SignedHIWORD(m.WParam);
                                IncrementValue(delta > 0, Increment, false);
                                return;
                            }
                        }

                        // 親に委譲する
                        Win32.NativeMethods.SendMessage(Parent.Handle, m.Msg, m.WParam, m.LParam);
                    }
                    return;

                default:
                    break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateEditText()
        {
            base.UpdateEditText();

            // 全選択
            if (Focused)
            {
                _internalTextBox.SelectAll();
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void UpButton()
        {
            IncrementValue(true, base.Increment, true);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override void DownButton()
        {
            IncrementValue(false, base.Increment, true);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            // 全選択（TABキー移動時のみ）
            if (Control.MouseButtons == MouseButtons.None)
            {
                SelectAll();
            }
            base.OnGotFocus(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnValueChanged(EventArgs e)
        {
            // 値変更イベント発行
            OnSequentialValueChanged(new SequentialValueChangedEventArgs(_valueChanging));

            // NumericUpDown.OnValueChanged は CanRaiseEvents が
            // 効かないので、呼び出し条件を直接判断する
            if (CanRaiseEvents)
            {
                base.OnValueChanged(e);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            // 最小値
            if (e.KeyData == Keys.Home)
            {
                base.Value = base.Minimum;
                e.Handled = true;
            }
            // 最大値
            else if (e.KeyData == Keys.End)
            {
                base.Value = base.Maximum;
                e.Handled = true;
            }
            // 大増加
            else if (e.KeyData == Keys.PageUp)
            {
                IncrementValue(true, _largeIncrement, true);
                e.Handled = true;
            }
            // 大減少
            else if (e.KeyData == Keys.PageDown)
            {
                IncrementValue(false, _largeIncrement, true);
                e.Handled = true;
            }
            base.OnKeyDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyUp(KeyEventArgs e)
        {
            // 値変更中解除
            if (e.KeyData == Keys.Up || e.KeyData == Keys.Down || e.KeyData == Keys.PageUp || e.KeyData == Keys.PageDown)
            {
                if (_valueChanging)
                {
                    _valueChanging = false;
                    OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
                }
            }
            base.OnKeyUp(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            // 値変更中解除
            if (e.Button == MouseButtons.Left)
            {
                if (_valueChanging)
                {
                    _valueChanging = false;
                    OnSequentialValueChanged(new SequentialValueChangedEventArgs(false));
                }
            }
            base.OnMouseUp(e);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override bool AutoSize
        {
            get { return base.AutoSize; }
            set { base.AutoSize = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool Hexadecimal
        {
            get { return base.Hexadecimal; }
            set { base.Hexadecimal = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool InterceptArrowKeys
        {
            get { return base.InterceptArrowKeys; }
            set { base.InterceptArrowKeys = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool ReadOnly
        {
            get { return base.ReadOnly; }
            set { base.ReadOnly = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft; }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(HorizontalAlignment.Right)]
        public new HorizontalAlignment TextAlign
        {
            get { return base.TextAlign; }
            set { base.TextAlign = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new LeftRightAlignment UpDownAlign
        {
            get { return base.UpDownAlign; }
            set { base.UpDownAlign = value; }
        }
        #endregion
    }

    //----------------------------------------------------------------------------
    // 値編集用
    #region IntUpDown
#if true
    /// <summary>
    /// 整数値アップダウンクラス。
    /// </summary>
    public sealed class IntUpDown : UINumericUpDown2<int, int>
    {
        const int DefaultMinValue = 0;
        const int DefaultMaxValue = 100;
        const int DefaultIncrement = 1;
        const int DefaultLargeIncrement = 10;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntUpDown()
        {
            Minimum = DefaultMinValue;
            Maximum = DefaultMaxValue;
            Increment = DefaultIncrement;
            LargeIncrement = DefaultLargeIncrement;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(DefaultMinValue)]
        [Description("最小値。int型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new int Minimum
        {
            get { return base.Minimum; }
            set { base.Minimum = value; }
        }
        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(DefaultMaxValue)]
        [Description("最大値。int型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new int Maximum
        {
            get { return base.Maximum; }
            set { base.Maximum = value; }
        }

        /// <summary>
        /// 増減値。
        /// </summary>
        [DefaultValue(DefaultIncrement)]
        [Description("増減値。int型で再定義しています。")]
        public new int Increment
        {
            get { return base.Increment; }
            set { base.Increment = value; }
        }

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(DefaultLargeIncrement)]
        [Description("大増減値。int型で再定義しています。")]
        public new int LargeIncrement
        {
            get { return base.LargeIncrement; }
            set { base.LargeIncrement = value; }
        }

        protected override string Format(int value)
        {
            return value.ToString();
        }

        protected override bool TryParseText(string text, out int value)
        {
            return StringUtility.TrySafeIntParse(text, out value);
        }

        protected override int increment(int original, int delta)
        {
            return original + delta;
        }

        protected override int decrement(int original, int delta)
        {
            return original - delta;
        }
    }
#else
        /// <summary>
    /// 整数値アップダウンクラス。
    /// </summary>
    public sealed class IntUpDown : UINumericUpDown
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntUpDown()
        {
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0)]
        [Description("値。int型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new int Value
        {
            get { return Convert.ToInt32(base.Value); }
            set
            {
                // 範囲拡張
                if (this.AutoRangeExpantion)
                {
                    IntRangeProperty range = this.RangeProperty;
                    if (!range.Contains(value))
                    {
                        if (value < 0 && 0 <= range.Minimum)
                        {
                            range.Minimum = value;
                        }
                        else if (range.Maximum <= 0 && 0 < value)
                        {
                            range.Maximum = value;
                        }
                        else if (0 <= value && value < range.Minimum)
                        {
                            range.Minimum = value;
                        }
                        else if (range.Maximum < value && value <= 0)
                        {
                            range.Maximum = value;
                        }
                        else
                        {
                            while (!range.Contains(value))
                            {
                                if (!range.Scale(true))
                                {
                                    range.Maximum = Math.Max(value, range.Maximum);
                                    range.Minimum = Math.Min(value, range.Minimum);
                                    break;
                                }
                            }
                        }
                        this.Minimum = range.Minimum;
                        this.Maximum = range.Maximum;
                        this.Increment = range.ChangeS;
                        this.LargeIncrement = range.ChangeL;
                    }
                }
                base.Value = Convert.ToDecimal(value);
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0)]
        [Description("最小値。int型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new int Minimum
        {
            get { return Convert.ToInt32(base.Minimum); }
            set { base.Minimum = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100)]
        [Description("最大値。int型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new int Maximum
        {
            get { return Convert.ToInt32(base.Maximum); }
            set { base.Maximum = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 増減値。
        /// </summary>
        [DefaultValue(1)]
        [Description("増減値。int型で再定義しています。")]
        public new int Increment
        {
            get { return Convert.ToInt32(base.Increment); }
            set { base.Increment = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(10)]
        [Description("大増減値。int型で再定義しています。")]
        public new int LargeIncrement
        {
            get { return Convert.ToInt32(base.LargeIncrement); }
            set { base.LargeIncrement = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 範囲プロパティ。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IntRangeProperty RangeProperty
        {
            get { return new IntRangeProperty(this.Minimum, this.Maximum, this.Increment, this.LargeIncrement); }
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new int DecimalPlaces
        {
            get { return base.DecimalPlaces; }
            set { base.DecimalPlaces = value; }
        }
        #endregion
    }
#endif
    #endregion

    #region FloatUpDown
#if true
        /// <summary>
    /// 実数値アップダウンクラス。
    /// </summary>
    public sealed class FloatUpDown : UINumericUpDown2<float, double>
    {
        const int DefaultDecimalPlaces = 2;
        const float DefaultMinValue = 0;
        const float DefaultMaxValue = 100;
        const double DefaultIncrement = 1;
        const double DefaultLargeIncrement = 10;
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatUpDown()
        {
            DecimalPlaces = 2;
            Minimum = DefaultMinValue;
            Maximum = DefaultMaxValue;
            Increment = DefaultIncrement;
            LargeIncrement = DefaultLargeIncrement;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(DefaultMinValue)]
        [Description("最小値。float型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new float Minimum
        {
            get { return base.Minimum; }
            set { base.Minimum = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(DefaultMaxValue)]
        [Description("最大値。float型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new float Maximum
        {
            get { return base.Maximum; }
            set { base.Maximum = value; }
        }

        /// <summary>
        /// 増減値。
        /// </summary>
        [DefaultValue(DefaultIncrement)]
        [Description("増減値。float型で再定義しています。")]
        public new double Increment
        {
            get { return base.Increment; }
            set { base.Increment = value; }
        }

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(DefaultLargeIncrement)]
        [Description("大増減値。float型で再定義しています。")]
        public new double LargeIncrement
        {
            get { return base.LargeIncrement; }
            set { base.LargeIncrement = value; }
        }

        protected override float decrement(float original, double delta)
        {
            return (float) (Math.Round((original - delta) / delta) * delta);
        }

        protected override float increment(float original, double delta)
        {
            return (float)(Math.Round((original + delta) / delta) * delta);
        }

        private static readonly Regex onlyDigits = new Regex(@"^-?[0-9]+$");
        protected override string Format(float value)
        {
            if (FixedDecimalPlaces)
            {
                return value.ToString("F2");
            }

            string text = value.ToString("R");

            // 整数の場合は小数表示に直す
            if (onlyDigits.IsMatch(text, 0))
            {
                text = value.ToString("F1");
            }
            return text;
        }

        protected override bool TryParseText(string text, out float value)
        {
            //return float.TryParse(text, out value);
            return StringUtility.TrySafeFloatParse(text, out value);
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        [DefaultValue(false)]
        public bool FixedDecimalPlaces
        {
            get;
            set;
        }
        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(DefaultDecimalPlaces)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public int DecimalPlaces
        {
            get;
            set;
        }
        #endregion

#else
    /// <summary>
    /// 実数値アップダウンクラス。
    /// </summary>
    public sealed class FloatUpDown : UINumericUpDown
    {
        // 小数点変更中フラグ
        private bool _decimalPlacesChanging = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatUpDown()
        {
            base.DecimalPlaces = 1;
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Description("値。float型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new float Value
        {
            get { return Convert.ToSingle(base.Value); }
            set
            {
                // 範囲拡張
                if (this.AutoRangeExpantion)
                {
                    FloatRangeProperty range = this.RangeProperty;
                    if (!range.Contains(value))
                    {
                        if (value < 0 && 0 <= range.Minimum)
                        {
                            range.Minimum = value;
                        }
                        else if (range.Maximum <= 0 && 0 < value)
                        {
                            range.Maximum = value;
                        }
                        else if (0 <=value && value < range.Minimum)
                        {
                            range.Minimum = value;
                        }
                        else if (range.Maximum < value && value <= 0)
                        {
                            range.Maximum = value;
                        }
                        else
                        {
                            while (!range.Contains(value))
                            {
                                if (!range.Scale(true))
                                {
                                    range.Maximum = Math.Max(value, range.Maximum);
                                    range.Minimum = Math.Min(value, range.Minimum);
                                    break;
                                }
                            }
                        }
                        this.Minimum        = range.Minimum;
                        this.Maximum        = range.Maximum;
                        this.Increment      = range.ChangeS;
                        this.LargeIncrement = range.ChangeL;
                    }
                }
                base.Value = Convert.ToDecimal(value);
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Description("最小値。float型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new float Minimum
        {
            get { return Convert.ToSingle(base.Minimum); }
            set { base.Minimum = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100.0f)]
        [Description("最大値。float型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new float Maximum
        {
            get { return Convert.ToSingle(base.Maximum); }
            set { base.Maximum = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 増減値。
        /// </summary>
        [DefaultValue(1.0f)]
        [Description("増減値。float型で再定義しています。")]
        public new float Increment
        {
            get { return Convert.ToSingle(base.Increment); }
            set { base.Increment = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(10.0f)]
        [Description("大増減値。float型で再定義しています。")]
        public new float LargeIncrement
        {
            get { return Convert.ToSingle(base.LargeIncrement); }
            set { base.LargeIncrement = Convert.ToDecimal(value); }
        }

        /// <summary>
        /// 範囲プロパティ。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public FloatRangeProperty RangeProperty
        {
            get { return new FloatRangeProperty(this.Minimum, this.Maximum, this.Increment, this.LargeIncrement); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateEditText()
        {
            // DecimalPlaces 変更中は呼ばせない
            if (!_decimalPlacesChanging)
            {
                base.UpdateEditText();
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnValueChanged(EventArgs e)
        {
            if (!FixedDecimalPlaces)
            {
                // 小数部桁数に応じて表示桁数を変更する
                int decimalPlaces = GetDecimalPlaces(GetCurrentValue());
                if (decimalPlaces == 0)
                {
                    decimalPlaces = 1;
                }
                if (base.DecimalPlaces != decimalPlaces)
                {
                    _decimalPlacesChanging = true;
                    {
                        base.DecimalPlaces = decimalPlaces;
                    }
                    _decimalPlacesChanging = false;
                }
            }
            base.OnValueChanged(e);
        }

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        [DefaultValue(false)]
        public bool FixedDecimalPlaces
        {
            get;
            set;
        }
#endif
    }
    #endregion

    /// <summary>
    /// 整数値アップダウンクラス。
    /// </summary>
    public sealed class UintUpDown : UINumericUpDown2<uint, uint>
    {
        const uint DefaultMinValue = 0;
        const uint DefaultMaxValue = 100;
        const uint DefaultIncrement = 1;
        const uint DefaultLargeIncrement = 10;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UintUpDown()
        {
            Minimum = DefaultMinValue;
            Maximum = DefaultMaxValue;
            Increment = DefaultIncrement;
            LargeIncrement = DefaultLargeIncrement;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(DefaultMinValue)]
        [Description("最小値。uint型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new uint Minimum
        {
            get { return base.Minimum; }
            set { base.Minimum = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(DefaultMaxValue)]
        [Description("最大値。uint型で再定義しています。")]
        [RefreshProperties(RefreshProperties.All)]
        public new uint Maximum
        {
            get { return base.Maximum; }
            set { base.Maximum = value; }
        }

        /// <summary>
        /// 増減値。
        /// </summary>
        [DefaultValue(DefaultIncrement)]
        [Description("増減値。uint型で再定義しています。")]
        public new uint Increment
        {
            get { return base.Increment; }
            set { base.Increment = value; }
        }

        /// <summary>
        /// 大増減値。
        /// </summary>
        [DefaultValue(DefaultLargeIncrement)]
        [Description("大増減値。uint型で再定義しています。")]
        public new uint LargeIncrement
        {
            get { return base.LargeIncrement; }
            set { base.LargeIncrement = value; }
        }

        protected override string Format(uint value)
        {
            return value.ToString();
        }

        protected override bool TryParseText(string text, out uint value)
        {
            //return uint.TryParse(text, out value);
            return StringUtility.TrySafeUintParse(text, out value);
        }

        protected override uint increment(uint original, uint delta)
        {
            return original + delta;
        }

        protected override uint decrement(uint original, uint delta)
        {
            return original > delta ? original - delta : 0;
        }
    }

    public static class RangeHelper
    {
        private static double r1 = Math.Log10(2);
        private static double r2 = Math.Log10(5);
        public static double MyLog(double v)
        {
            // Log をとって補正
            double a = Math.Log10(v);
            double b = Math.Floor(a);
            double r = a - b;
            double w;
            if (r < r1)
            {
                w = 3 * b + r / r1;
            }
            else if (r < r2)
            {
                w = 3 * b + 1 + (r - r1) / (r2 - r1);
            }
            else
            {
                w = 3 * b + 2 + (r - r2) / (1 - r2);
            }
            return w;
        }

        public static double MyPow(double w)
        {
            // 逆に補正してから Pow
            double b = Math.Floor(w / 3);
            double u = w - 3 * b;
            double r;
            if (u < 1)
            {
                r = u * r1;
            }
            else if (u < 2)
            {
                r = (u - 1) * (r2 - r1) + r1;
            }
            else
            {
                r = (u - 2) * (1 - r2) + r2;
            }
            double v = Math.Pow(10, b + r);
            return v;
        }

        // ToOrder(v0, v1, v0)=0、ToOrder(v0, v1, FromOrder(v0, v1, order)) = order を満たすようにする
        public static int ToOrder(double v0, double v1, double value)
        {
            double v = 0;
            if (0 < v0)
            {
                if (v0 <= v1)
                {
                    v = MyLog((double)value / v0);
                }
                else
                {
                    v = MyLog((double)v0 + 1 - value);
                }
            }
            else if (v0 < 0)
            {
                if (v1 <= v0)
                {
                    v = MyLog((double)value / v0);
                }
                else
                {
                    v = MyLog((double)value - v0 - 1);
                }
            }
            else
            {
                if (0 < v1)
                {
                    v = MyLog((double)value + 1);
                }
                else
                {
                    v = MyLog(1 - (double)value);
                }
            }
            if (value == v1)
            {
                return (int)Math.Ceiling(v);
            }
            else if (value == v0)
            {
                return (int)Math.Floor(v);
            }
            else
            {
                return (int)Math.Round(v);
            }
        }

        public static double FromOrder(double v0, double v1, int order)
        {
            double v = v0;
            if (0 < v0)
            {
                if (v0 <= v1)
                {
                    v = v0 * MyPow(order);
                }
                else
                {
                    v = -(MyPow(order) - v0 - 1);
                }
            }
            else if (v0 < 0)
            {
                if (v1 <= v0)
                {
                    v = v0 * MyPow(order);
                }
                else
                {
                    v = MyPow(order) + v0 + 1;
                }
            }
            else
            {
                if (0 < v1)
                {
                    v = MyPow(order) - 1;
                }
                else
                {
                    v = - (MyPow(order) - 1);
                }
            }

            return v;
        }

        public static double CalcIncrementValue(double min, double max)
        {
            return Math.Min(1, Math.Pow(10, Math.Round(Math.Log10(Math.Max(0.001, max - min))) - 2));
        }
    }
    #region IntRangeProperty
    /// <summary>
    /// 整数値範囲プロパティクラス。
    /// </summary>
    public sealed class IntRangeProperty
    {
        // 最小値
        private int _minimum = 0;
        // 最大値
        private int _maximum = 0;
        // 小変化量
        private readonly int _changeS = 0;
        // 大変化量
        private readonly int _changeL = 0;

        private readonly int _MinBound;
        private readonly int _MaxBound;
        private readonly int _MinDefault;
        private readonly int _MaxDefault;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntRangeProperty(int minimum, int maximum, int changeS, int changeL, int MinBound, int MaxBound, int MinDefault, int MaxDefault)
        {
            _minimum = minimum;
            _maximum = maximum;
            _changeS = changeS;
            _changeL = changeL;
            _MinBound = MinBound;
            _MaxBound = MaxBound;
            _MinDefault = MinDefault;
            _MaxDefault = MaxDefault;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        public int Minimum
        {
            get { return _minimum; }
            set { _minimum = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        public int Maximum
        {
            get { return _maximum; }
            set { _maximum = value; }
        }

        /// <summary>
        /// 小変化量。
        /// </summary>
        public int ChangeS
        {
            get { return _changeS; }
        }

        /// <summary>
        /// 大変化量。
        /// </summary>
        public int ChangeL
        {
            get { return _changeL; }
        }

        /// <summary>
        /// 指定した値が含まれるか。
        /// </summary>
        public bool Contains(int value)
        {
            return value >= _minimum && value <= _maximum;
        }

        /// <summary>
        /// スケーリング。
        /// </summary>
        public bool Scale(bool scaleUp)
        {
            //DebugConsole.WriteLine("Scale");
            //--------------------------------------------------------------------------
            // FloatRangeProperty と同じなので変更時は両方修正すること！
            //--------------------------------------------------------------------------

            var originalMin = _minimum;
            var originalMax = _maximum;

            try
            {
                int minOrder =  RangeHelper.ToOrder(_MinDefault, _MinBound, _minimum);
                int maxOrder =  RangeHelper.ToOrder(_MaxDefault, _MaxBound, _maximum);
                if (scaleUp)
                {
                    if (0 <= _MinDefault && minOrder == 0 && maxOrder < 0)
                    {
                        maxOrder++;
                    }
                    else if (_MaxDefault <= 0 && maxOrder == 0 && minOrder < 0)
                    {
                        minOrder++;
                    }
                    else
                    {
                        minOrder = maxOrder = Math.Max(minOrder, maxOrder) + 1;
                    }
                    _minimum = (int)Math.Max(Math.Min(RangeHelper.FromOrder(_MinDefault, _MinBound, minOrder), originalMin), _MinBound);
                    _maximum = (int)Math.Min(Math.Max(RangeHelper.FromOrder(_MaxDefault, _MaxBound, maxOrder), originalMax), _MaxBound);
                }
                else
                {
                    if (0 <= _MinDefault && minOrder == 0)
                    {
                        maxOrder--;
                    }
                    else if (_MaxDefault <= 0 && maxOrder == 0)
                    {
                        minOrder--;
                    }
                    else
                    {
                        minOrder = maxOrder = Math.Max(minOrder, maxOrder) - 1;
                    }
                    _minimum = (int)Math.Max(RangeHelper.FromOrder(_MinDefault, _MinBound, minOrder), originalMin);
                    _maximum = (int)Math.Min(RangeHelper.FromOrder(_MaxDefault, _MaxBound, maxOrder), originalMax);
                }
            }
            catch
            {
                Debug.Assert(false);
                _minimum = originalMin;
                _maximum = originalMax;
            }

            if (_maximum <= _minimum)
            {
                _minimum = originalMin;
                _maximum = originalMax;
            }

            if (_minimum == originalMin && _maximum == originalMax)
            {
                return false;
            }

            return true;
        }
    }

    #endregion

    #region UintRangeProperty
    /// <summary>
    /// 整数値範囲プロパティクラス。
    /// </summary>
    public sealed class UintRangeProperty
    {
        // 最小値
        private uint _minimum = 0;
        // 最大値
        private uint _maximum = 0;
        // 小変化量
        private readonly uint _changeS = 0;
        // 大変化量
        private readonly uint _changeL = 0;

        private uint _MinBound;
        private readonly uint _MaxBound;
        private uint _MinDefault;
        private readonly uint _MaxDefault;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UintRangeProperty(uint minimum, uint maximum, uint changeS, uint changeL, uint MinBound, uint MaxBound, uint MinDefault, uint MaxDefault)
        {
            _minimum = minimum;
            _maximum = maximum;
            _changeS = changeS;
            _changeL = changeL;
            _MinBound = MinBound;
            _MaxBound = MaxBound;
            _MinDefault = MinDefault;
            _MaxDefault = MaxDefault;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        public uint Minimum
        {
            get { return _minimum; }
            set { _minimum = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        public uint Maximum
        {
            get { return _maximum; }
            set { _maximum = value; }
        }

        /// <summary>
        /// 小変化量。
        /// </summary>
        public uint ChangeS
        {
            get { return _changeS; }
        }

        /// <summary>
        /// 大変化量。
        /// </summary>
        public uint ChangeL
        {
            get { return _changeL; }
        }

        /// <summary>
        /// 指定した値が含まれるか。
        /// </summary>
        public bool Contains(uint value)
        {
            return value >= _minimum && value <= _maximum;
        }

            /// <summary>
        /// スケーリング。
        /// </summary>
        public bool Scale(bool scaleUp)
        {
            //DebugConsole.WriteLine("Scale");
            //--------------------------------------------------------------------------
            // IntRangeProperty とまったく同じなので変更時は両方修正すること！
            //--------------------------------------------------------------------------

            var originalMin = _minimum;
            var originalMax = _maximum;

            try
            {
                int minOrder =  RangeHelper.ToOrder(_MinDefault, _MinBound, _minimum);
                int maxOrder =  RangeHelper.ToOrder(_MaxDefault, _MaxBound, _maximum);
                if (scaleUp)
                {
                    if (0 <= _MinDefault && minOrder == 0 && maxOrder < 0)
                    {
                        maxOrder++;
                    }
                    else if (_MaxDefault <= 0 && maxOrder == 0 && minOrder < 0)
                    {
                        minOrder++;
                    }
                    else
                    {
                        minOrder = maxOrder = Math.Max(minOrder, maxOrder) + 1;
                    }
                    _minimum = (uint)Math.Max(Math.Min(RangeHelper.FromOrder(_MinDefault, _MinBound, minOrder), originalMin), _MinBound);
                    _maximum = (uint)Math.Min(Math.Max(RangeHelper.FromOrder(_MaxDefault, _MaxBound, maxOrder), originalMax), _MaxBound);
                }
                else
                {
                    if (0 <= _MinDefault && minOrder == 0)
                    {
                        maxOrder--;
                    }
                    else if (_MaxDefault <= 0 && maxOrder == 0)
                    {
                        minOrder--;
                    }
                    else
                    {
                        minOrder = maxOrder = Math.Max(minOrder, maxOrder) - 1;
                    }
                    _minimum = (uint)Math.Max(RangeHelper.FromOrder(_MinDefault, _MinBound, minOrder), originalMin);
                    _maximum = (uint)Math.Min(RangeHelper.FromOrder(_MaxDefault, _MaxBound, maxOrder), originalMax);
                }
            }
            catch
            {
                Debug.Assert(false);
                _minimum = originalMin;
                _maximum = originalMax;
            }

            if (_maximum <= _minimum)
            {
                _minimum = originalMin;
                _maximum = originalMax;
            }

            if (_minimum == originalMin && _maximum == originalMax)
            {
                return false;
            }

            return true;
        }
    }
    #endregion
    #region FloatRangeProperty
    /// <summary>
    /// 実数値範囲プロパティクラス。
    /// </summary>
    public sealed class FloatRangeProperty
    {
        // 最小値
        private float _minimum = 0;
        // 最大値
        private float _maximum = 0;
        // 小変化量
        private double _changeS = 0;
        // 大変化量
        private double _changeL = 0;

        private readonly float _MinBound;
        private readonly float _MaxBound;
        private readonly float _MinDefault;
        private readonly float _MaxDefault;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatRangeProperty(float minimum, float maximum, double changeS, double changeL, float MinBound, float MaxBound, float MinDefault, float MaxDefault)
        {
            _minimum = minimum;
            _maximum = maximum;
            _changeL = Math.Min((double)Math.Max(Math.Abs(minimum), Math.Abs(maximum)), changeL);
            _changeS = Math.Min(changeS, changeL);
            _MinBound = MinBound;
            _MaxBound = MaxBound;
            _MinDefault = MinDefault;
            _MaxDefault = MaxDefault;
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        public float Minimum
        {
            get { return _minimum; }
            set { _minimum = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        public float Maximum
        {
            get { return _maximum; }
            set { _maximum = value; }
        }

        /// <summary>
        /// 小変化量。
        /// </summary>
        public double ChangeS
        {
            get { return _changeS; }
        }

        /// <summary>
        /// 大変化量。
        /// </summary>
        public double ChangeL
        {
            get { return _changeL; }
        }

        /// <summary>
        /// 指定した値が含まれるか。
        /// </summary>
        public bool Contains(float value)
        {
            return value >= _minimum && value <= _maximum;
        }

        /// <summary>
        /// スケーリング。
        /// </summary>
        public bool Scale(bool scaleUp)
        {
            //DebugConsole.WriteLine("Scale");
            //--------------------------------------------------------------------------
            // IntRangeProperty とまったく同じなので変更時は両方修正すること！
            //--------------------------------------------------------------------------

            var originalMin = _minimum;
            var originalMax = _maximum;

            try
            {
                int minOrder = RangeHelper.ToOrder(_MinDefault, _MinBound, _minimum);
                int maxOrder = RangeHelper.ToOrder(_MaxDefault, _MaxBound, _maximum);
                if (scaleUp)
                {
                    if (0 <= _MinDefault && minOrder == 0 && maxOrder < 0)
                    {
                        maxOrder++;
                    }
                    else if (_MaxDefault <= 0 && maxOrder == 0 && minOrder < 0)
                    {
                        minOrder++;
                    }
                    else
                    {
                        minOrder = maxOrder = Math.Max(minOrder, maxOrder) + 1;
                    }
                    _minimum = (float)Math.Max(Math.Min(RangeHelper.FromOrder(_MinDefault, _MinBound, minOrder), originalMin), _MinBound);
                    _maximum = (float)Math.Min(Math.Max(RangeHelper.FromOrder(_MaxDefault, _MaxBound, maxOrder), originalMax), _MaxBound);
                }
                else
                {
                    if (0 <= _MinDefault && minOrder == 0)
                    {
                        maxOrder--;
                    }
                    else if (_MaxDefault <= 0 && maxOrder == 0)
                    {
                        minOrder--;
                    }
                    else
                    {
                        minOrder = maxOrder = Math.Max(minOrder, maxOrder) - 1;
                    }
                    _minimum = (float)Math.Max(RangeHelper.FromOrder(_MinDefault, _MinBound, minOrder), originalMin);
                    _maximum = (float)Math.Min(RangeHelper.FromOrder(_MaxDefault, _MaxBound, maxOrder), originalMax);
                }
            }
            catch
            {
                Debug.Assert(false);
                _minimum = originalMin;
                _maximum = originalMax;
            }

            if (_maximum <= _minimum + 0.01999)
            {
                _minimum = originalMin;
                _maximum = originalMax;
            }

            if (_minimum == originalMin && _maximum == originalMax)
            {
                return false;
            }

            //_changeS = Math.Min(1, Math.Pow(10, Math.Round(Math.Log10(Math.Max(0.001, (double)_maximum - (double)_minimum))) - 2));
            _changeS = RangeHelper.CalcIncrementValue(_minimum, _maximum);
            _changeL = _changeS * 10;
            return true;
        }
    }
    #endregion
}
