﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Windows.Forms;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// ＵＩリストコントロールインタフェース。
    /// UIListBox、UICheckedListBox、UIComboBox で実装されます。
    /// </summary>
    public interface IUIListControl
    {
        /// <summary>
        /// イメージリスト。
        /// 今のところ UIListBox、UIComboBox でのみ使用されます。
        /// UICheckedListBox は OwnerDraw に対応していないので使用されません。
        /// </summary>
        ImageList ImageList
        {
            get;
            set;
        }
    }

    #region UIListControlItem
    /// <summary>
    /// ＵＩリストコントロール項目クラス。
    /// UIListBox、UICheckedListBox、UICombox の項目に使用します。
    /// </summary>
    public sealed class UIListControlItem
    {
        // テキスト
        private string _text;
        // データ
        private object _data;
        // イメージインデクス
        private int _imageIndex = -1;
        // アイコン
        private Image _icon = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIListControlItem(string text, object data)
            : this(text, data, -1, null)
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIListControlItem(string text, object data, int imageIndex)
            : this(text, data, imageIndex, null)
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIListControlItem(string text, object data, Image icon)
            : this(text, data, -1, icon)
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIListControlItem(string text, object data, int imageIndex, Image icon)
        {
            _text       = text;
            _data       = data;
            _imageIndex = imageIndex;
            _icon       = icon;
        }

        /// <summary>
        /// テキスト。
        /// </summary>
        public string Text
        {
            get { return _text;  }
            set { _text = value; }
        }

        /// <summary>
        /// データ。
        /// </summary>
        public object Data
        {
            get { return _data;  }
            set { _data = value; }
        }

        /// <summary>
        /// イメージインデクス。
        /// </summary>
        public int ImageIndex
        {
            get { return _imageIndex;  }
            set { _imageIndex = value; }
        }

        /// <summary>
        /// アイコン。
        /// </summary>
        public Image Icon
        {
            get { return _icon;  }
            set { _icon = value; }
        }

        /// <summary>
        /// イメージ付き項目か。
        /// </summary>
        public bool IsImageEnabled
        {
            get { return _imageIndex > -1 || _icon != null; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string ToString()
        {
            // CheckedListBox からテキスト表示用に呼ばれる
            return _text;
        }
    }
    #endregion

    #region CustomDrawListControlItemEvent
    /// <summary>
    /// リストコントロール項目カスタム描画イベントハンドラデリゲート。
    /// </summary>
    public delegate void CustomDrawListControlItemEventHandler(object sender, CustomDrawListControlItemEventArgs e);

    /// <summary>
    /// リストコントロール項目カスタム描画イベントデータクラス。
    /// </summary>
    public sealed class CustomDrawListControlItemEventArgs : EventArgs
    {
        // 内部イベントデータ
        private readonly DrawItemEventArgs _innerEventArgs = null;
        // オーナー
        private readonly IUIListControl _owner = null;
        // 項目
        private readonly UIListControlItem _item = null;
        // 既定処理を行うか
        private bool _doDefaultProcess = true;

        // 描画属性
        private Color _foreColor = Color.Empty;
        private Color _backColor = Color.Empty;
        private bool  _boldFont  = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public CustomDrawListControlItemEventArgs(DrawItemEventArgs innerEventArgs, IUIListControl owner, UIListControlItem item)
        {
            _innerEventArgs = innerEventArgs;
            _owner          = owner;
            _item           = item;
            _foreColor      = innerEventArgs.ForeColor;
            _backColor      = innerEventArgs.BackColor;
        }

        #region プロパティ
        /// <summary>
        /// 項目。
        /// 対応する項目がない場合は null になります。
        /// </summary>
        public UIListControlItem Item
        {
            get { return _item; }
        }

        /// <summary>
        /// グラフィックス。
        /// </summary>
        public Graphics Graphics
        {
            get { return _innerEventArgs.Graphics; }
        }

        /// <summary>
        /// フォント。
        /// </summary>
        public Font Font
        {
            get { return _innerEventArgs.Font; }
        }

        /// <summary>
        /// 領域。
        /// </summary>
        public Rectangle Bounds
        {
            get { return _innerEventArgs.Bounds; }
        }

        /// <summary>
        /// 項目番号。
        /// </summary>
        public int Index
        {
            get { return _innerEventArgs.Index; }
        }

        /// <summary>
        /// 前景色。
        /// </summary>
        public Color ForeColor
        {
            get { return _foreColor;  }
            set { _foreColor = value; }
        }

        /// <summary>
        /// 背景色。
        /// </summary>
        public Color BackColor
        {
            get { return _backColor;  }
            set { _backColor = value; }
        }

        /// <summary>
        /// 太字フォント使用フラグ。
        /// </summary>
        public bool BoldFont
        {
            get { return _boldFont;  }
            set { _boldFont = value; }
        }

        /// <summary>
        /// 既定処理を行うか。
        /// ユーザーが独自で描画処理を行った場合は false にして下さい。
        /// </summary>
        public bool DoDefaultProcess
        {
            get { return _doDefaultProcess;  }
            set { _doDefaultProcess = value; }
        }
        #endregion

        #region 状態確認
        /// <summary>
        /// 状態を確認。
        /// </summary>
        public bool CheckState(DrawItemState state)
        {
            return (_innerEventArgs.State & state) != 0;
        }
        #endregion

        //
        // TODO: 描画処理が中途半端なので見直す
        //

        #region 描画
        /// <summary>
        /// デフォルト描画処理。
        /// </summary>
        public void DrawDefault()
        {
//			DebugConsole.WriteLine("draw UIListControl item:[{0}] - {1}", _item != null ? _item.Text : "null", _innerEventArgs.State);

            // 背景
            DrawBackground();

            // 項目
            if (_item != null)
            {
                // 画像
                if (_item.IsImageEnabled)
                {
                    Rectangle rcImage = Bounds;
                    rcImage.X += 1;
                    rcImage.Y += (Bounds.Height - 16) / 2;
                    rcImage.Size = new Size(16, 16);

                    // アイコン（優先度：先）
                    if (_item.Icon != null)
                    {
                        Graphics.DrawImage(_item.Icon, rcImage);
                    }
                    // イメージリスト（優先度：後）
                    else if (_owner.ImageList != null)
                    {
                        ImageList imageList = _owner.ImageList;
                        if (imageList.Images.Count > _item.ImageIndex)
                        {
                            Graphics.DrawImage(imageList.Images[_item.ImageIndex], rcImage);
                        }
                    }
                }

                // テキスト
                Rectangle rcText = Bounds;
                if (_item.IsImageEnabled)
                {
                    RectangleUtility.OffsetLeft(ref rcText, Bounds.Height + 2);
                }
                else
                {
                    RectangleUtility.OffsetLeft(ref rcText, 1);
                }
                using (StringFormat sf = new StringFormat())
                {
                    sf.Alignment     = StringAlignment.Near;
                    sf.LineAlignment = StringAlignment.Center;
                    sf.FormatFlags  |= StringFormatFlags.NoWrap;

                    // カラー
                    Color color = SystemColors.WindowText;
                    if (((Control)_owner).Enabled)
                    {
                        if (_foreColor != Color.Empty)
                        {
                            color = _foreColor;
                        }
                        else if (CheckState(DrawItemState.Selected))
                        {
                            color = SystemColors.HighlightText;
                        }
                    }
                    else
                    {
                        color = SystemColors.GrayText;
                    }

                    // フォント
                    Font font = Font;
                    if (_boldFont)
                    {
                        font = TheApp.GuiFontBold;
                    }

                    // 描画
                    using (Brush brush = new SolidBrush(color))
                    {
                        Graphics.DrawString(_item.Text, font, brush, rcText, sf);
                    }
                }
            }

            // フォーカス
            DrawFocusRectangle();
        }

        /// <summary>
        /// 背景描画。
        /// </summary>
        public void DrawBackground()
        {
            // 有効状態
            if (((Control)_owner).Enabled)
            {
                // 背景色指定
                if (_backColor != Color.Empty && !CheckState(DrawItemState.Selected))
                {
                    using (Brush brush = new SolidBrush(_backColor))
                    {
                        Graphics.FillRectangle(brush, Bounds);
                    }
                }
                else
                {
                    _innerEventArgs.DrawBackground();
                }
            }
            // 無効状態
            else
            {
                Graphics.FillRectangle(SystemBrushes.Control, Bounds);
            }
        }

        /// <summary>
        /// フォーカス枠描画。
        /// </summary>
        public void DrawFocusRectangle()
        {
            _innerEventArgs.DrawFocusRectangle();
        }
        #endregion
    }
    #endregion
}
