﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

namespace App.Controls
{
    /// <summary>
    /// ＵＩリストボックスクラス。
    /// 項目は常に UIListControlItem 型で扱います。
    /// </summary>
    public class UIListBox : ListBox, IUIListControl
    {
        // イベント制御フラグ
        private bool _invokeEvent = true;
        // 選択番号保存値
        private int _preSelectedIndex = -1;
        // イメージリスト
        private ImageList _imageList = null;

        /// <summary>カスタム項目描画イベント。</summary>
        public event CustomDrawListControlItemEventHandler CustomDrawItem = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIListBox()
        {
            base.DrawMode       = DrawMode.OwnerDrawFixed;
            base.IntegralHeight = false;
            base.ItemHeight     = 16;
        }

        /// <summary>
        /// イメージリスト。
        /// </summary>
        [Localizable(false)]
        [DefaultValue(null)]
        [Description("項目の描画に使用するイメージリストです。")]
        public ImageList ImageList
        {
            get { return _imageList;  }
            set { _imageList = value; }
        }

        /// <summary>
        /// 再定義。
        /// UIListControlItem 型で扱う。
        /// </summary>
        [Browsable(false)]
        [DefaultValue(null)]
        public new UIListControlItem SelectedItem
        {
            get { return (UIListControlItem)base.SelectedItem; }
            set { base.SelectedItem = value;                   }
        }

        /// <summary>
        /// 選択項目データ。
        /// 各項目は UIListControlItem 型である必要があります。
        /// </summary>
        [Browsable(false)]
        [DefaultValue(null)]
        public object SelectedItemData
        {
            get
            {
                UIListControlItem item = (UIListControlItem)base.SelectedItem;
                if (item != null)
                {
                    return item.Data;
                }
                return null;
            }
            set
            {
                // データが null の項目を探す
                if (value == null)
                {
                    foreach (UIListControlItem item in base.Items)
                    {
                        if (item.Data == null)
                        {
                            base.SelectedItem = item;
                            return;
                        }
                    }
                }
                // データが等しい項目を探す
                else
                {
                    foreach (UIListControlItem item in Items)
                    {
                        if (item.Data != null && item.Data.Equals(value))
                        {
                            base.SelectedItem = item;
                            return;
                        }
                    }
                }

                // 該当する項目が見つからない
                base.SelectedItem = null;
            }
        }

        #region 項目追加
        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(string text, object data)
        {
            AddItem(new UIListControlItem(text, data));
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(string text, object data, int imageIndex)
        {
            AddItem(new UIListControlItem(text, data, imageIndex));
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(string text, object data, Image icon)
        {
            AddItem(new UIListControlItem(text, data, icon));
        }

        /// <summary>
        /// 項目追加（文字列リソース指定）。
        /// </summary>
        public void AddItemRes(string stringID, object data)
        {
            AddItem(new UIListControlItem(stringID, data));
        }

        /// <summary>
        /// 項目追加（文字列リソース指定）。
        /// </summary>
        public void AddItemRes(string stringID, object data, int imageIndex)
        {
            AddItem(new UIListControlItem(stringID, data, imageIndex));
        }

        /// <summary>
        /// 項目追加（文字列リソース指定）。
        /// </summary>
        public void AddItemRes(string stringID, object data, Image icon)
        {
            AddItem(new UIListControlItem(stringID, data, icon));
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(UIListControlItem item)
        {
            base.Items.Add(item);
        }
        #endregion

        /// <summary>
        /// 項目データ取得。
        /// </summary>
        public object GetItemData(int index)
        {
            UIListControlItem item = (UIListControlItem)base.Items[index];
            return item.Data;
        }

        /// <summary>
        /// SelectedIndexプロパティ設定。
        /// </summary>
        public void SetSelectedIndex(int index)
        {
            _invokeEvent = false;
            {
                base.SelectedIndex = index;
            }
            _invokeEvent = true;
        }

        /// <summary>
        /// SelectedItemプロパティ設定。
        /// </summary>
        public void SetSelectedItem(UIListControlItem item)
        {
            _invokeEvent = false;
            {
                base.SelectedItem = item;
            }
            _invokeEvent = true;
        }

        /// <summary>
        /// SelectedItemDataプロパティ設定。
        /// </summary>
        public void SetSelectedItemData(object data)
        {
            _invokeEvent = false;
            {
                SelectedItemData = data;
            }
            _invokeEvent = true;
        }

        /// <summary>
        /// カスタム項目描画ハンドラ。
        /// </summary>
        protected virtual void OnCustomDrawItem(CustomDrawListControlItemEventArgs e)
        {
            if (CustomDrawItem != null)
            {
                CustomDrawItem(this, e);
            }
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new int ColumnWidth
        {
            get { return base.ColumnWidth;  }
            set { base.ColumnWidth = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new object DataSource
        {
            get { return base.DataSource;  }
            set { base.DataSource = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new string DisplayMember
        {
            get { return base.DisplayMember;  }
            set { base.DisplayMember = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(DrawMode.OwnerDrawFixed)]
        public override DrawMode DrawMode
        {
            get { return base.DrawMode;  }
            set { base.DrawMode = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(true)]
        public new bool FormattingEnabled
        {
            // 基本クラスで true に初期化されるが DefaultValue が設定されていないため、
            // デザイナコードで設定コードが入ってしまうのを防ぐ
            get { return base.FormattingEnabled;  }
            set { base.FormattingEnabled = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new int HorizontalExtent
        {
            get { return base.HorizontalExtent;  }
            set { base.HorizontalExtent = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool HorizontalScrollbar
        {
            get { return base.HorizontalScrollbar;  }
            set { base.HorizontalScrollbar = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new ImeMode ImeMode
        {
            get { return base.ImeMode;  }
            set { base.ImeMode = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(false)]
        public new bool IntegralHeight
        {
            get { return base.IntegralHeight;  }
            set { base.IntegralHeight = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(16)]
        public override int ItemHeight
        {
            get { return base.ItemHeight;  }
            set { base.ItemHeight = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        public new ListBox.ObjectCollection Items
        {
            get { return base.Items; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool MultiColumn
        {
            get { return base.MultiColumn;  }
            set { base.MultiColumn = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft;  }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new SelectionMode SelectionMode
        {
            get { return base.SelectionMode;  }
            set { base.SelectionMode = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool Sorted
        {
            get { return base.Sorted;  }
            set { base.Sorted = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool TabStop
        {
            get { return base.TabStop; }
            set { base.TabStop = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseTabStops
        {
            get { return base.UseTabStops;  }
            set { base.UseTabStops = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new string ValueMember
        {
            get { return base.ValueMember;  }
            set { base.ValueMember = value; }
        }
        #endregion

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnSelectedIndexChanged(EventArgs e)
        {
            if (_invokeEvent && (_preSelectedIndex != base.SelectedIndex))
            {
                base.OnSelectedIndexChanged(e);
            }
            _preSelectedIndex = base.SelectedIndex;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnSelectedValueChanged(EventArgs e)
        {
            if (_invokeEvent && (_preSelectedIndex != base.SelectedIndex))
            {
                base.OnSelectedValueChanged(e);
            }
            _preSelectedIndex = base.SelectedIndex;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnDrawItem(DrawItemEventArgs e)
        {
            if (base.DesignMode) { return; }

            // ちらつき軽減
            // Todo:まだ多少ちらつく
            using (var bufferedGraphics = BufferedGraphicsManager.Current.Allocate(e.Graphics, e.Bounds))
            {
                var newArgs = new DrawItemEventArgs(
                    bufferedGraphics.Graphics,
                    e.Font, e.Bounds, e.Index, e.State, e.ForeColor, e.BackColor
                );

                OnDrawItemInternal(newArgs);

                bufferedGraphics.Render(e.Graphics);
            }
        }

        private void OnDrawItemInternal(DrawItemEventArgs e)
        {
            if (base.DesignMode) { return; }

            // 描画項目
            UIListControlItem item = null;
            if (e.Index >= 0)
            {
                item = (UIListControlItem)base.Items[e.Index];
            }

            // カスタム描画イベント発行
            CustomDrawListControlItemEventArgs ce = new CustomDrawListControlItemEventArgs(e, this, item);
            OnCustomDrawItem(ce);

            // 既定描画処理
            if (ce.DoDefaultProcess)
            {
                ce.DrawDefault();
            }
        }

        #endregion
    }
}
