﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Linq;

namespace App.Controls
{
    /// <summary>
    /// ＵＩグループボックスクラス。
    /// </summary>
    public class UIGroupBox : GroupBox
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIGroupBox()
        {
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        #region デザイナ制御
        /// <summary>
        /// ダブルバッファリングフラグ。
        /// </summary>
        [DefaultValue(false)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description(UIControlHelper.PropertyDesc_DoubleBufferring)]
        public bool DoubleBuffering
        {
            get { return base.DoubleBuffered; }
            set { base.DoubleBuffered = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new FlatStyle FlatStyle
        {
            get { return base.FlatStyle;  }
            set { base.FlatStyle = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft;  }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        public new bool TabStop
        {
            // 基本クラスで false に初期化されるが DefaultValue が設定されていないため、
            // デザイナコードで設定コードが入ってしまうのを防ぐ
            get { return base.TabStop;  }
            set { base.TabStop = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseCompatibleTextRendering
        {
            get { return base.UseCompatibleTextRendering;  }
            set { base.UseCompatibleTextRendering = value; }
        }
        #endregion
    }

    public class UIModifiedMarkGroupBox : UIGroupBox
    {
        public bool IsModified
        {
            get
            {
                return modified;
            }
            set
            {
                if (modified != value)
                {
                    modified = value;
                    Invalidate();
                }
            }
        }
        private bool modified;

        protected override void OnTextChanged(EventArgs e)
        {
            using (var g = CreateGraphics())
            {
                // サイズ、位置を計算し直す
                var textStringSize = TextRenderer.MeasureText(g, Text, Font);

                markPosX_ = textOffsetX + textStringSize.Width;
            }

            Invalidate();
        }

        private const int textOffsetX  = 5;
        private const int textOffsetY  = 1;
        private int markPosX_ = textOffsetX;

        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            if (IsModified || DesignMode)
            {
                e.Graphics.FillRectangle(SystemBrushes.Control, markPosX_ - 1, textOffsetY, ModifiedMark.StarMark.Width + 3, ModifiedMark.StarMark.Height);
                e.Graphics.DrawImage(ModifiedMark.StarMark, markPosX_, textOffsetY);
            }
        }
    }

    /// <summary>
    /// ＵＩグループボックスクラス。EffectMaker4F から流用
    /// </summary>
    public class UITitleBarGroupBox : UIUserControl
    {
        #region Constructor

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UITitleBarGroupBox()
        {
            DoubleBuffered = true;

            // Align middle left
            m_titleFormat.Alignment = StringAlignment.Near;
            m_titleFormat.LineAlignment = StringAlignment.Center;

            // Create the pens and brushes for rendering.
            m_penTitleBar = new Pen(m_titleBarColor);
            m_brushTitleBar = new SolidBrush(m_titleBarColor);
            m_brushTitleText = new SolidBrush(m_titleTextColor);
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the flag whether to draw the group box or not.
        /// </summary>
        [DefaultValue(true)]
        public bool DrawGroupBox
        {
            get { return m_bDrawGroupBox; }
            set
            {
                m_bDrawGroupBox = value;
                Invalidate();
            }
        }


        /// <summary>
        /// Get or set the radius of the rounded corners.
        /// </summary>
        [DefaultValue(6)]
        public int CornerRadius
        {
            get { return m_iCornerRadius; }
            set
            {
                m_iCornerRadius = value;
                Invalidate();
            }
        }


        /// <summary>
        /// Get or set the height of the title bar
        /// </summary>
        [DefaultValue(18)]
        public int TitleBarHeight
        {
            get { return m_iTitleBarHeight; }
            set
            {
                m_iTitleBarHeight = value;
                Invalidate();
            }
        }


        /// <summary>
        /// Get or set the color of the title bar
        /// </summary>
        public Color TitleBarColor
        {
            get { return m_titleBarColor; }
            set
            {
                m_titleBarColor = value;

                m_penTitleBar.Color = value;
                m_brushTitleBar.Color = value;

                Invalidate();
            }
        }


        /// <summary>
        /// Get or set the text color of the title
        /// </summary>
        public Color TitleTextColor
        {
            get { return m_titleTextColor; }
            set
            {
                m_titleTextColor = value;

                m_brushTitleText.Color = value;

                Invalidate();
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        public new bool TabStop
        {
            get { return base.TabStop; }
            set { base.TabStop = value; }
        }

        public bool IsModified
        {
            get
            {
                return modified;
            }
            set
            {
                if (modified != value)
                {
                    modified = value;
                    Invalidate();
                }
            }
        }

        private bool modified;
        #endregion

        #region Event handlers

        /// <summary>
        /// Handle paint event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            if (m_bDrawGroupBox == false)
            {
                base.OnPaint(e);
                return;
            }

            // Get the graphics object for rendering
            Graphics g = e.Graphics;

            // Smooth the lines
            g.SmoothingMode = SmoothingMode.HighQuality;

            base.OnPaint(e);

            // Measure text size
            int iControlWidth = Width - 1;
            int iControlHeight = Height - 1;

            // Setup the rectangles
            Rectangle contentRect = new Rectangle(0,
                                                   m_iTitleBarHeight,
                                                   iControlWidth,
                                                   iControlHeight - m_iTitleBarHeight);
            Rectangle titleBarRect = new Rectangle(0,
                                                    0,
                                                    iControlWidth,
                                                    m_iTitleBarHeight);
            Rectangle titleTextRect = new Rectangle(CornerRadius,
                                                     0,
                                                     iControlWidth - (CornerRadius * 2),
                                                     m_iTitleBarHeight);

            #region Create path for border

            // Create the path
            GraphicsPath pathBorder = new GraphicsPath();
            pathBorder.AddLine(contentRect.Left,
                                contentRect.Bottom - CornerRadius,
                                contentRect.Left,
                                contentRect.Top);
            pathBorder.AddLine(contentRect.Left,
                                contentRect.Top,
                                contentRect.Right,
                                contentRect.Top);
            pathBorder.AddLine(contentRect.Right,
                                contentRect.Top,
                                contentRect.Right,
                                contentRect.Bottom - CornerRadius);
            pathBorder.AddArc(contentRect.Right - CornerRadius,
                               contentRect.Bottom - CornerRadius,
                               CornerRadius,
                               CornerRadius,
                               0.0f,
                               90.0f);
            pathBorder.AddLine(contentRect.Right - CornerRadius,
                                contentRect.Bottom,
                                contentRect.Left + CornerRadius,
                                contentRect.Bottom);
            pathBorder.AddArc(contentRect.Left,
                               contentRect.Bottom - CornerRadius,
                               CornerRadius,
                               CornerRadius,
                               90.0f,
                               90.0f);
            pathBorder.CloseAllFigures();

            #endregion

            #region Create path for title bar

            // Create the path
            GraphicsPath pathTitle = new GraphicsPath();
            pathTitle.AddArc(titleBarRect.Left,
                              titleBarRect.Top,
                              CornerRadius,
                              CornerRadius,
                              180.0f,
                              90.0f);
            pathTitle.AddLine(titleBarRect.Left + CornerRadius,
                               titleBarRect.Top,
                               titleBarRect.Right - CornerRadius,
                               titleBarRect.Top);
            pathTitle.AddArc(titleBarRect.Right - CornerRadius,
                              titleBarRect.Top,
                              CornerRadius,
                              CornerRadius,
                              270.0f,
                              90.0f);
            pathTitle.AddLine(titleBarRect.Right,
                               titleBarRect.Top + CornerRadius,
                               titleBarRect.Right,
                               titleBarRect.Bottom);
            pathTitle.AddLine(titleBarRect.Right,
                               titleBarRect.Bottom,
                               titleBarRect.Left,
                               titleBarRect.Bottom);
            pathTitle.AddLine(titleBarRect.Left,
                               titleBarRect.Bottom,
                               titleBarRect.Left,
                               titleBarRect.Top + CornerRadius);
            pathTitle.CloseAllFigures();

            #endregion

            g.DrawPath(m_penTitleBar, pathBorder);
            g.FillPath(m_brushTitleBar, pathTitle);
            g.DrawPath(m_penTitleBar, pathTitle);

            g.DrawString(Text,
                          Font,
                          m_brushTitleText,
                          titleTextRect,
                          m_titleFormat);

            if (modified || DesignMode)
            {
                var size = g.MeasureString(Text, Font);
                var image = ModifiedMark.StarMark;
                e.Graphics.DrawImage(image, (int)(titleTextRect.Left + size.Width + 1), (int)((titleTextRect.Top + titleTextRect.Bottom - image.Height) / 2));
            }
        }


        /// <summary>
        /// Handle size changed event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnSizeChanged(System.EventArgs e)
        {
            base.OnSizeChanged(e);

            Invalidate();
        }

        protected override void OnTextChanged(EventArgs e)
        {
            base.OnTextChanged(e);

            // 画像配置位置は Text に依存する。
            // 配置は ShowImage で行っているので、すべての画像を削除してから再表示する。
            var imgPanels = showingImagePanels_.ToArray();
            foreach (var imgPanel in imgPanels)
            {
                HideImage(imgPanel.BackgroundImage);
            }
            foreach (var imgPanel in imgPanels)
            {
                ShowImage(imgPanel.BackgroundImage, imgPanel.ToolTipText);
            }
        }

        #endregion

        // リストの並び順は ImagePanel 配置順 (左詰め) を前提。
        private System.Collections.Generic.List<ImagePanel> showingImagePanels_ = new System.Collections.Generic.List<ImagePanel>();

        /// <summary>
        /// 画像を表示する。
        /// </summary>
        /// <param name="image">表示画像</param>
        /// <param name="imageToolTipString">ツールチップ文字列</param>
        public void ShowImage(Image image, string imageToolTipString)
        {
            HideImage(image);

            var showingImagePanel = new ImagePanel();
            {
                showingImagePanel.BackgroundImage = image;
                showingImagePanel.ToolTipText = imageToolTipString;
                showingImagePanel.Size = image.Size;
                showingImagePanel.BackColor = TitleBarColor;
            }

            using (var g = CreateGraphics())
            {
                int iControlWidth = Width - 1;

                Rectangle titleTextRect = new Rectangle(
                    CornerRadius,
                    0,
                    iControlWidth - (CornerRadius * 2),
                    m_iTitleBarHeight);

                var textSize = g.MeasureString(Text, Font);
                int buttonWidth = 30;
                var x = showingImagePanels_.Any() ? (showingImagePanels_.Last().Bounds.Right + 6) : ((int)(titleTextRect.Left + textSize.Width + 1) + (ModifiedMark.StarMark.Width + 6) + buttonWidth);
                var y = (int)((titleTextRect.Top + titleTextRect.Bottom - image.Height) / 2);

                showingImagePanel.Location = new Point(x, y);
            }

            showingImagePanels_.Add(showingImagePanel);
            Controls.Add(showingImagePanel);
        }

        /// <summary>
        /// 画像を非表示にする
        /// </summary>
        /// <param name="image">ShowImage() に設定した画像</param>
        public void HideImage(Image image)
        {
            int n = 0;
            foreach (var imgPanel in showingImagePanels_.ToArray())
            {
                if (imgPanel.BackgroundImage == image)
                {
                    // コントロールを取り除く。
                    var imgPanelParent = imgPanel.Parent;
                    if (imgPanelParent != null)
                    {
                        imgPanelParent.Controls.Remove(imgPanel);
                    }

                    // 後続コントロールを左詰めする幅を計算。
                    var currIdx = showingImagePanels_.IndexOf(imgPanel);
                    var nextIdx = currIdx + 1;
                    if (nextIdx < showingImagePanels_.Count)
                    {
                        var nextImgPanel = showingImagePanels_[nextIdx];
                        n += nextImgPanel.Location.X - imgPanel.Location.X;
                    }
                    showingImagePanels_.Remove(imgPanel);
                }
                else
                {
                    // 前方で取り除いた分を左詰めする。
                    var newLocation = new Point(imgPanel.Location.X - n, imgPanel.Location.Y);
                    imgPanel.Location = newLocation;
                }
            }
        }

        public int ModifiedMarkRight()
        {
            using (var g = CreateGraphics())
            {
                int iControlWidth = Width - 1;

                Rectangle titleTextRect = new Rectangle(CornerRadius,
                                                         0,
                                                         iControlWidth - (CornerRadius * 2),
                                                         m_iTitleBarHeight);

                var textSize = g.MeasureString(Text, Font);
                var x = (int)(titleTextRect.Left + textSize.Width + 1) + (ModifiedMark.StarMark.Width + 6);
                return x;
            }
        }

        #region Member variables

        private bool m_bDrawGroupBox = true;

        private Color m_titleTextColor = Color.FromArgb(255, 255, 255, 255);
        public readonly static Color DefaultTitleTextColor = Color.FromArgb(255, 255, 255, 255);
        private Color m_titleBarColor = DefaultTitleBarColor;

        public readonly static Color DefaultTitleBarColor = Color.FromArgb(255, 71, 91, 128);

        private Color m_borderColor = Color.FromArgb(255, 71, 91, 128);
        private int m_iCornerRadius = 6;
        private int m_iTitleBarHeight = 18;

        private readonly Pen m_penTitleBar = null;
        private readonly SolidBrush m_brushTitleBar = null;
        private readonly SolidBrush m_brushTitleText = null;

        private readonly StringFormat m_titleFormat = new StringFormat();

        #endregion
    }
}
