﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using System.Linq;

namespace App.Controls
{
    /// <summary>
    /// ＵＩコンボボックスクラス。
    /// </summary>
    public class UIComboBox2 : ComboBox, KeyEditableControl
    {
        // 値
        private string _value = string.Empty;

        public bool CanKeyEdit		// interface KeyEditableControl
        {
            get
            {
                return DropDownStyle != ComboBoxStyle.DropDownList;
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIComboBox2()
        {
            base.MaxDropDownItems = 30;
            ExpandDropDownWidth = true;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            // リターン入力処理
            if (keyData == Keys.Return)
            {
                ParseText(true);

                var handler = (EventHandler)base.Events[EVENT_RETURNKEYDOWN];
                if (handler != null)
                {
                    handler(this, EventArgs.Empty);
                }

                return true;
            }
            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // マウスホイール
                case Win32.WM.WM_MOUSEWHEEL:
                    {
                        if (!Focused)
                        {
                            // 親に委譲する
                            Win32.NativeMethods.SendMessage(Parent.Handle, m.Msg, m.WParam, m.LParam);
                            return;
                        }
                        // フォーカスありでドロップダウンが出ていない
                        else if (!DroppedDown)
                        {
                            // コントロール領域外なら親に委譲
                            Point point = PointToClient(Win32.Utility.LParamToPoint(m.LParam));
                            if (!ClientRectangle.Contains(point))
                            {
                                Win32.NativeMethods.SendMessage(Parent.Handle, m.Msg, m.WParam, m.LParam);
                                return;
                            }
                        }
                        break;
                    }
                default:
                    break;
            }
            base.WndProc(ref m);
        }

        protected override void OnSelectedIndexChanged(EventArgs e)
        {
            // CanRaiseEvents が効かないので、呼び出し条件を直接判断する
            if (CanRaiseEvents)
            {
                ParseText(true);
                base.OnSelectedIndexChanged(e);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            ParseText(false);
            base.OnLostFocus(e);
        }

        /// <summary>
        /// テキストのパース処理。
        /// </summary>
        private void ParseText(bool withSelectAll)
        {
            // 値を更新して再度テキストに反映
            bool valueChanged = UpdateValue();
            UpdateText();

            // 全選択状態にする
            if (withSelectAll)
            {
                SelectAll();
            }

            // 最後にイベントを発行
            if (valueChanged)
            {
                OnValueChanged(EventArgs.Empty);
            }
        }

        // Value と Text の違い
        //   => Valueは内部で保持している値
        //   => Text は表示用の値

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue("")]
        [Description("値。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public string Value
        {
            get { return _value; }
            set
            {
                if (_value != (value ?? ""))
                {
                    _value = value ?? "";
//					UpdateText();
                    OnValueChanged(EventArgs.Empty);
                }

                UpdateText();
            }
        }

        private bool UpdateValue()
        {
            string prev = _value;
            _value = Text;
            return _value != prev;
        }

        private void UpdateText()
        {
            Text = _value;
        }

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_VALUECHANGED = new object();
        private static readonly object EVENT_RETURNKEYDOWN = new object();

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("値が変更された時に発生します。")]
        public event EventHandler ValueChanged
        {
            add { base.Events.AddHandler(EVENT_VALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_VALUECHANGED, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        protected virtual void OnValueChanged(EventArgs e)
        {
            EventHandler handler = (EventHandler)base.Events[EVENT_VALUECHANGED];
            if (handler != null)
            {
                DebugConsole.WriteLine("ValueInputBox: OnValueChanged");
                UIControlHelper.EventMessage("ValueChanged", this, e);
                handler(this, e);
            }
        }

        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("リターンキーが押された時に発生します。")]
        public event EventHandler ReturnKeyDown
        {
            add { base.Events.AddHandler(EVENT_RETURNKEYDOWN, value); }
            remove { base.Events.RemoveHandler(EVENT_RETURNKEYDOWN, value); }
        }

        protected override void OnDropDown(EventArgs e)
        {
            if (ExpandDropDownWidth)
            {
                // float の加算や int が収まるように double 型を用いる。
                double max_width = (double)Width;

                using (Graphics g = CreateGraphics())
                {
                    var char_width = g.MeasureString("■", Font).Width;        // キツキツになりすぎるので１文字分幅を広げる

                    foreach (var item in Items)
                    {
                        var itemType = item.GetType();
                        var dispMemInfo = !string.IsNullOrEmpty(DisplayMember) ? itemType.GetProperty(DisplayMember) : null;
                        var s = (dispMemInfo != null) ? dispMemInfo.GetValue(item).ToString() : item.ToString();
                        max_width = Math.Max(max_width, (double)g.MeasureString(s, Font).Width + (double)char_width);
                    }
                }

                // 画面サイズを拡大限界にする。
                // 画面サイズ取得はマルチディスプレイ環境を考慮する。
                var bounds = Screen.AllScreens.Any() ? Screen.AllScreens.Select(x => x.Bounds).Aggregate(Rectangle.Union) : Screen.PrimaryScreen.Bounds;

                DropDownWidth = (int)Math.Min(max_width, (double)bounds.Width);
            }

            base.OnDropDown(e);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(30)]
        public new int MaxDropDownItems
        {
            get { return base.MaxDropDownItems; }
            set { base.MaxDropDownItems = value; }
        }

        /// <summary>
        /// ドロップダウン時に幅を自動拡大するかどうか。
        /// </summary>
        [DefaultValue(true)]
        public bool ExpandDropDownWidth { get; set; }
        #endregion
    }

    /// <summary>
    /// ＵＩコンボボックスクラス。
    /// 項目は常に UIListControlItem 型で扱います。
    /// </summary>
    public class UIComboBox : ComboBox, IUIListControl, KeyEditableControl
    {
        // イメージリスト
        private ImageList _imageList = null;

        private static readonly object EVENT_RETURNKEYDOWN = new object();

        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("リターンキーが押された時に発生します。")]
        public event EventHandler ReturnKeyDown
        {
            add { base.Events.AddHandler(EVENT_RETURNKEYDOWN, value); }
            remove { base.Events.RemoveHandler(EVENT_RETURNKEYDOWN, value); }
        }

        /// <summary>
        /// カスタム項目描画イベント。
        /// イベント抑制対象にはなりません。
        /// </summary>
        public event CustomDrawListControlItemEventHandler CustomDrawItem = null;

        public bool CanKeyEdit		// interface KeyEditableControl
        {
            get
            {
                return DropDownStyle != ComboBoxStyle.DropDownList;
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIComboBox()
        {
            base.DrawMode         = DrawMode.OwnerDrawFixed;
            base.DropDownStyle    = ComboBoxStyle.DropDownList;
            base.ItemHeight       = 16;
            base.MaxDropDownItems = 30; // それなりの適当値
        }

        /// <summary>
        /// イメージリスト。
        /// </summary>
        [Localizable(false)]
        [DefaultValue(null)]
        [Description("項目の描画に使用するイメージリストです。")]
        public ImageList ImageList
        {
            get { return _imageList;  }
            set { _imageList = value; }
        }

        /// <summary>
        /// 選択項目。
        /// UIListControlItem 型で再定義しています。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new UIListControlItem SelectedItem
        {
            get { return (UIListControlItem)base.SelectedItem; }
            set { base.SelectedItem = value; }
        }

        /// <summary>
        /// 選択項目データ。
        /// 各項目は UIListControlItem 型である必要があります。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object SelectedItemData
        {
            get
            {
                UIListControlItem item = (UIListControlItem)base.SelectedItem;
                if (item != null)
                {
                    return item.Data;
                }
                return null;
            }
            set
            {
                // データが null の項目を探す
                if (value == null)
                {
                    foreach (UIListControlItem item in base.Items)
                    {
                        if (item.Data == null)
                        {
                            base.SelectedItem = item;
                            return;
                        }
                    }
                }
                // データが等しい項目を探す
                else
                {
                    foreach (UIListControlItem item in Items)
                    {
                        if (item.Data != null && item.Data.Equals(value))
                        {
                            base.SelectedItem = item;
                            return;
                        }
                    }
                }

                // 該当する項目が見つからない
                base.SelectedItem = null;
            }
        }

        /// <summary>
        /// 選択項目データ。
        /// 各項目は UIListControlItem 型である必要があります。
        /// </summary>
        [DefaultValue(true)]
        public bool ExpandDropDownWidth { get; set; }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(string text, object data)
        {
            AddItem(new UIListControlItem(text, data));
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(string text, object data, int imageIndex)
        {
            AddItem(new UIListControlItem(text, data, imageIndex));
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(string text, object data, Image icon)
        {
            AddItem(new UIListControlItem(text, data, icon));
        }

        /// <summary>
        /// 項目追加（文字列リソース指定）。
        /// </summary>
        public void AddItemRes(string stringID, object data)
        {
            AddItem(new UIListControlItem(stringID, data));
        }

        /// <summary>
        /// 項目追加（文字列リソース指定）。
        /// </summary>
        public void AddItemRes(string stringID, object data, int imageIndex)
        {
            AddItem(new UIListControlItem(stringID, data, imageIndex));
        }

        /// <summary>
        /// 項目追加（文字列リソース指定）。
        /// </summary>
        public void AddItemRes(string stringID, object data, Image icon)
        {
            AddItem(new UIListControlItem(stringID, data, icon));
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void AddItem(UIListControlItem item)
        {
            base.Items.Add(item);
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void InsertItem(int index, string text, object data)
        {
            InsertItem(index, new UIListControlItem(text, data));
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void InsertItem(int index, string text, object data, int imageIndex)
        {
            InsertItem(index, new UIListControlItem(text, data, imageIndex));
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void InsertItem(int index, string text, object data, Image icon)
        {
            InsertItem(index, new UIListControlItem(text, data, icon));
        }

        /// <summary>
        /// 項目追加（文字列リソース指定）。
        /// </summary>
        public void InsertItemRes(int index, string stringID, object data)
        {
            InsertItem(index, new UIListControlItem(stringID, data));
        }

        /// <summary>
        /// 項目追加（文字列リソース指定）。
        /// </summary>
        public void InsertItemRes(int index, string stringID, object data, int imageIndex)
        {
            InsertItem(index, new UIListControlItem(stringID, data, imageIndex));
        }

        /// <summary>
        /// 項目追加（文字列リソース指定）。
        /// </summary>
        public void InsertItemRes(int index, string stringID, object data, Image icon)
        {
            InsertItem(index, new UIListControlItem(stringID, data, icon));
        }

        /// <summary>
        /// 項目追加。
        /// </summary>
        public void InsertItem(int index, UIListControlItem item)
        {
            base.Items.Insert(index, item);
        }

        /// <summary>
        /// データが含まれているか？
        /// </summary>
        public UIListControlItem FindItemFromItemData(object data)
        {
            foreach (UIListControlItem item in base.Items)
            {
                if (item.Data == data)
                {
                    return item;
                }
            }

            return null;
        }

        /// <summary>
        /// 項目データ取得。
        /// </summary>
        public object GetItemData(int index)
        {
            UIListControlItem item = (UIListControlItem)base.Items[index];
            return item.Data;
        }

        /// <summary>
        /// SelectedIndexプロパティ設定。
        /// </summary>
        public void SetSelectedIndex(int index)
        {
            // TODO:
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                base.SelectedIndex = index;
            }
        }

        /// <summary>
        /// SelectedItemDataプロパティ設定。
        /// </summary>
        public void SetSelectedItemData(object data)
        {
            // TODO:
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                SelectedItemData = data;
            }
        }

        /// <summary>
        /// カスタム項目描画ハンドラ。
        /// </summary>
        protected virtual void OnCustomDrawItem(CustomDrawListControlItemEventArgs e)
        {
            if (CustomDrawItem != null)
            {
                CustomDrawItem(this, e);
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            // リターン入力処理
            if (keyData == Keys.Return)
            {
                //ParseText(true);

                var handler = (EventHandler)base.Events[EVENT_RETURNKEYDOWN];
                if (handler != null)
                {
                    handler(this, EventArgs.Empty);
                }

                return true;
            }
            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // マウスホイール
                case Win32.WM.WM_MOUSEWHEEL:
                    {
                        if (!Focused)
                        {
                            // 親に委譲する
                            Win32.NativeMethods.SendMessage(Parent.Handle, m.Msg, m.WParam, m.LParam);
                            return;
                        }
                        // フォーカスありでドロップダウンが出ていない
                        else if (!DroppedDown)
                        {
                            // コントロール領域外なら親に委譲
                            Point point = PointToClient(Win32.Utility.LParamToPoint(m.LParam));
                            if (!ClientRectangle.Contains(point))
                            {
                                Win32.NativeMethods.SendMessage(Parent.Handle, m.Msg, m.WParam, m.LParam);
                                return;
                            }
                        }
                        break;
                    }
                case Win32.WM.WM_CTLCOLORLISTBOX:
                    // http://www.microsoft.com/japan/msdn/net/winforms/bettercombobox.aspx
                    {
                        if (!ExpandDropDownWidth)
                        {
                            break;
                        }

                        //画面内に収まるようにします
                        int left = PointToScreen(new Point(0, 0)).X;

                        int screen_right = Screen.PrimaryScreen.WorkingArea.Width;
                        int screen_bottom = Screen.PrimaryScreen.WorkingArea.Height;
                        {
                            foreach (Screen s in Screen.AllScreens)
                            {
                                screen_right = Math.Max(screen_right, s.WorkingArea.Right);
                                screen_bottom = Math.Max(screen_bottom, s.WorkingArea.Bottom);
                            }
                        }

                        //ドロップダウンが画面の右端からはみ出す場合のみ、以下の処理を実行します
                        //					if(DropDownWidth > Screen.PrimaryScreen.WorkingArea.Width - left)
                        if (DropDownWidth > screen_right - left)
                        {
                            //ComboBox の現在の位置とサイズを取得します
                            Rectangle comboRect = RectangleToScreen(ClientRectangle);

                            int dropHeight = 0;
                            int topOfDropDown = 0;
                            int leftOfDropDown = 0;

                            //ドロップ リストの高さを計算します
                            for (int i = 0; (i < Items.Count && i < MaxDropDownItems); i++)
                            {
                                dropHeight += ItemHeight;
                            }

                            //ドロップ リストが画面の下端からはみ出す場合には
                            //ドロップダウンの上位置を設定します
                            //						if(dropHeight > Screen.PrimaryScreen.WorkingArea.Height - PointToScreen(new Point(0, 0)).Y)
                            if (dropHeight > screen_bottom - PointToScreen(new Point(0, 0)).Y)
                            {
                                topOfDropDown = comboRect.Top - dropHeight - 2;
                            }
                            else
                            {
                                topOfDropDown = comboRect.Bottom;
                            }

                            //移動先の左位置を計算します
                            //						leftOfDropDown = comboRect.Left - (DropDownWidth - (Screen.PrimaryScreen.WorkingArea.Width - left));
                            leftOfDropDown = comboRect.Left - (DropDownWidth - (screen_right - left));

                            //ドロップダウンの配置とサイズを決定します
                            //SetWindowPos(HWND hWnd,
                            //	  HWND hWndInsertAfter,
                            //	  int X,
                            //	  int Y,
                            //	  int cx,
                            //	  int cy,
                            //	  UINT uFlags);
                            //SWP_NOSIZE フラグを使用する場合、cx および cy パラメーターは無視されます
                            App.Win32.NativeMethods.SetWindowPos(m.LParam,
                              IntPtr.Zero,
                              leftOfDropDown,
                              topOfDropDown,
                              0,
                              0,
                              App.Win32.NativeMethods.SWP_NOSIZE);
                        }
                    }
                    break;
                default:
                    break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override void OnDrawItem(DrawItemEventArgs e)
        {
            if (base.DesignMode) { return; }

            // 描画項目
            UIListControlItem item = null;
            if (e.Index >= 0)
            {
                item = (UIListControlItem)base.Items[e.Index];
            }

            // カスタム描画イベント発行
            CustomDrawListControlItemEventArgs ce = new CustomDrawListControlItemEventArgs(e, this, item);
            OnCustomDrawItem(ce);

            // 既定描画処理
            if (ce.DoDefaultProcess)
            {
                ce.DrawDefault();
            }

            // base.OnDrawItem は呼ばない
        }

        protected override void OnDropDown(EventArgs e)
        {

            if (ExpandDropDownWidth)
            {
                float max_width = (float)Width;

                using (Graphics g = CreateGraphics())
                {

                    float char_width = g.MeasureString("■", Font).Width;		// キツキツになりすぎるので１文字分幅を広げる

                    foreach (UIListControlItem s in Items)
                    {
                        max_width = Math.Max(max_width, g.MeasureString(s.Text, Font).Width + char_width);
                    }
                }

                DropDownWidth = (int)max_width;
            }

            base.OnDropDown(e);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new object DataSource
        {
            get { return base.DataSource; }
            set { base.DataSource = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new string DisplayMember
        {
            get { return base.DisplayMember; }
            set { base.DisplayMember = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(DrawMode.OwnerDrawFixed)]
        public new DrawMode DrawMode
        {
            get { return base.DrawMode; }
            set { base.DrawMode = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(ComboBoxStyle.DropDownList)]
        public new ComboBoxStyle DropDownStyle
        {
            get { return base.DropDownStyle; }
            set { base.DropDownStyle = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new ImeMode ImeMode
        {
            get { return base.ImeMode; }
            set { base.ImeMode = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool IntegralHeight
        {
            get { return base.IntegralHeight; }
            set { base.IntegralHeight = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(16)]
        public new int ItemHeight
        {
            get { return base.ItemHeight; }
            set { base.ItemHeight = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        public new ComboBox.ObjectCollection Items
        {
            get { return base.Items; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(30)]
        public new int MaxDropDownItems
        {
            get { return base.MaxDropDownItems; }
            set { base.MaxDropDownItems = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new int MaxLength
        {
            get { return base.MaxLength; }
            set { base.MaxLength = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft; }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool Sorted
        {
            get { return base.Sorted; }
            set { base.Sorted = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override string Text
        {
            get { return base.Text; }
            set { base.Text = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new string ValueMember
        {
            get { return base.ValueMember; }
            set { base.ValueMember = value; }
        }
        #endregion
    }

    /// <summary>
    /// 特定条件でのクリック動作を変えたコンボボックス
    /// </summary>
    public class UICustomClickCombobox : UIComboBox
    {
        /// <summary>
        /// デフォルトの処理を行うか
        /// </summary>
        public delegate void CustomMouseClickEvent(Point p);

        /// <summary>
        /// カスタムクリックイベント
        /// </summary>
        public event CustomMouseClickEvent CustomMouseClick;

        /// <summary>
        /// カスタムクリックイベントを実行できるか?
        /// </summary>
        public Predicate<Point> IsInCustomArea = ((x)=>false);

        /// <summary>
        /// オーバーライド
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                case Win32.WM.WM_LBUTTONDOWN:
                    {
                        Point p = Win32.Utility.LParamToPoint(m.LParam);
                        if (IsInCustomArea(p))
                        {
                            CustomMouseClick(p);
                            // デフォルトの処理を行わない
                            return;
                        }
                    }
                    break;
                case Win32.WM.WM_LBUTTONDBLCLK:
                    {
                        Point p = Win32.Utility.LParamToPoint(m.LParam);
                        if (IsInCustomArea(p))
                        {
                            // デフォルトの処理を行わない
                            return;
                        }
                    }
                    break;
                case Win32.WM.WM_MOUSEMOVE:
                    {
                        Point p = Win32.Utility.LParamToPoint(m.LParam);
                        if (IsInCustomArea(p))
                        {
                            Cursor = System.Windows.Forms.Cursors.Hand;
                        }
                        else
                        {
                            Cursor = DefaultCursor;
                        }
                    }
                    break;
            }
            base.WndProc(ref m);
        }
    }
}
