﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

namespace App.Controls
{
    /// <summary>
    /// ＵＩボタンクラス。
    /// </summary>
    public class UIButton : Button
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIButton()
        {
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override bool AutoSize
        {
            get { return base.AutoSize; }
            set { base.AutoSize = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new AutoSizeMode AutoSizeMode
        {
            get { return base.AutoSizeMode; }
            set { base.AutoSizeMode = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override RightToLeft RightToLeft
        {
            get { return base.RightToLeft; }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseCompatibleTextRendering
        {
            get { return base.UseCompatibleTextRendering; }
            set { base.UseCompatibleTextRendering = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseVisualStyleBackColor
        {
            get { return base.UseVisualStyleBackColor; }
            set { base.UseVisualStyleBackColor = value; }
        }
        #endregion
    }

    /// <summary>
    /// ドロップダウンボタンクラス。
    /// </summary>
    public sealed class DropDownButton : UIButton
    {
        // ドロップダウンメニュー
        private UIContextMenuStrip _dropDownMenu = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DropDownButton()
        {
        }

        /// <summary>
        /// ドロップダウンメニュー。
        /// </summary>
        [DefaultValue(null)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("クリック時に表示するコンテキストメニュー。")]
        public UIContextMenuStrip DropDownMenu
        {
            get { return _dropDownMenu; }
            set
            {
                if (_dropDownMenu != value)
                {
                    _dropDownMenu = value;
                }
            }
        }

        private bool showDropDownMark = true;

        /// <summary>
        /// ▼マークを見せるかどうか
        /// </summary>
        public bool ShowDropDownMark {
            get { return showDropDownMark; }
            set
            {
                if (showDropDownMark != value)
                {
                    showDropDownMark = value;
                    Invalidate();
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            // 基本描画処理を行なう必要があるので先に呼ぶ
            base.OnPaint(e);

            // ドロップダウンマーク
            if (showDropDownMark)
            {
                Rectangle rcClient = ClientRectangle;
                Rectangle rcMark = new Rectangle(0, 0, 8, 4);
                rcMark.X = rcClient.Width - (rcMark.Width + 6);
                rcMark.Y = (rcClient.Height - rcMark.Height) / 2;
                {
                    Pen pen = SystemPens.ControlText;
                    if (!Enabled)
                    {
                        pen = SystemPens.GrayText;
                    }
                    for (int i = 0; i < rcMark.Height; i++)
                    {
                        e.Graphics.DrawLine(pen, rcMark.Left + i, rcMark.Top + i, (rcMark.Right - 1) - i, rcMark.Top + i);
                    }
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnClick(EventArgs e)
        {
            if (BeforeShowMenu != null)
            {
                BeforeShowMenu();
            }

            // ドロップダウンメニュー表示
            if (_dropDownMenu != null && _dropDownMenu.Items.Count > 0)
            {
                Rectangle rcButton = Parent.RectangleToScreen(Bounds);
                _dropDownMenu.Show(new Point(rcButton.Left + 1, rcButton.Bottom));
            }
            base.OnClick(e);
        }

        /// <summary>
        /// メニュー表示前のアクション
        /// </summary>
        public Action BeforeShowMenu;
    }
}
