﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using App.ConfigData;
using App.Data;
using App.Utility;
using App.res;

namespace App.Controls
{
    /// <summary>
    /// テクスチャ選択パネルクラス。
    /// </summary>
    public sealed partial class TextureSelectPanel : UIUserControl
    {
        // テクスチャ
        private Texture texture_ { get { return lbxTexture.SelectedItem != null ? lbxTexture.SelectedItem.Data as Texture: null; } }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TextureSelectPanel()
        {
            InitializeComponent();

            UpdateComponent();
        }

        /// <summary>
        /// テクスチャ項目数。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public int TextureItemCount
        {
            get { return lbxTexture.Items.Count; }
        }

        public Texture SelectedTexture
        {
            //get { return lbxTexture.SelectedItem != null ? lbxTexture.SelectedItem.Text : null; }
            get
            {
                if (lbxTexture.SelectedItem == null) return null;

                var tex = lbxTexture.SelectedItem.Data as Texture;
                if (lbxTexture.DisabledTextures.Contains(tex))
                {
                    return null;
                }

                return tex;
            }
        }

        /// <summary>
        /// テクスチャ項目追加。
        /// </summary>
        public void AddTextureItem(Texture texture, bool canSelect)
        {
            if (!canSelect)
            {
                lbxTexture.DisabledTextures.Add(texture);
            }
            lbxTexture.AddItem(texture);
        }

        public void AddTextureItem(string name)
        {
            lbxTexture.AddDummyItem(name);
        }

        public int SelectedIndex
        {
            get
            {
                return lbxTexture.SelectedIndex;
            }
            set
            {
                lbxTexture.SelectedIndex = value;
            }
        }

        public static readonly Dictionary<string, string>	mipmapGenFilters_ = new Dictionary<string, string>()
        {
            {"point",	Strings.MipmapGenFilter_Point},
            {"linear",	Strings.MipmapGenFilter_Linear},
            {"cubic",	Strings.MipmapGenFilter_Cubic},
        };

        /// <summary>
        /// コンポーネント更新。
        /// </summary>
        private void UpdateComponent()
        {
            // イメージ
            tvpImage.Target = texture_;

            // テクスチャ
            if (texture_ != null)
            {
                var preset = ApplicationConfig.Preset.TexturePresets.FirstOrDefault(x => x.quantize_type == texture_.Data.texture_info.quantize_type);

                ltbFullPath.Text = texture_.FilePath;
                ltbHint.Text				= texture_.Data.texture_info.hint;

                ltbDccPreset.Text			= texture_.Data.texture_info.dcc_preset;
                ltbPresetFormat.Text		= (preset == null) ? string.Empty : preset.Name;

                ltbFormat.Text				= UIText.EnumValue(texture_.Data.texture_info.quantize_type);
                ltbMipmapGenFilter.Text		= mipmapGenFilters_.ContainsKey(texture_.Data.texture_info.mip_gen_filter) ? mipmapGenFilters_[texture_.Data.texture_info.mip_gen_filter] : string.Empty;
                ccsPanel.Value				= texture_.Data.texture_info.comp_sel;
                ltbWeightedCompress.Text	= UIText.FlagEnableDisable(texture_.Data.texture_info.weighted_compress);
                rgbaTable1.Values			= texture_.Data.texture_info.linear;

                ltbDimension.Text			= UIText.EnumValue(texture_.Data.texture_info.dimension);
                ltbWidth.Text				= texture_.Data.texture_info.width.ToString();
                ltbHeight.Text				= texture_.Data.texture_info.height.ToString();

                ltbDepth.Text				= texture_.Data.texture_info.depth.ToString();
                ltbVramSize.Text			= DataSize.XBytesText(texture_.Data.texture_info.size);
                ltbTexMipmapLevel.Text		= texture_.Data.texture_info.mip_level.ToString();
                ltbTexMipmapMinSize.Text	= UIText.MipmapMinSize(texture_.Data.texture_info);

                ltbInitialSwizzle.Text		= texture_.Data.texture_info.initial_swizzle.ToString();
            }
            else
            {
                ltbFullPath.Text = string.Empty;
                ltbHint.Text				= string.Empty;

                ltbDccPreset.Text			= string.Empty;
                ltbPresetFormat.Text		= string.Empty;

                ltbFormat.Text				= string.Empty;
                ltbMipmapGenFilter.Text		= string.Empty;
                ccsPanel.Value				= null;
                ltbWeightedCompress.Text	= string.Empty;
                rgbaTable1.Values			= null;

                ltbDimension.Text			= string.Empty;
                ltbWidth.Text				= string.Empty;
                ltbHeight.Text				= string.Empty;

                ltbDepth.Text				= string.Empty;
                ltbVramSize.Text			= string.Empty;
                ltbTexMipmapLevel.Text		= string.Empty;
                ltbTexMipmapMinSize.Text	= string.Empty;

                ltbInitialSwizzle.Text		= string.Empty;
            }

            {
                ltbHint.Enabled				= texture_ != null;

                ltbDccPreset.Enabled		= texture_ != null;
                ltbPresetFormat.Enabled		= texture_ != null;

                ltbFormat.Enabled			= texture_ != null;
                ltbMipmapGenFilter.Enabled	= texture_ != null;
                ccsPanel.Enabled			= texture_ != null;
                ltbWeightedCompress.Enabled	= texture_ != null;
                rgbaTable1.Enabled			= texture_ != null;

                ltbDimension.Enabled		= texture_ != null;
                ltbWidth.Enabled			= texture_ != null;
                ltbHeight.Enabled			= texture_ != null;

                ltbDepth.Enabled			= texture_ != null;
                ltbVramSize.Enabled			= texture_ != null;
                ltbTexMipmapLevel.Enabled	= texture_ != null;
                ltbTexMipmapMinSize.Enabled	= texture_ != null;

                ltbInitialSwizzle.Enabled	= texture_ != null;
            }

            if (ComponentUpdated != null)
            {
                ComponentUpdated();
            }
        }

        public event Action ComponentUpdated;

        /// <summary>
        /// マウスのダブルクリックでテクスチャを選択したときのイベント。
        /// </summary>
        public event Action TextureSelected;

        #region イベントハンドラ
        //---------------------------------------------------------------------
        // テクスチャ選択
        private void lbxTexture_SelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateComponent();
        }

        //---------------------------------------------------------------------
        // テクスチャリストボックスダブルクリック
        private void lbxTexture_MouseDoubleClick(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            // ダブルクリックされたリスト番号を取得。
            // TextureImageListBox では、非選択状態が存在しないので
            // テクスチャが選択されたときにのみ選択の切り替えとイベントの呼び出しを行う。
            int index = lbxTexture.IndexFromPoint(e.Location);
            if (index != -1)
            {
                SelectedIndex = index;
                if (TextureSelected != null)
                {
                    TextureSelected();
                }
            }
        }
        #endregion
    }
}
