﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using App.Data;
using App.Utility;
using App.res;

namespace App.Controls
{
    public class TextureImagePanel : UIPanel
    {
        private Texture	target_;
        private TextureImagePanelDisplayMode DisplayMode_;

        private Bitmap	textureImage_;
        private Bitmap	depthTextureImage_;

        // クリック可能状態フラグ
        private bool clickableState_ = false;

        // ツールチップ
        private UIToolTip ttpHint_ = null;

        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Texture Target
        {
            get { return target_; }
            set
            {
                target_ = value;
                DepthIndex = -1;

                if (value == null)
                {
                    textureImage_ = null;
                    depthTextureImage_ = null;

                    clickableState_ = false;
                    Cursor = Cursors.Default;

                    if (ttpHint_ != null)
                    {
                        ttpHint_.RemoveAll();
                    }
                }
                else
                {
#if false
                    switch (DisplayMode)
                    {
                        case TextureImagePanelDisplayMode.Color:
                        {
                            textureImage_ = Target.ColorLargeThumbnail;
                            break;
                        }

                        case TextureImagePanelDisplayMode.Alpha:
                        {
                            textureImage_ = Target.AlphaLargeThumbnail;
                            break;
                        }
                    }

                    UpdateDepthTextureImage();

#endif
                }

                Invalidate();
            }
        }

        [DefaultValue(TextureImagePanelDisplayMode.Color)]
        public TextureImagePanelDisplayMode DisplayMode
        {
            get { return DisplayMode_; }
            set {
                if (DisplayMode_ != value)
                {
                    DisplayMode_ = value;
                    //UpdateDepthTextureImage();
                    Invalidate();
                }
            }
        }

        // 表示する深さ。-1で全部表示
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public int DepthIndex
        {
            get { return depthIndex_; }
            set
            {
                depthIndex_ = value;
                //UpdateDepthTextureImage();
                Invalidate();
            }
        }
        private int depthIndex_ = -1;

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnCreateControl()
        {
            if (DesignMode == false)
            {
                Cursor = Cursors.Hand;
            }

            base.OnCreateControl();
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            // 背景
            e.Graphics.Clear(SystemColors.Control);

            // 枠線
            Rectangle rcClient = ClientRectangle;
            ControlPaint.DrawBorder3D(e.Graphics, rcClient, Border3DStyle.SunkenOuter);

            // 画像
            if (Target != null)
            {
                //BitmapUtility.DrawTextureImage(e.Graphics, DepthIndex == -1 ? textureImage_ : depthTextureImage_, Width, Height,
                //                               Target.Is1d);
                var rect = new Rectangle(1, 1, Width, Height);
                switch (DisplayMode)
                {
                    case TextureImagePanelDisplayMode.Color:
                        {
                            if (DepthIndex == -1)
                            {
                                Target.DrawColorThumbnail(e.Graphics, rect);
                            }
                            else
                            {
                                Target.DrawDepthThumbnail(e.Graphics, rect, true, DepthIndex);
                            }
                            break;
                        }

                    case TextureImagePanelDisplayMode.Alpha:
                        {
                            if (DepthIndex == -1)
                            {
                                Target.DrawAlphaThumbnail(e.Graphics, rect);
                            }
                            else
                            {
                                Target.DrawDepthThumbnail(e.Graphics, rect, false, DepthIndex);
                            }
                            break;
                        }
                }

                // デコードできなくてオリジナルイメージが存在しない場合はメッセージを表示
                if (Target.IsOriginalImage && Target.OrigianlImageCount == 0)
                {
                    GraphicsUtility.DrawText(e.Graphics, Strings.TextureImagePanel_OnPaint_NoOriginalImage, this.Font, Color.Black, rect, HorizontalAlignment.Center);
                    rect.Offset(-1, -1);
                    GraphicsUtility.DrawText(e.Graphics, Strings.TextureImagePanel_OnPaint_NoOriginalImage, this.Font, Color.White, rect, HorizontalAlignment.Center);
                }
            }

            // クリック可能を示唆する枠線
            if (clickableState_)
            {
                // 太さ２で強調
                GraphicsUtility.DrawRectangle(e.Graphics, Const.ClickableControlFramePen, rcClient);
                rcClient.Inflate(-1, -1);
                GraphicsUtility.DrawRectangle(e.Graphics, Const.ClickableControlFramePen, rcClient);
            }

            base.OnPaint(e);
        }

        protected override void OnMouseEnter(EventArgs e)
        {
            if (ttpHint_ == null)
            {
                ttpHint_ = new UIToolTip();
            }

            // ツールチップ設定
            if(	(Target != null) && (
                (Target.HasColor && (DisplayMode == TextureImagePanelDisplayMode.Color)) ||
                (Target.HasAlpha && (DisplayMode == TextureImagePanelDisplayMode.Alpha))))
            {
                ttpHint_.SetToolTip(this, Strings.TextureImagePanel_Hint);
            }
            else
            {
                ttpHint_.RemoveAll();
            }

            {
                switch (DisplayMode)
                {
                    case TextureImagePanelDisplayMode.Color:	clickableState_ = (Target != null) ? Target.HasColor : false;	break;
                    case TextureImagePanelDisplayMode.Alpha:	clickableState_ = (Target != null) ? Target.HasAlpha : false;	break;
                    default:									clickableState_ = false;										break;
                }

                Cursor = clickableState_ ? Cursors.Hand : Cursors.Default;

                Invalidate();
            }

            base.OnMouseEnter(e);
        }

        protected override void OnMouseLeave(EventArgs e)
        {
            if (clickableState_)
            {
                clickableState_ = false;
                Invalidate();
            }

            base.OnMouseLeave(e);
        }

        protected override void OnClick(EventArgs e)
        {
            if (clickableState_)
            {
                TextureDetailsDialog.ShowWindow(Target, DisplayMode == TextureImagePanelDisplayMode.Alpha);
            }

            base.OnClick(e);
        }

        private void UpdateDepthTextureImage()
        {
            if ((DepthIndex == -1) || (Target == null))// || (textureImage_ == null))
            {
                depthTextureImage_ = null;
            }
            else
            {
#if false
                // 未実装のためここには来ないはず
                Debug.Assert(false, "UpdateDepthTextureImage() not implimented");
#else
                //
                var width  = Target.Data.texture_info.width;
                var height = Target.Data.texture_info.height;

                // キューブテクスチャの時は、展開図として扱う
                if (Target.IsCube)
                {
                    width  *= 4;
                    height *= 3;
                }

                depthTextureImage_ = new Bitmap(width, height);

                {
                    var allWidth  = textureImage_.Width;
                    var allHeight = textureImage_.Height;

                    var xCount = (int)(allWidth  / width);
                    var yCount = (int)(allHeight / height);

                    var x = (DepthIndex % xCount) * width;
                    var y = (DepthIndex / yCount) * height;

                    using (var g = Graphics.FromImage(depthTextureImage_))
                    {
                        g.DrawImage(
                            textureImage_,
                            new Rectangle(0, 0, width, height),
                            x, y, width, height, GraphicsUnit.Pixel
                        );
                    }
                }

#endif
            }
        }
    }

    public enum TextureImagePanelDisplayMode
    {
        Color,
        Alpha
    }
}
