﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// 分割パネルフォルダクラス。
    /// </summary>
    public abstract class SplitPanelFolder : UnselectableControl
    {
        // ターゲットコンテナ
        private SplitContainer _targetContainer = null;
        // ターゲットパネル
        private SplitterPanelFolderTargetPanel _targetPanel = SplitterPanelFolderTargetPanel.Panel1;
        // マウスポイント中フラグ
        private bool _mouseEnter = false;

        /// <summary>パネル開閉状態変更イベント。</summary>
        public event EventHandler PanelCollapsedChanged = null;

        /// <summary>デフォルト長さ。</summary>
        protected const int _defaultLength = 120;
        /// <summary>デフォルト厚み。</summary>
        protected const int _defaultThickness = 6;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public SplitPanelFolder()
        {
        }

        /// <summary>
        /// ターゲットコンテナ。
        /// </summary>
        [DefaultValue(null)]
        [Description("ターゲットコンテナ。")]
        public SplitContainer TargetContainer
        {
            get { return _targetContainer; }
            set { _targetContainer = value; }
        }

        /// <summary>
        /// ターゲットパネル。
        /// </summary>
        [DefaultValue(SplitterPanelFolderTargetPanel.Panel1)]
        [Description("ターゲットパネル。")]
        public SplitterPanelFolderTargetPanel TargetPanel
        {
            get { return _targetPanel; }
            set
            {
                _targetPanel = value;
                Invalidate();
            }
        }

        /// <summary>
        /// パネルの開閉状態を切り替える。
        /// </summary>
        public void TogglePanelCollapsed()
        {
            // パネル開閉
            if (_targetContainer != null)
            {
                // パネル１
                if (_targetPanel == SplitterPanelFolderTargetPanel.Panel1)
                {
                    _targetContainer.Panel1Collapsed = !_targetContainer.Panel1Collapsed;
                }
                // パネル２
                else
                {
                    _targetContainer.Panel2Collapsed = !_targetContainer.Panel2Collapsed;
                }

                // イベント発行
                if (PanelCollapsedChanged != null)
                {
                    PanelCollapsedChanged(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            // 背景
            if (_mouseEnter)
            {
                e.Graphics.Clear(ColorUtility.Blend(BackColor, SystemColors.Highlight, 0.25f));
                GraphicsUtility.DrawRectangle(e.Graphics, SystemPens.Highlight, ClientRectangle);
            }
            else
            {
                e.Graphics.Clear(BackColor);
            }

            // トリガーマーク
            DrawTrigger(e.Graphics, SystemColors.ControlText);
            base.OnPaint(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseEnter(EventArgs e)
        {
            _mouseEnter = true;
            Invalidate();
            base.OnMouseEnter(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseLeave(EventArgs e)
        {
            _mouseEnter = false;
            Invalidate();
            base.OnMouseLeave(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnClick(EventArgs e)
        {
            _mouseEnter = false;
            TogglePanelCollapsed();
            base.OnClick(e);
        }

        /// <summary>
        /// トリガーマーク描画。
        /// </summary>
        protected abstract void DrawTrigger(Graphics g, Color color);
    }

    #region HSplitPanelFolder
    /// <summary>
    /// 水平分割パネルフォルダクラス。
    /// </summary>
    [ToolboxItem(true)]
    public sealed class HSplitPanelFolder : SplitPanelFolder
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public HSplitPanelFolder()
        {
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(_defaultLength, _defaultThickness); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void DrawTrigger(Graphics g, Color color)
        {
            int center = Width / 2;

            // 方向を決める
            bool up = true;
            if (TargetContainer != null)
            {
                if (TargetPanel == SplitterPanelFolderTargetPanel.Panel1)
                {
                    up = !TargetContainer.Panel1Collapsed;
                }
                else
                {
                    up = TargetContainer.Panel2Collapsed;
                }
            }

            // 描画
            using (Pen pen = new Pen(color))
            {
                if (up)
                {
                    g.DrawLine(pen, center - 0, 1, center + 1, 1);
                    g.DrawLine(pen, center - 1, 2, center + 2, 2);
                    g.DrawLine(pen, center - 2, 3, center + 3, 3);
                    g.DrawLine(pen, center - 3, 4, center + 4, 4);
                }
                else
                {
                    g.DrawLine(pen, center - 3, 1, center + 4, 1);
                    g.DrawLine(pen, center - 2, 2, center + 3, 2);
                    g.DrawLine(pen, center - 1, 3, center + 2, 3);
                    g.DrawLine(pen, center - 0, 4, center + 1, 4);
                }
            }
        }
    }
    #endregion

    #region VSplitPanelFolder
    /// <summary>
    /// 垂直分割パネルフォルダクラス。
    /// </summary>
    [ToolboxItem(true)]
    public sealed class VSplitPanelFolder : SplitPanelFolder
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public VSplitPanelFolder()
        {
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(_defaultThickness, _defaultLength); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void DrawTrigger(Graphics g, Color color)
        {
            int center = Height / 2;

            // 方向を決める
            bool left = true;
            if (TargetContainer != null)
            {
                if (TargetPanel == SplitterPanelFolderTargetPanel.Panel1)
                {
                    left = !TargetContainer.Panel1Collapsed;
                }
                else
                {
                    left = TargetContainer.Panel2Collapsed;
                }
            }

            // 描画
            using (Pen pen = new Pen(color))
            {
                if (left)
                {
                    g.DrawLine(pen, 1, center - 0, 1, center + 1);
                    g.DrawLine(pen, 2, center - 1, 2, center + 2);
                    g.DrawLine(pen, 3, center - 2, 3, center + 3);
                    g.DrawLine(pen, 4, center - 3, 4, center + 4);
                }
                else
                {
                    g.DrawLine(pen, 1, center - 3, 1, center + 4);
                    g.DrawLine(pen, 2, center - 2, 2, center + 3);
                    g.DrawLine(pen, 3, center - 1, 3, center + 2);
                    g.DrawLine(pen, 4, center - 0, 4, center + 1);
                }
            }
        }
    }
    #endregion

    /// <summary>
    /// スプリッタパネルフォルダ用ターゲットパネル。
    /// </summary>
    public enum SplitterPanelFolderTargetPanel
    {
        /// <summary>パネル１。</summary>
        Panel1,
        /// <summary>パネル２。</summary>
        Panel2,
    }
}
