﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using System.Linq;
using App.ConfigData;
using App.Data;
using App.Utility;
using App.res;
using nw.g3d.nw4f_3dif;
using ConfigCommon;

namespace App.Controls
{
    /// <summary>
    /// ディレクトリ保存ダイアログクラス。
    /// </summary>
    public sealed partial class SaveDirectoryDialog : OkCancelDialog
    {
        // リスト初期化フラグ
        private bool _listInitialized = false;
        // 保存場所履歴数
        private const int _recentDirectoryCount = 8;
        // 更新日時列番号
        private const int _overwriteDateColumnIndex = 2;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public SaveDirectoryDialog(List<Document> documents)
        {
            InitializeComponent();

            DialogUtility.SetHelp(this, HelpUtility.PageKey.p_file_menu_a_specified_save);

            // ドキュメントリスト
            Debug.Assert((documents != null) && (documents.Count > 0));

            // ファイルリスト
            foreach (Document document in documents)
            {
                var name = document.FileName + DocumentManager.GetSameNameIndexText(document, true);

                ListViewItem item = new ListViewItem(name);
                item.Tag = document;
                item.Checked = true;
                item.ImageKey = PathUtility.MakeNoFormatIfName(document.FileExt);
                item.UseItemStyleForSubItems = false;
                item.SubItems.Add(document.FileExt.ToLower());
                item.SubItems.Add(string.Empty);
                lvwFileList.Items.Add(item);
            }
            lvwFileList.Sort(false);
            lvwFileList_VisibleChanged(lvwFileList, EventArgs.Empty);

            // チェックコンテキストメニュー
            InitializeContextMenuItem(cmiCheckF3pj, ProjectDocument.DefaultDotExt.Substring(1));
            InitializeContextMenuItem(cmiCheckFmd,  G3dPath.ModelExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFtx,  G3dPath.TextureExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsk,  G3dPath.SkeletalAnimExtension.Substring(1));
            if (ApplicationConfig.Preset.EnableMaterialAnimCreation)
            {
                InitializeContextMenuItem(cmiCheckFma, G3dPath.MaterialAnimExtension.Substring(1));
                cmiCheckFsp.Visible =
                cmiCheckFcl.Visible =
                cmiCheckFts.Visible =
                cmiCheckFtp.Visible =
                cmiCheckFvm.Visible = false;
            }
            else
            {
                cmiCheckFma.Visible = false;
                InitializeContextMenuItem(cmiCheckFsp, G3dPath.ShaderParamAnimExtension.Substring(1));
                InitializeContextMenuItem(cmiCheckFcl, G3dPath.ColorAnimExtension.Substring(1));
                InitializeContextMenuItem(cmiCheckFts, G3dPath.TexSrtAnimExtension.Substring(1));
                InitializeContextMenuItem(cmiCheckFtp, G3dPath.TexPatternAnimExtension.Substring(1));
                InitializeContextMenuItem(cmiCheckFvm, G3dPath.MatVisibilityAnimExtension.Substring(1));
            }
            InitializeContextMenuItem(cmiCheckFvb,  G3dPath.BoneVisibilityAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsh, G3dPath.ShapeAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsn, G3dPath.SceneAnimExtension.Substring(1));

            // クリアコンテキストメニュー
            InitializeContextMenuItem(cmiClearF3pj, ProjectDocument.DefaultDotExt.Substring(1));
            InitializeContextMenuItem(cmiClearFmd,  G3dPath.ModelExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFtx,  G3dPath.TextureExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsk,  G3dPath.SkeletalAnimExtension.Substring(1));
            if (ApplicationConfig.Preset.EnableMaterialAnimCreation)
            {
                InitializeContextMenuItem(cmiClearFma, G3dPath.MaterialAnimExtension.Substring(1));
                cmiClearFsp.Visible =
                cmiClearFcl.Visible =
                cmiClearFts.Visible =
                cmiClearFtp.Visible =
                cmiClearFvm.Visible = false;
            }
            else
            {
                cmiClearFma.Visible = false;
                InitializeContextMenuItem(cmiClearFsp, G3dPath.ShaderParamAnimExtension.Substring(1));
                InitializeContextMenuItem(cmiClearFcl, G3dPath.ColorAnimExtension.Substring(1));
                InitializeContextMenuItem(cmiClearFts, G3dPath.TexSrtAnimExtension.Substring(1));
                InitializeContextMenuItem(cmiClearFtp, G3dPath.TexPatternAnimExtension.Substring(1));
                InitializeContextMenuItem(cmiClearFvm, G3dPath.MatVisibilityAnimExtension.Substring(1));
            }
            InitializeContextMenuItem(cmiClearFvb,  G3dPath.BoneVisibilityAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsh, G3dPath.ShapeAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsn, G3dPath.SceneAnimExtension.Substring(1));

            // コンフィグ読み込み
            {
                var config = ApplicationConfig.Setting.SaveDirectory;

                foreach (string recentDirectory in config.RecentDirectory)
                {
                    cmbOutput.Items.Add(recentDirectory);
                }

                chkEnableTextureDirectory.Checked = config.EnableTextureDirectory;
                chkOverwriteConfirmation.Checked = config.OverwriteConfirmation;
            }

            // 初期表示設定
            if (cmbOutput.Items.Count > 0)
            {
                cmbOutput.Text = cmbOutput.Items[0].ToString();
            }
            else
            {
                UpdateState();
            }
            ActiveControl = btnOK;
        }

        /// <summary>
        /// コンテキストメニュー項目初期化。
        /// </summary>
        private void InitializeContextMenuItem(ToolStripMenuItem menuItem, string fileExt)
        {
            menuItem.Tag = fileExt;
            menuItem.Enabled = lvwFileList.ContainsItem(fileExt) || lvwFileList.ContainsItem(fileExt + G3dPath.Text) || lvwFileList.ContainsItem(fileExt + G3dPath.Binary);
            menuItem.Image = Const.IconImages[fileExt];
        }

        /// <summary>
        /// 状態更新。
        /// </summary>
        private void UpdateState()
        {
            // 出力先に応じて上書き情報を更新
            if (Directory.Exists(cmbOutput.Text))
            {
                UpdateOverwriteInformation();
            }
            // 上書き情報をクリア
            else
            {
                ClearOverwriteInformation();
            }

            // 保存カウンタを更新
            UpdateCounter();
        }

        /// <summary>
        /// テクスチャパス変換
        /// </summary>
        private static string ConvertTexturePath(string path)
        {
            string extension = Path.GetExtension(path);
            if ((extension == G3dPath.TextureTextExtension) ||
                (extension == G3dPath.TextureBinaryExtension))
            {
                return Path.GetDirectoryName(path) + "\\" +
                    Const.TexturesDirectoryName + "\\" + Path.GetFileName(path);
            }
            return path;
        }

        /// <summary>
        /// 上書き情報更新。
        /// </summary>
        private void UpdateOverwriteInformation()
        {
            string basePath = Path.GetFullPath(cmbOutput.Text) + "\\";

            using(var ub = new UpdateBlock(lvwFileList))
            {
                foreach (ListViewItem item in lvwFileList.Items)
                {
                    // ファイルパス（テクスチャパスで補正する）
                    var doc = (Document)item.Tag;
                    string filePath = basePath + doc.FileName;
                    if (chkEnableTextureDirectory.Checked)
                    {
                        filePath = ConvertTexturePath(filePath);
                    }

                    // 同じパスに同名ファイルがあれば更新日時取得
                    ListViewItem.ListViewSubItem subItem = item.SubItems[_overwriteDateColumnIndex];
                    if (File.Exists(filePath))
                    {
                        DateTime date = File.GetLastWriteTime(filePath);
                        subItem.Text = date.ToShortDateString() + " " + date.ToShortTimeString();
                        subItem.ForeColor = item.Checked ? Color.Red : SystemColors.WindowText;
                    }
                    else
                    {
                        subItem.Text = string.Empty;
                        subItem.ForeColor = SystemColors.WindowText;
                    }
                }
            }
        }

        /// <summary>
        /// 上書き情報クリア。
        /// </summary>
        private void ClearOverwriteInformation()
        {
            using(var ub = new UpdateBlock(lvwFileList))
            {
                foreach (ListViewItem item in lvwFileList.Items)
                {
                    ListViewItem.ListViewSubItem subItem = item.SubItems[_overwriteDateColumnIndex];
                    subItem.Text = string.Empty;
                    subItem.ForeColor = SystemColors.WindowText;
                }
            }
        }

        /// <summary>
        /// カウンタ更新。
        /// </summary>
        private void UpdateCounter()
        {
            // プロパティグリッドを更新するだけ
            pgdFileCount.SelectedObject = new SaveFileCountPropertyData(lvwFileList);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // 初期化終了
            // ListView.ItemChecked イベントが初回表示時に必ず呼ばれてしまうのを回避...
            _listInitialized = true;
            base.OnLoad(e);
        }

        public List<Tuple<Document, string>> SaveDocuments = new List<Tuple<Document, string>>();
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            // 出力ディレクトリ確認
            string outputPath = cmbOutput.Text;
            if (!DocumentIOUtility.ConfirmSaveDirectory(ref outputPath))
            {
                return false;
            }

            // 対象ファイル確認
            if (lvwFileList.CheckedItems.Count == 0)
            {
                UIMessageBox.Warning(Strings.IO_SaveTargetNotFound);
                return false;
            }

            // 重複チェック
            var group = from document in lvwFileList.CheckedItems.OfType<ListViewItem>().Select(x => (Document)x.Tag)
                        group document by new Tuple<GuiObjectID, string>(document.ObjectID, document.Name);

            var duplicate = group.FirstOrDefault(x => x.Count() > 1);
            if (duplicate != null)
            {
                //duplicate.Key.Item1;
                var value = UIText.EnumValue(duplicate.Key.Item1);
                UIMessageBox.Warning(Strings.IO_SaveSameName, value + " " + duplicate.Key.Item2);
                return false;
            }

            // 上書き確認
            if (chkOverwriteConfirmation.Checked)
            {
                if (!ConfirmOverwrite())
                {
                    return false;
                }
            }

            // 書き込み権限がない
            if (FileUtility.HasWritePermission(outputPath) == false)
            {
                UIMessageBox.Warning(outputPath + "\n" + Strings.IO_BinaryResourceFilenameConnotWriteAccessPermission);
                return false;
            }

            // ドキュメントリストを整理
            bool enableTextureDirectory = chkEnableTextureDirectory.Checked;
            string textureOutputPath = outputPath + "\\" + Const.TexturesDirectoryName;
            bool createTextureDirectory = false;
            foreach (ListViewItem item in lvwFileList.Items)
            {
                Document document = (Document)item.Tag;

                // 保存対象ならファイルパスを設定
                if (item.Checked)
                {
                    if (enableTextureDirectory && (document is Texture))
                    {
                        // テクスチャ出力ディレクトリが無い状態でロケーションを設定すると例外発生
                        //file4C.SetFileLocation(textureOutputPath);
                        createTextureDirectory = true;
                        SaveDocuments.Add(new Tuple<Document, string>(document, Path.Combine(textureOutputPath, document.FileName)));
                    }
                    else
                    {
                        SaveDocuments.Add(new Tuple<Document,string>(document, Path.Combine(outputPath, document.FileName)));
                    }
                }
            }

            // テクスチャ出力ディレクトリを作成
            if (createTextureDirectory)
            {
                if (!Directory.Exists(textureOutputPath))
                {
                    Directory.CreateDirectory(textureOutputPath);
                }
            }

            // コンフィグ書き込み
            {
                var config = ApplicationConfig.Setting.SaveDirectory;
                {
                    // ドライブのルートディレクトリに対応するため最後にパス記号をつける
                    outputPath = outputPath + "\\";

                    ArrayUtility.Remove(ref config.RecentDirectory, outputPath);
                    ArrayUtility.Insert(ref config.RecentDirectory, 0, outputPath);

                    if (config.RecentDirectory.Length > _recentDirectoryCount)
                    {
                        Array.Resize(ref config.RecentDirectory, _recentDirectoryCount);
                    }

                    config.EnableTextureDirectory = chkEnableTextureDirectory.Checked;
                    config.OverwriteConfirmation = chkOverwriteConfirmation.Checked;
                }
            }

            return true;
        }

        /// <summary>
        /// 上書き確認。
        /// </summary>
        private bool ConfirmOverwrite()
        {
            // 上書き対象ドキュメントリスト
            List<Document> documents = new List<Document>();
            foreach (ListViewItem item in lvwFileList.Items)
            {
                if (item.Checked)
                {
                    if (item.SubItems[_overwriteDateColumnIndex].Text != string.Empty)
                    {
                        documents.Add((Document)item.Tag);
                    }
                }
            }

            // 確認
            if (documents.Count == 0)
            {
                return true;
            }
            else
            {
                return UIMessageBox.OkCancel(string.Format(Strings.IO_SaveConfirmOverwrite, documents.Count));
            }
        }

        #region イベントハンドラ
        //---------------------------------------------------------------------
        // 出力先設定
        private void cmbOutput_TextChanged(object sender, EventArgs e)
        {
            UpdateState();
        }

        private void cmbOutput_SelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateState();
        }

        private void cmbOutput_DragEnter(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                // ファイルであればドラッグ可能
                e.Effect = DragDropEffects.Copy;
            }
            else
            {
                // その他の場合はドロップさせない
                e.Effect = DragDropEffects.None;
            }
        }

        private void cmbOutput_DragDrop(object sender, DragEventArgs e)
        {
            string[] paths = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            if ((paths == null) || (paths.Length == 0)) { return; }
            string path = paths[0];

            if (Directory.Exists(path))
            {
                cmbOutput.Text = path;
            }
            else
            {
                cmbOutput.Text = Path.GetDirectoryName(path);
            }
        }

        //---------------------------------------------------------------------
        // 出力先選択
        private void btnOutput_Click(object sender, EventArgs args)
        {
            using(FolderBrowserDialog dialog = new FolderBrowserDialog())
            {
                dialog.Description = Strings.IO_SaveDirectorySpecify;

                if (Directory.Exists(cmbOutput.Text))
                {
                    dialog.SelectedPath = Path.GetFullPath(cmbOutput.Text);
                }
                else if (cmbOutput.Items.Count > 0)
                {
                    dialog.SelectedPath = cmbOutput.Items[0].ToString();
                }
                var mainFrame = TheApp.MainFrame.IsDisposed ? null : TheApp.MainFrame;
                if (dialog.ShowDialog(mainFrame) == DialogResult.OK)
                {
                    cmbOutput.Text = dialog.SelectedPath;
                }
            }
        }

        //---------------------------------------------------------------------
        // ファイルリスト
        private void lvwFileList_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            if (_listInitialized)
            {
                UpdateState();
            }
        }

        //---------------------------------------------------------------------
        // チェック状態変更
        private void cmiCheck_Click(object sender, EventArgs e)
        {
            string fileExt = (string)((ToolStripMenuItem)sender).Tag;

            // 全て
            if (fileExt == null)
            {
                lvwFileList.CheckAllItems();
            }
            // 種類別
            else
            {
                lvwFileList.CheckAllItems(fileExt);
                lvwFileList.CheckAllItems(fileExt + G3dPath.Text);
                lvwFileList.CheckAllItems(fileExt + G3dPath.Binary);
            }
            UpdateState();
        }

        private void cmiClear_Click(object sender, EventArgs e)
        {
            string fileExt = (string)((ToolStripMenuItem)sender).Tag;

            // 全て
            if (fileExt == null)
            {
                lvwFileList.ClearAllItems();
            }
            // 種類別
            else
            {
                lvwFileList.ClearAllItems(fileExt);
                lvwFileList.ClearAllItems(fileExt + G3dPath.Text);
                lvwFileList.ClearAllItems(fileExt + G3dPath.Binary);
            }
            UpdateState();
        }

        //---------------------------------------------------------------------
        // オプション
        private void chkEnableTextureDirectory_CheckedChanged(object sender, EventArgs args)
        {
            UpdateState();
        }
        #endregion

        private void lvwFileList_VisibleChanged(object sender, EventArgs e)
        {
            lvwFileList.BeginInvokeOrExecute(new Action(() =>
                {
                    if (lvwFileList.ClientSize.Width > 0)
                    {
                        int total = lvwFileList.ClientSize.Width;
                        int right = lvwFileList.Columns[1].Width + lvwFileList.Columns[2].Width;
                        lvwFileList.Columns[0].Width = (total - right - 2);
                    }
                }));
        }
    }
}
