﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace App.Controls
{
    [ToolboxItem(true)]
    public partial class ListViewEditor : UIUserControl
    {
        private class CustomBlock : IDisposable
        {
            private readonly Action Exit;

            public CustomBlock(Action enter, Action exit)
            {
                enter();
                Exit = exit;
            }

            public void Dispose()
            {
                Exit();
            }
        }

        public event EventHandler MoveUpButtonClicked;
        public event EventHandler MoveDownButtonClicked;

        public event EventHandler EditButtonClicked
        {
            add
            {
                btnEdit.Click += value;
            }
            remove
            {
                btnEdit.Click -= value;
            }
        }

        public event EventHandler AddButtonClicked
        {
            add
            {
                btnAdd.Click += value;
            }
            remove
            {
                btnAdd.Click -= value;
            }
        }

        public event EventHandler RemoveButtonClicked
        {
            add
            {
                btnRemove.Click += value;
            }
            remove
            {
                btnRemove.Click -= value;
            }
        }

        private bool Moving = false;

        private UIListView listView;
        public UIListView ListView
        {
            get
            {
                return listView;
            }
            set
            {
                if (listView != null)
                {
                    listView.SelectedIndexChanged -= ListView_SelectedIndexChanged;
                }
                listView = value;
                if (listView != null)
                {
                    listView.SelectedIndexChanged += ListView_SelectedIndexChanged;
                }
                UpdateState();
            }
        }

        public ListViewEditor()
        {
            InitializeComponent();

            btnMoveUp.Click += (ss, ee) =>
            {
                MoveUpSelectedItems();

                if (MoveUpButtonClicked != null)
                {
                    MoveUpButtonClicked(ss, ee);
                }
            };
            btnMoveDown.Click += (ss, ee) =>
            {
                MoveDownSelectedItems();

                if (MoveDownButtonClicked != null)
                {
                    MoveDownButtonClicked(ss, ee);
                }
            };
        }

        private void UpdateState()
        {
            if (ListView.SelectedIndices.Count > 0)
            {
                btnMoveUp.Enabled = true;
                btnMoveDown.Enabled = true;
                btnEdit.Enabled = true;
                btnAdd.Enabled = true;
                btnRemove.Enabled = true;
            }
            else
            {
                btnMoveUp.Enabled = false;
                btnMoveDown.Enabled = false;
                btnEdit.Enabled = false;
                btnAdd.Enabled = true;
                btnRemove.Enabled = false;
            }
        }

        private void MoveUpSelectedItems()
        {
            using (var cb = new CustomBlock(() => { Moving = true; }, () => { Moving = false; }))
            {
                // リスト操作後の選択状態復帰のために選択アイテムを保持しておく。
                var selectedItems = ListView.SelectedItems.Cast<ListViewItem>().ToArray();

                // 選択アイテムを上から順に上に移動させるので、インデックスを昇順ソート。
                var selectedIndices = selectedItems.Select(x => x.Index).OrderBy(y => y).ToArray();

                // リスト操作。
                // アイテムを取り除いても、アイテム側で選択状態が保持されるようなので、それをリストに再登録すると選択状態を復帰できる。
                // しかし、この挙動が仕様なのか不明なため、ここでは選択状態の復帰を手動で行っている。
                using (var ub = new UpdateBlock(ListView))
                {
                    // 自動ソート無効。
                    ListView.ListViewItemSorter = null;

                    // リスト操作前に選択を解除。
                    ListView.SelectedIndices.Clear();

                    // 選択アイテムを移動。
                    foreach (var i in selectedIndices)
                    {
                        var item = ListView.Items[i];
                        ListView.Items.RemoveAt(i);

                        var pos = Math.Max(0, i - 1);
                        if (pos < ListView.Items.Count)
                        {
                            ListView.Items.Insert(pos, item);
                        }
                        else
                        {
                            ListView.Items.Add(item);
                        }
                    }

                    // 保持しておいた選択アイテムで選択状態を復帰。
                    foreach (var item in selectedItems)
                    {
                        var pos = ListView.Items.IndexOf(item);
                        ListView.SelectedIndices.Add(pos);
                    }
                }
            }
        }

        private void MoveDownSelectedItems()
        {
            using (var cb = new CustomBlock(() => { Moving = true; }, () => { Moving = false; }))
            {
                // リスト操作後の選択状態復帰のために選択アイテムを保持しておく。
                var selectedItems = ListView.SelectedItems.Cast<ListViewItem>().ToArray();

                // 選択アイテムを下から順に下に移動させるので、インデックスを降順ソート。
                var selectedIndices = selectedItems.Select(x => x.Index).OrderBy(y => -y).ToArray();

                // リスト操作。
                // アイテムを取り除いても、アイテム側で選択状態が保持されるようなので、それをリストに再登録すると選択状態を復帰できる。
                // しかし、この挙動が仕様なのか不明なため、ここでは選択状態の復帰を手動で行っている。
                using (var ub = new UpdateBlock(ListView))
                {
                    // 自動ソート無効。
                    ListView.ListViewItemSorter = null;

                    // リスト操作前に選択を解除。
                    ListView.SelectedIndices.Clear();

                    // 選択アイテムを移動。
                    foreach (var i in selectedIndices)
                    {
                        var item = ListView.Items[i];
                        ListView.Items.RemoveAt(i);

                        var pos = Math.Max(0, i + 1);
                        if (pos < ListView.Items.Count)
                        {
                            ListView.Items.Insert(pos, item);
                        }
                        else
                        {
                            ListView.Items.Add(item);
                        }
                    }

                    // 保持しておいた選択アイテムで選択状態を復帰。
                    foreach (var item in selectedItems)
                    {
                        var pos = ListView.Items.IndexOf(item);
                        ListView.SelectedIndices.Add(pos);
                    }
                }
            }
        }

        private void ListView_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (!Moving)
            {
                UpdateState();
            }
        }
    }
}
