﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Security.Permissions;
using System.Drawing;

namespace App.Controls
{
    [ToolboxBitmap(typeof(FolderOrFileBrowserDialog))]
    public partial class FolderOrFileBrowserDialog : Component
    {
        public enum FolderID
        {
            Desktop = 0x0000,
            Printers = 0x0004,
            MyDocuments = 0x0005,
            Favorites = 0x0006,
            Recent = 0x0008,
            SendTo = 0x0009,
            StartMenu = 0x000b,
            MyComputer = 0x0011,
            NetworkNeighborhood = 0x0012,
            Templates = 0x0015,
            MyPictures = 0x0027,
            NetAndDialUpConnections = 0x0031,
        }

        private const int S_OK = 0;

        private string description = string.Empty;

        /// <summary>
        /// 説明を取得または設定する。
        /// </summary>
        public string Description
        {
            get
            {
                return description;
            }
            set
            {
                description = (value != null) ? value : string.Empty;
            }
        }

        private bool includeFiles = true;

        /// <summary>
        /// ファイルを対象に含めるかどうかを取得または設定する。
        /// </summary>
        public bool IncludeFiles
        {
            get
            {
                return includeFiles;
            }
            set
            {
                includeFiles = value;
            }
        }

        private FolderID rootFolder = FolderID.Desktop;

        /// <summary>
        /// ダイアログに表示するツリーのルートフォルダを取得または設定する。
        /// </summary>
        public FolderID RootFolder
        {
            get
            {
                return rootFolder;
            }
            set
            {
                rootFolder = value;
            }
        }

        private string selectedPath = string.Empty;

        /// <summary>
        /// 選択フォルダを取得または設定する。
        /// </summary>
        public string SelectedPath
        {
            get
            {
                return selectedPath;
            }
            set
            {
                selectedPath = (value != null) ? value : string.Empty;
            }
        }

        public FolderOrFileBrowserDialog()
        {
            InitializeComponent();
        }

        public FolderOrFileBrowserDialog(IContainer container)
        {
            container.Add(this);

            InitializeComponent();
        }

        public DialogResult ShowDialog()
        {
            return ShowDialog(null);
        }

        public DialogResult ShowDialog(IWin32Window owner)
        {
            var malloc = GetSHMalloc();
            if (malloc == null)
            {
                // アロケータを取得できない場合は、メモリの解放ができないのでダイアログを開かない。
                return DialogResult.Cancel;
            }

            var pidlRoot = IntPtr.Zero;
            var initPath = IntPtr.Zero;
            var buffer = IntPtr.Zero;
            var pidlRet = IntPtr.Zero;

            try
            {
                var options =
                    Win32.BIF.BIF_RETURNONLYFSDIRS |
                    Win32.BIF.BIF_VALIDATE |
                    Win32.BIF.BIF_NEWDIALOGSTYLE;
                if (IncludeFiles)
                {
                    options |= Win32.BIF.BIF_BROWSEINCLUDEFILES;
                }
                if ((options & Win32.BIF.BIF_NEWDIALOGSTYLE) != 0)
                {
                    if (System.Threading.ApartmentState.MTA == Application.OleRequired())
                    {
                        options &= ~Win32.BIF.BIF_NEWDIALOGSTYLE;
                    }
                }

                // 表示ツリーのルードディレクトリ。
                IntPtr hWndOwner = (owner != null) ? owner.Handle : Win32.NativeMethods.GetActiveWindow();
                Win32.NativeMethods.SHGetSpecialFolderLocation(hWndOwner, (int)rootFolder, out pidlRoot);
                if (pidlRoot == IntPtr.Zero)
                {
                    return DialogResult.Cancel;
                }

                // SelectedPath に存在しないパスが設定されていることもあり得る。
                // その場合 Win32.Constants.MAX_PATH に収まっているとは限らないので、長い方を使う。
                StringBuilder pathBuilder = new StringBuilder(Math.Max(SelectedPath.Length, Win32.Constants.MAX_PATH));

                // 初期パス。
                pathBuilder.Clear();
                if (Win32.NativeMethods.PathCanonicalize(pathBuilder, SelectedPath))
                {
                    // 有効なパスになるまでディレクトリ階層を上がる。
                    var correctedPath = pathBuilder.ToString();
                    while (!string.IsNullOrEmpty(correctedPath))
                    {
                        try
                        {
                            var attr = System.IO.File.GetAttributes(correctedPath);
                            break;
                        }
                        catch
                        {
                            correctedPath = System.IO.Path.GetDirectoryName(correctedPath);
                        }
                    }
                    initPath = Marshal.StringToBSTR(!string.IsNullOrEmpty(correctedPath) ? correctedPath : string.Empty);
                }
                else
                {
                    initPath = Marshal.StringToBSTR(string.Empty);
                }

                // パス用バッファ。
                buffer = Marshal.AllocHGlobal(Win32.Constants.MAX_PATH);

                var bi = new Win32.NativeMethods.BROWSEINFO
                {
                    pidlRoot = pidlRoot,
                    hwndOwner = hWndOwner,
                    pszDisplayName = buffer,
                    lpszTitle = Description,
                    ulFlags = options,
                    lpfn = BrowseCallbackProc,
                    lParam = initPath
                };

                // ダイアログを表示。
                pidlRet = Win32.NativeMethods.SHBrowseForFolder(ref bi);

                if (pidlRet == IntPtr.Zero)
                {
                    // キャンセルされた。
                    return DialogResult.Cancel;
                }

                pathBuilder.Clear();
                if (Win32.NativeMethods.SHGetPathFromIDList(pidlRet, pathBuilder) != Convert.ToInt32(true))
                {
                    // 変換に失敗。
                    // キャンセルとみなす。
                    return DialogResult.Cancel;
                }
                SelectedPath = pathBuilder.ToString();
            }
            finally
            {
                if (pidlRet != IntPtr.Zero)
                {
                    malloc.Free(pidlRet);
                }

                Marshal.FreeHGlobal(buffer);

                Marshal.FreeBSTR(initPath);

                if (pidlRoot != IntPtr.Zero)
                {
                    malloc.Free(pidlRoot);
                }
            }

            return DialogResult.OK;
        }

        private static Win32.NativeMethods.IMalloc GetSHMalloc()
        {
            Win32.NativeMethods.IMalloc malloc;
            var ret = Win32.NativeMethods.SHGetMalloc(out malloc);
            return (ret == S_OK) ? malloc : null;
        }

        private static int BrowseCallbackProc(IntPtr hwnd, uint uMsg, IntPtr lParam, IntPtr lpData)
        {
            switch (uMsg)
            {
                case Win32.BFFM.BFFM_INITIALIZED:
                    Win32.NativeMethods.SendMessage(hwnd, Win32.BFFM.BFFM_SETSELECTIONW, new IntPtr(Convert.ToInt32(true)), lpData);
                    break;
            }
            return 0;
        }
    }
}
