﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using App.Data;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// シェーダーパラメーターアニメーションの修正選択ダイアログクラス。
    /// </summary>
    public sealed partial class FixShaderParamAnimDialog : OkCancelDialog
    {
        public List<string> Messages = new List<string>();
        public List<Material> Materials = new List<Material>();
        public bool CanFix = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FixShaderParamAnimDialog(IEnumerable<ShaderAssignUtility.MaterialAnimationError> materials)
        {
            InitializeComponent();
            lvwFileList.SetItemCount(materials.Count());

            int index = 0;
            // リスト構築
            foreach (var material in materials)
            {
                ListViewItem item = lvwFileList.Items[index];
                item.Tag = material;
                item.Checked = material.canFix;
                CanFix |= material.canFix;

                var builder = new StringBuilder();
                foreach (var error in material.errors.Select(x => x.message))
                {
                    builder.AppendLine(error);
                }
                var message = builder.ToString();

                item.SubItems[clhMaterial.Index].Text = material.matAnim.mat_name;
                item.SubItems[clhModel.Index].Text = ((Document)material.anim).FileName;
                item.SubItems[clhShaderDefinition.Index].Text = material.shaderDefinition.FileName;
                item.SubItems[clhShadingModel.Index].Text = material.shading_model.name;
                item.SubItems[clhItem.Index].Text = material.errors.Count.ToString();
                item.SubItems[clhFix.Index].Text = material.errors.Where(x => x.table != null).Count().ToString();
                Messages.Add(message);
                Debug.Assert(!string.IsNullOrEmpty(message));
                index++;
            }
            lvwFileList.CheckBoxes = CanFix;
            lblDocumentsTitle.Text = CanFix ? res.Strings.FixShaderParamAnimCheck : res.Strings.FixShaderParamAnimNoCheck;
            if (!CanFix)
            {
                btnOK.Location = btnCancel.Location;
                btnCancel.Visible = false;
            }
            btnOK.Enabled = true;
        }

        /// <summary>
        /// 修正対象
        /// </summary>
        public IEnumerable<ShaderAssignUtility.MaterialAnimationError> CheckedMaterials
        {
            get { return lvwFileList.CheckedItems.OfType<ListViewItem>().Select(x => (ShaderAssignUtility.MaterialAnimationError)x.Tag).Where(x => x.canFix); }
        }

        private void lvwFileList_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            // チェックされていなければ OK ボタンは有効にならない
            btnOK.Enabled = !CanFix || CheckedMaterials.Any();
        }

        private void lvwFileList_SelectionChanged(object sender, EventArgs e)
        {
            var index = lvwFileList.SelectedIndex;
            if (0 <= index && index < Messages.Count)
            {
                tbxDescription.Text = Messages[index];
            }
            else
            {
                tbxDescription.Text = "";
            }
        }

        private void lvwFileList_DrawSubItem(object sender, DrawListViewSubItemEventArgs e)
        {
            // チェックボックスの表示を混在させるために自前で描画
            if (CanFix && e.ColumnIndex == 0 && e.Item != null && e.Item.Tag is ShaderAssignUtility.MaterialAnimationError)
            {
                if (!((ShaderAssignUtility.MaterialAnimationError)e.Item.Tag).canFix)
                {
                    Color foreColor = e.Item.ForeColor;
                    Color backColor = e.Item.BackColor;
                    if (e.Item.Selected)
                    {
                        if (e.ItemState.HasFlag(ListViewItemStates.Focused))
                        {
                            backColor = SystemColors.Highlight;
                            foreColor = SystemColors.HighlightText;
                        }
                        else
                        {
                            backColor = SystemColors.Control;
                            foreColor = SystemColors.WindowText;
                        }
                    }

                    var rect = new Rectangle(e.Item.Position.X, e.Item.Position.Y, clhMaterial.Width - e.Item.Position.X, e.Item.Bounds.Height);
                    using (var brush = new SolidBrush(backColor))
                    {
                        e.Graphics.FillRectangle(brush, rect);
                    }

                    // e.DrawText(); だとずれるので自前で書く
                    StringFormat sf = new StringFormat();
                    sf.Alignment = StringAlignment.Near;
                    sf.LineAlignment = StringAlignment.Center;
                    sf.Trimming = StringTrimming.EllipsisCharacter;
                    sf.FormatFlags |= StringFormatFlags.NoWrap;
                    using (var brush = new SolidBrush(foreColor))
                    {

                        e.Graphics.DrawString(e.Item.Text, lvwFileList.Font, brush, rect, sf);
                    }

                    return;
                }
            }
            e.DrawDefault = true;
        }
    }
}
