﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using App.Data;
using nw.g3d.nw4f_3dif;
using System.Linq.Expressions;

namespace App.Controls
{
    /// <summary>
    /// 修正するライトアニメーションを選択するダイアログです。
    /// </summary>
    public partial class FixLightAnimationAssignDialog : OkCancelDialog
    {
        /// <summary>
        /// ライトアニメーションの修正情報です。
        /// </summary>
        public class LightAnimationInconsistentInfo
        {
            /// <summary>
            /// ライトアニメーションの名前を取得または設定します。
            /// </summary>
            public string LightAnimationName { get; set; }

            /// <summary>
            /// シーンの名前を取得または設定します。
            /// </summary>
            public string LightAnimationOwnerName { get; set; }

            /// <summary>
            /// タイプを取得または設定します。
            /// </summary>
            public string LightAnimationTypeName { get; set; }

            /// <summary>
            /// 修正項目の件数を取得または設定します。
            /// </summary>
            public int Count { get; set; }

            /// <summary>
            /// 修正内容の説明を取得または設定します。
            /// </summary>
            public string Message { get; set; }
        }

        /// <summary>
        /// 修正内容の説明の種類です。
        /// </summary>
        public enum InconsistentMessageType
        {
            /// <summary>
            /// 修正予定の説明です。
            /// </summary>
            Plan,

            /// <summary>
            /// 完了した修正の説明です。
            /// </summary>
            Result,
        };

        /// <summary>
        /// ライトアニメーションの修正内容の説明です。
        /// </summary>
        public List<string> Messages = new List<string>();

        /// <summary>
        /// コンストラクタです。
        /// 修正項目の選択用。
        /// </summary>
        public FixLightAnimationAssignDialog(IEnumerable<LightAnimation> lightAnims)
        {
            InitializeComponent();

            lvwFileList.SetItemCount(lightAnims.Count());
            int index = 0;

            // ファイルリストを構築
            foreach (LightAnimation lightAnim in lightAnims)
            {
                ListViewItem item = lvwFileList.Items[index];
                item.Tag = lightAnim;
                item.Checked = true;
                int count;
                var message = FixLightAnimationAssignDialog.InconsistentMessage(lightAnim, out count);

                item.SubItems[clhScene.Index].Text = lightAnim.Owner.Name;
                item.SubItems[clhLightAnim.Index].Text = lightAnim.Name;
                item.SubItems[clhType.Index].Text = lightAnim.Data.type;
                item.SubItems[clhItem.Index].Text = count.ToString();

                Messages.Add(message);
                Debug.Assert(string.IsNullOrEmpty(message) == false);

                index++;
            }

            lvwFileList.CheckBoxes = true;
        }

        /// <summary>
        /// コンストラクタです。
        /// 修正項目の通知用。リードオンリー。
        /// </summary>
        /// <param name="infos">修正情報</param>
        public FixLightAnimationAssignDialog(IEnumerable<LightAnimationInconsistentInfo> infos)
        {
            InitializeComponent();

            lvwFileList.SetItemCount(infos.Count());

            int index = 0;

            // ファイルリストを構築
            foreach (var info in infos)
            {
                ListViewItem item = lvwFileList.Items[index];
                item.Tag = null;
                item.Checked = true;

                item.SubItems[clhScene.Index].Text = info.LightAnimationOwnerName;
                item.SubItems[clhLightAnim.Index].Text = info.LightAnimationName;
                item.SubItems[clhType.Index].Text = info.LightAnimationTypeName;
                item.SubItems[clhItem.Index].Text = info.Count.ToString();

                Messages.Add(info.Message);
                Debug.Assert(string.IsNullOrEmpty(info.Message) == false);

                index++;
            }

            // 編集系コントーロールを無効化
            {
                cbxAutoFix.Visible = false;
                btnCancel.Visible = false;
                btnOK.Location = btnCancel.Location;
                btnOK.Enabled = true;

                pnlBottom.Height -= cbxAutoFix.Height;
            }

            // ウィンドウ上部のラベルの文面を差し替える
            {
                lblDocumentsTitle.Text = res.Strings.FixLightAnimationAssignDialog_AutoFix;
            }
        }

        /// <summary>
        /// ライトアニメーションを選択したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void lvwFileList_SelectionChanged(object sender, EventArgs e)
        {
            var index = lvwFileList.SelectedIndex;

            // 選択されたライトアニメーションの修正内容を表示する
            if (0 <= index && index < Messages.Count)
            {
                tbxDescription.Text = Messages[index];
            }
            else
            {
                tbxDescription.Text = string.Empty;
            }
        }

        /// <summary>
        /// 次回以降自動的に修正を行うかどうか取得します。
        /// </summary>
        public bool IsAutoFix
        {
            get { return cbxAutoFix.Checked; }
        }

        /// <summary>
        /// 修正対象のライトアニメーションを取得します。
        /// </summary>
        public IEnumerable<LightAnimation> CheckeLightAnims
        {
            get { return lvwFileList.CheckedItems.OfType<ListViewItem>().Select(x => (LightAnimation)x.Tag); }
        }

        /// <summary>
        /// 修正対象外のライトアニメーションを取得します。
        /// </summary>
        public IEnumerable<LightAnimation> NotCheckedLightAnims
        {
            get { return lvwFileList.Items.OfType<ListViewItem>().Where(x => !x.Checked).Select(x =>(LightAnimation)x.Tag); }
        }

        /// <summary>
        /// 修正内容の説明を取得します。
        /// </summary>
        /// <param name="lightAnim">ライトアニメーション</param>
        /// <param name="count">メッセージの件数</param>
        /// <param name="type">これから行う修正か、完了した修正か</param>
        /// <returns></returns>
        public static string InconsistentMessage(LightAnimation lightAnim, out int count, InconsistentMessageType type = InconsistentMessageType.Plan)
        {
            var builder = new StringBuilder();

            // メッセージを取得
            var messages = FixLightAnimationAssignDialog.UpdateMessages(lightAnim, type);

            // メッセージの件数を取得
            count = messages.Count;

            // メッセージを連結
            foreach (var message in messages)
            {
                builder.AppendLine(message);
            }

            return builder.ToString();
        }

        /// <summary>
        /// ライトアニメーションの修正項目についての説明を取得します。
        /// </summary>
        /// <param name="lightAnim">ライトアニメーション</param>
        /// <param name="type">これから行う修正か、完了した修正か</param>
        /// <returns>ライトアニメーションの修正項目についての説明。</returns>
        public static List<string> UpdateMessages(LightAnimation lightAnim, InconsistentMessageType type = InconsistentMessageType.Plan)
        {
            List<string> messages = new List<string>();
            var lightAnimTargets = lightAnim.LightAnimTargets;
            var lightAnimTargetTypes = LightAnimation.GetTargetTypes(lightAnim.Data.type);

            // プリセットに設定されていないターゲットを処理
            foreach (var lightAnimTarget in lightAnimTargets)
            {
                if (lightAnimTargetTypes.Contains(lightAnimTarget.targetType) == false)
                {
                    if (lightAnimTarget.KeyFrames.Count == 0) continue;

                    string target = Enum.GetName(typeof(light_anim_target_targetType), lightAnimTarget.targetType);
                    messages.Add(string.Format(GetResourceString(() => res.Strings.FixLightAnimationAssignDialog_LightAnimation_OverTarget, type), target));
                }
            }

            // プリセットに設定されていて割り当てられていないターゲットを処理
            foreach (var lightAnimTargetType in lightAnimTargetTypes)
            {
                var lightAnimTarget = lightAnimTargets.FirstOrDefault(x => x.targetType == lightAnimTargetType);
                if (lightAnimTarget.KeyFrames.Count == 0)
                {
                    string target = Enum.GetName(typeof(light_anim_target_targetType), lightAnimTargetType);
                    messages.Add(string.Format(GetResourceString(() => res.Strings.FixLightAnimationAssignDialog_LightAnimation_NoTarget, type), target));
                }
            }

            return messages;
        }

        /// <summary>
        /// タイプごとのリソース文字列を得る
        /// </summary>
        internal static string GetResourceString<T>(Expression<Func<T>> nameExpression, InconsistentMessageType type)
        {
            Debug.Assert(nameExpression.Body is MemberExpression);

            string str = null;
            {
                var memberExpression = nameExpression.Body as MemberExpression;
                var name = memberExpression.Member.Name;

                if (type == InconsistentMessageType.Result)
                {
                    str = res.Strings.ResourceManager.GetString(name + "_Result", res.Strings.Culture);
                    if (str != null)
                    {
                        return str;
                    }
                }

                str = res.Strings.ResourceManager.GetString(name, res.Strings.Culture);
                Debug.Assert(str != null);
            }
            return str;
        }
    }
}
