﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Media;
using System.Threading;
using System.Windows.Forms;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// モーダルダイアログクラス。
    /// </summary>
    public partial class ModalDialog : UIForm // abstract
    {
        // サウンド効果
        private DialogSoundEffect _soundEffect = DialogSoundEffect.None;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ModalDialog()
        {
            InitializeComponent();
        }

        /// <summary>
        /// サウンド効果。
        /// </summary>
        [DefaultValue(DialogSoundEffect.None)]
        [Description("ダイアログ表示時に鳴らすサウンド効果です。")]
        public DialogSoundEffect SoundEffect
        {
            get { return _soundEffect; }
            set { _soundEffect = value; }
        }

        /// <summary>
        /// ダイアログを表示（再定義しています）。
        /// 表示終了後にオーナー側の無効領域更新を行います。
        /// </summary>
        public new DialogResult ShowDialog()
        {
            var result = DialogResult.OK;
            {
                // 別スレッドからの呼び出し用
                if (Thread.CurrentThread != TheApp.MainThread && TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
                {
                    TheApp.MainFrame.BeginInvoke(
                        new MethodInvoker(
                            () =>
                                {
                                    result = this.ShowDialogInternal();
                                })
                    ).AsyncWaitHandle.WaitOne();
                }
                else
                {
                    result = this.ShowDialogInternal();
                }
            }
            return result;
        }

        /// <summary>
        /// モーダルダイアログとして表示する
        /// </summary>
        /// <returns></returns>
        private DialogResult ShowDialogInternal()
        {
            DialogResult result;
            using (new WaitCursor(false))
            {
                if (TheApp.MainFrame != null && TheApp.MainFrame.IsHandleCreated)
                {
                    //最小化されている場合は最大化する
                    if (TheApp.MainFrame.WindowState == FormWindowState.Minimized)
                    {
                        TheApp.MainFrame.WindowState = FormWindowState.Normal;
                    }
                    TheApp.MainFrame.BringToFront();
                    result = base.ShowDialog(TheApp.MainFrame);
                }
                else
                {
                    this.StartPosition = FormStartPosition.CenterScreen;
                    result = base.ShowDialog();
                }
            }
            Application.DoEvents();
            return result;
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool IsMdiContainer
        {
            get { return base.IsMdiContainer;  }
            set { base.IsMdiContainer = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool MaximizeBox
        {
            get { return base.MaximizeBox;  }
            set { base.MaximizeBox = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool MinimizeBox
        {
            get { return base.MinimizeBox;  }
            set { base.MinimizeBox = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool ShowInTaskbar
        {
            get { return base.ShowInTaskbar;  }
            set { base.ShowInTaskbar = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool TopMost
        {
            get { return base.TopMost;  }
            set { base.TopMost = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new FormWindowState WindowState
        {
            get { return base.WindowState;  }
            set { base.WindowState = value; }
        }
        #endregion

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            if (!DesignMode)
            {
                // サイズ変更モード時の設定
                if (FormBorderStyle == FormBorderStyle.Sizable)
                {
                    // アイコン
                    Icon = TheApp.MainFrame.Icon;
                }
            }
            base.OnLoad(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnShown(EventArgs e)
        {
            if (!DesignMode)
            {
                // サウンド効果
                switch (_soundEffect)
                {
                    case DialogSoundEffect.Information:
                        SystemSounds.Asterisk.Play();
                        break;
                    case DialogSoundEffect.Question:
                        // WindowsXP 標準設定では問い合わせサウンドは未設定なので警告音を出しておく
                        // 特に違和感はないので気づかれないでしょ...
                        //SystemSounds.Question.Play();
                        SystemSounds.Exclamation.Play();
                        break;
                    case DialogSoundEffect.Warning:
                        SystemSounds.Exclamation.Play();
                        break;
                    case DialogSoundEffect.Error:
                        SystemSounds.Hand.Play();
                        break;
                    default:
                        break;
                }
            }
            base.OnShown(e);
            // スタートアップ時のエラーなどが他のウィンドウに隠れる場合があるため手前に表示する
            this.BringToFront();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnClosing(CancelEventArgs e)
        {
            // 結果コード別ハンドラ呼び出し
            switch (DialogResult)
            {
                case DialogResult.OK:     e.Cancel = !OnResultOk();     break;
                case DialogResult.Cancel: e.Cancel = !OnResultCancel(); break;
                case DialogResult.Yes:    e.Cancel = !OnResultYes();    break;
                case DialogResult.No:     e.Cancel = !OnResultNo();     break;
                default:                  e.Cancel = !OnResultCancel(); break;
            }
            base.OnClosing(e);
        }
        #endregion

        #region 派生クラスオーバーライド用
        /// <summary>
        /// 「ＯＫ」ハンドラ。
        /// ダイアログを閉じる場合は true を返します。
        /// 基本クラスのメソッドを呼び出す必要はありません。
        /// </summary>
        protected virtual bool OnResultOk()
        {
            return true;
        }

        /// <summary>
        /// 「キャンセル」ハンドラ。
        /// ダイアログを閉じる場合は true を返します。
        /// 基本クラスのメソッドを呼び出す必要はありません。
        /// </summary>
        protected virtual bool OnResultCancel()
        {
            return true;
        }

        /// <summary>
        /// 「はい」ハンドラ。
        /// ダイアログを閉じる場合は true を返します。
        /// 基本クラスのメソッドを呼び出す必要はありません。
        /// </summary>
        protected virtual bool OnResultYes()
        {
            return true;
        }

        /// <summary>
        /// 「いいえ」ハンドラ。
        /// ダイアログを閉じる場合は true を返します。
        /// 基本クラスのメソッドを呼び出す必要はありません。
        /// </summary>
        protected virtual bool OnResultNo()
        {
            return true;
        }
        #endregion
    }

    /// <summary>
    /// ダイアログサウンド効果。
    /// </summary>
    public enum DialogSoundEffect
    {
        /// <summary>なし。</summary>
        None,
        /// <summary>情報。</summary>
        Information,
        /// <summary>応答。</summary>
        Question,
        /// <summary>警告。</summary>
        Warning,
        /// <summary>エラー。</summary>
        Error,
    }
}
