﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Windows.Forms;
using App.ConfigData;
using App.Data;
using App.Utility;
using App.res;
using System.ComponentModel;
using Microsoft.WindowsAPICodePack.Dialogs;

namespace App.Controls
{
    public sealed partial class TeamConfigSettingDialog : OkCancelDialog
    {
        private class CustomBlock : IDisposable
        {
            private readonly Action Exit;

            public CustomBlock(Action enter, Action exit)
            {
                enter();
                Exit = exit;
            }

            public void Dispose()
            {
                Exit();
            }
        }

        const int EM_SETCUEBANNER = 0x1500 + 1;

        private readonly UIPanel[] configPanels_;

        private UIComboBox2 cbxShaderConvertLogTime = new UIComboBox2() { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxCreateShaderVariationCommandAllowNoOutput = new UIComboBox2() { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxEnableMaterialAnimCreation = new UIComboBox2() { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxSeparateMaterialAnimCreationMenu = new UIComboBox2() { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxShowOnlyBinarizeEnabledBoneVisAnimCurve = new UIComboBox2() { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxFollowDccSamplerNameRule = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxDisableTextureSrtWithOriginal = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxOptimizeShaderAfterMaterialChanged = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxDefaultValue_FilterMag = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxDefaultValue_FilterMin = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxDefaultValue_FilterMipmap = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxDefaultValue_FilterMaxAniso = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxDefaultValue_WrapU = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxDefaultValue_WrapV = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxDefaultValue_WrapW = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxDefaultValue_Clamp = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxColor_GammaCorrection = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxDisableAnimationQuantize = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };
        private UIComboBox2 cbxRenderStateInfoVisible = new UIComboBox2 { DropDownStyle = ComboBoxStyle.DropDownList };

        private bool editMode_ = false;

        private readonly TeamConfig.TeamSettings srcTeamSettings_;
        private readonly System.Collections.Generic.List<TeamConfig.PlatformPreset> srcPlatformPresets_;
        private readonly bool useExternalPlatformPresets_;
        private TeamConfig.TeamSettings teamSettings_;
        private System.Collections.Generic.List<TeamConfig.PlatformPreset> platformPresets_;

        public TeamConfigSettingDialog()
        {
            InitializeComponent();

            {
                // 変更前の設定への参照。
                srcTeamSettings_ = new TeamConfig.TeamSettings()
                {
                    FileIo = ApplicationConfig.FileIo,
                    Preset = ApplicationConfig.Preset,
                    DefaultValue = ApplicationConfig.DefaultValue,
                    Error = ApplicationConfig.Error,
                    Preview = ApplicationConfig.Preview,
                    Color = ApplicationConfig.Color
                };
                srcPlatformPresets_ = ApplicationConfig.PlatformPresets;

                // プラットフォーム設定を外部ファイルとして使っているかどうか。
                useExternalPlatformPresets_ = (srcTeamSettings_.Preset.PlatformPresets == null) || !srcTeamSettings_.Preset.PlatformPresets.Any();

                // 編集用のインスタンスを作る。
                teamSettings_ = new TeamConfig.TeamSettings()
                {
                    FileIo = ObjectUtility.Clone(srcTeamSettings_.FileIo),
                    Preset = ObjectUtility.Clone(srcTeamSettings_.Preset),
                    DefaultValue = ObjectUtility.Clone(srcTeamSettings_.DefaultValue),
                    Error = ObjectUtility.Clone(srcTeamSettings_.Error),
                    Preview = ObjectUtility.Clone(srcTeamSettings_.Preview),
                    Color = ObjectUtility.Clone(srcTeamSettings_.Color)
                };
                platformPresets_ = new System.Collections.Generic.List<TeamConfig.PlatformPreset>(srcPlatformPresets_.Select(x => ObjectUtility.Clone(x)));
            }

            DialogUtility.SetHelp(this, HelpUtility.PageKey.p_tools_menu_a_team_config);

            if (!string.IsNullOrEmpty(ApplicationConfig.TeamSettingsFilePath))
            {
                ltbTeamConfigPath.Text = ApplicationConfig.TeamSettingsFilePath;
            }
            else
            {
                ltbTeamConfigPath.Text = ApplicationConfig.TeamConfigDefaultPath;
            }

            tbxDefaultValue_LodMin.Minimum = 0.0f;
            tbxDefaultValue_LodMin.Maximum = 13.0f;
            tbxDefaultValue_LodMax.Minimum = 0.0f;
            tbxDefaultValue_LodMax.Maximum = 13.0f;
            tbxDefaultValue_LodBias.Minimum = -32.0f;
            tbxDefaultValue_LodBias.Maximum = 31.984375f;
            tbxPreview_Fps.Maximum = double.MaxValue;
            tbxMessageInterval.Maximum = int.MaxValue;

            var booleans = new bool[] { false, true };
            cbxShaderConvertLogTime.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxShaderConvertLogTime.SelectedIndexChanged += cbxShaderConvertLogTime_SelectedIndexChanged;
            cbxCreateShaderVariationCommandAllowNoOutput.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxCreateShaderVariationCommandAllowNoOutput.SelectedIndexChanged += cbxCreateShaderVariationCommandAllowNoOutput_SelectedIndexChanged;
            cbxEnableMaterialAnimCreation.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxEnableMaterialAnimCreation.SelectedIndexChanged += cbxEnableMaterialAnimCreation_SelectedIndexChanged;
            cbxSeparateMaterialAnimCreationMenu.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxSeparateMaterialAnimCreationMenu.SelectedIndexChanged += cbxSeparateMaterialAnimCreationMenu_SelectedIndexChanged;
            cbxShowOnlyBinarizeEnabledBoneVisAnimCurve.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxShowOnlyBinarizeEnabledBoneVisAnimCurve.SelectedIndexChanged += cbxShowOnlyBinarizeEnabledBoneVisAnimCurve_SelectedIndexChanged;
            cbxFollowDccSamplerNameRule.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxFollowDccSamplerNameRule.SelectedIndexChanged += cbxFollowDccSamplerNameRule_SelectedIndexChanged;
            cbxDisableTextureSrtWithOriginal.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxDisableTextureSrtWithOriginal.SelectedIndexChanged += cbxDisableTextureSrtWithOriginal_SelectedIndexChanged;
            cbxOptimizeShaderAfterMaterialChanged.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxOptimizeShaderAfterMaterialChanged.SelectedIndexChanged += cbxOptimizeShaderAfterMaterialChanged_SelectedIndexChanged;
            cbxDefaultValue_FilterMag.Items.AddRange(Enum.GetValues(typeof(nw.g3d.nw4f_3dif.filter_mag_minType)).Cast<nw.g3d.nw4f_3dif.filter_mag_minType>().Select(x => new UIListControlItem(UIText.EnumValue(x), x)).ToArray());
            cbxDefaultValue_FilterMag.SelectedIndexChanged += cbxDefaultValue_FilterMag_SelectedIndexChanged;
            cbxDefaultValue_FilterMin.Items.AddRange(Enum.GetValues(typeof(nw.g3d.nw4f_3dif.filter_mag_minType)).Cast<nw.g3d.nw4f_3dif.filter_mag_minType>().Select(x => new UIListControlItem(UIText.EnumValue(x), x)).ToArray());
            cbxDefaultValue_FilterMin.SelectedIndexChanged += cbxDefaultValue_FilterMin_SelectedIndexChanged;
            cbxDefaultValue_FilterMipmap.Items.AddRange(Enum.GetValues(typeof(nw.g3d.nw4f_3dif.filter_mipType)).Cast<nw.g3d.nw4f_3dif.filter_mipType>().Select(x => new UIListControlItem(UIText.EnumValue(x), x)).ToArray());
            cbxDefaultValue_FilterMipmap.SelectedIndexChanged += cbxDefaultValue_FilterMipmap_SelectedIndexChanged;
            cbxDefaultValue_FilterMaxAniso.Items.AddRange(Enum.GetValues(typeof(nw.g3d.nw4f_3dif.filter_max_anisoType)).Cast<nw.g3d.nw4f_3dif.filter_max_anisoType>().Select(x => new UIListControlItem(UIText.EnumValue(x), x)).ToArray());
            cbxDefaultValue_FilterMaxAniso.SelectedIndexChanged += cbxDefaultValue_FilterMaxAniso_SelectedIndexChanged;
            cbxDefaultValue_WrapU.Items.AddRange(Enum.GetValues(typeof(nw.g3d.nw4f_3dif.wrap_uvwType)).Cast<nw.g3d.nw4f_3dif.wrap_uvwType>().Select(x => new UIListControlItem(UIText.EnumValue(x), x)).ToArray());
            cbxDefaultValue_WrapU.SelectedIndexChanged += cbxDefaultValue_WrapU_SelectedIndexChanged;
            cbxDefaultValue_WrapV.Items.AddRange(Enum.GetValues(typeof(nw.g3d.nw4f_3dif.wrap_uvwType)).Cast<nw.g3d.nw4f_3dif.wrap_uvwType>().Select(x => new UIListControlItem(UIText.EnumValue(x), x)).ToArray());
            cbxDefaultValue_WrapV.SelectedIndexChanged += cbxDefaultValue_WrapV_SelectedIndexChanged;
            cbxDefaultValue_WrapW.Items.AddRange(Enum.GetValues(typeof(nw.g3d.nw4f_3dif.wrap_uvwType)).Cast<nw.g3d.nw4f_3dif.wrap_uvwType>().Select(x => new UIListControlItem(UIText.EnumValue(x), x)).ToArray());
            cbxDefaultValue_WrapW.SelectedIndexChanged += cbxDefaultValue_WrapW_SelectedIndexChanged;
            tbxDefaultValue_LodMin.TextAlign = HorizontalAlignment.Left;
            tbxDefaultValue_LodMax.TextAlign = HorizontalAlignment.Left;
            tbxDefaultValue_LodBias.TextAlign = HorizontalAlignment.Left;
            cbxDefaultValue_Clamp.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagEnableDisable(x), x)).ToArray());
            cbxDefaultValue_Clamp.SelectedIndexChanged += cbxDefaultValue_Clamp_SelectedIndexChanged;
            tbxPreview_Fps.TextAlign = HorizontalAlignment.Left;
            tbxMessageInterval.TextAlign = HorizontalAlignment.Left;
            cbxColor_GammaCorrection.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxColor_GammaCorrection.SelectedIndexChanged += cbxColor_GammaCorrection_SelectedIndexChanged;
            cbxPlatform.Items.AddRange(platformPresets_.Select(x => new UIListControlItem(x.Name, x)).ToArray());
            cbxDisableAnimationQuantize.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxDisableAnimationQuantize.SelectedIndexChanged += cbxDisableAnimationQuantize_SelectedIndexChanged;
            cbxRenderStateInfoVisible.Items.AddRange(booleans.Select(x => new UIListControlItem(UIText.FlagYesNo(x), x)).ToArray());
            cbxRenderStateInfoVisible.SelectedIndexChanged += cbxRenderStateInfoVisible_SelectedIndexChanged;

            configPanels_ = new []
            {
                pnlFileIo,
                pnlPreset,
                pnlDefaultValue,
                pnlError,
                pnlPreview,
                pnlColor,
                pnlPlatform,
                pnlHidden,
            };

            foreach(var panel in configPanels_)
            {
                panel.Parent = null;
            }

            lbxMenu.AddItem(Strings.TeamConfigSettingDialog_Config_FileIo,						pnlFileIo);
            lbxMenu.AddItem(Strings.TeamConfigSettingDialog_Config_Preset,						pnlPreset);
            lbxMenu.AddItem(Strings.TeamConfigSettingDialog_Config_DefaultValue,				pnlDefaultValue);
            lbxMenu.AddItem(Strings.TeamConfigSettingDialog_Config_Error,						pnlError);
            lbxMenu.AddItem(Strings.TeamConfigSettingDialog_Config_Preview,						pnlPreview);
            lbxMenu.AddItem(Strings.TeamConfigSettingDialog_Config_Color,						pnlColor);
            lbxMenu.AddItem(Strings.TeamConfigSettingDialog_Config_Platform,                    pnlPlatform);
#if DEBUG
            lbxMenu.AddItem(Strings.TeamConfigSettingDialog_Config_Hidden,                      pnlHidden);
#else
            if (teamSettings_.DefaultValue.ShowHiddenSettingsPage)
            {
                lbxMenu.AddItem(Strings.TeamConfigSettingDialog_Config_Hidden, pnlHidden);
            }
#endif

            lbxMenu.SelectedItemData = pnlFileIo;
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            btnOK.AutoSize = true;
            btnOK.Text = Strings.TeamConfigSettingDialog_ChangeToEditableMode;
        }

        protected override void OnClosing(CancelEventArgs e)
        {
            base.OnClosing(e);

            if (DialogResult == DialogResult.OK)
            {
                if (!editMode_)
                {
                    // 編集モードに切り替える。
                    if (UIMessageBox.YesNo(Strings.TeamConfigSettingDialog_PreEditConfirmation))
                    {
                        ChangeToEditMode();
                    }
                    e.Cancel = true;
                }
                else if (IsModified)
                {
                    if (UIMessageBox.YesNo(Strings.TeamConfigSettingDialog_PostEditConfirmation))
                    {
                        do
                        {
                            var needsReboot = NeedsReboot;

                            // まず、再起動が必要な場合でのドキュメントの保存を行う。
                            if (needsReboot)
                            {
                                // ドキュメントの保存確認。
                                if (!DocumentManager.CheckAndSaveModified(DocumentManager.Documents))
                                {
                                    // 保存キャンセル。
                                    e.Cancel = true;
                                    break;
                                }
                            }

                            // 次に、設定を保存する。
                            {
                                // プラットフォーム設定の保存もしくは保存準備。
                                var internalPlatformPresets = teamSettings_.Preset.PlatformPresets;
                                if (useExternalPlatformPresets_)
                                {
                                    // プラットフォーム設定をチーム設定ファイルとは別のファイルとして保存。
                                    var srcPlatformPresets = srcPlatformPresets_.ToDictionary(x => x.FilePath, y => ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(y)));
                                    var modifiedPlatformPresets = platformPresets_.Where(x => !srcPlatformPresets.ContainsKey(x.FilePath) || !ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(x)).SequenceEqual(srcPlatformPresets[x.FilePath]));
                                    if (modifiedPlatformPresets.Any())
                                    {
                                        ApplicationConfig.SavePlatformPresets(modifiedPlatformPresets);
                                    }
                                }
                                else
                                {
                                    // プラットフォーム設定をチーム設定として保存するための準備。
                                    teamSettings_.Preset.PlatformPresets = platformPresets_;
                                }

                                // チーム設定を保存。
                                var saved = ApplicationConfig.SaveTeamSettings(teamSettings_);
                                teamSettings_.Preset.PlatformPresets = internalPlatformPresets;

                                if (!saved)
                                {
                                    // 保存失敗。
                                    e.Cancel = true;
                                    break;
                                }

                                // チーム設定の保存先が変更されているかもしれいないので、ラベルを更新。
                                // フォームを閉じるので変更は不要だが念のため。
                                ltbTeamConfigPath.Text = ApplicationConfig.TeamSettingsFilePath;
                            }

                            if (needsReboot)
                            {
                                // 再起動処理を登録。
                                Disposed += (ss, ee) =>
                                {
                                    // 再起動はアプリケーションの終了処理完了後に行わなければならない。
                                    Program.TheProgram.PostShutdown += (sss, eee) =>
                                    {
                                        Application.Restart();
                                    };

                                    // ドキュメントの保存確認は済んでいるので、MainFrame 側での確認は行わない。
                                    TheApp.MainFrame.DocumentSavingConfirmation = false;

                                    // MainFrame を閉じてアプリケーションの終了処理を行わせる。
                                    // Close による終了処理完了後に Program.PostShutdown イベントに設定した Application.Restart() が呼ばれる。
                                    TheApp.MainFrame.Close();
                                };
                            }
                            else
                            {
                                // 変更を適用する。
                                if (IsPlatformPresetModified)
                                {
                                    ApplicationConfig.PlatformPresets = platformPresets_;
                                    if (!useExternalPlatformPresets_)
                                    {
                                        teamSettings_.Preset.PlatformPresets = platformPresets_;
                                    }
                                }
                                if (IsFileIoModified)
                                {
                                    ApplicationConfig.FileIo = teamSettings_.FileIo;
                                }
                                if (IsPresetModified)
                                {
                                    ApplicationConfig.Preset = teamSettings_.Preset;
                                }
                                if (IsDefaultValueModified)
                                {
                                    ApplicationConfig.DefaultValue = teamSettings_.DefaultValue;
                                }
                                if (IsErrorModified)
                                {
                                    ApplicationConfig.Error = teamSettings_.Error;
                                }
                                if (IsPreviewModified)
                                {
                                    ApplicationConfig.Preview = teamSettings_.Preview;

                                    // メッセージ送信間隔を更新。
                                    Viewer.Manager.Instance.SetMessageInterval(ApplicationConfig.Preview.MessageInterval);
                                }
                                if (IsColorModified)
                                {
                                    ApplicationConfig.Color = teamSettings_.Color;
                                }
                            }
                        }
                        while (false);
                    }
                    else
                    {
                        // 変更の確定がキャンセルされた。
                        e.Cancel = true;
                    }
                }
            }
        }

        private static void ReplaceControl(UITextBox oldCtrl, UIComboBox2 newCtrl)
        {
            var parent = oldCtrl.Parent;
            if (parent != null)
            {
                newCtrl.Anchor = oldCtrl.Anchor;
                newCtrl.Bounds = oldCtrl.Bounds;
                newCtrl.MinimumSize = new System.Drawing.Size(System.Math.Max(oldCtrl.MinimumSize.Width, newCtrl.MinimumSize.Width), newCtrl.MinimumSize.Height);
                newCtrl.TabIndex = oldCtrl.TabIndex;
                parent.Controls.Remove(oldCtrl);
                parent.Controls.Add(newCtrl);
            }
        }

        private static void AddListViewEditor(UIPanel panel, UIListView listView, EventHandler up, EventHandler down, EventHandler edit, EventHandler add, EventHandler remove)
        {
            Action<Control, int> adjustLayout = null;
            adjustLayout = (ctrl, yOffset) =>
            {
                var p = ((ctrl != null) && (ctrl != panel)) ? ctrl.Parent : null;
                if (p != null)
                {
                    p.Height += yOffset;
                    var threshold = ctrl.Bottom - yOffset;
                    foreach (var c in p.Controls.OfType<Control>().Where(x => x.Top >= threshold))
                    {
                        c.Top += yOffset;

                        adjustLayout(c, yOffset);
                    }
                }
            };

            var parent = listView.Parent;
            if (parent != null)
            {
                var editor = new ListViewEditor()
                {
                    ListView = listView
                };
                var yOffset = Math.Max(editor.Margin.Top, listView.Margin.Bottom);
                editor.Location = new System.Drawing.Point(
                    Math.Max(0, listView.Width - editor.Width),
                    listView.Bottom + yOffset);
                EventHandler layout = (ss, ee) =>
                {
                    editor.Location = new System.Drawing.Point(
                        Math.Max(0, listView.Width - editor.Width),
                        listView.Bottom + yOffset);
                };
                listView.SizeChanged += layout;
                listView.LocationChanged += layout;
                listView.DoubleClick += edit;
                editor.MoveUpButtonClicked += up;
                editor.MoveDownButtonClicked += down;
                editor.EditButtonClicked += edit;
                editor.AddButtonClicked += add;
                editor.RemoveButtonClicked += remove;

                parent.Height += yOffset + editor.Height;
                foreach (var ctrl in parent.Controls.OfType<Control>().Where(x => x.Top >= listView.Bottom))
                {
                    ctrl.Top += yOffset + editor.Height;
                }

                parent.Controls.Add(editor);

                adjustLayout(parent, yOffset + editor.Height);
            }
        }

        private System.Collections.Generic.List<TeamConfig.Preset.AttenuationFunction> GetAttenuationPresets(string type)
        {
            if (type == Strings.TeamConfigSettingDialog_pnlPreset_LightDistAttnFunc)
            {
                return teamSettings_.Preset.LightDistAttnPresets;
            }
            else if (type == Strings.TeamConfigSettingDialog_pnlPreset_LightAngleAttnFunc)
            {
                return teamSettings_.Preset.LightAngAttnPresets;
            }
            else if (type == Strings.TeamConfigSettingDialog_pnlPreset_FogDistAttnFunc)
            {
                return teamSettings_.Preset.FogDistAttnPresets;
            }
            throw new ArgumentOutOfRangeException();
        }

        private void ChangeToEditMode()
        {
            if (editMode_)
            {
                return;
            }

            using (var block = new UIControlEventSuppressBlock())
            {
                #region FileIo
                using (var cb = new CustomBlock(() => { pnlFileIo.SuspendLayout(); }, () => { pnlFileIo.ResumeLayout(); }))
                {
                    Action updateSearchPaths = () =>
                    {
                        teamSettings_.FileIo.SearchPaths.Clear();
                        teamSettings_.FileIo.SearchPaths.AddRange(lvwFileIoSearchPath.Items.Cast<ListViewItem>().Select(x => (TeamConfig.FileIo.SearchPath)x.Tag).ToArray());

                        // スタートアップパスは追加参照パスを参照することがあるので、その更新も必要。
                        foreach (var item in lvwFileIoStartUpPath.Items.Cast<ListViewItem>())
                        {
                            var startUpPath = (ConfigCommon.StartUp)item.Tag;
                            item.SubItems[clhlvwFileIoStartUpPath_FullPath.Index].Text = startUpPath.FullPath(teamSettings_.FileIo.SearchPaths);
                        }

                        isFileIoModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlFileIo,
                        lvwFileIoSearchPath,
                        delegate (object ss, EventArgs ee)
                        {
                            updateSearchPaths();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateSearchPaths();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwFileIoSearchPath.SelectedItems.Count == 1)
                            {
                                var item = lvwFileIoSearchPath.SelectedItems[0];
                                using (var dialog = new FileIoSearchPathDialog()
                                {
                                    Tag = new FileIoSearchPathDialog.TagData((TeamConfig.FileIo.SearchPath)item.Tag)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (FileIoSearchPathDialog.TagData)dialog.Tag;
                                        item.SubItems[clhlvwFileIoSearchPath_Name.Index].Text = data.SearchPath.Name;
                                        item.SubItems[clhlvwFileIoSearchPath_Path.Index].Text = data.SearchPath.pathXml;
                                        item.SubItems[clhlvwFileIoSearchPath_Recursive.Index].Text = UIText.FlagYesNo(data.SearchPath.Recursive);
                                        item.SubItems[clhlvwFileIoSearchPath_FullPath.Index].Text = System.Environment.ExpandEnvironmentVariables(data.SearchPath.pathXml);

                                        updateSearchPaths();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new FileIoSearchPathDialog()
                            {
                                Tag = new FileIoSearchPathDialog.TagData(new TeamConfig.FileIo.SearchPath())
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwFileIoSearchPath.Items.Count;
                                    lvwFileIoSearchPath.SetItemCount(n + 1);
                                    var item = lvwFileIoSearchPath.Items[n];

                                    var data = (FileIoSearchPathDialog.TagData)dialog.Tag;
                                    item.SubItems[clhlvwFileIoSearchPath_Name.Index].Text = data.SearchPath.Name;
                                    item.SubItems[clhlvwFileIoSearchPath_Path.Index].Text = data.SearchPath.pathXml;
                                    item.SubItems[clhlvwFileIoSearchPath_Recursive.Index].Text = UIText.FlagYesNo(data.SearchPath.Recursive);
                                    item.SubItems[clhlvwFileIoSearchPath_FullPath.Index].Text = System.Environment.ExpandEnvironmentVariables(data.SearchPath.pathXml);

                                    item.Tag = data.SearchPath;
                                    teamSettings_.FileIo.SearchPaths.Add(data.SearchPath);

                                    updateSearchPaths();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwFileIoSearchPath.SelectedItems.Count == 1)
                            {
                                var item = lvwFileIoSearchPath.SelectedItems[0];
                                lvwFileIoSearchPath.Items.Remove(item);
                                teamSettings_.FileIo.SearchPaths.Remove((TeamConfig.FileIo.SearchPath)item.Tag);

                                updateSearchPaths();
                            }
                        });

                    Action updateStartUpPath = () =>
                    {
                        teamSettings_.FileIo.StartUpPaths.Clear();
                        teamSettings_.FileIo.StartUpPaths.AddRange(lvwFileIoStartUpPath.Items.Cast<ListViewItem>().Select(x => (ConfigCommon.StartUp)x.Tag).ToArray());

                        isFileIoModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlFileIo,
                        lvwFileIoStartUpPath,
                        delegate (object ss, EventArgs ee)
                        {
                            updateStartUpPath();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateStartUpPath();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwFileIoStartUpPath.SelectedItems.Count == 1)
                            {
                                var item = lvwFileIoStartUpPath.SelectedItems[0];
                                using (var dialog = new FileIoStartUpPathDialog()
                                {
                                    Tag = new FileIoStartUpPathDialog.TagData((ConfigCommon.StartUp)item.Tag, teamSettings_.FileIo.SearchPaths)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (FileIoStartUpPathDialog.TagData)dialog.Tag;
                                        item.SubItems[clhlvwFileIoStartUpPath_Base.Index].Text = data.StartUp.Base;
                                        item.SubItems[clhlvwFileIoStartUpPath_Path.Index].Text = data.StartUp.pathXml;
                                        item.SubItems[clhlvwFileIoStartUpPath_FullPath.Index].Text = data.StartUp.FullPath(teamSettings_.FileIo.SearchPaths);

                                        updateStartUpPath();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new FileIoStartUpPathDialog()
                            {
                                Tag = new FileIoStartUpPathDialog.TagData(new ConfigCommon.StartUp(), teamSettings_.FileIo.SearchPaths)
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwFileIoStartUpPath.Items.Count;
                                    lvwFileIoStartUpPath.SetItemCount(n + 1);
                                    var item = lvwFileIoStartUpPath.Items[n];

                                    var data = (FileIoStartUpPathDialog.TagData)dialog.Tag;
                                    item.SubItems[clhlvwFileIoStartUpPath_Base.Index].Text = data.StartUp.Base;
                                    item.SubItems[clhlvwFileIoStartUpPath_Path.Index].Text = data.StartUp.pathXml;
                                    item.SubItems[clhlvwFileIoStartUpPath_FullPath.Index].Text = data.StartUp.FullPath(teamSettings_.FileIo.SearchPaths);

                                    item.Tag = data.StartUp;
                                    teamSettings_.FileIo.StartUpPaths.Add(data.StartUp);

                                    updateStartUpPath();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwFileIoStartUpPath.SelectedItems.Count == 1)
                            {
                                var item = lvwFileIoStartUpPath.SelectedItems[0];
                                lvwFileIoStartUpPath.Items.Remove(item);
                                teamSettings_.FileIo.StartUpPaths.Remove((ConfigCommon.StartUp)item.Tag);

                                updateStartUpPath();
                            }
                        });

                    Action updateAttachPath = () =>
                    {
                        teamSettings_.FileIo.AttachPaths.Clear();
                        teamSettings_.FileIo.AttachPaths.AddRange(lvwAttachPath.Items.Cast<ListViewItem>().Select(x => (TeamConfig.FileIo.AttachPath)x.Tag).ToArray());

                        isFileIoModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlFileIo,
                        lvwAttachPath,
                        delegate (object ss, EventArgs ee)
                        {
                            updateAttachPath();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateAttachPath();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwAttachPath.SelectedItems.Count == 1)
                            {
                                var item = lvwAttachPath.SelectedItems[0];
                                using (var dialog = new FileIoAttachPathDialog()
                                {
                                    Tag = new FileIoAttachPathDialog.TagData((TeamConfig.FileIo.AttachPath)item.Tag)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (FileIoAttachPathDialog.TagData)dialog.Tag;
                                        item.SubItems[clhAttachPath_Filter.Index].Text = data.AttachPath.Filter;
                                        item.SubItems[clhAttachPath_Path.Index].Text = data.AttachPath.pathXml;
                                        item.SubItems[clhAttachPath_FullPath.Index].Text = data.AttachPath.FullPath();

                                        updateAttachPath();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new FileIoAttachPathDialog()
                            {
                                Tag = new FileIoAttachPathDialog.TagData(new TeamConfig.FileIo.AttachPath())
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwAttachPath.Items.Count;
                                    lvwAttachPath.SetItemCount(n + 1);
                                    var item = lvwAttachPath.Items[n];

                                    var data = (FileIoAttachPathDialog.TagData)dialog.Tag;
                                    item.SubItems[clhAttachPath_Filter.Index].Text = data.AttachPath.Filter;
                                    item.SubItems[clhAttachPath_Path.Index].Text = data.AttachPath.pathXml;
                                    item.SubItems[clhAttachPath_FullPath.Index].Text = data.AttachPath.FullPath();

                                    item.Tag = data.AttachPath;
                                    teamSettings_.FileIo.AttachPaths.Add(data.AttachPath);

                                    updateAttachPath();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwAttachPath.SelectedItems.Count == 1)
                            {
                                var item = lvwAttachPath.SelectedItems[0];
                                lvwAttachPath.Items.Remove(item);
                                teamSettings_.FileIo.AttachPaths.Remove((TeamConfig.FileIo.AttachPath)item.Tag);

                                updateAttachPath();
                            }
                        });

                    Action updateParentMaterialPath = () =>
                    {
                        teamSettings_.FileIo.ParentMaterialPaths.Clear();
                        teamSettings_.FileIo.ParentMaterialPaths.AddRange(lvwParentMaterialPath.Items.Cast<ListViewItem>().Select(x => (TeamConfig.FileIo.ParentMaterialPath)x.Tag).ToArray());

                        isFileIoModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlFileIo,
                        lvwParentMaterialPath,
                        delegate (object ss, EventArgs ee)
                        {
                            updateParentMaterialPath();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateParentMaterialPath();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwParentMaterialPath.SelectedItems.Count == 1)
                            {
                                var item = lvwParentMaterialPath.SelectedItems[0];
                                using (var dialog = new FileIoParentMaterialPathDialog()
                                {
                                    Tag = new FileIoParentMaterialPathDialog.TagData((TeamConfig.FileIo.ParentMaterialPath)item.Tag)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (FileIoParentMaterialPathDialog.TagData)dialog.Tag;
                                        item.SubItems[clhlvwFileIoParentPath_Name.Index].Text = data.ParentMaterialPath.Name;
                                        item.SubItems[clhlvwFileIoParentPath_Recursive.Index].Text = UIText.FlagYesNo(data.ParentMaterialPath.Recursive);
                                        item.SubItems[clhlvwFileIoParentPath_Path.Index].Text = data.ParentMaterialPath.pathXml;
                                        item.SubItems[clhlvwFileIoParentPath_FullPath.Index].Text = data.ParentMaterialPath.path;

                                        updateParentMaterialPath();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new FileIoParentMaterialPathDialog()
                            {
                                Tag = new FileIoParentMaterialPathDialog.TagData(new TeamConfig.FileIo.ParentMaterialPath())
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwParentMaterialPath.Items.Count;
                                    lvwParentMaterialPath.SetItemCount(n + 1);
                                    var item = lvwParentMaterialPath.Items[n];

                                    var data = (FileIoParentMaterialPathDialog.TagData)dialog.Tag;
                                    item.SubItems[clhlvwFileIoParentPath_Name.Index].Text = data.ParentMaterialPath.Name;
                                    item.SubItems[clhlvwFileIoParentPath_Recursive.Index].Text = UIText.FlagYesNo(data.ParentMaterialPath.Recursive);
                                    item.SubItems[clhlvwFileIoParentPath_Path.Index].Text = data.ParentMaterialPath.pathXml;
                                    item.SubItems[clhlvwFileIoParentPath_FullPath.Index].Text = data.ParentMaterialPath.path;

                                    item.Tag = data.ParentMaterialPath;
                                    teamSettings_.FileIo.ParentMaterialPaths.Add(data.ParentMaterialPath);

                                    updateParentMaterialPath();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwParentMaterialPath.SelectedItems.Count == 1)
                            {
                                var item = lvwParentMaterialPath.SelectedItems[0];
                                lvwParentMaterialPath.Items.Remove(item);
                                teamSettings_.FileIo.ParentMaterialPaths.Remove((TeamConfig.FileIo.ParentMaterialPath)item.Tag);

                                updateParentMaterialPath();
                            }
                        });

                    tbxPreSaveCommand.ReadOnly = false;
                    tbxPostSaveCommand.ReadOnly = false;
                    tbxPreOpenCommand.ReadOnly = false;
                    tbxPostCloseCommand.ReadOnly = false;
                    tbxPreBinarizeCommand.ReadOnly = false;
                    tbxPreBinarizeCommandFilter.ReadOnly = false;
                    tbxShaderConverterAdditionalArgs.ReadOnly = false;
                    tbxShaderConverterAdditionalArgsExpanded.ReadOnly = true;
                    tbxShaderConverterAdditionalArgsForOptimizedShader.ReadOnly = false;
                    tbxShaderConverterAdditionalArgsForOptimizedShaderExpanded.ReadOnly = true;
                    ReplaceControl(tbxShaderConvertLogTime, cbxShaderConvertLogTime);
                    cbxShaderConvertLogTime.SelectedItem = cbxShaderConvertLogTime.Items.OfType<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.FileIo.LogShaderConvertTime);
                    tbxCreateShaderVariationCommand.ReadOnly = false;
                    ReplaceControl(tbxCreateShaderVariationCommandAllowNoOutput, cbxCreateShaderVariationCommandAllowNoOutput);
                    cbxCreateShaderVariationCommandAllowNoOutput.SelectedItem = cbxCreateShaderVariationCommandAllowNoOutput.Items.OfType<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.FileIo.CreateShaderVariationCommand.AllowNoOutput);

                    Action updateUserCommand = () =>
                    {
                        teamSettings_.FileIo.UserCommands.Clear();
                        teamSettings_.FileIo.UserCommands.AddRange(lvwUserCommand.Items.Cast<ListViewItem>().Select(x => (TeamConfig.FileIo.UserCommand)x.Tag).ToArray());

                        isFileIoModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlFileIo,
                        lvwUserCommand,
                        delegate (object ss, EventArgs ee)
                        {
                            updateUserCommand();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateUserCommand();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwUserCommand.SelectedItems.Count == 1)
                            {
                                var item = lvwUserCommand.SelectedItems[0];
                                using (var dialog = new FileIoUserCommandDialog()
                                {
                                    Tag = new FileIoUserCommandDialog.TagData((TeamConfig.FileIo.UserCommand)item.Tag)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (FileIoUserCommandDialog.TagData)dialog.Tag;
                                        item.SubItems[clhUserCommand_Name.Index].Text = data.UserCommand.Name;
                                        item.SubItems[clhUserCommand_Filter.Index].Text = data.UserCommand.Filter;
                                        item.SubItems[clhUserCommand_Args.Index].Text = data.UserCommand.Args;
                                        item.SubItems[clhUserCommand_Reload.Index].Text = UIText.FlagYesNo(data.UserCommand.Reload);
                                        item.SubItems[clhUserCommand_ForceUnloadAndLoad.Index].Text = UIText.FlagYesNo(data.UserCommand.ForceUnloadAndLoad);
                                        item.SubItems[clhUserCommand_Temporary.Index].Text = UIText.FlagYesNo(data.UserCommand.Temporary);
                                        item.SubItems[clhUserCommand_ShortCut.Index].Text = data.UserCommand.ShortCut;
                                        item.SubItems[clhUserCommand_FileTreeShortCut.Index].Text = UIText.FlagYesNo(data.UserCommand.FileTreeShortCut);
                                        item.SubItems[clhUserCommand_Path.Index].Text = data.UserCommand.PathXml;

                                        updateUserCommand();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new FileIoUserCommandDialog()
                            {
                                Tag = new FileIoUserCommandDialog.TagData(new TeamConfig.FileIo.UserCommand())
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwUserCommand.Items.Count;
                                    lvwUserCommand.SetItemCount(n + 1);
                                    var item = lvwUserCommand.Items[n];

                                    var data = (FileIoUserCommandDialog.TagData)dialog.Tag;
                                    item.SubItems[clhUserCommand_Name.Index].Text = data.UserCommand.Name;
                                    item.SubItems[clhUserCommand_Filter.Index].Text = data.UserCommand.Filter;
                                    item.SubItems[clhUserCommand_Args.Index].Text = data.UserCommand.Args;
                                    item.SubItems[clhUserCommand_Reload.Index].Text = UIText.FlagYesNo(data.UserCommand.Reload);
                                    item.SubItems[clhUserCommand_ForceUnloadAndLoad.Index].Text = UIText.FlagYesNo(data.UserCommand.ForceUnloadAndLoad);
                                    item.SubItems[clhUserCommand_Temporary.Index].Text = UIText.FlagYesNo(data.UserCommand.Temporary);
                                    item.SubItems[clhUserCommand_ShortCut.Index].Text = data.UserCommand.ShortCut;
                                    item.SubItems[clhUserCommand_FileTreeShortCut.Index].Text = UIText.FlagYesNo(data.UserCommand.FileTreeShortCut);
                                    item.SubItems[clhUserCommand_Path.Index].Text = data.UserCommand.PathXml;

                                    item.Tag = data.UserCommand;
                                    teamSettings_.FileIo.UserCommands.Add(data.UserCommand);

                                    updateUserCommand();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwUserCommand.SelectedItems.Count == 1)
                            {
                                var item = lvwUserCommand.SelectedItems[0];
                                lvwUserCommand.Items.Remove(item);
                                teamSettings_.FileIo.UserCommands.Remove((TeamConfig.FileIo.UserCommand)item.Tag);

                                updateUserCommand();
                            }
                        });

                    Action updateRuntimeUserScript = () =>
                    {
                        teamSettings_.FileIo.RuntimeUserScripts.Clear();
                        teamSettings_.FileIo.RuntimeUserScripts.AddRange(lvwRuntimeUserScript.Items.Cast<ListViewItem>().Select(x => (TeamConfig.FileIo.RuntimeUserScript)x.Tag).ToArray());

                        isFileIoModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlFileIo,
                        lvwRuntimeUserScript,
                        delegate (object ss, EventArgs ee)
                        {
                            updateRuntimeUserScript();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateRuntimeUserScript();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwRuntimeUserScript.SelectedItems.Count == 1)
                            {
                                var item = lvwRuntimeUserScript.SelectedItems[0];
                                using (var dialog = new FileIoRuntimeUserScriptDialog()
                                {
                                    Tag = new FileIoRuntimeUserScriptDialog.TagData((TeamConfig.FileIo.RuntimeUserScript)item.Tag)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (FileIoRuntimeUserScriptDialog.TagData)dialog.Tag;
                                        item.SubItems[clhRuntimeUserScript_Name.Index].Text = data.RuntimeUserScript.Name;
                                        item.SubItems[clhRuntimeUserScript_Path.Index].Text = data.RuntimeUserScript.PathXml;

                                        updateRuntimeUserScript();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new FileIoRuntimeUserScriptDialog()
                            {
                                Tag = new FileIoRuntimeUserScriptDialog.TagData(new TeamConfig.FileIo.RuntimeUserScript())
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwRuntimeUserScript.Items.Count;
                                    lvwRuntimeUserScript.SetItemCount(n + 1);
                                    var item = lvwRuntimeUserScript.Items[n];

                                    var data = (FileIoRuntimeUserScriptDialog.TagData)dialog.Tag;
                                    item.SubItems[clhRuntimeUserScript_Name.Index].Text = data.RuntimeUserScript.Name;
                                    item.SubItems[clhRuntimeUserScript_Path.Index].Text = data.RuntimeUserScript.PathXml;

                                    item.Tag = data.RuntimeUserScript;
                                    teamSettings_.FileIo.RuntimeUserScripts.Add(data.RuntimeUserScript);

                                    updateRuntimeUserScript();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwRuntimeUserScript.SelectedItems.Count == 1)
                            {
                                var item = lvwRuntimeUserScript.SelectedItems[0];
                                lvwRuntimeUserScript.Items.Remove(item);
                                teamSettings_.FileIo.RuntimeUserScripts.Remove((TeamConfig.FileIo.RuntimeUserScript)item.Tag);

                                updateRuntimeUserScript();
                            }
                        });
                }
                #endregion FileIo

                #region Preset
                using (var cb = new CustomBlock(() => { pnlPreset.SuspendLayout(); }, () => { pnlPreset.ResumeLayout(); }))
                {
                    tbxSettingName.ReadOnly = false;
                    ReplaceControl(tbxEnableMaterialAnimCreation, cbxEnableMaterialAnimCreation);
                    cbxEnableMaterialAnimCreation.SelectedItem = cbxEnableMaterialAnimCreation.Items.OfType<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.Preset.EnableMaterialAnimCreation);
                    ReplaceControl(tbxSeparateMaterialAnimCreationMenu, cbxSeparateMaterialAnimCreationMenu);
                    cbxSeparateMaterialAnimCreationMenu.SelectedItem = cbxSeparateMaterialAnimCreationMenu.Items.OfType<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.Preset.SeparateMaterialAnimCreationMenu);
                    ReplaceControl(tbxShowOnlyBinarizeEnabledBoneVisAnimCurve, cbxShowOnlyBinarizeEnabledBoneVisAnimCurve);
                    cbxShowOnlyBinarizeEnabledBoneVisAnimCurve.SelectedItem = cbxShowOnlyBinarizeEnabledBoneVisAnimCurve.Items.OfType<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.Preset.ShowOnlyBinarizeEnabledBoneVisibilityAnimCurve);

                    Action updateTexture = () =>
                    {
                        teamSettings_.Preset.TexturePresets.Clear();
                        teamSettings_.Preset.TexturePresets.AddRange(lvwPresetTexture.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.TexturePreset)x.Tag).ToArray());

                        isPresetModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlPreset,
                        lvwPresetTexture,
                        delegate (object ss, EventArgs ee)
                        {
                            updateTexture();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateTexture();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwPresetTexture.SelectedItems.Count == 1)
                            {
                                var item = lvwPresetTexture.SelectedItems[0];
                                using (var dialog = new PresetTextureDialog()
                                {
                                    Tag = new PresetTextureDialog.TagData((TeamConfig.Preset.TexturePreset)item.Tag)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (PresetTextureDialog.TagData)dialog.Tag;
                                        item.SubItems[clhPresetTexture_TextureName.Index].Text = data.TexturePreset.Name;
                                        item.SubItems[clhPresetTexture_TextureFormat.Index].Text = UIText.EnumValue(data.TexturePreset.quantize_type);

                                        updateTexture();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new PresetTextureDialog()
                            {
                                Tag = new PresetTextureDialog.TagData(new TeamConfig.Preset.TexturePreset())
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwPresetTexture.Items.Count;
                                    lvwPresetTexture.SetItemCount(n + 1);
                                    var item = lvwPresetTexture.Items[n];

                                    var data = (PresetTextureDialog.TagData)dialog.Tag;
                                    item.SubItems[clhPresetTexture_TextureName.Index].Text = data.TexturePreset.Name;
                                    item.SubItems[clhPresetTexture_TextureFormat.Index].Text = UIText.EnumValue(data.TexturePreset.quantize_type);

                                    item.Tag = data.TexturePreset;
                                    teamSettings_.Preset.TexturePresets.Add(data.TexturePreset);

                                    updateTexture();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwPresetTexture.SelectedItems.Count == 1)
                            {
                                var item = lvwPresetTexture.SelectedItems[0];
                                lvwPresetTexture.Items.Remove(item);
                                teamSettings_.Preset.TexturePresets.Remove((TeamConfig.Preset.TexturePreset)item.Tag);

                                updateTexture();
                            }
                        });

                    ReplaceControl(tbxFollowDccSamplerNameRule, cbxFollowDccSamplerNameRule);
                    cbxFollowDccSamplerNameRule.SelectedItem = cbxFollowDccSamplerNameRule.Items.OfType<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.Preset.FollowDccSamplerNameRule);
                    ReplaceControl(tbxDisableTextureSrtWithOriginal, cbxDisableTextureSrtWithOriginal);
                    cbxDisableTextureSrtWithOriginal.SelectedItem = cbxDisableTextureSrtWithOriginal.Items.OfType<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.Preset.DisableEditTextureSrtWithOriginal);

                    Action updateSamplerName = () =>
                    {
                        teamSettings_.Preset.SamplerNamePresets.Clear();
                        teamSettings_.Preset.SamplerNamePresets.AddRange(lvwPresetSamplerName.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.SamplerNamePreset)x.Tag).ToArray());

                        isPresetModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlPreset,
                        lvwPresetSamplerName,
                        delegate (object ss, EventArgs ee)
                        {
                            updateSamplerName();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateSamplerName();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwPresetSamplerName.SelectedItems.Count == 1)
                            {
                                var item = lvwPresetSamplerName.SelectedItems[0];
                                using (var dialog = new PresetSamplerNameDialog()
                                {
                                    Tag = new PresetSamplerNameDialog.TagData((TeamConfig.Preset.SamplerNamePreset)item.Tag)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (PresetSamplerNameDialog.TagData)dialog.Tag;
                                        item.SubItems[clhPresetSamplerName_Name.Index].Text = data.SamplerNamePreset.Name;

                                        updateSamplerName();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new PresetSamplerNameDialog()
                            {
                                Tag = new PresetSamplerNameDialog.TagData(new TeamConfig.Preset.SamplerNamePreset())
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwPresetSamplerName.Items.Count;
                                    lvwPresetSamplerName.SetItemCount(n + 1);
                                    var item = lvwPresetSamplerName.Items[n];

                                    var data = (PresetSamplerNameDialog.TagData)dialog.Tag;
                                    item.SubItems[clhPresetSamplerName_Name.Index].Text = data.SamplerNamePreset.Name;

                                    item.Tag = data.SamplerNamePreset;
                                    teamSettings_.Preset.SamplerNamePresets.Add(data.SamplerNamePreset);

                                    updateSamplerName();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwPresetSamplerName.SelectedItems.Count == 1)
                            {
                                var item = lvwPresetSamplerName.SelectedItems[0];
                                lvwPresetSamplerName.Items.Remove(item);
                                teamSettings_.Preset.SamplerNamePresets.Remove((TeamConfig.Preset.SamplerNamePreset)item.Tag);

                                updateSamplerName();
                            }
                        });

                    Action updateVertexAttributeAssign = () =>
                    {
                        teamSettings_.Preset.VertexAttributeAssignPresets.Clear();
                        teamSettings_.Preset.VertexAttributeAssignPresets.AddRange(lvwPresetVertexAttributeAssign.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.VertexAttributeAssignPreset)x.Tag).ToArray());

                        isPresetModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlPreset,
                        lvwPresetVertexAttributeAssign,
                        delegate (object ss, EventArgs ee)
                        {
                            updateVertexAttributeAssign();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateVertexAttributeAssign();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwPresetVertexAttributeAssign.SelectedItems.Count == 1)
                            {
                                var item = lvwPresetVertexAttributeAssign.SelectedItems[0];
                                using (var dialog = new PresetVertexAttributeAssignDialog()
                                {
                                    Tag = new PresetVertexAttributeAssignDialog.TagData((TeamConfig.Preset.VertexAttributeAssignPreset)item.Tag)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (PresetVertexAttributeAssignDialog.TagData)dialog.Tag;
                                        item.SubItems[clhPresetVertexAttributeAssign_Name.Index].Text = data.VertexAttributeAssignPreset.Name;

                                        updateVertexAttributeAssign();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new PresetVertexAttributeAssignDialog()
                            {
                                Tag = new PresetVertexAttributeAssignDialog.TagData(new TeamConfig.Preset.VertexAttributeAssignPreset())
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwPresetVertexAttributeAssign.Items.Count;
                                    lvwPresetVertexAttributeAssign.SetItemCount(n + 1);
                                    var item = lvwPresetVertexAttributeAssign.Items[n];

                                    var data = (PresetVertexAttributeAssignDialog.TagData)dialog.Tag;
                                    item.SubItems[clhPresetVertexAttributeAssign_Name.Index].Text = data.VertexAttributeAssignPreset.Name;

                                    item.Tag = data.VertexAttributeAssignPreset;
                                    teamSettings_.Preset.VertexAttributeAssignPresets.Add(data.VertexAttributeAssignPreset);

                                    updateVertexAttributeAssign();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwPresetVertexAttributeAssign.SelectedItems.Count == 1)
                            {
                                var item = lvwPresetVertexAttributeAssign.SelectedItems[0];
                                lvwPresetVertexAttributeAssign.Items.Remove(item);
                                teamSettings_.Preset.VertexAttributeAssignPresets.Remove((TeamConfig.Preset.VertexAttributeAssignPreset)item.Tag);

                                updateVertexAttributeAssign();
                            }
                        });

                    Action updateLightAnim = () =>
                    {
                        teamSettings_.Preset.LightAnimPresets.Clear();
                        teamSettings_.Preset.LightAnimPresets.AddRange(lvwPresetLight.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.LightAnimPreset)x.Tag).ToArray());

                        isPresetModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlPreset,
                        lvwPresetLight,
                        delegate (object ss, EventArgs ee)
                        {
                            updateLightAnim();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateLightAnim();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwPresetLight.SelectedItems.Count == 1)
                            {
                                var item = lvwPresetLight.SelectedItems[0];
                                using (var dialog = new PresetLightAnimDialog()
                                {
                                    Tag = new PresetLightAnimDialog.TagData((TeamConfig.Preset.LightAnimPreset)item.Tag, teamSettings_.Preset.LightAnimPresets.Where(x => x != item.Tag).ToArray())
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (PresetLightAnimDialog.TagData)dialog.Tag;
                                        item.SubItems[clhPresetLight_LightType.Index].Text = data.LightAnimPreset.Type;
                                        item.SubItems[clhPresetLight_LightName.Index].Text = data.LightAnimPreset.Label;
                                        item.SubItems[clhPresetLight_LightTargets.Index].Text = string.Join(
                                            ", ",
                                            data.LightAnimPreset.LightAnimTargets.Select(
                                                x =>
                                                x.Target.ToString() + (x.HasDefault() ? "(" + x.DefaultValue + ")" : "")));

                                        updateLightAnim();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new PresetLightAnimDialog()
                            {
                                Tag = new PresetLightAnimDialog.TagData(new TeamConfig.Preset.LightAnimPreset(), teamSettings_.Preset.LightAnimPresets)
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwPresetLight.Items.Count;
                                    lvwPresetLight.SetItemCount(n + 1);
                                    var item = lvwPresetLight.Items[n];

                                    var data = (PresetLightAnimDialog.TagData)dialog.Tag;
                                    item.SubItems[clhPresetLight_LightType.Index].Text = data.LightAnimPreset.Type;
                                    item.SubItems[clhPresetLight_LightName.Index].Text = data.LightAnimPreset.Label;
                                    item.SubItems[clhPresetLight_LightTargets.Index].Text = string.Join(
                                        ", ",
                                        data.LightAnimPreset.LightAnimTargets.Select(
                                            x =>
                                            x.Target.ToString() + (x.HasDefault() ? "(" + x.DefaultValue + ")" : "")));

                                    item.Tag = data.LightAnimPreset;
                                    teamSettings_.Preset.LightAnimPresets.Add(data.LightAnimPreset);

                                    updateLightAnim();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwPresetLight.SelectedItems.Count == 1)
                            {
                                var item = lvwPresetLight.SelectedItems[0];
                                lvwPresetLight.Items.Remove(item);
                                teamSettings_.Preset.LightAnimPresets.Remove((TeamConfig.Preset.LightAnimPreset)item.Tag);

                                updateLightAnim();
                            }
                        });

                    Action updateAttnFunc = () =>
                    {
                        teamSettings_.Preset.LightDistAttnPresets.Clear();
                        teamSettings_.Preset.LightAngAttnPresets.Clear();
                        teamSettings_.Preset.FogDistAttnPresets.Clear();
                        foreach (var item in lvwPresetAttn.Items.Cast<ListViewItem>())
                        {
                            GetAttenuationPresets(item.SubItems[clhPresetAttn_AttnKind.Index].Text).Add((TeamConfig.Preset.AttenuationFunction)item.Tag);
                        }

                        isPresetModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlPreset,
                        lvwPresetAttn,
                        delegate (object ss, EventArgs ee)
                        {
                            updateAttnFunc();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateAttnFunc();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwPresetAttn.SelectedItems.Count == 1)
                            {
                                var item = lvwPresetAttn.SelectedItems[0];
                                using (var dialog = new PresetAttnDialog()
                                {
                                    Tag = new PresetAttnDialog.TagData(
                                        item.SubItems[clhPresetAttn_AttnKind.Index].Text,
                                        (TeamConfig.Preset.AttenuationFunction)item.Tag,
                                        GetAttenuationPresets)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (PresetAttnDialog.TagData)dialog.Tag;

                                        var oAttnType = item.SubItems[clhPresetAttn_AttnKind.Index].Text;
                                        if (data.AttnType != oAttnType)
                                        {
                                            var srcAttnPresets = GetAttenuationPresets(oAttnType);
                                            srcAttnPresets.Remove(data.AttnFunc);
                                            var dstAttnPresets = GetAttenuationPresets(data.AttnType);
                                            dstAttnPresets.Add(data.AttnFunc);
                                        }
                                        item.SubItems[clhPresetAttn_AttnKind.Index].Text = data.AttnType;
                                        item.SubItems[clhPresetAttn_AttnName.Index].Text = data.AttnFunc.Label;
                                        item.SubItems[clhPresetAttn_AttnType.Index].Text = data.AttnFunc.Function;

                                        updateAttnFunc();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new PresetAttnDialog()
                            {
                                Tag = new PresetAttnDialog.TagData(
                                    string.Empty,
                                    new TeamConfig.Preset.AttenuationFunction(),
                                    GetAttenuationPresets)
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwPresetAttn.Items.Count;
                                    lvwPresetAttn.SetItemCount(n + 1);
                                    var item = lvwPresetAttn.Items[n];

                                    var data = (PresetAttnDialog.TagData)dialog.Tag;
                                    item.SubItems[clhPresetAttn_AttnKind.Index].Text = data.AttnType;
                                    item.SubItems[clhPresetAttn_AttnName.Index].Text = data.AttnFunc.Label;
                                    item.SubItems[clhPresetAttn_AttnType.Index].Text = data.AttnFunc.Function;

                                    item.Tag = data.AttnFunc;
                                    var attnPresets = GetAttenuationPresets(data.AttnType);
                                    attnPresets.Add(data.AttnFunc);

                                    updateAttnFunc();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwPresetAttn.SelectedItems.Count == 1)
                            {
                                var item = lvwPresetAttn.SelectedItems[0];
                                lvwPresetAttn.Items.Remove(item);
                                var attnPresets = GetAttenuationPresets(item.SubItems[clhPresetAttn_AttnKind.Index].Text);
                                attnPresets.Remove((TeamConfig.Preset.AttenuationFunction)item.Tag);

                                updateAttnFunc();
                            }
                        });
                }
                #endregion Preset

                #region DefaultValue
                using (var cb = new CustomBlock(() => { pnlDefaultValue.SuspendLayout(); }, () => { pnlDefaultValue.ResumeLayout(); }))
                {
                    ReplaceControl(tbxOptimizeShaderAfterMaterialChanged, cbxOptimizeShaderAfterMaterialChanged);
                    cbxOptimizeShaderAfterMaterialChanged.SelectedItem = cbxOptimizeShaderAfterMaterialChanged.Items.OfType<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.DefaultValue.OptimizeShaderAfterMaterialChanged);

                    ReplaceControl(tbxDefaultValue_FilterMag, cbxDefaultValue_FilterMag);
                    cbxDefaultValue_FilterMag.SelectedItem = cbxDefaultValue_FilterMag.Items.OfType<UIListControlItem>().FirstOrDefault(x => (nw.g3d.nw4f_3dif.filter_mag_minType)x.Data == teamSettings_.DefaultValue.filter.mag);
                    ReplaceControl(tbxDefaultValue_FilterMin, cbxDefaultValue_FilterMin);
                    cbxDefaultValue_FilterMin.SelectedItem = cbxDefaultValue_FilterMin.Items.OfType<UIListControlItem>().FirstOrDefault(x => (nw.g3d.nw4f_3dif.filter_mag_minType)x.Data == teamSettings_.DefaultValue.filter.min);
                    ReplaceControl(tbxDefaultValue_FilterMipmap, cbxDefaultValue_FilterMipmap);
                    cbxDefaultValue_FilterMipmap.SelectedItem = cbxDefaultValue_FilterMipmap.Items.OfType<UIListControlItem>().FirstOrDefault(x => (nw.g3d.nw4f_3dif.filter_mipType)x.Data == teamSettings_.DefaultValue.filter.mip);
                    ReplaceControl(tbxDefaultValue_FilterMaxAniso, cbxDefaultValue_FilterMaxAniso);
                    cbxDefaultValue_FilterMaxAniso.SelectedItem = cbxDefaultValue_FilterMaxAniso.Items.OfType<UIListControlItem>().FirstOrDefault(x => (nw.g3d.nw4f_3dif.filter_max_anisoType)x.Data == teamSettings_.DefaultValue.filter.max_aniso);

                    ReplaceControl(tbxDefaultValue_WrapU, cbxDefaultValue_WrapU);
                    cbxDefaultValue_WrapU.SelectedItem = cbxDefaultValue_WrapU.Items.OfType<UIListControlItem>().FirstOrDefault(x => (nw.g3d.nw4f_3dif.wrap_uvwType)x.Data == teamSettings_.DefaultValue.wrap.u);
                    ReplaceControl(tbxDefaultValue_WrapV, cbxDefaultValue_WrapV);
                    cbxDefaultValue_WrapV.SelectedItem = cbxDefaultValue_WrapV.Items.OfType<UIListControlItem>().FirstOrDefault(x => (nw.g3d.nw4f_3dif.wrap_uvwType)x.Data == teamSettings_.DefaultValue.wrap.v);
                    ReplaceControl(tbxDefaultValue_WrapW, cbxDefaultValue_WrapW);
                    cbxDefaultValue_WrapW.SelectedItem = cbxDefaultValue_WrapW.Items.OfType<UIListControlItem>().FirstOrDefault(x => (nw.g3d.nw4f_3dif.wrap_uvwType)x.Data == teamSettings_.DefaultValue.wrap.w);

                    tbxDefaultValue_LodMin.ReadOnly = false;
                    tbxDefaultValue_LodMax.ReadOnly = false;
                    tbxDefaultValue_LodBias.ReadOnly = false;

                    ReplaceControl(tbxDefaultValue_Clamp, cbxDefaultValue_Clamp);
                    cbxDefaultValue_Clamp.SelectedItem = cbxDefaultValue_Clamp.Items.OfType<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.DefaultValue.UIRange.Clamp);
                }
                #endregion DefaultValue

                #region Error
                using (var cb = new CustomBlock(() => { pnlError.SuspendLayout(); }, () => { pnlError.ResumeLayout(); }))
                {
                    tbxError_To.ReadOnly = false;
                    tbxError_Cc.ReadOnly = false;
                    tbxError_Bcc.ReadOnly = false;
                    tbxError_Subject.ReadOnly = false;
                }
                #endregion Error

                #region Preview
                using (var cb = new CustomBlock(() => { pnlPreview.SuspendLayout(); }, () => { pnlPreview.ResumeLayout(); }))
                {
                    tbxPreview_Fps.ReadOnly = false;
                    tbxMessageInterval.ReadOnly = false;
                }
                #endregion Preview

                #region Color
                using (var cb = new CustomBlock(() => { pnlColor.SuspendLayout(); }, () => { pnlColor.ResumeLayout(); }))
                {
                    ReplaceControl(tbxColor_GammaCorrection, cbxColor_GammaCorrection);
                    cbxColor_GammaCorrection.SelectedItem = cbxColor_GammaCorrection.Items.OfType<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.Color.GammaCorrection);
                }
                #endregion Color

                #region Platform
                using (var cb = new CustomBlock(() => { pnlPlatform.SuspendLayout(); }, () => { pnlPlatform.ResumeLayout(); }))
                {
                    tbxPcPeerType.ReadOnly = false;
                    tbxPcShaderArgs.ReadOnly = false;
                    tbxPcShaderArgsExpanded.ReadOnly = true;
                    tbxPcShaderForOptimizedShaderArgs.ReadOnly = false;
                    tbxPcShaderArgsForOptimizedShaderExpanded.ReadOnly = true;
                    tbxPcTextureArgs.ReadOnly = false;
                    tbxPcTextureArgsExpanded.ReadOnly = true;

                    tbxDevDeviceType.ReadOnly = false;
                    tbxDevPeerType.ReadOnly = false;
                    tbxDevShaderArgs.ReadOnly = false;
                    tbxDevShaderArgsExpanded.ReadOnly = true;
                    tbxDevShaderArgsForOptimizedShader.ReadOnly = false;
                    tbxDevShaderArgsForOptimizedShaderExpanded.ReadOnly = true;
                    tbxDevTextureArgs.ReadOnly = false;
                    tbxDevTextureArgsExpanded.ReadOnly = true;
                }
                #endregion Platform

                #region Hidden
                using (var cb = new CustomBlock(() => { pnlHidden.SuspendLayout(); }, () => { pnlHidden.ResumeLayout(); }))
                {
                    tbxBuildFsvScriptFilePath.ReadOnly = false;
                    btnBuildFsvScriptFilePath.Enabled = true;
                    tbxBuildFsvTemplateFilePath.ReadOnly = false;
                    btnBuildFsvTemplateFilePath.Enabled = true;

                    Action updateShaderPatameterFilter= () =>
                    {
                        teamSettings_.Preset.ShaderParameterFilters.Clear();
                        teamSettings_.Preset.ShaderParameterFilters.AddRange(lvwShaderParameterFilter.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.ShaderParameterFilter)x.Tag));

                        isPresetModified_ = null;
                        UpdateOkButtonText();
                    };
                    AddListViewEditor(
                        pnlPreset,
                        lvwShaderParameterFilter,
                        delegate (object ss, EventArgs ee)
                        {
                            updateShaderPatameterFilter();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            updateShaderPatameterFilter();
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwShaderParameterFilter.SelectedItems.Count == 1)
                            {
                                var item = lvwShaderParameterFilter.SelectedItems[0];
                                using (var dialog = new PresetShaderParameterFilterDialog()
                                {
                                    Tag = new PresetShaderParameterFilterDialog.TagData((TeamConfig.Preset.ShaderParameterFilter)item.Tag)
                                })
                                {
                                    if (dialog.ShowDialog(this) == DialogResult.OK)
                                    {
                                        var data = (PresetShaderParameterFilterDialog.TagData)dialog.Tag;
                                        item.SubItems[clhShaderParameterFilter_Name.Index].Text = data.ShaderParameterFilter.Name;
                                        item.SubItems[clhShaderParameterFilter_Include.Index].Text = data.ShaderParameterFilter.Include;
                                        item.SubItems[clhShaderParameterFilter_Exclude.Index].Text = data.ShaderParameterFilter.Exclude;
                                        item.SubItems[clhShaderParameterFilter_IgnorePage.Index].Text = UIText.FlagYesNo(data.ShaderParameterFilter.IgnorePage);

                                        updateShaderPatameterFilter();
                                    }
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            using (var dialog = new PresetShaderParameterFilterDialog()
                            {
                                Tag = new PresetShaderParameterFilterDialog.TagData(new TeamConfig.Preset.ShaderParameterFilter())
                            })
                            {
                                if (dialog.ShowDialog(this) == DialogResult.OK)
                                {
                                    var n = lvwShaderParameterFilter.Items.Count;
                                    lvwShaderParameterFilter.SetItemCount(n + 1);
                                    var item = lvwShaderParameterFilter.Items[n];

                                    var data = (PresetShaderParameterFilterDialog.TagData)dialog.Tag;
                                    item.SubItems[clhShaderParameterFilter_Name.Index].Text = data.ShaderParameterFilter.Name;
                                    item.SubItems[clhShaderParameterFilter_Include.Index].Text = data.ShaderParameterFilter.Include;
                                    item.SubItems[clhShaderParameterFilter_Exclude.Index].Text = data.ShaderParameterFilter.Exclude;
                                    item.SubItems[clhShaderParameterFilter_IgnorePage.Index].Text = UIText.FlagYesNo(data.ShaderParameterFilter.IgnorePage);

                                    item.Tag = data.ShaderParameterFilter;
                                    teamSettings_.Preset.ShaderParameterFilters.Add(data.ShaderParameterFilter);

                                    updateShaderPatameterFilter();
                                }
                            }
                        },
                        delegate (object ss, EventArgs ee)
                        {
                            if (lvwShaderParameterFilter.SelectedItems.Count == 1)
                            {
                                var item = lvwShaderParameterFilter.SelectedItems[0];
                                lvwShaderParameterFilter.Items.Remove(item);
                                teamSettings_.Preset.ShaderParameterFilters.Remove((TeamConfig.Preset.ShaderParameterFilter)item.Tag);

                                updateShaderPatameterFilter();
                            }
                        });

                    ReplaceControl(tbxDisableAnimationQuantize, cbxDisableAnimationQuantize);
                    cbxDisableAnimationQuantize.SelectedItem = cbxDisableAnimationQuantize.Items.Cast<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.DefaultValue.DisableAnimationQuantize);
                    ReplaceControl(tbxRenderStateInfoVisible, cbxRenderStateInfoVisible);
                    cbxRenderStateInfoVisible.SelectedItem = cbxRenderStateInfoVisible.Items.Cast<UIListControlItem>().FirstOrDefault(x => (bool)x.Data == teamSettings_.DefaultValue.RenderStateInfoVisible);
                }
                #endregion Hidden
            }

            editMode_ = true;
            UpdateOkButtonText();
        }

        private bool? isFileIoModified_;
        private bool IsFileIoModified
        {
            get
            {
                if (!isFileIoModified_.HasValue)
                {
                    // XML 化して比較する。
                    // プロパティへのアクセス時にシリアライズ対象外の要素が更新されるものがあり、インスタンスのバイナリデータでは正しく比較できないため。
                    var src = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(srcTeamSettings_.FileIo));
                    var cur = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(teamSettings_.FileIo));
                    isFileIoModified_ = !cur.SequenceEqual(src);
                }
                return isFileIoModified_.Value;
            }
        }

        private bool? isPresetModified_;
        private bool IsPresetModified
        {
            get
            {
                if (!isPresetModified_.HasValue)
                {
                    // XML 化して比較する。
                    // プロパティへのアクセス時にシリアライズ対象外の要素が更新されるものがあり、インスタンスのバイナリデータでは正しく比較できないため。
                    var src = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(srcTeamSettings_.Preset));
                    var cur = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(teamSettings_.Preset));
                    isPresetModified_ = !cur.SequenceEqual(src);
                }
                return isPresetModified_.Value;
            }
        }

        private bool? isDefaultValueModified_;
        private bool IsDefaultValueModified
        {
            get
            {
                if (!isDefaultValueModified_.HasValue)
                {
                    // XML 化して比較する。
                    // プロパティへのアクセス時にシリアライズ対象外の要素が更新されるものがあり、インスタンスのバイナリデータでは正しく比較できないため。
                    var src = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(srcTeamSettings_.DefaultValue));
                    var cur = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(teamSettings_.DefaultValue));
                    isDefaultValueModified_ = !cur.SequenceEqual(src);
                }
                return isDefaultValueModified_.Value;
            }
        }

        private bool? isErrorModified_;
        private bool IsErrorModified
        {
            get
            {
                if (!isErrorModified_.HasValue)
                {
                    // XML 化して比較する。
                    // プロパティへのアクセス時にシリアライズ対象外の要素が更新されるものがあり、インスタンスのバイナリデータでは正しく比較できないため。
                    var src = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(srcTeamSettings_.Error));
                    var cur = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(teamSettings_.Error));
                    isErrorModified_ = !cur.SequenceEqual(src);
                }
                return isErrorModified_.Value;
            }
        }

        private bool? isPreviewModified_;
        private bool IsPreviewModified
        {
            get
            {
                if (!isPreviewModified_.HasValue)
                {
                    // XML 化して比較する。
                    // プロパティへのアクセス時にシリアライズ対象外の要素が更新されるものがあり、インスタンスのバイナリデータでは正しく比較できないため。
                    var src = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(srcTeamSettings_.Preview));
                    var cur = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(teamSettings_.Preview));
                    isPreviewModified_ = !cur.SequenceEqual(src);
                }
                return isPreviewModified_.Value;
            }
        }

        private bool? isColorModified_;
        private bool IsColorModified
        {
            get
            {
                if (!isColorModified_.HasValue)
                {
                    // XML 化して比較する。
                    // プロパティへのアクセス時にシリアライズ対象外の要素が更新されるものがあり、インスタンスのバイナリデータでは正しく比較できないため。
                    var src = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(srcTeamSettings_.Color));
                    var cur = ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(teamSettings_.Color));
                    isColorModified_ = !cur.SequenceEqual(src);
                }
                return isColorModified_.Value;
            }
        }

        private bool? isPlatformPresetModified_;
        private bool IsPlatformPresetModified
        {
            get
            {
                if (!isPlatformPresetModified_.HasValue)
                {
                    if (platformPresets_.Count != srcPlatformPresets_.Count)
                    {
                        isPlatformPresetModified_ = true;
                    }
                    else
                    {
                        // XML 化して比較する。
                        // プロパティへのアクセス時にシリアライズ対象外の要素が更新されるものがあり、インスタンスのバイナリデータでは正しく比較できないため。
                        var src = srcPlatformPresets_.Select(x => ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(x))).ToArray();
                        var cur = platformPresets_.Select(x => ObjectUtility.ToBytes(ObjectUtility.CloneWithXmlSerializer(x))).ToArray();
                        isPlatformPresetModified_ = !Enumerable.Range(0, cur.Length).All(x => cur[x].SequenceEqual(src[x]));
                    }
                }
                return isPlatformPresetModified_.Value;
            }
        }

        private bool IsModified
        {
            get
            {
                return
                    IsFileIoModified ||
                    IsPresetModified ||
                    IsDefaultValueModified ||
                    IsErrorModified ||
                    IsPreviewModified ||
                    IsColorModified ||
                    IsPlatformPresetModified;
            }
        }

        private bool NeedsReboot
        {
            get
            {
                // TeamConfigPreview.Fps と TeamConfigPreview.MessageInterval 以外が変更された場合は再起動が必要。
                // 現状の Preview には上記以外が含まれないので、ここでの評価式に IsPreviewModified を含めていない。
                return
                    IsFileIoModified ||
                    IsPresetModified ||
                    IsDefaultValueModified ||
                    IsErrorModified ||
                    IsColorModified ||
                    IsPlatformPresetModified;
            }
        }

        private void UpdateOkButtonText()
        {
            if (editMode_)
            {
                btnOK.Text =
                    NeedsReboot ? Strings.TeamConfigSettingDialog_SaveAndReboot :
                    IsModified ? Strings.TeamConfigSettingDialog_ApplyAndSave :
                    Strings.TeamConfigSettingDialog_NoChange;
            }
            else
            {
                btnOK.Text = Strings.TeamConfigSettingDialog_ChangeToEditableMode;
            }
        }

        private void lbxMenu_SelectedValueChanged(object sender, EventArgs e)
        {
            foreach (var panel in configPanels_.Where(x => x != lbxMenu.SelectedItemData))
            {
                panel.Parent = null;
            }

            (lbxMenu.SelectedItemData as UIPanel).Parent = pnlConfigPanelBase;
        }

        private void pnlFileIo_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlFileIo.Tag == null) && (pnlFileIo.Parent != null))
            {
                var fileIo = teamSettings_.FileIo;
                using(var ub = new UpdateBlock(lvwFileIoSearchPath))
                {
                    lvwFileIoSearchPath.SetItemCount(fileIo.SearchPaths.Count);

                    int index = 0;
                    foreach(var searchPath in fileIo.SearchPaths)
                    {
                        var item = lvwFileIoSearchPath.Items[index];

                        item.Tag = searchPath;

                        item.SubItems[clhlvwFileIoSearchPath_Name.Index].Text		= searchPath.Name;
                        item.SubItems[clhlvwFileIoSearchPath_Recursive.Index].Text	= UIText.FlagYesNo(searchPath.Recursive);
                        item.SubItems[clhlvwFileIoSearchPath_Path.Index].Text		= searchPath.pathXml;
                        item.SubItems[clhlvwFileIoSearchPath_FullPath.Index].Text   = searchPath.path;

                        ++index;
                    }
                }

                using(var ub = new UpdateBlock(lvwFileIoStartUpPath))
                {
                    lvwFileIoStartUpPath.SetItemCount(fileIo.StartUpPaths.Count);

                    int index = 0;
                    foreach(var startUpPath in fileIo.StartUpPaths)
                    {
                        var item = lvwFileIoStartUpPath.Items[index];

                        item.Tag = startUpPath;

                        item.SubItems[clhlvwFileIoStartUpPath_Base.Index].Text		= startUpPath.Base;
                        item.SubItems[clhlvwFileIoStartUpPath_Path.Index].Text		= startUpPath.pathXml;
                        item.SubItems[clhlvwFileIoStartUpPath_FullPath.Index].Text	= startUpPath.FullPath(teamSettings_.FileIo.SearchPaths);

                        ++index;
                    }
                }

                using(var ub = new UpdateBlock(lvwAttachPath))
                {
                    lvwAttachPath.SetItemCount(fileIo.AttachPaths.Count);

                    int index = 0;
                    foreach(var attachPath in fileIo.AttachPaths)
                    {
                        var item = lvwAttachPath.Items[index];

                        item.Tag = attachPath;

                        item.SubItems[clhAttachPath_Filter.Index].Text		= attachPath.Filter;
                        item.SubItems[clhAttachPath_Path.Index].Text		= attachPath.pathXml;
                        item.SubItems[clhAttachPath_FullPath.Index].Text	= attachPath.FullPath();

                        ++index;
                    }
                }

                using (var ub = new UpdateBlock(lvwParentMaterialPath))
                {
                    lvwParentMaterialPath.SetItemCount(fileIo.ParentMaterialPaths.Count);

                    int index = 0;
                    foreach (var parentMaterialPath in fileIo.ParentMaterialPaths)
                    {
                        var item = lvwParentMaterialPath.Items[index];

                        item.Tag = parentMaterialPath;

                        item.SubItems[clhlvwFileIoParentPath_Name.Index].Text = parentMaterialPath.Name ?? string.Empty;
                        item.SubItems[clhlvwFileIoParentPath_Recursive.Index].Text = UIText.FlagYesNo(parentMaterialPath.Recursive);
                        item.SubItems[clhlvwFileIoParentPath_Path.Index].Text = parentMaterialPath.pathXml;
                        item.SubItems[clhlvwFileIoParentPath_FullPath.Index].Text = parentMaterialPath.path;

                        ++index;
                    }
                }


                tbxPreSaveCommand.Text = fileIo.PreSaveCommand;

                tbxPostSaveCommand.Text = fileIo.PostSaveCommand;

                tbxPreOpenCommand.Text = fileIo.PreOpenCommand;

                tbxPostCloseCommand.Text = fileIo.PostCloseCommand;

                tbxPreBinarizeCommand.Text = fileIo.PreBinarizeCommand.PathXml ?? "";
                tbxPreBinarizeCommandFilter.Text = fileIo.PreBinarizeCommand.Filter ?? "";

                tbxShaderConverterAdditionalArgs.Text = fileIo.ShaderConverterAdditionalArgs;
                tbxShaderConverterAdditionalArgsExpanded.Text = Environment.ExpandEnvironmentVariables(fileIo.ShaderConverterAdditionalArgs ?? string.Empty);
                tbxShaderConverterAdditionalArgsForOptimizedShader.Text = fileIo.ShaderConverterAdditionalArgsForOptimizedShader;
                tbxShaderConverterAdditionalArgsForOptimizedShaderExpanded.Text = Environment.ExpandEnvironmentVariables(fileIo.ShaderConverterAdditionalArgsForOptimizedShader ?? string.Empty);

                tbxShaderConvertLogTime.Text = UIText.FlagYesNo(fileIo.LogShaderConvertTime);

                tbxCreateShaderVariationCommand.Text = fileIo.CreateShaderVariationCommand.HasCommand ?
                    fileIo.CreateShaderVariationCommand.PathXml.Replace("\n", "\r\n") : string.Empty;
                tbxCreateShaderVariationCommandAllowNoOutput.Text = UIText.FlagYesNo(fileIo.CreateShaderVariationCommand.AllowNoOutput);

                using (var ub = new UpdateBlock(lvwUserCommand))
                {
                    lvwUserCommand.SetItemCount(teamSettings_.FileIo.UserCommands.Count);

                    var index = 0;
                    foreach (var command in teamSettings_.FileIo.UserCommands)
                    {
                        var item = lvwUserCommand.Items[index];

                        item.Tag = command;

                        item.SubItems[clhUserCommand_Name.Index].Text = command.Name;
                        item.SubItems[clhUserCommand_Filter.Index].Text = command.Filter;
                        item.SubItems[clhUserCommand_Args.Index].Text = command.Args;
                        item.SubItems[clhUserCommand_Reload.Index].Text = UIText.FlagYesNo(command.Reload);
                        item.SubItems[clhUserCommand_ForceUnloadAndLoad.Index].Text =
                            UIText.FlagYesNo(command.ForceUnloadAndLoad);
                        item.SubItems[clhUserCommand_Temporary.Index].Text = UIText.FlagYesNo(command.Temporary);
                        item.SubItems[clhUserCommand_ShortCut.Index].Text = command.ShortCut;
                        item.SubItems[clhUserCommand_FileTreeShortCut.Index].Text =
                            UIText.FlagYesNo(command.FileTreeShortCut);
                        item.SubItems[clhUserCommand_Path.Index].Text = command.PathXml;

                        ++index;
                    }

                    lvwUserCommand.AutoResizeColumnsByHeaderAndContents();
                }

                using (var ub = new UpdateBlock(lvwRuntimeUserScript))
                {
                    lvwRuntimeUserScript.SetItemCount(teamSettings_.FileIo.RuntimeUserScripts.Count);

                    var index = 0;
                    foreach (var script in teamSettings_.FileIo.RuntimeUserScripts)
                    {
                        var item = lvwRuntimeUserScript.Items[index];

                        item.Tag = script;

                        item.SubItems[clhRuntimeUserScript_Name.Index].Text = script.Name;
                        item.SubItems[clhRuntimeUserScript_Path.Index].Text = script.PathXml;

                        ++index;
                    }

                    lvwRuntimeUserScript.AutoResizeColumnsByHeaderAndContents();
                }

                pnlFileIo.Tag = true;
            }
        }

        private void pnlPreset_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlPreset.Tag == null) && (pnlPreset.Parent != null))
            {
                tbxSettingName.Text = teamSettings_.Preset.SettingName;
                tbxSettingNameExpanded.Text = teamSettings_.Preset.FullSettingName;
                tbxEnableMaterialAnimCreation.Text = UIText.FlagYesNo(teamSettings_.Preset.EnableMaterialAnimCreation);
                tbxSeparateMaterialAnimCreationMenu.Text = UIText.FlagYesNo(teamSettings_.Preset.SeparateMaterialAnimCreationMenu);
                tbxShowOnlyBinarizeEnabledBoneVisAnimCurve.Text = UIText.FlagYesNo(teamSettings_.Preset.ShowOnlyBinarizeEnabledBoneVisibilityAnimCurve);

                using(var ub = new UpdateBlock(lvwPresetTexture))
                {
                    lvwPresetTexture.SetItemCount(teamSettings_.Preset.TexturePresets.Count);

                    int index = 0;
                    foreach(var texture in teamSettings_.Preset.TexturePresets)
                    {
                        var item = lvwPresetTexture.Items[index];

                        item.Tag = texture;

                        item.SubItems[clhPresetTexture_TextureName.Index].Text		= texture.Name;
                        item.SubItems[clhPresetTexture_TextureFormat.Index].Text	= UIText.EnumValue(texture.quantize_type);

                        ++ index;
                    }
                }

                tbxFollowDccSamplerNameRule.Text = UIText.FlagYesNo(teamSettings_.Preset.FollowDccSamplerNameRule);
                tbxDisableTextureSrtWithOriginal.Text = UIText.FlagYesNo(teamSettings_.Preset.DisableEditTextureSrtWithOriginal);

                using (new UpdateBlock(lvwPresetSamplerName))
                {
                    lvwPresetSamplerName.SetItemCount(teamSettings_.Preset.SamplerNamePresets.Count);

                    int index = 0;
                    foreach (var samplerName in teamSettings_.Preset.SamplerNamePresets)
                    {
                        var item = lvwPresetSamplerName.Items[index];

                        item.Tag = samplerName;

                        item.SubItems[clhPresetSamplerName_Name.Index].Text = samplerName.Name;

                        index++;
                    }
                }

                using (new UpdateBlock(lvwPresetVertexAttributeAssign))
                {
                    lvwPresetVertexAttributeAssign.SetItemCount(teamSettings_.Preset.VertexAttributeAssignPresets.Count);

                    int index = 0;
                    foreach (var vertexAttributeAssign in teamSettings_.Preset.VertexAttributeAssignPresets)
                    {
                        var item = lvwPresetVertexAttributeAssign.Items[index];

                        item.Tag = vertexAttributeAssign;

                        item.SubItems[clhPresetVertexAttributeAssign_Name.Index].Text = vertexAttributeAssign.Name;

                        index++;
                    }
                }

                using (var ub = new UpdateBlock(lvwPresetLight))
                {
                    lvwPresetLight.SetItemCount(teamSettings_.Preset.LightAnimPresets.Count);

                    int index = 0;
                    foreach (var light in teamSettings_.Preset.LightAnimPresets)
                    {
                        var item = lvwPresetLight.Items[index];

                        item.Tag = light;

                        item.SubItems[clhPresetLight_LightName.Index].Text = light.Label;
                        item.SubItems[clhPresetLight_LightType.Index].Text = light.Type;
                        item.SubItems[clhPresetLight_LightTargets.Index].Text
                            = string.Join(", ",
                                          light.LightAnimTargets.Select(
                                              x =>
                                              x.Target.ToString() + (x.HasDefault() ? "(" + x.DefaultValue + ")" : "")));

                        ++index;
                    }
                    lvwPresetLight.AutoResizeColumn(
                        clhPresetLight_LightTargets.Index,
                        ColumnHeaderAutoResizeStyle.ColumnContent);
                }

                using (var ub = new UpdateBlock(lvwPresetAttn))
                {
                    lvwPresetAttn.SetItemCount(
                          teamSettings_.Preset.LightDistAttnPresets.Count
                        + teamSettings_.Preset.LightAngAttnPresets.Count
                        + teamSettings_.Preset.FogDistAttnPresets.Count);

                    int index = 0;
                    foreach (var attn in teamSettings_.Preset.LightDistAttnPresets)
                    {
                        var item = lvwPresetAttn.Items[index];

                        item.Tag = attn;

                        item.SubItems[clhPresetAttn_AttnKind.Index].Text = Strings.TeamConfigSettingDialog_pnlPreset_LightDistAttnFunc;
                        item.SubItems[clhPresetAttn_AttnName.Index].Text = attn.Label;
                        item.SubItems[clhPresetAttn_AttnType.Index].Text = attn.Function;

                        ++index;
                    }

                    foreach (var attn in teamSettings_.Preset.LightAngAttnPresets)
                    {
                        var item = lvwPresetAttn.Items[index];

                        item.Tag = attn;

                        item.SubItems[clhPresetAttn_AttnKind.Index].Text = Strings.TeamConfigSettingDialog_pnlPreset_LightAngleAttnFunc;
                        item.SubItems[clhPresetAttn_AttnName.Index].Text = attn.Label;
                        item.SubItems[clhPresetAttn_AttnType.Index].Text = attn.Function;

                        ++index;
                    }

                    foreach (var attn in teamSettings_.Preset.FogDistAttnPresets)
                    {
                        var item = lvwPresetAttn.Items[index];

                        item.Tag = attn;

                        item.SubItems[clhPresetAttn_AttnKind.Index].Text = Strings.TeamConfigSettingDialog_pnlPreset_FogDistAttnFunc;
                        item.SubItems[clhPresetAttn_AttnName.Index].Text = attn.Label;
                        item.SubItems[clhPresetAttn_AttnType.Index].Text = attn.Function;

                        ++index;
                    }

                    //lvwPresetAttn.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent);
                }

                pnlPreset.Tag = true;
            }
        }

        private void pnlDefaultValue_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlDefaultValue.Tag == null) && (pnlDefaultValue.Parent != null))
            {
                tbxOptimizeShaderAfterMaterialChanged.Text = UIText.FlagYesNo(teamSettings_.DefaultValue.OptimizeShaderAfterMaterialChanged);

                tbxDefaultValue_FilterMag.Text = UIText.EnumValue(teamSettings_.DefaultValue.filter.mag);
                tbxDefaultValue_FilterMin.Text = UIText.EnumValue(teamSettings_.DefaultValue.filter.min);
                tbxDefaultValue_FilterMipmap.Text = UIText.EnumValue(teamSettings_.DefaultValue.filter.mip);
                tbxDefaultValue_FilterMaxAniso.Text = UIText.EnumValue(teamSettings_.DefaultValue.filter.max_aniso);

                tbxDefaultValue_WrapU.Text = UIText.EnumValue(teamSettings_.DefaultValue.wrap.u);
                tbxDefaultValue_WrapV.Text = UIText.EnumValue(teamSettings_.DefaultValue.wrap.v);
                tbxDefaultValue_WrapW.Text = UIText.EnumValue(teamSettings_.DefaultValue.wrap.w);

                tbxDefaultValue_LodMin.Value = teamSettings_.DefaultValue.lod.min;
                tbxDefaultValue_LodMax.Value = teamSettings_.DefaultValue.lod.max;
                tbxDefaultValue_LodBias.Value = teamSettings_.DefaultValue.lod.bias;


                tbxDefaultValue_Clamp.Text = UIText.FlagEnableDisable(teamSettings_.DefaultValue.UIRange.Clamp);

                pnlDefaultValue.Tag = true;
            }
        }

        private void pnlError_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlError.Tag == null) && (pnlError.Parent != null))
            {
                if (teamSettings_.Error.Mail != null)
                {
                    tbxError_To.Text		= teamSettings_.Error.Mail.to;
                    tbxError_Cc.Text		= teamSettings_.Error.Mail.cc;
                    tbxError_Bcc.Text		= teamSettings_.Error.Mail.bcc;
                    tbxError_Subject.Text	= teamSettings_.Error.Mail.subject;
                }

                pnlError.Tag = true;
            }
        }

        private void pnlPreview_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlPreview.Tag == null) && (pnlPreview.Parent != null))
            {
                tbxPreview_Fps.Value = teamSettings_.Preview.Fps;
                tbxMessageInterval.Value = teamSettings_.Preview.MessageInterval;

                pnlPreview.Tag = true;
            }
        }

        private void pnlColor_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlColor.Tag == null) && (pnlColor.Parent != null))
            {
                tbxColor_GammaCorrection.Text	= UIText.FlagYesNo(teamSettings_.Color.GammaCorrection);

                pnlColor.Tag = true;
            }
        }

        private void pnlPlatform_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlPlatform.Tag == null) && (pnlPlatform.Parent != null))
            {
                grpPcOption.Enabled = false;
                grpDevOption.Enabled = false;

                cbxPlatform.SelectedItem = cbxPlatform.Items.Cast<UIListControlItem>().FirstOrDefault(x => x.Data == null) ?? cbxPlatform.Items.Cast<UIListControlItem>().FirstOrDefault();

                pnlPlatform.Tag = true;
            }
        }

        private void pnlHidden_ParentChanged(object sender, EventArgs e)
        {
            if ((pnlHidden.Tag == null) && (pnlHidden.Parent != null))
            {
                tbxBuildFsvScriptFilePath.Text = teamSettings_.FileIo.BuildFsv.ScriptFilePath;
                tbxBuildFsvTemplateFilePath.Text = teamSettings_.FileIo.BuildFsv.TemplateFilePath;

                using (var ub = new UpdateBlock(lvwShaderParameterFilter))
                {
                    lvwShaderParameterFilter.SetItemCount(
                        teamSettings_.Preset.ShaderParameterFilters.Count);

                    var index = 0;
                    foreach (var filter in teamSettings_.Preset.ShaderParameterFilters)
                    {
                        var item = lvwShaderParameterFilter.Items[index];

                        item.Tag = filter;

                        item.SubItems[clhShaderParameterFilter_Name.Index].Text = filter.Name;
                        item.SubItems[clhShaderParameterFilter_Include.Index].Text = filter.Include;
                        item.SubItems[clhShaderParameterFilter_Exclude.Index].Text = filter.Exclude;
                        item.SubItems[clhShaderParameterFilter_IgnorePage.Index].Text = UIText.FlagYesNo(filter.IgnorePage);

                        ++index;
                    }

                    lvwShaderParameterFilter.AutoResizeColumnsByHeaderAndContents();
                }

                tbxDisableAnimationQuantize.Text = UIText.FlagYesNo(teamSettings_.DefaultValue.DisableAnimationQuantize);
                tbxRenderStateInfoVisible.Text = UIText.FlagYesNo(teamSettings_.DefaultValue.RenderStateInfoVisible);

                pnlHidden.Tag = true;
            }
        }

        #region FileIo EventHandler
        private void lvwFileIoSearchPath_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            if (lvwFileIoSearchPath.SortOnColumnClick)
            {
                teamSettings_.FileIo.SearchPaths.Clear();
                teamSettings_.FileIo.SearchPaths.AddRange(lvwFileIoSearchPath.Items.Cast<ListViewItem>().Select(x => (TeamConfig.FileIo.SearchPath)x.Tag).ToArray());

                isFileIoModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void lvwFileIoStartUpPath_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            if (lvwFileIoStartUpPath.SortOnColumnClick)
            {
                teamSettings_.FileIo.StartUpPaths.Clear();
                teamSettings_.FileIo.StartUpPaths.AddRange(lvwFileIoStartUpPath.Items.Cast<ListViewItem>().Select(x => (ConfigCommon.StartUp)x.Tag).ToArray());

                isFileIoModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void lvwAttachPath_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            if (lvwAttachPath.SortOnColumnClick)
            {
                teamSettings_.FileIo.AttachPaths.Clear();
                teamSettings_.FileIo.AttachPaths.AddRange(lvwAttachPath.Items.Cast<ListViewItem>().Select(x => (TeamConfig.FileIo.AttachPath)x.Tag).ToArray());

                isFileIoModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void lvwParentMaterialPath_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            if (lvwParentMaterialPath.SortOnColumnClick)
            {
                teamSettings_.FileIo.ParentMaterialPaths.Clear();
                teamSettings_.FileIo.ParentMaterialPaths.AddRange(lvwParentMaterialPath.Items.Cast<ListViewItem>().Select(x => (TeamConfig.FileIo.ParentMaterialPath)x.Tag).ToArray());

                isFileIoModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxPreSaveCommand_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.PreSaveCommand = tbxPreSaveCommand.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.PreSaveCommand) &&
                (srcTeamSettings_.FileIo.PreSaveCommand == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.PreSaveCommand = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxPostSaveCommand_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.PostSaveCommand = tbxPostSaveCommand.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.PostSaveCommand) &&
                (srcTeamSettings_.FileIo.PostSaveCommand == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.PostSaveCommand = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxPreOpenCommand_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.PreOpenCommand = tbxPreOpenCommand.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.PreOpenCommand) &&
                (srcTeamSettings_.FileIo.PreOpenCommand == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.PreOpenCommand = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxPostCloseCommand_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.PostCloseCommand = tbxPostCloseCommand.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.PostCloseCommand) &&
                (srcTeamSettings_.FileIo.PostCloseCommand == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.PostCloseCommand = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxPreBinarizeCommand_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.PreBinarizeCommand.PathXml = tbxPreBinarizeCommand.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.PreBinarizeCommand.PathXml) &&
                (srcTeamSettings_.FileIo.PreBinarizeCommand.PathXml == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.PreBinarizeCommand.PathXml = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxPreBinarizeCommandFilter_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.PreBinarizeCommand.Filter = tbxPreBinarizeCommandFilter.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.PreBinarizeCommand.Filter) &&
                (srcTeamSettings_.FileIo.PreBinarizeCommand.Filter == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.PreBinarizeCommand.Filter = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxShaderConverterAdditionalArgs_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.ShaderConverterAdditionalArgs = tbxShaderConverterAdditionalArgs.Text;
            tbxShaderConverterAdditionalArgsExpanded.Text = System.Environment.ExpandEnvironmentVariables(teamSettings_.FileIo.ShaderConverterAdditionalArgs);

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.ShaderConverterAdditionalArgs) &&
                (srcTeamSettings_.FileIo.ShaderConverterAdditionalArgs == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.ShaderConverterAdditionalArgs = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxShaderConverterAdditionalArgsForOptimizedShader_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.ShaderConverterAdditionalArgsForOptimizedShader = tbxShaderConverterAdditionalArgsForOptimizedShader.Text;
            tbxShaderConverterAdditionalArgsForOptimizedShaderExpanded.Text = System.Environment.ExpandEnvironmentVariables(teamSettings_.FileIo.ShaderConverterAdditionalArgsForOptimizedShader);

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.ShaderConverterAdditionalArgsForOptimizedShader) &&
                (srcTeamSettings_.FileIo.ShaderConverterAdditionalArgsForOptimizedShader == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.ShaderConverterAdditionalArgsForOptimizedShader = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void cbxShaderConvertLogTime_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxShaderConvertLogTime.SelectedItem != null)
            {
                teamSettings_.FileIo.LogShaderConvertTime = (bool)((UIListControlItem)cbxShaderConvertLogTime.SelectedItem).Data;

                isFileIoModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxCreateShaderVariationCommand_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.CreateShaderVariationCommand.PathXml = tbxCreateShaderVariationCommand.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.CreateShaderVariationCommand.PathXml) &&
                (srcTeamSettings_.FileIo.CreateShaderVariationCommand.PathXml == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.CreateShaderVariationCommand.PathXml = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void cbxCreateShaderVariationCommandAllowNoOutput_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxCreateShaderVariationCommandAllowNoOutput.SelectedItem != null)
            {
                teamSettings_.FileIo.CreateShaderVariationCommand.AllowNoOutput = (bool)((UIListControlItem)cbxCreateShaderVariationCommandAllowNoOutput.SelectedItem).Data;

                isFileIoModified_ = null;
                UpdateOkButtonText();
            }
        }
        #endregion FileIo EventHandler

        #region Preset EventHandler
        private void tbxSettingName_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.Preset.SettingName = tbxSettingName.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.Preset.SettingName) &&
                (srcTeamSettings_.Preset.SettingName == null);
            if (needsNullReset)
            {
                teamSettings_.Preset.SettingName = null;
            }

            tbxSettingNameExpanded.Text = teamSettings_.Preset.FullSettingName;

            isPresetModified_ = null;
            UpdateOkButtonText();
        }

        private void cbxEnableMaterialAnimCreation_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxEnableMaterialAnimCreation.SelectedItem != null)
            {
                teamSettings_.Preset.EnableMaterialAnimCreation = (bool)((UIListControlItem)cbxEnableMaterialAnimCreation.SelectedItem).Data;

                isPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxSeparateMaterialAnimCreationMenu_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxSeparateMaterialAnimCreationMenu.SelectedItem != null)
            {
                teamSettings_.Preset.SeparateMaterialAnimCreationMenu = (bool)((UIListControlItem)cbxSeparateMaterialAnimCreationMenu.SelectedItem).Data;

                isPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxShowOnlyBinarizeEnabledBoneVisAnimCurve_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxShowOnlyBinarizeEnabledBoneVisAnimCurve.SelectedItem != null)
            {
                teamSettings_.Preset.ShowOnlyBinarizeEnabledBoneVisibilityAnimCurve = (bool)((UIListControlItem)cbxShowOnlyBinarizeEnabledBoneVisAnimCurve.SelectedItem).Data;

                isPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void lvwPresetTexture_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            if (lvwPresetTexture.SortOnColumnClick)
            {
                teamSettings_.Preset.TexturePresets.Clear();
                teamSettings_.Preset.TexturePresets.AddRange(lvwPresetTexture.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.TexturePreset)x.Tag).ToArray());

                isPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxFollowDccSamplerNameRule_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxFollowDccSamplerNameRule.SelectedItem != null)
            {
                teamSettings_.Preset.FollowDccSamplerNameRule = (bool)((UIListControlItem)cbxFollowDccSamplerNameRule.SelectedItem).Data;

                isPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxDisableTextureSrtWithOriginal_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxDisableTextureSrtWithOriginal.SelectedItem != null)
            {
                teamSettings_.Preset.DisableEditTextureSrtWithOriginal = (bool)((UIListControlItem)cbxDisableTextureSrtWithOriginal.SelectedItem).Data;

                isPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void lvwPresetSamplerName_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            if (lvwPresetSamplerName.SortOnColumnClick)
            {
                teamSettings_.Preset.SamplerNamePresets.Clear();
                teamSettings_.Preset.SamplerNamePresets.AddRange(lvwPresetSamplerName.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.SamplerNamePreset)x.Tag).ToArray());

                isPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void lvwPresetVertexAttributeAssign_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            if (lvwPresetVertexAttributeAssign.SortOnColumnClick)
            {
                teamSettings_.Preset.VertexAttributeAssignPresets.Clear();
                teamSettings_.Preset.VertexAttributeAssignPresets.AddRange(lvwPresetVertexAttributeAssign.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.VertexAttributeAssignPreset)x.Tag).ToArray());

                isPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void lvwPresetLight_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            if (lvwPresetLight.SortOnColumnClick)
            {
                teamSettings_.Preset.LightAnimPresets.Clear();
                teamSettings_.Preset.LightAnimPresets.AddRange(lvwPresetLight.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.LightAnimPreset)x.Tag).ToArray());

                isPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void lvwPresetAttn_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            if (lvwPresetAttn.SortOnColumnClick)
            {
                teamSettings_.Preset.LightDistAttnPresets.Clear();
                teamSettings_.Preset.LightAngAttnPresets.Clear();
                teamSettings_.Preset.FogDistAttnPresets.Clear();
                foreach (var item in lvwPresetAttn.Items.Cast<ListViewItem>())
                {
                    GetAttenuationPresets(item.SubItems[clhPresetAttn_AttnKind.Index].Text).Add((TeamConfig.Preset.AttenuationFunction)item.Tag);
                }

                isPresetModified_ = null;
                UpdateOkButtonText();
            }
        }
        #endregion Preset EventHandler

        #region DefaultValue EventHandler
        private void cbxOptimizeShaderAfterMaterialChanged_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxOptimizeShaderAfterMaterialChanged.SelectedItem != null)
            {
                teamSettings_.DefaultValue.OptimizeShaderAfterMaterialChanged = (bool)((UIListControlItem)cbxOptimizeShaderAfterMaterialChanged.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxDefaultValue_FilterMag_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxDefaultValue_FilterMag.SelectedItem != null)
            {
                teamSettings_.DefaultValue.filter.mag = (nw.g3d.nw4f_3dif.filter_mag_minType)((UIListControlItem)cbxDefaultValue_FilterMag.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxDefaultValue_FilterMin_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxDefaultValue_FilterMin.SelectedItem != null)
            {
                teamSettings_.DefaultValue.filter.min = (nw.g3d.nw4f_3dif.filter_mag_minType)((UIListControlItem)cbxDefaultValue_FilterMin.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxDefaultValue_FilterMipmap_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxDefaultValue_FilterMipmap.SelectedItem != null)
            {
                teamSettings_.DefaultValue.filter.mip = (nw.g3d.nw4f_3dif.filter_mipType)((UIListControlItem)cbxDefaultValue_FilterMipmap.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxDefaultValue_FilterMaxAniso_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxDefaultValue_FilterMaxAniso.SelectedItem != null)
            {
                teamSettings_.DefaultValue.filter.max_aniso = (nw.g3d.nw4f_3dif.filter_max_anisoType)((UIListControlItem)cbxDefaultValue_FilterMaxAniso.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxDefaultValue_WrapU_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxDefaultValue_WrapU.SelectedItem != null)
            {
                teamSettings_.DefaultValue.wrap.u = (nw.g3d.nw4f_3dif.wrap_uvwType)((UIListControlItem)cbxDefaultValue_WrapU.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxDefaultValue_WrapV_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxDefaultValue_WrapV.SelectedItem != null)
            {
                teamSettings_.DefaultValue.wrap.v = (nw.g3d.nw4f_3dif.wrap_uvwType)((UIListControlItem)cbxDefaultValue_WrapV.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxDefaultValue_WrapW_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxDefaultValue_WrapW.SelectedItem != null)
            {
                teamSettings_.DefaultValue.wrap.w = (nw.g3d.nw4f_3dif.wrap_uvwType)((UIListControlItem)cbxDefaultValue_WrapW.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxDefaultValue_LodMin_ValueChanged(object sender, EventArgs e)
        {
            teamSettings_.DefaultValue.lod.min = tbxDefaultValue_LodMin.Value;

            isDefaultValueModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxDefaultValue_LodMin_TextChanged(object sender, EventArgs e)
        {
            float value;
            if (float.TryParse(tbxDefaultValue_LodMin.Text, out value))
            {
                teamSettings_.DefaultValue.lod.min = value;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxDefaultValue_LodMax_ValueChanged(object sender, EventArgs e)
        {
            teamSettings_.DefaultValue.lod.max = tbxDefaultValue_LodMax.Value;

            isDefaultValueModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxDefaultValue_LodMax_TextChanged(object sender, EventArgs e)
        {
            float value;
            if (float.TryParse(tbxDefaultValue_LodMax.Text, out value))
            {
                teamSettings_.DefaultValue.lod.max = value;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxDefaultValue_LodBias_ValueChanged(object sender, EventArgs e)
        {
            teamSettings_.DefaultValue.lod.bias = tbxDefaultValue_LodBias.Value;

            isDefaultValueModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxDefaultValue_LodBias_TextChanged(object sender, EventArgs e)
        {
            float value;
            if (float.TryParse(tbxDefaultValue_LodBias.Text, out value))
            {
                teamSettings_.DefaultValue.lod.bias = value;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxDefaultValue_Clamp_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxDefaultValue_Clamp.SelectedItem != null)
            {
                teamSettings_.DefaultValue.UIRange.Clamp = (bool)((UIListControlItem)cbxDefaultValue_Clamp.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }
        #endregion DefaultValue EventHandler

        #region Error EventHandler
        private void tbxError_To_TextChanged(object sender, EventArgs e)
        {
            if (teamSettings_.Error.Mail == null)
            {
                teamSettings_.Error.Mail = new TeamConfig.Mail();
            }
            teamSettings_.Error.Mail.to = tbxError_To.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            if (string.IsNullOrEmpty(teamSettings_.Error.Mail.to))
            {
                if (srcTeamSettings_.Error.Mail != null)
                {
                    bool needsNulllReset = (srcTeamSettings_.Error.Mail.to == null);
                    if (needsNulllReset)
                    {
                        teamSettings_.Error.Mail.to = null;
                    }
                }
                else
                {
                    bool needsNullReset =
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.to) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.cc) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.bcc) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.subject);
                    if (needsNullReset)
                    {
                        teamSettings_.Error.Mail = null;
                    }
                }
            }

            isErrorModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxError_Cc_TextChanged(object sender, EventArgs e)
        {
            if (teamSettings_.Error.Mail == null)
            {
                teamSettings_.Error.Mail = new TeamConfig.Mail();
            }
            teamSettings_.Error.Mail.cc = tbxError_Cc.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            if (string.IsNullOrEmpty(teamSettings_.Error.Mail.cc))
            {
                if (srcTeamSettings_.Error.Mail != null)
                {
                    bool needsNulllReset = (srcTeamSettings_.Error.Mail.cc == null);
                    if (needsNulllReset)
                    {
                        teamSettings_.Error.Mail.cc = null;
                    }
                }
                else
                {
                    bool needsNullReset =
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.to) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.cc) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.bcc) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.subject);
                    if (needsNullReset)
                    {
                        teamSettings_.Error.Mail = null;
                    }
                }
            }

            isErrorModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxError_Bcc_TextChanged(object sender, EventArgs e)
        {
            if (teamSettings_.Error.Mail == null)
            {
                teamSettings_.Error.Mail = new TeamConfig.Mail();
            }
            teamSettings_.Error.Mail.bcc = tbxError_Bcc.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            if (string.IsNullOrEmpty(teamSettings_.Error.Mail.bcc))
            {
                if (srcTeamSettings_.Error.Mail != null)
                {
                    bool needsNulllReset = (srcTeamSettings_.Error.Mail.bcc == null);
                    if (needsNulllReset)
                    {
                        teamSettings_.Error.Mail.bcc = null;
                    }
                }
                else
                {
                    bool needsNullReset =
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.to) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.cc) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.bcc) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.subject);
                    if (needsNullReset)
                    {
                        teamSettings_.Error.Mail = null;
                    }
                }
            }

            isErrorModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxError_Subject_TextChanged(object sender, EventArgs e)
        {
            if (teamSettings_.Error.Mail == null)
            {
                teamSettings_.Error.Mail = new TeamConfig.Mail();
            }
            teamSettings_.Error.Mail.subject = tbxError_Subject.Text;

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            if (string.IsNullOrEmpty(teamSettings_.Error.Mail.subject))
            {
                if (srcTeamSettings_.Error.Mail != null)
                {
                    bool needsNulllReset = (srcTeamSettings_.Error.Mail.subject == null);
                    if (needsNulllReset)
                    {
                        teamSettings_.Error.Mail.subject = null;
                    }
                }
                else
                {
                    bool needsNullReset =
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.to) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.cc) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.bcc) &&
                        string.IsNullOrEmpty(teamSettings_.Error.Mail.subject);
                    if (needsNullReset)
                    {
                        teamSettings_.Error.Mail = null;
                    }
                }
            }

            isErrorModified_ = null;
            UpdateOkButtonText();
        }
        #endregion Error EventHandler

        #region Preview EventHandler
        private void tbxPreview_Fps_ValueChanged(object sender, EventArgs e)
        {
            teamSettings_.Preview.Fps = tbxPreview_Fps.Value;

            isPreviewModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxPreview_Fps_TextChanged(object sender, EventArgs e)
        {
            double value;
            if (double.TryParse(tbxPreview_Fps.Text, out value))
            {
                teamSettings_.Preview.Fps = value;

                isPreviewModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxMessageInterval_ValueChanged(object sender, EventArgs e)
        {
            teamSettings_.Preview.MessageInterval = tbxMessageInterval.Value;

            isPreviewModified_ = null;
            UpdateOkButtonText();
        }

        private void tbxMessageInterval_TextChanged(object sender, EventArgs e)
        {
            int value;
            if (int.TryParse(tbxMessageInterval.Text, out value))
            {
                teamSettings_.Preview.MessageInterval = value;

                isPreviewModified_ = null;
                UpdateOkButtonText();
            }
        }
        #endregion Preview EventHandler

        #region Color EventHandler
        private void cbxColor_GammaCorrection_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxColor_GammaCorrection.SelectedItem != null)
            {
                teamSettings_.Color.GammaCorrection = (bool)((UIListControlItem)cbxColor_GammaCorrection.SelectedItem).Data;

                isColorModified_ = null;
                UpdateOkButtonText();
            }
        }
        #endregion Color EventHandler

        #region Platform EventHandler
        private void cbxPlatform_SelectedIndexChanged(object sender, EventArgs e)
        {
            var platformPreset = (TeamConfig.PlatformPreset)cbxPlatform.SelectedItemData;

            TeamConfig.PlatformPreset.DeviceOption pcOption = null, devOption = null;

            if (platformPreset != null)
            {
                pcOption = platformPreset.PcOption;
                devOption = platformPreset.DeviceOptions.Where(i => i != pcOption).FirstOrDefault();

                lblPlatformConfigPath.Enabled = true;
                tbxPlatformConfigPath.Enabled = true;
                tbxPlatformConfigPath.Text = platformPreset.FilePath;
            }
            else
            {
                lblPlatformConfigPath.Enabled = false;
                tbxPlatformConfigPath.Enabled = false;
                tbxPlatformConfigPath.Text = string.Empty;
            }

            if (pcOption != null)
            {
                grpPcOption.Enabled = true;

                tbxPcDeviceType.Text = pcOption.DeviceType;
                tbxPcPeerType.Text = pcOption.PeerType;
                tbxPcShaderArgs.Text = pcOption.ShaderConverterAdditionalArgs;
                tbxPcShaderArgsExpanded.Text = Environment.ExpandEnvironmentVariables(pcOption.ShaderConverterAdditionalArgs ?? string.Empty);
                tbxPcShaderForOptimizedShaderArgs.Text = pcOption.ShaderConverterAdditionalArgsForOptimizedShader;
                tbxPcShaderArgsForOptimizedShaderExpanded.Text = Environment.ExpandEnvironmentVariables(pcOption.ShaderConverterAdditionalArgsForOptimizedShader ?? string.Empty);
                tbxPcTextureArgs.Text = pcOption.TextureConverterAdditionalArgs;
                tbxPcTextureArgsExpanded.Text = Environment.ExpandEnvironmentVariables(pcOption.TextureConverterAdditionalArgs ?? string.Empty);
            }
            else
            {
                grpPcOption.Enabled = false;

                tbxPcDeviceType.Text = string.Empty;
                tbxPcPeerType.Text = string.Empty;
                tbxPcShaderArgs.Text = string.Empty;
                tbxPcShaderArgsExpanded.Text = string.Empty;
                tbxPcShaderForOptimizedShaderArgs.Text = string.Empty;
                tbxPcShaderArgsForOptimizedShaderExpanded.Text = string.Empty;
                tbxPcTextureArgs.Text = string.Empty;
                tbxPcTextureArgsExpanded.Text = string.Empty;
            }

            if (devOption != null)
            {
                grpDevOption.Enabled = true;

                tbxDevDeviceType.Text = devOption.DeviceType;
                tbxDevPeerType.Text = devOption.PeerType;
                tbxDevShaderArgs.Text = devOption.ShaderConverterAdditionalArgs;
                tbxDevShaderArgsExpanded.Text = Environment.ExpandEnvironmentVariables(devOption.ShaderConverterAdditionalArgs ?? string.Empty);
                tbxDevShaderArgsForOptimizedShader.Text = devOption.ShaderConverterAdditionalArgsForOptimizedShader;
                tbxDevShaderArgsForOptimizedShaderExpanded.Text = Environment.ExpandEnvironmentVariables(devOption.ShaderConverterAdditionalArgsForOptimizedShader ?? string.Empty);
                tbxDevTextureArgs.Text = devOption.TextureConverterAdditionalArgs;
                tbxDevTextureArgsExpanded.Text = Environment.ExpandEnvironmentVariables(devOption.TextureConverterAdditionalArgs ?? string.Empty);
            }
            else
            {
                grpDevOption.Enabled = false;

                tbxDevDeviceType.Text = string.Empty;
                tbxDevPeerType.Text = string.Empty;
                tbxDevShaderArgs.Text = string.Empty;
                tbxDevShaderArgsExpanded.Text = string.Empty;
                tbxDevShaderArgsForOptimizedShader.Text = string.Empty;
                tbxDevShaderArgsForOptimizedShaderExpanded.Text = string.Empty;
                tbxDevTextureArgs.Text = string.Empty;
                tbxDevTextureArgsExpanded.Text = string.Empty;
            }
        }

        private void tbxPcPeerType_TextChanged(object sender, EventArgs e)
        {
            var platformPreset = (TeamConfig.PlatformPreset)cbxPlatform.SelectedItemData;
            if (platformPreset == null)
            {
                return;
            }

            var pcOption = platformPreset.PcOption;
            if (pcOption != null)
            {
                pcOption.PeerType = tbxPcPeerType.Text;

                // 元が null 文字で、string.Empty への変更では null を維持させる。
                var needsNullReset =
                    string.IsNullOrEmpty(pcOption.PeerType) &&
                    (srcPlatformPresets_[cbxPlatform.SelectedIndex].PcOption.PeerType == null);
                if (needsNullReset)
                {
                    pcOption.PeerType = null;
                }

                isPlatformPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxPcShaderArgs_TextChanged(object sender, EventArgs e)
        {
            var platformPreset = (TeamConfig.PlatformPreset)cbxPlatform.SelectedItemData;
            if (platformPreset == null)
            {
                return;
            }

            var pcOption = platformPreset.PcOption;
            if (pcOption != null)
            {
                pcOption.ShaderConverterAdditionalArgs = tbxPcShaderArgs.Text;
                tbxPcShaderArgsExpanded.Text = System.Environment.ExpandEnvironmentVariables(pcOption.ShaderConverterAdditionalArgs);

                // 元が null 文字で、string.Empty への変更では null を維持させる。
                var needsNullReset =
                    string.IsNullOrEmpty(pcOption.ShaderConverterAdditionalArgs) &&
                    (srcPlatformPresets_[cbxPlatform.SelectedIndex].PcOption.ShaderConverterAdditionalArgs == null);
                if (needsNullReset)
                {
                    pcOption.ShaderConverterAdditionalArgs = null;
                }

                isPlatformPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxPcShaderForOptimizedShaderArgs_TextChanged(object sender, EventArgs e)
        {
            var platformPreset = (TeamConfig.PlatformPreset)cbxPlatform.SelectedItemData;
            if (platformPreset == null)
            {
                return;
            }

            var pcOption = platformPreset.PcOption;
            if (pcOption != null)
            {
                pcOption.ShaderConverterAdditionalArgsForOptimizedShader = tbxPcShaderForOptimizedShaderArgs.Text;
                tbxPcShaderArgsForOptimizedShaderExpanded.Text = System.Environment.ExpandEnvironmentVariables(pcOption.ShaderConverterAdditionalArgsForOptimizedShader);

                // 元が null 文字で、string.Empty への変更では null を維持させる。
                var needsNullReset =
                    string.IsNullOrEmpty(pcOption.ShaderConverterAdditionalArgsForOptimizedShader) &&
                    (srcPlatformPresets_[cbxPlatform.SelectedIndex].PcOption.ShaderConverterAdditionalArgsForOptimizedShader == null);
                if (needsNullReset)
                {
                    pcOption.ShaderConverterAdditionalArgsForOptimizedShader = null;
                }

                isPlatformPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxPcTextureArgs_TextChanged(object sender, EventArgs e)
        {
            var platformPreset = (TeamConfig.PlatformPreset)cbxPlatform.SelectedItemData;
            if (platformPreset == null)
            {
                return;
            }

            var pcOption = platformPreset.PcOption;
            if (pcOption != null)
            {
                pcOption.TextureConverterAdditionalArgs = tbxPcTextureArgs.Text;
                tbxPcTextureArgsExpanded.Text = System.Environment.ExpandEnvironmentVariables(pcOption.TextureConverterAdditionalArgs);

                // 元が null 文字で、string.Empty への変更では null を維持させる。
                var needsNullReset =
                    string.IsNullOrEmpty(pcOption.TextureConverterAdditionalArgs) &&
                    (srcPlatformPresets_[cbxPlatform.SelectedIndex].PcOption.TextureConverterAdditionalArgs == null);
                if (needsNullReset)
                {
                    pcOption.TextureConverterAdditionalArgs = null;
                }

                isPlatformPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxDevDeviceType_TextChanged(object sender, EventArgs e)
        {
            var platformPreset = (TeamConfig.PlatformPreset)cbxPlatform.SelectedItemData;
            if (platformPreset == null)
            {
                return;
            }

            var devOption = platformPreset.DeviceOptions.FirstOrDefault(x => x != platformPreset.PcOption);
            if (devOption != null)
            {
                devOption.DeviceType = tbxDevDeviceType.Text;

                // 元が null 文字で、string.Empty への変更では null を維持させる。
                var needsNullReset =
                    string.IsNullOrEmpty(devOption.DeviceType) &&
                    (srcPlatformPresets_[cbxPlatform.SelectedIndex].DeviceOptions[platformPreset.DeviceOptions.IndexOf(devOption)].DeviceType == null);
                if (needsNullReset)
                {
                    devOption.DeviceType = null;
                }

                isPlatformPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxDevPeerType_TextChanged(object sender, EventArgs e)
        {
            var platformPreset = (TeamConfig.PlatformPreset)cbxPlatform.SelectedItemData;
            if (platformPreset == null)
            {
                return;
            }

            var devOption = platformPreset.DeviceOptions.FirstOrDefault(x => x != platformPreset.PcOption);
            if (devOption != null)
            {
                devOption.PeerType = tbxDevPeerType.Text;

                // 元が null 文字で、string.Empty への変更では null を維持させる。
                var needsNullReset =
                    string.IsNullOrEmpty(devOption.PeerType) &&
                    (srcPlatformPresets_[cbxPlatform.SelectedIndex].DeviceOptions[platformPreset.DeviceOptions.IndexOf(devOption)].PeerType == null);
                if (needsNullReset)
                {
                    devOption.PeerType = null;
                }

                isPlatformPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxDevShaderArgs_TextChanged(object sender, EventArgs e)
        {
            var platformPreset = (TeamConfig.PlatformPreset)cbxPlatform.SelectedItemData;
            if (platformPreset == null)
            {
                return;
            }

            var devOption = platformPreset.DeviceOptions.FirstOrDefault(x => x != platformPreset.PcOption);
            if (devOption != null)
            {
                devOption.ShaderConverterAdditionalArgs = tbxDevShaderArgs.Text;
                tbxDevShaderArgsExpanded.Text = System.Environment.ExpandEnvironmentVariables(devOption.ShaderConverterAdditionalArgs);

                // 元が null 文字で、string.Empty への変更では null を維持させる。
                var needsNullReset =
                    string.IsNullOrEmpty(devOption.ShaderConverterAdditionalArgs) &&
                    (srcPlatformPresets_[cbxPlatform.SelectedIndex].DeviceOptions[platformPreset.DeviceOptions.IndexOf(devOption)].ShaderConverterAdditionalArgs == null);
                if (needsNullReset)
                {
                    devOption.ShaderConverterAdditionalArgs = null;
                }

                isPlatformPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxDevShaderArgsForOptimizedShader_TextChanged(object sender, EventArgs e)
        {
            var platformPreset = (TeamConfig.PlatformPreset)cbxPlatform.SelectedItemData;
            if (platformPreset == null)
            {
                return;
            }

            var devOption = platformPreset.DeviceOptions.FirstOrDefault(x => x != platformPreset.PcOption);
            if (devOption != null)
            {
                devOption.ShaderConverterAdditionalArgsForOptimizedShader = tbxDevShaderArgsForOptimizedShader.Text;
                tbxDevShaderArgsForOptimizedShaderExpanded.Text = System.Environment.ExpandEnvironmentVariables(devOption.ShaderConverterAdditionalArgsForOptimizedShader);

                // 元が null 文字で、string.Empty への変更では null を維持させる。
                var needsNullReset =
                    string.IsNullOrEmpty(devOption.ShaderConverterAdditionalArgsForOptimizedShader) &&
                    (srcPlatformPresets_[cbxPlatform.SelectedIndex].DeviceOptions[platformPreset.DeviceOptions.IndexOf(devOption)].ShaderConverterAdditionalArgsForOptimizedShader == null);
                if (needsNullReset)
                {
                    devOption.ShaderConverterAdditionalArgsForOptimizedShader = null;
                }

                isPlatformPresetModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void tbxDevTextureArgs_TextChanged(object sender, EventArgs e)
        {
            var platformPreset = (TeamConfig.PlatformPreset)cbxPlatform.SelectedItemData;
            if (platformPreset == null)
            {
                return;
            }

            var devOption = platformPreset.DeviceOptions.FirstOrDefault(x => x != platformPreset.PcOption);
            if (devOption != null)
            {
                devOption.TextureConverterAdditionalArgs = tbxDevTextureArgs.Text;
                tbxDevTextureArgsExpanded.Text = System.Environment.ExpandEnvironmentVariables(devOption.TextureConverterAdditionalArgs);

                // 元が null 文字で、string.Empty への変更では null を維持させる。
                var needsNullReset =
                    string.IsNullOrEmpty(devOption.TextureConverterAdditionalArgs) &&
                    (srcPlatformPresets_[cbxPlatform.SelectedIndex].DeviceOptions[platformPreset.DeviceOptions.IndexOf(devOption)].TextureConverterAdditionalArgs == null);
                if (needsNullReset)
                {
                    devOption.TextureConverterAdditionalArgs = null;
                }

                isPlatformPresetModified_ = null;
                UpdateOkButtonText();
            }
        }
        #endregion Platform EventHandler

        #region Hidden EventHandler
        private void tbxBuildFsvScriptFilePath_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.BuildFsv.ScriptFilePath = tbxBuildFsvScriptFilePath.Text;

            // ScriptFilePath 入力時には TemplateFilePath も入力必須。
            if (!string.IsNullOrEmpty(tbxBuildFsvScriptFilePath.Text))
            {
                var ptr = System.Runtime.InteropServices.Marshal.StringToBSTR(App.res.Strings.MandatoryField);
                Win32.NativeMethods.SendMessage(tbxBuildFsvTemplateFilePath.Handle, EM_SETCUEBANNER, IntPtr.Zero, ptr);
            }
            else
            {
                var ptr = System.Runtime.InteropServices.Marshal.StringToBSTR(string.Empty);
                Win32.NativeMethods.SendMessage(tbxBuildFsvTemplateFilePath.Handle, EM_SETCUEBANNER, IntPtr.Zero, ptr);
            }

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.BuildFsv.ScriptFilePath) &&
                (srcTeamSettings_.FileIo.BuildFsv.ScriptFilePath == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.BuildFsv.ScriptFilePath = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void btnBuildFsvScriptFilePath_Click(object sender, EventArgs e)
        {
            using (var dialog = new CommonOpenFileDialog()
            {
                Title = res.Strings.IO_SpecifyTargetFile,
                IsFolderPicker = false,
                EnsureFileExists = true
            })
            {
                var initPath = System.Environment.ExpandEnvironmentVariables(tbxBuildFsvScriptFilePath.Text);

                if (!string.IsNullOrEmpty(initPath))
                {
                    var initDir = System.IO.Path.GetDirectoryName(initPath);
                    if (!string.IsNullOrEmpty(initDir))
                    {
                        dialog.InitialDirectory = initDir;
                    }

                    var initFile = System.IO.Path.GetFileName(initPath);
                    if (!string.IsNullOrEmpty(initFile))
                    {
                        dialog.DefaultFileName = initFile;
                    }
                }

                if (dialog.ShowDialog(Handle) == CommonFileDialogResult.Ok)
                {
                    tbxBuildFsvScriptFilePath.Text = dialog.FileName;
                }
            }
        }

        private void tbxBuildFsvTemplateFilePath_TextChanged(object sender, EventArgs e)
        {
            teamSettings_.FileIo.BuildFsv.TemplateFilePath = tbxBuildFsvTemplateFilePath.Text;

            // TemplateFilePath 入力時には ScriptFilePath も入力必須。
            if (!string.IsNullOrEmpty(tbxBuildFsvTemplateFilePath.Text))
            {
                var ptr = System.Runtime.InteropServices.Marshal.StringToBSTR(App.res.Strings.MandatoryField);
                Win32.NativeMethods.SendMessage(tbxBuildFsvScriptFilePath.Handle, EM_SETCUEBANNER, IntPtr.Zero, ptr);
            }
            else
            {
                var ptr = System.Runtime.InteropServices.Marshal.StringToBSTR(string.Empty);
                Win32.NativeMethods.SendMessage(tbxBuildFsvScriptFilePath.Handle, EM_SETCUEBANNER, IntPtr.Zero, ptr);
            }

            // 元が null 文字で、string.Empty への変更では null を維持させる。
            var needsNullReset =
                string.IsNullOrEmpty(teamSettings_.FileIo.BuildFsv.TemplateFilePath) &&
                (srcTeamSettings_.FileIo.BuildFsv.TemplateFilePath == null);
            if (needsNullReset)
            {
                teamSettings_.FileIo.BuildFsv.TemplateFilePath = null;
            }

            isFileIoModified_ = null;
            UpdateOkButtonText();
        }

        private void btnBuildFsvTemplateFilePath_Click(object sender, EventArgs e)
        {
            using (var dialog = new CommonOpenFileDialog()
            {
                Title = res.Strings.IO_SpecifyTargetFile,
                IsFolderPicker = false,
                EnsureFileExists = true
            })
            {
                var initPath = System.Environment.ExpandEnvironmentVariables(tbxBuildFsvTemplateFilePath.Text);

                if (!string.IsNullOrEmpty(initPath))
                {
                    var initDir = System.IO.Path.GetDirectoryName(initPath);
                    if (!string.IsNullOrEmpty(initDir))
                    {
                        dialog.InitialDirectory = initDir;
                    }

                    var initFile = System.IO.Path.GetFileName(initPath);
                    if (!string.IsNullOrEmpty(initFile))
                    {
                        dialog.DefaultFileName = initFile;
                    }
                }

                if (dialog.ShowDialog(Handle) == CommonFileDialogResult.Ok)
                {
                    tbxBuildFsvTemplateFilePath.Text = dialog.FileName;
                }
            }
        }

        private void cbxDisableAnimationQuantize_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxDisableAnimationQuantize.SelectedItem != null)
            {
                teamSettings_.DefaultValue.DisableAnimationQuantize = (bool)((UIListControlItem)cbxDisableAnimationQuantize.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }

        private void cbxRenderStateInfoVisible_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbxRenderStateInfoVisible.SelectedItem != null)
            {
                teamSettings_.DefaultValue.RenderStateInfoVisible = (bool)((UIListControlItem)cbxRenderStateInfoVisible.SelectedItem).Data;

                isDefaultValueModified_ = null;
                UpdateOkButtonText();
            }
        }
        #endregion Hidden EventHandler
    }
}
