﻿// ---------------------------------------------------------------------------
//  Project:  NintendoWare
//
//  Copyright (C)Nintendo/HAL Laboratory, Inc.  All rights reserved.
//
//  These coded instructions, statements, and computer programs contain proprietary
//  information of Nintendo and/or its licensed developers and are protected by
//  national and international copyright laws. They may not be disclosed to third
//  parties or copied or duplicated in any form, in whole or in part, without the
//  prior written consent of Nintendo.
//
//  The content herein is highly confidential and should be handled accordingly.
// ---------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Windows.Forms;

namespace ColorPicker
{
    /// <summary>
    /// カラースライダーの基底クラス。
    /// </summary>
    public class ColorSlider : Control
    {
        protected float		m_currentPoint	= 0;
        protected float		m_currentValue	= 0;
        private float		m_max	= 1f;
        private int		m_changeCount	= 0;
        /// <summary>スライダー表示用Bitmap</summary>
        protected Bitmap	m_imageBMP	= null;
        /// <summary>選択保存カラー</summary>
        protected RGB m_color = new RGB(Color.Black);
        /// <summary>マウスダウンフラグ</summary>
        protected bool	m_isMouseDown	= false;
        /// <summary>描画フラグ</summary>
        protected bool	m_changePaint	= false;

        public float Gamma {
            private get
            {
                return gamma;
            }
            set
            {
                gamma = value;
                Invalidate();
            }
        }
        private float gamma;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ColorSlider()
        {
            // ちらつきの防止用
            DoubleBuffered = true;

            Gamma = 1;
        }

        #region イベント

        internal delegate void ColorSelectedHandler( object sender, ColorChangedEventArgs e );
        internal event ColorSelectedHandler ColorSelected;
        internal event EventHandler MaxValueChanged;

        #endregion

        #region オーバーライド

        /// <summary>
        /// OnMouseDownのオーバーライド（色選択イベント発行）
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (e.Button != MouseButtons.Left) { return; }

            m_isMouseDown = true;
            SliderValue = GetMousePoint(e);
            if (ColorSelected != null)
            {
                ColorSelected(this, new ColorChangedEventArgs(GetPointColor(SliderValue), false, false /* 両方ありえるけどイベント側で判断 */));
            }
            m_changeCount = 0;
        }

        /// <summary>
        /// OnMouseUpのオーバーライド
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            //if (e.Button != MouseButtons.Left) { return; }

            m_isMouseDown = false;
            if (ColorSelected != null)
            {
                ColorSelected(this, new ColorChangedEventArgs(GetPointColor(SliderValue), true, false /* 両方ありえるけどイベント側で判断 */));
            }
        }

        /// <summary>
        /// OnMouseMoveのオーバーライド（色選択イベント発行）
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);
            if (!m_isMouseDown)
            {
                return;
            }
            if (e.Button != MouseButtons.Left)
            {
                return;
            }

            int num = GetMousePoint(e);
            if (num < 0)
            {
                num = 0;
            }
            else if (num > GetBarSize() - 1)
            {
                num = GetBarSize() - 1;
            }
            if ((++m_changeCount) > 0)
            {
                m_changeCount = 0;
            }
            SliderValue = num;
            if (m_changeCount == 0 && ColorSelected != null)
            {
                ColorSelected(this, new ColorChangedEventArgs(GetPointColor(SliderValue), false, false /* 両方ありえるけどイベント側で判断 */));
            }
        }

        /// <summary>
        /// OnPaintのオーバーライド（ビットマップ表示）
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            if (DesignMode) { return; }

            if (Enabled && m_imageBMP != null)
            {
                Graphics g = e.Graphics;
                // 補間モード変更
                InterpolationMode im = g.InterpolationMode;
                g.InterpolationMode = InterpolationMode.NearestNeighbor;

                using (var attr = new ImageAttributes())
                {
                    attr.SetGamma(Gamma);
                    g.DrawImage(m_imageBMP, DisplayRectangle, -0.5f, -0.5f, m_imageBMP.Width, m_imageBMP.Height, GraphicsUnit.Pixel, attr);
                }

                // 補間モード復帰
                g.InterpolationMode = im;

                m_changePaint = false;
            }
            else
            {
                base.OnPaint(e);
            }
        }

        /// <summary>
        /// ColorValue計算用の幅を返す
        /// </summary>
        protected virtual int GetBarSize()
        {
            return ClientSize.Width;
        }

        /// <summary>
        /// デザインモード時は親を再描画
        /// </summary>
        protected override void OnSizeChanged(EventArgs e)
        {
            if (DesignMode)
            {
                if (Parent != null)
                {
                    Parent.Invalidate();
                }
            }
            base.OnSizeChanged(e);
        }

        #endregion

        #region プロパティ
        /// <summary>
        /// RGBAのいずれか？
        /// </summary>
        public float MaxValue
        {
            get { return m_max; }
            set
            {
                if (m_max != value)
                {
                    m_max = value;
                    if (MaxValueChanged != null)
                    {
                        MaxValueChanged(this, EventArgs.Empty);
                    }
                }
            }
        }

        /// <summary>
        /// カラー値（整数）
        /// </summary>
        public float ColorValue
        {
            get { return m_currentValue; }
            set
            {
                m_currentValue = value;
                m_currentPoint = (float)((double)value * (GetBarSize() - 1) / m_max);
            }
        }

        /// <summary>
        /// スライダー値（横向きのスライダーならX座標、縦向きのスライダーならY座標）
        /// </summary>
        protected virtual float SliderValue
        {
            get { return m_currentPoint; }
            set
            {
                m_currentPoint = value;
                m_currentValue = (float)((double)value * m_max / (GetBarSize() - 1));
            }
        }

        #endregion

        /// <summary>
        /// 指定位置色の取得
        /// </summary>
        public virtual RGB GetPointColor(float p)
        {
            return new RGB(Color.Empty);
        }

        /// <summary>
        /// マウスの位置を返す
        /// </summary>
        protected virtual int GetMousePoint(MouseEventArgs e)
        {
            return e.X;
        }

        /// <summary>
        /// 保存されている選択カラー
        /// </summary>
        public RGB GetSelectColor()
        {
            return m_color;
        }
    }
}
