﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;

namespace ColorPicker
{
    /// <summary>
    /// HSVカラー
    /// </summary>
    public class HSV
    {
        private float		m_h;
        private float		m_s;
        private float		m_v;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="h">H値（0〜360）</param>
        /// <param name="s">S値（0〜100）</param>
        /// <param name="v">V値（0〜100）</param>
        public HSV( float h, float s, float v )
        {
            m_h	= h;	m_s	= s;	m_v	= v;
        }

        /// <summary>
        /// 比較演算子
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            bool	result	= false;
            if( obj is HSV )
            {
                HSV	tmp	= (HSV)obj;
                if( tmp.H == H && tmp.S == S && tmp.V == V )
                    result	= true;
            }
            return result;
        }

        /// <summary>
        /// GetHashCodeのオーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode ();
        }

        /// <summary>
        /// RGBからHSVを作成
        /// </summary>
        /// <param name="rgb">RGB値（各0〜255）</param>
        /// <returns>HSV</returns>
        public static HSV FromRGB( RGB rgb )
        {
            float       m   = Math.Max(Math.Max(Math.Max(rgb.R, rgb.G), rgb.B), 1.0f);
            float		r	= rgb.R / m;
            float		g	= rgb.G / m;
            float		b	= rgb.B / m;

            float		minValue	= Math.Min( Math.Min( r, g ), b );
            float		maxValue	= Math.Max( Math.Max( r, g ), b );
            float		delta		= maxValue - minValue;

            double	hh	= 0;
            float		ss	= 0;
            float		vv	= (float)(maxValue * m * 100);

            if( maxValue != 0 && delta != 0 )
            {
                if( minValue == 0 )
                    ss	= 100;
                else
                    ss	=  (float)((double)delta / maxValue * 100);
                if( r == maxValue )
                    hh	= ((double)( g - b ) / delta );
                else if( g == maxValue )
                    hh	= (2 + (double)( b - r ) / delta);
                else if( b == maxValue )
                    hh	= (4 + (double)( r - g ) / delta);
            }

            hh	*= 60;
            if( hh < 0 )
                hh	+= 360;

            return new HSV( (float)hh, ss, vv );
        }

        /// <summary>
        /// HSVカラーH要素
        /// </summary>
        public float H
        {
            get	{	return m_h;		}
            set	{	m_h	= value;	}
        }
        /// <summary>
        /// HSVカラーS要素
        /// </summary>
        public float S
        {
            get	{	return m_s;		}
            set	{	m_s	= value;	}
        }
        /// <summary>
        /// HSVカラーV要素
        /// </summary>
        public float V
        {
            get	{	return m_v;		}
            set	{	m_v	= value;	}
        }
    }
    /// <summary>
    /// RGBカラー
    /// </summary>
    public class RGB
    {
        private float		m_r;
        private float		m_g;
        private float		m_b;
        private float m_a;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="r">R値（0〜1）</param>
        /// <param name="g">G値（0〜1）</param>
        /// <param name="b">B値（0〜1）</param>
        public RGB( float r, float g, float b )
        {
            m_r = r; m_g = g; m_b = b; m_a = 1;
        }

        public RGB(float a, float r, float g, float b)
        {
            m_a = a; m_r = r; m_g = g; m_b = b;
        }

        public RGB(float a, RGB rgb)
        {
            m_a = a;
            m_r = rgb.m_r;
            m_g = rgb.m_g;
            m_b = rgb.m_b;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="col">Color</param>
        public RGB( Color col )
        {
            m_r = (float)col.R / 255.0f; m_g = (float)col.G / 255.0f; m_b = (float)col.B / 255.0f; m_a = (float)col.A / 255.0f;
        }

        /// <summary>
        /// 比較演算子
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            bool	result	= false;
            if( obj is RGB )
            {
                RGB	tmp	= (RGB)obj;
                if( tmp.R == R && tmp.G == G && tmp.B == B && tmp.A == A)
                    result	= true;
            }
            return result;
        }

        /// <summary>
        /// GetHashCodeのオーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode ();
        }

        /// <summary>
        /// HSVからRGBを作成
        /// </summary>
        /// <param name="hsb">HSV値（H:0〜360,S:0〜100,V:0〜100）</param>
        /// <returns>RGB</returns>
        public static RGB FromHSV( HSV hsb )
        {
            float h = hsb.H;
            float s = (float)(hsb.S / 100f);
            float b = (float)(hsb.V / 100f);
            if (s == 0)
            {
                return new RGB(b, b, b);
            }
            double dif = (h * 6) % 360;
            float num1 = (float)(b * (1 - s));
            float num2 = (float)(b * (1 - s * dif / 360d));
            float num3 = (float)(b * (1 - s * (360 - dif) / 360d));

            switch ((int)(h / 60))
            {
                case 1:
                    return new RGB(num2, b, num1);
                case 2:
                    return new RGB(num1, b, num3);
                case 3:
                    return new RGB(num1, num2, b);
                case 4:
                    return new RGB(num3, num1, b);
                case 5:
                    return new RGB(b, num1, num2);
                default:
                    return new RGB(b, num3, num1);
            }
        }

        /// <summary>
        /// RGBカラーR要素
        /// </summary>
        public float R
        {
            get	{	return m_r;		}
            set	{	m_r	= value;	}
        }
        /// <summary>
        /// RGBカラーG要素
        /// </summary>
        public float G
        {
            get	{	return m_g;		}
            set	{	m_g	= value;	}
        }
        /// <summary>
        /// RGBカラーB要素
        /// </summary>
        public float B
        {
            get	{	return m_b;		}
            set	{	m_b	= value;	}
        }
        /// <summary>
        /// RGBカラーA要素
        /// </summary>
        public float A
        {
            get { return m_a; }
            set { m_a = value; }
        }

        /// <summary>
        /// RGBカラーからColor型を取得（αは255）
        /// </summary>
        /// <returns></returns>
        public Color GetColor()
        {
            var v = Math.Max(Math.Max(m_r, m_g), m_b);
            float r, g, b;
            if (v > 1.0f)
            {
                r = m_r / v;
                g = m_g / v;
                b = m_b / v;
            }
            else
            {
                r = m_r;
                g = m_g;
                b = m_b;
            }
            return Color.FromArgb((int)Math.Round(r * 255), (int)Math.Round(g * 255), (int)Math.Round(b * 255));
        }

        public RGB ToLinearFromSrgb()
        {
            var m = (double)Math.Max(Math.Max(R, G), B);
            double r, g, b;
            if (m > 1.0)
            {
                r = App.Utility.ColorUtility.ToLinearFromSrgb(R / m) * m;
                g = App.Utility.ColorUtility.ToLinearFromSrgb(G / m) * m;
                b = App.Utility.ColorUtility.ToLinearFromSrgb(B / m) * m;
            }
            else
            {
                r = App.Utility.ColorUtility.ToLinearFromSrgb(R);
                g = App.Utility.ColorUtility.ToLinearFromSrgb(G);
                b = App.Utility.ColorUtility.ToLinearFromSrgb(B);
            }
            return new RGB(A, (float)r, (float)g, (float)b);
        }

        public RGB ToSrgbFromLinear()
        {
            var m = (double)Math.Max(Math.Max(R, G), B);
            double r, g, b;
            if (m > 1.0)
            {
                r = App.Utility.ColorUtility.ToSrgbFromLinear(R / m) * m;
                g = App.Utility.ColorUtility.ToSrgbFromLinear(G / m) * m;
                b = App.Utility.ColorUtility.ToSrgbFromLinear(B / m) * m;
            }
            else
            {
                r = App.Utility.ColorUtility.ToSrgbFromLinear(R);
                g = App.Utility.ColorUtility.ToSrgbFromLinear(G);
                b = App.Utility.ColorUtility.ToSrgbFromLinear(B);
            }
            return new RGB(A, (float)r, (float)g, (float)b);
        }
    }
}
