﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Windows.Forms;
using App.Data;
using App.res;

namespace App.Controls
{
    /// <summary>
    /// カラーピッカーアダプタクラス。
    /// </summary>
    public sealed class ColorPickerAdapter
    {
        // トリガー
        private readonly IColorPickerTrigger _trigger;
        // 関連コントロールリスト
        private readonly List<Control> _controls = new List<Control>();
        // ロストフォーカス処理フラグ
        private bool _handleLostFocus = true;

        /// <summary>カラー編集イベント。</summary>
        public event ColorEditEventHandler ColorEdit = null;

        public event EventHandler ConnectionStart = null;

        public event Action Nullify = null;

        public bool NullifyVisible { get { return Nullify != null; } }

        /// <summary>
        /// カラーピッカーのタイトル
        /// </summary>
        public string ColorPickerText { get; set; }

        /// <summary>
        /// HDR の上限。
        /// </summary>
        public float HDRUpperBound {
            get
            {
                return _trigger.HDRUpperBound;
            }
        }

        // ColorEdit実行中かどうか？
        private bool isInColorEdit_ = false;

        public bool IsDefaultLinear
        {
            get
            {
                return _trigger.IsDefaultLinear;
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ColorPickerAdapter(IColorPickerTrigger trigger)
        {
            // テキストを登録
            ColorPickerText = Strings.ColorPicker;
            _trigger = trigger;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorPickerAdapter(ColorButton trigger)
        {
            // テキストを登録
            ColorPickerText = Strings.ColorPicker;

            // トリガー登録
            _trigger = trigger;
            trigger.Click += new EventHandler(Event_trigger_Click);

            // トリガーも関連コントロールと同様なイベント処理をさせる
            AddControl(trigger);
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorPickerAdapter(ColorEditSummary trigger)
        {
            // テキストを登録
            ColorPickerText = Strings.ColorPicker;

            // トリガー登録
            _trigger = trigger;
            trigger.MouseClick += new MouseEventHandler(Event_trigger_MouseClick);
        }

        #region アダプタ利用者アクセス用
        /// <summary>
        /// コントロール追加。
        /// </summary>
        public void AddControl(Control control)
        {
            if (!_controls.Contains(control))
            {
                _controls.Add(control);

                // イベント登録
                control.GotFocus        += new EventHandler(Event_controls_GotFocus);
                control.LostFocus       += new EventHandler(Event_controls_LostFocus);
                control.HandleDestroyed += new EventHandler(Event_controls_HandleDestroyed);
            }
        }

        /// <summary>
        /// 更新通知。
        /// </summary>
        public void NotifyUpdate()
        {
            if (isInColorEdit_ == false)
            {
                // 接続更新
                ColorPickerDialog.UpdateConnection(this);
            }
        }
        #endregion

        #region カラーピッカーアクセス用
        /// <summary>
        /// カラー。
        /// </summary>
        public RgbaColor Color
        {
            get { return _trigger.Color; }
        }

        /// <summary>
        /// 編集用カラーかどうか。
        /// </summary>
        public bool IsMarkColor
        {
            get { return _trigger.IsMarkColor; }
        }

        /// <summary>
        /// アルファ値有効フラグ。
        /// </summary>
        public bool EnableAlpha
        {
            get { return _trigger.EnableAlpha; }
        }

        /// <summary>
        /// ロストフォーカス処理をするかどうか。
        /// </summary>
        public bool HandleLostFocus
        {
            get { return _handleLostFocus; }
            set { _handleLostFocus = value; }
        }

        /// <summary>
        /// カラー編集イベント発行。
        /// </summary>
        public void InvokeColorEdit(ColorEditEventArgs e)
        {
            // 利用者にイベント発行
            if (ColorEdit != null)
            {
                isInColorEdit_ = true;

                ColorEdit(this, e);

                isInColorEdit_ = false;
            }
        }

        public void InvokeNullify()
        {
            if (Nullify != null)
            {
                Nullify();
            }
        }

        /// <summary>
        /// Undo/Redo を実行を認めるか
        /// </summary>
        public Func<bool> CanUndoRedo { get; set; }
        #endregion

        #region イベントハンドラ
        //---------------------------------------------------------------------------
        // トリガーボタン
        private void Event_trigger_Click(object sender, EventArgs e)
        {
            // ボタンクリックで接続開始
            ColorPickerDialog.StartConnection(this);

            if (ConnectionStart != null)
            {
                ConnectionStart(this, EventArgs.Empty);
            }
        }

        private void Event_trigger_MouseClick(object sender, MouseEventArgs e)
        {
            if (_trigger.ReadOnly == false)
            {
                // 左クリックで接続開始
                if (e.Button == MouseButtons.Left)
                {
                    ColorPickerDialog.StartConnection(this);

                    if (ConnectionStart != null)
                    {
                        ConnectionStart(this, EventArgs.Empty);
                    }
                }
            }
        }

        //---------------------------------------------------------------------------
        // 関連コントロール
        private void Event_controls_GotFocus(object sender, EventArgs e)
        {
            // 接続変更
            ColorPickerDialog.ChangeConnection(this);
        }

        private void Event_controls_LostFocus(object sender, EventArgs e)
        {
            if (_handleLostFocus)
            {
                // 接続終了
                ColorPickerDialog.EndConnection(this);
            }
            // フラグを元に戻す
            _handleLostFocus = true;
        }

        private void Event_controls_HandleDestroyed(object sender, EventArgs e)
        {
            // 接続終了
            ColorPickerDialog.EndConnection(this);
        }
        #endregion

        /// <summary>
        /// 切断
        /// </summary>
        public void EndConnection()
        {
            ColorPickerDialog.EndConnection(this);
        }

        // TODO: public で公開しなくてもいい形にしたい
        /// <summary>
        /// ColorPickerDialog が切断直前に呼ぶ
        /// </summary>
        public void OnBeforeDisconnect()
        {
            if (BeforeDisconnect != null)
            {
                BeforeDisconnect(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// 切断直前イベント
        /// </summary>
        public event EventHandler BeforeDisconnect;
    }

    /// <summary>
    /// カラーピッカートリガーインタフェース。
    /// </summary>
    public interface IColorPickerTrigger
    {
        /// <summary>
        /// カラー。
        /// </summary>
        RgbaColor Color
        {
            get;
        }

        /// <summary>
        /// 編集用カラーかどうか。
        /// </summary>
        bool IsMarkColor
        {
            get;
        }

        /// <summary>
        /// アルファ値有効フラグ。
        /// </summary>
        bool EnableAlpha
        {
            get;
        }

        bool IsDefaultLinear
        {
            get;
        }

        bool ReadOnly
        {
            get;
        }

        /// <summary>
        /// HDR の上限。
        /// </summary>
        float HDRUpperBound
        {
            get;
        }
    }
}
