﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using App.Data;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// カラーボタンクラス。
    /// </summary>
    public class ColorButton : UIButton, IColorPickerTrigger
    {
        // カラー
        private RgbaColor _color = RgbaColor.FromColor(System.Drawing.Color.White);
        // アルファ値有効フラグ
        private bool _enableAlpha = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorButton()
        {
        }

        /// <summary>
        /// カラー。
        /// </summary>
        [DefaultValue(typeof(Color), "White")]
        [Description("カラー。")]
        public RgbaColor Color
        {
            get { return _color; }
            set
            {
                _color = value;
                Invalidate();
            }
        }

        /// <summary>
        /// 編集用カラーかどうか。
        /// </summary>
        [DefaultValue(false)]
        [Description("編集用カラーかどうか。")]
        public bool IsMarkColor
        {
            get;
            set;
        }

        /// <summary>
        /// アルファ値有効フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Description("アルファ値有効フラグ。")]
        public bool EnableAlpha
        {
            get { return _enableAlpha; }
            set
            {
                _enableAlpha = value;
                Invalidate();
            }
        }

        /// <summary>
        /// HDR の上限。
        /// </summary>
        [DefaultValue(1.0)]
        [Description("HDR の上限。")]
        public float HDRUpperBound
        {
            get;
            set;
        }

        public bool IsDefaultLinear
        {
            get;
            set;
        }

        public bool ReadOnly
        {
            get
            {
                return false;
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            // 既定描画処理で背景等を描画する
            // 状態に応じて VisualStyle で描画される
            base.OnPaint(e);

            // 描画領域
            Rectangle rect = DisplayRectangle;
            rect.Inflate(-5, -5);

            // 有効状態
            if (Enabled)
            {
                // 枠線色
                Color lineColor = System.Drawing.Color.Black;
                if (_color.R == 0 && _color.G == 0 && _color.B == 0)
                {
                    lineColor = System.Drawing.Color.DimGray;
                }

                var color = App.Utility.ColorUtility.Pow(System.Drawing.Color.FromArgb(255, _color.ToColor()), 1);

                // カラー部
                using (Brush brush = new SolidBrush(color))
                {
                    e.Graphics.FillRectangle(brush, rect);
                }

                // アルファ部
                if (_enableAlpha)
                {
                    // 枠線色（再調整）
                    if (_color.A == 0)
                    {
                        lineColor = System.Drawing.Color.DimGray;
                    }

                    Rectangle rcAlpha = rect;
                    RectangleUtility.OffsetLeft(ref rcAlpha, rect.Width / 4 * 3);
                    using (Brush brush = new SolidBrush(System.Drawing.Color.FromArgb(_color.ToColor().A, _color.ToColor().A, _color.ToColor().A)))
                    {
                        e.Graphics.FillRectangle(brush, rcAlpha);
                    }

                    // 罫線
                    using (Pen pen = new Pen(lineColor))
                    {
                        e.Graphics.DrawLine(pen, rcAlpha.Left, rcAlpha.Top, rcAlpha.Left, rcAlpha.Bottom - 1);
                    }
                }

                // 枠線
                using (Pen pen = new Pen(lineColor))
                {
                    GraphicsUtility.DrawRectangle(e.Graphics, pen, rect);
                }
            }
            // 無効状態
            else
            {
                // 枠線
                GraphicsUtility.DrawRectangle(e.Graphics, SystemPens.GrayText, rect);
            }
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override Color ForeColor
        {
            get { return base.ForeColor;  }
            set { base.ForeColor = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override string Text
        {
            get { return base.Text;  }
            set { base.Text = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override ContentAlignment TextAlign
        {
            get { return base.TextAlign;  }
            set { base.TextAlign = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new TextImageRelation TextImageRelation
        {
            get { return base.TextImageRelation;  }
            set { base.TextImageRelation = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseMnemonic
        {
            get { return base.UseMnemonic;  }
            set { base.UseMnemonic = value; }
        }
        #endregion
    }
}
