﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;
using System.Linq;
using App.ConfigData;
using App.Data;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.Controls
{
    /// <summary>
    /// バイナリ保存ダイアログクラス。
    /// </summary>
    public sealed partial class BinarySaveDialog : OkCancelDialog
    {
        // リスト初期化フラグ
        private bool _listInitialized = false;
        // ドキュメントリスト
        private readonly List<Document> _documents = null;
        // 出力パス
        private string _outputPath = string.Empty;
        // 履歴数
        private const int _recentCount = 8;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public BinarySaveDialog(List<Document> documents)
        {
            InitializeComponent();

            // ドキュメントリスト
            Debug.Assert((documents != null) && (documents.Count > 0));
            _documents = documents;

            // ファイルリスト
            foreach (Document document in _documents)
            {
                var name = document.FileName + DocumentManager.GetSameNameIndexText(document, true);

                ListViewItem item = new ListViewItem(name);
                item.Tag = document;
                item.Checked = true;
                item.ImageKey = PathUtility.MakeNoFormatIfName(document.FileExt);
                item.SubItems.Add(document.FileExt.ToLower());
                lvwFileList.Items.Add(item);
            }
            lvwFileList.Sort(false);
            lvwFileList_VisibleChanged(lvwFileList, EventArgs.Empty);

            // チェックコンテキストメニュー
            InitializeContextMenuItem(cmiCheckFmd, G3dPath.ModelExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFtx, G3dPath.TextureExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsk, G3dPath.SkeletalAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsp, G3dPath.ShaderParamAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFcl, G3dPath.ColorAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFts, G3dPath.TexSrtAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFtp, G3dPath.TexPatternAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFvb, G3dPath.BoneVisibilityAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFvm, G3dPath.MatVisibilityAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsh, G3dPath.ShapeAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiCheckFsn, G3dPath.SceneAnimExtension.Substring(1));

            // クリアコンテキストメニュー
            InitializeContextMenuItem(cmiClearFmd, G3dPath.ModelExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFtx, G3dPath.TextureExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsk, G3dPath.SkeletalAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsp, G3dPath.ShaderParamAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFcl, G3dPath.ColorAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFts, G3dPath.TexSrtAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFtp, G3dPath.TexPatternAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFvb, G3dPath.BoneVisibilityAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFvm, G3dPath.MatVisibilityAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsh, G3dPath.ShapeAnimExtension.Substring(1));
            InitializeContextMenuItem(cmiClearFsn, G3dPath.SceneAnimExtension.Substring(1));

            // コンフィグ読み込み
            var config = ApplicationConfig.Setting.BinarySave;

            {
//				Debug.Assert(config.RecentPath.Length > 0);
                foreach (string recentPath in config.RecentPath)
                {
                    cmbPath.Items.Add(recentPath);
                }
//				cmbPath.Text = cmbPath.Items[0].ToString();
            }

            // 初期表示設定
            UpdateCounter();
            ActiveControl = btnOK;
        }

        /// <summary>
        /// 保存処理。
        /// ダイアログ表示終了後に呼び出します。
        /// </summary>
        public void Save()
        {
            // 出力ファイルリスト作成
            List<Document> documents = new List<Document>();
            foreach (ListViewItem item in lvwFileList.CheckedItems)
            {
                documents.Add((Document)item.Tag);
            }

            // 保存
            DocumentSaver saver = new DocumentSaver();
            using (WaitCursor wait = new WaitCursor())
            {
                List<string> errMsgs = new List<string>();
                // リソースはまとめて出力します。
                int stepCount = documents.Count + 1;
                using (DocumentSaverStatusIndicator indicator = new DocumentSaverStatusIndicator(saver, stepCount))
                {
                    if (!saver.BinarySave(documents, _outputPath, errMsgs, App.AppContext.SelectedPlatformPreset))
                    {
                        // エラーログ出力
                        ErrorLog.WriteBinaryErrorLog(errMsgs);
                        // バイナリ化失敗エラーメッセージ
                        BinaryConverterManager.ShowBinalizeErrorMsg(errMsgs);
                        return;
                    }
                }
            }
        }

        /// <summary>
        /// コンテキストメニュー項目初期化。
        /// </summary>
        private void InitializeContextMenuItem(ToolStripMenuItem menuItem, string fileExt)
        {
            menuItem.Tag = fileExt;
            menuItem.Enabled = lvwFileList.ContainsItem(fileExt + G3dPath.Text) || lvwFileList.ContainsItem(fileExt + G3dPath.Binary);
            menuItem.Image = Const.IconImages[fileExt];
        }

        /// <summary>
        /// カウンタ更新。
        /// </summary>
        private void UpdateCounter()
        {
            // プロパティグリッドを更新するだけ
            pgdFileCount.SelectedObject = new IntermediateFileCountPropertyData(lvwFileList);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // 初期化終了
            // ListView.ItemChecked イベントが初回表示時に必ず呼ばれてしまうのを回避...
            _listInitialized = true;
            base.OnLoad(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            // 対象ファイル確認
            if (lvwFileList.CheckedItems.Count == 0)
            {
                UIMessageBox.Warning(Strings.IO_SaveTargetNotFound);
                return false;
            }

            // 重複チェック
            var group = from document in lvwFileList.CheckedItems.OfType<ListViewItem>().Select(x => (Document)x.Tag)
                        group document by new Tuple<GuiObjectID, string>(document.ObjectID, document.Name);

            var duplicate = group.FirstOrDefault(x => x.Count() > 1);
            if (duplicate != null)
            {
                //duplicate.Key.Item1;
                var value = UIText.EnumValue(duplicate.Key.Item1);
                UIMessageBox.Warning(Strings.IO_SaveSameName, value + " " + duplicate.Key.Item2);
                return false;
            }

            foreach (ListViewItem item in lvwFileList.CheckedItems)
            {
                var document = (Document)item.Tag;
                if (!document.ConfirmBeforeBinarySave())
                {
                    return false;
                }
            }

            string output = string.Empty;
            {
                output = cmbPath.Text;
                if (output == string.Empty)
                {
                    UIMessageBox.Warning(Strings.IO_BinaryResourceFilepathIsEmpty);
                    return false;
                }

                // 拡張子のみのファイルははじく
                try
                {
                    if (Path.GetFileNameWithoutExtension(output) == string.Empty)
                    {
                        UIMessageBox.Warning(Strings.IO_BinaryResourceFilenameIsEmpty);
                        return false;
                    }
                }
                catch (ArgumentException e)
                {
                    // パスに無効な文字が指定されています。
                    UIMessageBox.Warning(e.Message);
                    return false;
                }

                if (!Path.IsPathRooted(output))
                {
                    UIMessageBox.Warning(Strings.IO_NotPathRooted);
                    return false;
                }

                if (!Directory.Exists(Path.GetDirectoryName(output)))
                {
                    UIMessageBox.Warning(Strings.IO_DirectoryNotExist);
                    return false;
                }

                try
                {
                    if (Path.GetExtension(output) != Const.DotBfres)
                    {
                        output += Const.DotBfres;
                    }

                    string fullPath = Path.GetFullPath(output);
                    if (output != fullPath)
                    {
                        if (!UIMessageBox.YesNo(Strings.IO_SaveConfirmOutputPath, fullPath))
                        {
                            return false;
                        }
                        output = fullPath;
                    }
                }
                catch (Exception)
                {
                    UIMessageBox.Warning(string.Format(Strings.IO_InvalidFileName, output));
                    return false;
                }
                if (File.Exists(output))
                {
                    if (!UIMessageBox.OkCancel(Strings.IO_BinaryResourceFileOverwrite))
                    {
                        return false;
                    }
                }

                // 保存前コマンド
                if (!DocumentSaver.PreSaveCommand(new[] { output }))
                {
                    return false;
                }

                // リードオンリーファイルに上書きしようとしている
                if (File.Exists(output) && ((File.GetAttributes(output) & FileAttributes.ReadOnly) != 0))
                {
                    UIMessageBox.Warning(output + "\n" + Strings.IO_BinaryResourceFilenameIsReadonly);
                    return false;
                }

                // 書き込み権限がない
                if (FileUtility.HasWritePermission(output) == false)
                {
                    UIMessageBox.Warning(output + "\n" + Strings.IO_BinaryResourceFilenameConnotWriteAccessPermission);
                    return false;
                }

                _outputPath = output;
            }

            // コンフィグ書き込み
            {
                var config = ApplicationConfig.Setting.BinarySave;

                AddConfigPath(ref config.RecentPath, output);
            }
            return true;
        }

        // コンフィグパスの追加
        private void AddConfigPath(ref string[] recents, string path)
        {
            ArrayUtility.Remove(ref recents, path);
            ArrayUtility.Insert(ref recents, 0, path);
            if (recents.Length > _recentCount)
            {
                Array.Resize(ref recents, _recentCount);
            }
        }

        #region イベントハンドラ
        //---------------------------------------------------------------------
        private void btnPath_Click(object sender, EventArgs e)
        {
            using(SaveFileDialog dialog = new SaveFileDialog())
            {
                dialog.Filter = string.Format("{0} (*.{1})|*.{1}", Strings.FileFilter_Bfres, Const.Bfres);
                dialog.DefaultExt = Const.Bfres;
                dialog.OverwritePrompt = false;
                if (File.Exists(cmbPath.Text)) { dialog.FileName = cmbPath.Text; }
                if (dialog.ShowDialog(TheApp.MainFrame) == DialogResult.OK)
                {
                    cmbPath.Text = dialog.FileName;
                }
            }
        }

        // ドラッグ判定
        private void cmbPath_DragEnter(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                e.Effect = DragDropEffects.Copy;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        // ドラッグ＆ドロップ
        private void cmbPath_DragDrop(object sender, DragEventArgs e)
        {
            string[] paths = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            if ((paths == null) || (paths.Length == 0)) { return; }
            cmbPath.Text = paths[0];
        }

        //---------------------------------------------------------------------
        // ファイルリスト
        private void lvwFileList_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            if (_listInitialized)
            {
                UpdateCounter();
            }
        }

        //---------------------------------------------------------------------
        // チェック状態設定
        private void cmiCheckExt_Click(object sender, EventArgs e)
        {
            string fileExt = (string)((ToolStripMenuItem)sender).Tag;

            // 全て
            if (fileExt == null)
            {
                lvwFileList.CheckAllItems();
            }
            // 種類別
            else
            {
                lvwFileList.CheckAllItems(fileExt + G3dPath.Text);
                lvwFileList.CheckAllItems(fileExt + G3dPath.Binary);
            }
            UpdateCounter();
        }

        private void cmiClearExt_Click(object sender, EventArgs e)
        {
            string fileExt = (string)((ToolStripMenuItem)sender).Tag;

            // 全て
            if (fileExt == null)
            {
                lvwFileList.ClearAllItems();
            }
            // 種類別
            else
            {
                lvwFileList.ClearAllItems(fileExt + G3dPath.Text);
                lvwFileList.ClearAllItems(fileExt + G3dPath.Binary);
            }
            UpdateCounter();
        }

        #endregion

        private void lvwFileList_VisibleChanged(object sender, EventArgs e)
        {
            lvwFileList.BeginInvokeOrExecute(new Action(() =>
                {
                    int total = lvwFileList.ClientSize.Width;
                    int right = lvwFileList.Columns[1].Width;
                    lvwFileList.Columns[0].Width = (total - right - 2);
                }));
        }
    }
}

