﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using App.Data;
using ConfigCommon;

namespace App.Command
{
    /// <summary>
    /// 編集コマンド
    /// </summary>
    public abstract class EditCommand : ICommand
    {
        /// <summary>
        /// 実行
        /// </summary>
        public ICommand Execute()
        {
            using (var block = new App.AppContext.PropertyChangedSuppressBlock())
            {
                DocumentPropertyChangedEventArgs eventArgs = new DocumentPropertyChangedEventArgs();
                ICommand result = Edit(eventArgs);
                if (eventArgs.HasArgs())
                {
                    App.AppContext.NotifyPropertyChanged(this, eventArgs);
                }

                NotifyComponentChanged(this);
                this.IsExecuted = true;

                return result;
            }
        }

        /// <summary>
        /// 編集
        /// </summary>
        public abstract ICommand Edit(DocumentPropertyChangedEventArgs eventArgs);

        protected static void NotifyComponentChanged(ICommand command)
        {
            if (command is IHasDocumentComponentChangedInfo)
            {
                IHasDocumentComponentChangedInfo info = command as IHasDocumentComponentChangedInfo;

                if (info.ChangeSource.Objects.Count > 0)
                {
                    DocumentComponentChangedEventArgs changed = new DocumentComponentChangedEventArgs();
                    {
                        foreach(Document doc in info.ChangeSource.Objects)
                        {
                            changed.AddClosedDocument(doc);
                        }
                    }
                }
            }
        }

        public abstract bool IsValid();

        public event Func<bool> canUndo;
        public virtual bool CanUndo()
        {
            if (canUndo != null)
            {
                return canUndo();
            }

            return true;
        }

        public event Func<bool> canRedo;
        public virtual bool CanRedo()
        {
            if (canRedo != null)
            {
                return canRedo();
            }

            return true;
        }

        /// <summary>
        /// コマンドが実行されたか
        /// </summary>
        public virtual bool IsExecuted { get; private set; }
    }

    //-------------------------------------------------------------------------
    /// <summary>
    /// グループ編集コマンド
    /// </summary>
    public abstract class GroupEditCommand : EditCommand
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GroupEditCommand(GuiObject guiObject, object data) :
            this(new GuiObjectGroup(guiObject), (object)data){}

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GroupEditCommand(GuiObjectGroup group, object data)
        {
            Debug.Assert(group != null);
            Initialize(group.Objects, data);
        }

        public GroupEditCommand(GuiObjectGroup group, GuiObjectID? id, IEnumerable<object> data)
        {
            if (id.HasValue)
            {
                _targets = group.GetObjects(id.Value);
            }
            else
            {
                _targets = group.Objects;
            }

            _data = new object[_targets.Count];
            int i = 0;
            foreach (object obj in data)
            {
                if (i >= _data.Length)
                {
                    break;
                }
                _data[i] = obj;
                i++;
            }
        }

        public GroupEditCommand(GuiObjectGroup group, IEnumerable<object> data)
        {
            _targets = group.Objects;
            _data = new object[_targets.Count];
            int i = 0;
            foreach (object obj in data)
            {
                Debug.Assert(i < _data.Length);
                _data[i] = obj;
                i++;
            }

            //Debug.Assert(i == _data.Length);
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GroupEditCommand(GuiObjectGroup group, GuiObjectID? objectID, object data)
        {
            Debug.Assert(group != null);
            if (objectID.HasValue)
            {
                Initialize(group.GetObjects(objectID.Value), data);
            }
            else
            {
                Initialize(group.Objects, data);
            }
        }

        public bool UpdateModified = true;

        //---------------------------------------------------------------------
        /// <summary>
        /// 編集
        /// </summary>
        public sealed override ICommand Edit(DocumentPropertyChangedEventArgs eventArgs)
        {
            if (!IsValid())
            {
                return this;
            }

            object[] swap = new object[_targets.Count];

            ICommand result = Edit(_targets, _data, swap);
            _data = swap;

            foreach (GuiObject target in _targets)
            {
                eventArgs.AddArgs(CreateEventArgs(target));
                if (UpdateModified)
                {
                    target.SetMaybeModified();
                }
            }

            return result;
        }

        /// <summary>
        /// 編集
        /// </summary>
        protected abstract ICommand Edit(ReadOnlyList<GuiObject> targets, object[] data, object[] swap);

        //---------------------------------------------------------------------
        /// <summary>
        /// 初期化
        /// </summary>
        protected void Initialize(ReadOnlyList<GuiObject> targets, object data)
        {
            // 編集するオブジェクトの数だけデータを用意する
            //Debug.Assert((data != null) && (targets.Count > 0));

            _targets = targets;
            _data = new object[_targets.Count];
            for (int i = 0; i < _targets.Count; i++)
            {
                _data[i] = data;
            }
        }

        /// <summary>
        /// イベント引数の作成
        /// </summary>
        protected virtual DocumentPropertyChangedArgs CreateEventArgs(GuiObject target)
        {
            return new DocumentContentsChangedArgs(target, this);
        }

        //---------------------------------------------------------------------
        // ターゲット
        protected ReadOnlyList<GuiObject> _targets;

        // データ
        private object[] _data;

        public override bool IsValid()
        {
            return _targets.Count > 0;
        }
    }

    /// <summary>
    /// 汎用的なグループ編集コマンド
    /// </summary>
    public class GeneralGroupValueEditCommand<T> : GroupEditCommand where T : struct
    {
        public delegate void EditDelegate(ref GuiObject target, ref object data, ref object swap);
        public delegate void PreEditDelegate(ArrayList targets, object[] data);
        public delegate void PostEditDelegate(ArrayList targets, object[] data);
        public delegate DocumentPropertyChangedArgs CreateEventArgsDelegate(GuiObject target, EditCommand command);

        private readonly EditDelegate		_editDelegate;
        private readonly PreEditDelegate		_preEditDelegate;
        private readonly PostEditDelegate	_postEditDelegate;
        private readonly CreateEventArgsDelegate _createEventArgsDelegate;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GeneralGroupValueEditCommand(
            GuiObjectGroup group,
            GuiObjectID? id,
            T data,
            EditDelegate editDelegate,
            PreEditDelegate preEditDelegate = null,
            PostEditDelegate postEditDelegate = null,
            CreateEventArgsDelegate createEventArgsDelegate = null,
            bool updateModified = true) :
            base(group, id, (object)data)
        {
            Debug.Assert(editDelegate != null);

            _editDelegate = editDelegate;
            _preEditDelegate = preEditDelegate;
            _postEditDelegate = postEditDelegate;
            _createEventArgsDelegate = createEventArgsDelegate;
            UpdateModified = updateModified;
        }

        protected override ICommand Edit(ReadOnlyList<GuiObject> targets, object[] data, object[] swap)
        {
            if (_preEditDelegate != null)
            {
                _preEditDelegate(new ArrayList(targets), data);
            }

            Debug.Assert(data.Length == targets.Count);
            for (int i = 0; i < targets.Count; ++i)
            {
                GuiObject target = targets[i];

#if false
                if (Viewer.Select.IsValidSelectType(target.ObjectID))
                {
                    Viewer.Select.Send(target);
                }
#endif

                _editDelegate(ref target, ref data[i], ref swap[i]);
            }

            if (_postEditDelegate != null)
            {
                _postEditDelegate(new ArrayList(targets), data);
            }

            return this;
        }

        protected override DocumentPropertyChangedArgs CreateEventArgs(GuiObject target)
        {
            if (_createEventArgsDelegate != null)
            {
                return _createEventArgsDelegate(target, this);
            }
            else
            {
                return base.CreateEventArgs(target);
            }
        }
    }

    /// <summary>
    /// 汎用的なグループ編集コマンド
    /// 参照など編集対象個別に別の値を与える場合用
    /// </summary>
    public class GeneralGroupReferenceEditCommand<T> : GroupEditCommand where T: class
    {
        public delegate void EditDelegate(ref GuiObject target, ref object data, ref object swap);
        public delegate void PreEditDelegate(ArrayList targets, object[] data);
        public delegate void PostEditDelegate(ArrayList targets, object[] data);
        public delegate DocumentPropertyChangedArgs CreateEventArgsDelegate(GuiObject target, EditCommand command);

        private readonly EditDelegate		_editDelegate;
        private readonly PreEditDelegate		_preEditDelegate;
        private readonly PostEditDelegate	_postEditDelegate;
        private readonly CreateEventArgsDelegate _createEventArgsDelegate;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GeneralGroupReferenceEditCommand(
            GuiObjectGroup group,
            GuiObjectID? id,
            IEnumerable<T> data,
            EditDelegate editDelegate,
            PreEditDelegate preEditDelegate = null,
            PostEditDelegate postEditDelegate = null,
            CreateEventArgsDelegate createEventArgsDelegate = null,
            bool updateModified = true) :
            base(group, id, (IEnumerable<object>)data)
        {
            Debug.Assert(editDelegate != null);

            _editDelegate = editDelegate;
            _preEditDelegate = preEditDelegate;
            _postEditDelegate = postEditDelegate;
            _createEventArgsDelegate = createEventArgsDelegate;
            UpdateModified = updateModified;
        }

        protected override ICommand Edit(ReadOnlyList<GuiObject> targets, object[] data, object[] swap)
        {
            if (_preEditDelegate != null)
            {
                _preEditDelegate(new ArrayList(targets), data);
            }

            Debug.Assert(data.Length == targets.Count);
            for (int i = 0; i < targets.Count; ++i)
            {
                GuiObject target = targets[i];

#if false
                if (Viewer.Select.IsValidSelectType(target.ObjectID))
                {
                    Viewer.Select.Send(target);
                }
#endif

                _editDelegate(ref target, ref data[i], ref swap[i]);
            }

            if (_postEditDelegate != null)
            {
                _postEditDelegate(new ArrayList(targets), data);
            }

            return this;
        }

        protected override DocumentPropertyChangedArgs CreateEventArgs(GuiObject target)
        {
            if (_createEventArgsDelegate != null)
            {
                return _createEventArgsDelegate(target, this);
            }
            else
            {
                return base.CreateEventArgs(target);
            }
        }
    }
}
