﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.Remoting;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Channels.Ipc;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Xml;
using System.Xml.Serialization;
using App.Command;
using App.ConfigData;
using App.Data;
using App.FileView;
using App.PropertyEdit;
using App.res;
using App.Utility;
using Bridge;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using Viewer;

namespace App.Bridge
{
    public static class BridgeManager
    {
        private const string processName_ = "3dEditor";
        private const string windowTitleHeader_ = "3DEditor";

        // Bridgeとのやり取り
        private static IpcServerChannel	bridgeChannel_	= null;
        private static BridgeInfo		bridgeInfo_		= null;

        private const uint MessageId = 0x19770322;

        public static void Initialize()
        {
            StartServer();
        }

        public static void Destroy()
        {
            var i = StopServer();
            if (i)
            {
                // 他の3DEditorがいれば、サーバーになってもらう
                StartOther3DEditor();
            }
        }

        private static void StartServer()
        {
            Debug.Assert(bridgeChannel_ == null);

            try
            {
                // Bridgeとのやり取り
                {
                    bridgeChannel_	= new IpcServerChannel(BridgeInfo.IpcChannelName);
                    bridgeInfo_		= new BridgeInfo();

                    ChannelServices.RegisterChannel(bridgeChannel_, true);
                    RemotingServices.Marshal(bridgeInfo_, BridgeInfo.IpcUrl, typeof(BridgeInfo));

                    bridgeInfo_.Wrote += (s, e) =>
                        {
                            // ハンドルが作られるまで待つ。
                            // なお MainFrame のハンドルが作られる前にモーダルダイアログが表示されることがある。
                            while (!TheApp.IsInitializedMainFrame || !TheApp.MainFrame.IsHandleCreated)
                            {
                                Thread.Sleep(100);
                            }

                            switch (bridgeInfo_.Mode)
                            {
                                default:
                                    TheApp.MainFrame.InvokeAfterCanFocus(new MethodInvoker(DoBridge), true);
                                    break;
                                case BridgeInfo.ModeType.ExportInfo:
                                case BridgeInfo.ModeType.ExportTemporaryFiles:
                                case BridgeInfo.ModeType.QueryConnectionState:
                                    TheApp.MainFrame.BeginInvoke(new MethodInvoker(DoBridge));
                                    break;
                            }
                        };
                }
            }
            catch (System.Runtime.Remoting.RemotingException e)
            {
                // サーバーになれなかったら無視
                DebugConsole.WriteLine(e.Message);
            }
        }

        private static bool StopServer()
        {
            if (bridgeChannel_ != null)
            {
                ChannelServices.UnregisterChannel(bridgeChannel_);

                bridgeChannel_	= null;
                bridgeInfo_		= null;

                return true;
            }
            else
            {
                return false;
            }
        }

        public static bool WndProc(ref Message message)
        {
            // エディタスタータからのメッセージ受信
            if ((message.Msg	== App.Win32.WM.WM_COPYDATA) &&
                (message.WParam	== (IntPtr)MessageId))
            {
                var data = (App.Win32.NativeMethods.COPYDATASTRUCT)message.GetLParam(typeof(App.Win32.NativeMethods.COPYDATASTRUCT));
                message.Result = (IntPtr)data.cbData;

                StartServer();
                return true;
            }
            return false;
        }

        private static IntPtr SearchWindow()
        {
            foreach (var process in System.Diagnostics.Process.GetProcessesByName(processName_))
            {
                if (process.MainWindowHandle != IntPtr.Zero)
                {
                    if (process.MainWindowTitle.StartsWith(windowTitleHeader_))
                    {
                        return process.MainWindowHandle;
                    }
                }
            }
            return IntPtr.Zero;
        }

        // 他の3DEditorがいれば、サーバーになってもらう
        private static void StartOther3DEditor()
        {
            var hwnd = SearchWindow();
            if (hwnd != IntPtr.Zero)
            {
                App.Win32.NativeMethods.COPYDATASTRUCT data;
                {
                    data.dwData	= IntPtr.Zero;
                    data.lpData	= string.Empty;
                    data.cbData	= App.Win32.NativeMethods.COPYDATASTRUCT.DataLength;
                }

                Win32.NativeMethods.SendMessage(hwnd, App.Win32.WM.WM_COPYDATA, (IntPtr)MessageId, ref data);
            }
        }

        /// <summary>
        /// スタートアップ。
        /// </summary>
        public static void DoBridge()
        {
            Application.DoEvents();

            Process(bridgeInfo_);
        }

        private static void WriteXml(object obj, string filePath)
        {
            var xmlWriterSettings = new XmlWriterSettings()
            {
                Encoding = new UTF8Encoding(true),			// BOM付き
                Indent = true,
                IndentChars = "\t",
                CloseOutput = false,
            };

            using (var file = new FileStream(filePath, FileMode.Create))
            {
                using (var xmlWriter = XmlTextWriter.Create(file, xmlWriterSettings))
                {
                    var ns = new XmlSerializerNamespaces();
                    ns.Add(string.Empty, string.Empty);

                    (new XmlSerializer(obj.GetType())).Serialize(xmlWriter, obj, ns);
                }
            }
        }

        private static void Process(BridgeInfo bridgeInfo)
        {
            // ExportInfo だけ別枠
            if (bridgeInfo.Mode == BridgeInfo.ModeType.ExportInfo)
            {
                // 途中でモデルが読み込まれたりする可能性があるので一応現状のモデルを取得
                var models = DocumentManager.Models.ToArray();

                // モデルの配置情報を問い合わせ
                foreach (var model in models)
                {
                    Viewer.QueryModelLayoutMessage.Send(model);
                }

                new Task(() =>
                {
                    // メッセージ送信完了を待つ
                    while (Viewer.ViewerDrawSuppressBlock.InBlock ||
                        Viewer.Manager.Instance.IsExistMessages ||
                        Viewer.Manager.Instance.IsExecuting)
                    {
                        System.Threading.Thread.Sleep(20);
                    }

                    // 問い合わせ結果を待つ
                    foreach (var model in models)
                    {
                        if (!model.WaitQueryModelLayout(10000))
                        {
                            DebugConsole.WriteLine("QueryModelLayout timeouted");
                            // 10秒以内に結果が戻ってこなかったら抜ける
                            break;
                        }
                    }

                    // メインスレッドに戻って情報出力
                    TheApp.MainFrame.BeginInvoke(new MethodInvoker(() =>
                    {
                        bridgeInfo.Results = Process_ExportInfo(bridgeInfo);
                        System.Threading.Thread.MemoryBarrier();
                        bridgeInfo.IsReceived = true;
                        DebugConsole.WriteLine("brige info exported");
                    }));
                }).Start();
                return;
            }


            var tmp = App.AppContext.NotificationHandler;
            var bridgeNotificationHandler = new BridgeNotificationHandler();
            App.AppContext.NotificationHandler = bridgeNotificationHandler;
            try
            {
            switch(bridgeInfo.Mode)
            {
                case BridgeInfo.ModeType.Close3DEditor:
                    {
                        break;
                    }
                case BridgeInfo.ModeType.ExportInfo:
                    {
                        throw new NotImplementedException();
                    };
                case BridgeInfo.ModeType.ExportTemporaryFiles:
                    {
                        bridgeInfo.Results = Process_ExportTemporaryFiles(bridgeInfo);
                        break;
                    };
                case BridgeInfo.ModeType.ReloadTemporaryFiles:
                    {
                        bridgeInfo.Results = Process_ReloadTemporaryFiles(bridgeInfo);
                        break;
                    }
                case BridgeInfo.ModeType.LoadFiles:
                    {
                        bridgeInfo.Results = Process_LoadFiles(bridgeInfo);
                        break;
                    }
                case BridgeInfo.ModeType.CloseAll:
                    {
                        bridgeInfo.Results = Process_CloseAll(bridgeInfo);
                        break;
                    }
                case BridgeInfo.ModeType.SaveFiles:
                    {
                        bridgeInfo.Results = Process_SaveFiles(bridgeInfo);
                        break;
                    }
                case BridgeInfo.ModeType.SaveAll:
                    {
                        bridgeInfo.Results = Process_SaveAll(bridgeInfo);
                        break;
                    }
                case BridgeInfo.ModeType.OptimizeShader:
                    {
                        bridgeInfo.Results = Process_OptimizeShader(bridgeInfo);
                        break;
                    }
                case BridgeInfo.ModeType.Connect:
                    {
                        bridgeInfo.Results = Process_Connect(bridgeInfo);
                        break;
                    }
                case BridgeInfo.ModeType.Disconnect:
                    {
                        bridgeInfo.Results = Process_Disconnect(bridgeInfo);
                        break;
                    }
                case BridgeInfo.ModeType.ClearHistory:
                    {
                        bridgeInfo.Results = Process_ClearHistory(bridgeInfo);
                        break;
                    }
                case BridgeInfo.ModeType.QueryConnectionState:
                    {
                        bridgeInfo.Results = Process_QueryConnectionState(bridgeInfo);
                        break;
                    }
                default:
                    throw new NotImplementedException();
            }
            }
            finally
            {
                bridgeInfo.Results.AddRange(bridgeNotificationHandler.results);
                System.Threading.Thread.MemoryBarrier();
                bridgeInfo.IsReceived = true;
                // Close3DEditor のときはここでアプリをクローズ
                if (bridgeInfo.Mode == BridgeInfo.ModeType.Close3DEditor)
                {
                    TheApp.MainFrame.Close();
                }
                App.AppContext.NotificationHandler = tmp;
            }
        }

        private static List<BridgeInfo.Result> Process_QueryConnectionState(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();
            if (Manager.Instance.IsConnected)
            {
                if (Manager.Instance.IsExistMessages || Manager.Instance.IsExecuting)
                {
                    results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.ConnectionBusy, error = false });
                }
                else
                {
                    results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.ConnectionIdle, error = false });
                }
            }
            else
            {
                results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.Disconnected, error = false });
            }
            return results;
        }

        private static List<BridgeInfo.Result> Process_Disconnect(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();
            if (!Viewer.Manager.Instance.IsConnected)
            {
                results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.AlreadyDisconnect, error = false });
                return results;
            }

            G3dHioLibProxy.Hio.IsAutoConnection = false;
            TheApp.MainFrame.ConnectToHio(false);
            if (Viewer.Manager.Instance.IsConnected)
            {
                results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.FailedToDisconnect, });
            }
            return results;
        }

        private static List<BridgeInfo.Result> Process_ClearHistory(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();
            TheApp.CommandManager.Clear();
            GC.Collect();
            return results;
        }

        private static List<BridgeInfo.Result> Process_Connect(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();
            // platform の選択
            if (!string.IsNullOrEmpty(bridgeInfo.Platform))
            {
                var idx = ApplicationConfig.PlatformPresets.FindIndex(x => string.Compare(x.Name, bridgeInfo.Platform, StringComparison.OrdinalIgnoreCase) == 0);
                if (idx < 0)
                {
                    results.Add(new BridgeInfo.Result
                    {
                        Type = BridgeInfo.Result.ResultType.Error,
                        error = true,
                        Message = string.Format(Strings.Bridge_PlatformNotSupported, bridgeInfo.Platform),
                    });
                    return results;
                }
                TheApp.MainFrame.SelectPlatform(ApplicationConfig.PlatformPresets[idx].Name);
            }


            if (bridgeInfo.IsDeviceTarget)
            {
                if (App.AppContext.SelectedPlatformPreset.DeviceOptions.All(x => x.DeviceType == "Pc"))
                {
                    results.Add(new BridgeInfo.Result {
                        Type = BridgeInfo.Result.ResultType.Error,
                        error = true,
                        Message = string.Format(Strings.Bridge_DeviceNotSupported, App.AppContext.SelectedPlatformPreset.Name),
                    });
                    return results;
                }
            }
            else
            {
                if (App.AppContext.SelectedPlatformPreset.DeviceOptions.All(x => x.DeviceType != "Pc"))
                {
                    results.Add(new BridgeInfo.Result
                    {
                        Type = BridgeInfo.Result.ResultType.Error,
                        error = true,
                        Message = string.Format(res.Strings.Bridge_PcNotSupported, App.AppContext.SelectedPlatformPreset.Name),
                    });
                    return results;
                }
            }

            var autoConnection = G3dHioLibProxy.Hio.IsAutoConnection;
            if (Viewer.Manager.Instance.IsConnected)
            {
                if (Viewer.Manager.IsDeviceTargeted == bridgeInfo.IsDeviceTarget)
                {
                    results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.AlreadyConnected, error = false });
                    return results;
                }

                G3dHioLibProxy.Hio.IsAutoConnection = false;
                TheApp.MainFrame.ConnectToHio(false);

                if (Viewer.Manager.Instance.IsConnected)
                {
                    results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.FailedToConnect });
                    G3dHioLibProxy.Hio.IsAutoConnection = autoConnection;
                    return results;
                }
            }

            if ((ConfigData.ApplicationConfig.Setting.Preview.Target == HioTarget.Device) != bridgeInfo.IsDeviceTarget)
            {
                ConfigData.ApplicationConfig.Setting.Preview.Target = bridgeInfo.IsDeviceTarget
                                                                          ? HioTarget.Device
                                                                          : HioTarget.Pc;
                TheApp.MainFrame.UpdateIcon();
            }

            TheApp.MainFrame.ConnectToHio(true);
            G3dHioLibProxy.Hio.IsAutoConnection = autoConnection;

            if (!Viewer.Manager.Instance.IsConnected)
            {
                results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.FailedToConnect });
            }

            return results;
        }

        private static string DocumentPath(Document doc)
        {
            return doc == null ? "" : (string.IsNullOrEmpty(doc.FilePath) ? doc.FileName : doc.FilePath);
        }

        private static List<BridgeInfo.Result> Process_ExportInfo(BridgeInfo bridgeInfo)
        {
            var exportInfo = new export_info()
            {
                current_frame = (float)App.AppContext.PreviewCurrentFrame,
                scene_animation_sets = DocumentManager.SceneAnimationSetsWithDefault.Select(
                    y => new export_info.animation_set()
                    {
                        name = y.Name,
                        default_animation_set = y == DocumentManager.DefaultSceneAnimationSet,
                        preview = y == DocumentManager.PreviewSceneAnimSet,
                        animations = y.Animations
                            .Select(z => DocumentManager.Animations.FirstOrDefault(a => new AnimationSetItem(a.FileName, a.FileLocation) == z))
                            .Select(w => new export_info.animation() { path = DocumentPath(w) })
                            .ToArray(),
                    }).ToArray(),
                models =
                    DocumentManager.Models.Select(
                        x => new export_info.model()
                        {
                            name			= x.Name,
                            path			= x.FilePath,
                            animation_sets	= x.AnimationSetsWithDefault.Select(
                                                y => new export_info.animation_set()
                                                {
                                                    name					= y.Name,
                                                    default_animation_set	= y == x.DefaultAnimationSet,
                                                    preview = x.PreviewAnimSet == y,
                                                    animations = y.Animations
                                                        .Select(z => DocumentManager.Animations.FirstOrDefault(a => new AnimationSetItem(a.FileName, a.FileLocation) == z))
                                                        .Select(w => new export_info.animation() { path = DocumentPath(w) })
                                                        .ToArray(),
                                                }
                                            ).ToArray(),
                            preview = new export_info.preview()
                            {
                                bind_model_name = x.PreviewInfo.BindModelName ?? "",
                                bind_bone_name = x.PreviewInfo.BindBoneName ?? "",
                                scale = new float[3] { x.PreviewInfo.Scale.X, x.PreviewInfo.Scale.Y, x.PreviewInfo.Scale.Z },
                                rotate = new float[3] { x.PreviewInfo.Rotate.X, x.PreviewInfo.Rotate.Y, x.PreviewInfo.Rotate.Z },
                                translate = new float[3] { x.PreviewInfo.Translate.X, x.PreviewInfo.Translate.Y, x.PreviewInfo.Translate.Z }
                            }
                        }
                    ).ToArray(),
                animations          = DocumentManager.Animations.Where(a => (a is SceneAnimation == false)).Select(x => new export_info.animation() { path = DocumentPath(x) }).ToArray(),
                textures            = DocumentManager.Textures.Select(x => new export_info.texture() { path = DocumentPath(x) }).ToArray(),
                shader_definitions  = DocumentManager.ShaderDefinitions.Select(x => new export_info.shader_definition() { path = DocumentPath(x) }).ToArray(),
                scene_animations    = DocumentManager.Animations.Where(a => a is SceneAnimation).Select(x => new export_info.animation() { path = DocumentPath(x) }).ToArray()
            };

            try
            {
                WriteXml(exportInfo, bridgeInfo.Output);
                return new List<BridgeInfo.Result>();
            }
            catch(Exception e)
            {
                // ファイル書き込みに失敗
                return new List<BridgeInfo.Result>() { new BridgeInfo.Result(){ Type = BridgeInfo.Result.ResultType.CannotWrite, Message = bridgeInfo.Output + " " + e.Message } };
            }
        }

        private static List<BridgeInfo.Result> Process_ExportTemporaryFiles(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();

            var saver = new DocumentSaver(bridgeInfo.DisableFileInfo);

            foreach(var fileName in bridgeInfo.Files)
            {
                var docs = DocumentManager.Documents.Where(x => x.FileName == fileName).OfType<IntermediateFileDocument>().ToArray();

                if (docs.Length > 1)
                {
                    // 重複
                    results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.FileDuplicated, Message = fileName });
                    continue;
                }

                var doc = docs.FirstOrDefault();
                if (doc != null)
                {
                    var outpuPath = Path.Combine(bridgeInfo.Path, fileName);

                    try
                    {
                        saver.WriteDocument(doc, outpuPath, false, true);
                    }
                    catch(Exception)
                    {
                        // ファイル書き込みに失敗
                        results.Add(new BridgeInfo.Result(){ Type = BridgeInfo.Result.ResultType.CannotWrite, Message = outpuPath});
                    }
                }
                else
                {
                    // ファイルが見つからない
                    results.Add(new BridgeInfo.Result(){ Type = BridgeInfo.Result.ResultType.FileNotFoundIn3DEditor, Message = fileName });
                }
            }

            return results;
        }

        public static void ExecuteReload(List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>>> newAndOlds, bool forceUnloadAndLoad)
        {

            using (var propertyChangedSuppressBlock = new App.AppContext.PropertyChangedSuppressBlock())
            {
                var commandSet = new EditCommandSet();
                commandSet.SetViewerDrawSuppressBlockDelegate(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages);

                // 再送信するモデル
                Model[] models = newAndOlds.Where(x => x.Item1.ObjectID == GuiObjectID.Model).Select(x => x.Item1).OfType<Model>().ToArray();

                // 同じ構造かどうか？
                HashSet<Document> structureChangedFiles = new HashSet<Document>();

                List<Document> referencedDocuments = new List<Document>();
                using (var viewerBlock = new Viewer.ViewerDrawSuppressBlock(Viewer.ViewerDrawSuppressBlock.DiscardAllMessages))
                {
                    // シェーダープールの保存
                    if (models.Any())
                    {
                        commandSet.Add(MaterialShaderPage.CreateEditCommand_ShaderPool(new GuiObjectGroup(models.SelectMany(x => x.Materials))).Execute());
                    }

                    // 読み込んだ内容に更新
                    ExecuteResetCommand(newAndOlds, commandSet, structureChangedFiles);

                    // テクスチャの読み込み
                    // TODO: 読み込みエラー処理
                    List<DocumentManager.Error> errors;
                    var referencingDocuments = new Dictionary<Document, App.Data.DocumentManager.PathWithName[]>();
                    using (var wait = new WaitCursor())
                    using (var watch = new DebugStopWatch("Load Files 2"))
                    {
                        // 参照の列挙
                        var pathWithNames = Enumerable.Empty<App.Data.DocumentManager.PathWithName>();
                        foreach (var openedDocument in newAndOlds.Select(x => (Document)x.Item1))
                        {
                            if (openedDocument is IReferTexture)
                            {
                                // 既に読み込まれているものは読み込まない
                                var iRefer = (IReferTexture)openedDocument;
                                var textures = DocumentManager.MakeReferenceTextureFilename(openedDocument)
                                    .Where(x => !iRefer.ReferenceTexturePaths
                                    .ContainsKey(Path.GetFileNameWithoutExtension(x.path))).ToArray();
                                if (textures.Length > 0)
                                {
                                    pathWithNames = pathWithNames.Concat(textures);
                                    referencingDocuments[openedDocument] = textures;
                                }
                            }

                            pathWithNames = pathWithNames.Concat(DocumentManager.MakeReferenceShaderDefinitionFilename(openedDocument));
                        }

                        DocumentManager.ExecuteLoadFromFileCommand(referencedDocuments,
                            pathWithNames.Distinct(),
                            out errors,
                            false,
                            commandSet,
                            false);


                        watch.SetMessage(string.Format("Load {0} Files", referencedDocuments.Count));
                    }

                    foreach (var refer in newAndOlds.Select(x => (Document)x.Item1).OfType<IReferTexture>())
                    {
                        var target = new GuiObjectGroup((Document)refer);
                        App.Data.DocumentManager.PathWithName[] pathWithNames;

                        // 参照を追加
                        if (referencingDocuments.TryGetValue((Document)refer, out pathWithNames))
                        {
                            foreach (var pathWithName in pathWithNames)
                            {
                                var texturePath = Path.GetFullPath(pathWithName.path);
                                var texture = DocumentManager.Textures.FirstOrDefault(x => string.Compare(texturePath, x.FilePath, true) == 0);

                                if (texture != null)
                                {
                                    if (refer is Model)
                                    {
                                        commandSet.Add(
                                            MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(
                                            target, GuiObjectID.Model, texture.Name, texture.FilePath).Execute());
                                    }
                                    else if (refer is SeparateMaterial)
                                    {
                                        commandSet.Add(
                                            MaterialSamplerPage.CreateEditCommand_ReferenceTexturePath(
                                            target, GuiObjectID.SeparateMaterial, texture.Name, texture.FilePath).Execute());
                                    }
                                    else if (refer is TexturePatternAnimation)
                                    {
                                        commandSet.Add(
                                            TexturePatternAnimationPatternPage.CreateEditCommand_ReferenceTexturePath(
                                            target, texture.Name, texture.FilePath).Execute());
                                    }
                                }
                            }
                        }

                        // いらない参照は削除
                        if (refer is Model)
                        {
                            commandSet.Add(
                                MaterialSamplerPage.CreateEditCommand_RemoveTexturePaths(target, GuiObjectID.Model).Execute());
                        }
                        else if (refer is SeparateMaterial)
                        {
                            commandSet.Add(
                                MaterialSamplerPage.CreateEditCommand_RemoveTexturePaths(target, GuiObjectID.SeparateMaterial).Execute());
                        }
                        else if (refer is TexturePatternAnimation)
                        {
                            commandSet.Add(
                                TexturePatternAnimationPatternPage.CreateEditCommand_RemoveTexturePaths(target).Execute());
                        }
                    }

                    referencedDocuments.AddRange(newAndOlds.Where(x => x.Item1.ObjectID == GuiObjectID.Texture).Select(x => (Document)x.Item1));

                    // マテリアルの修正
                    {
                        var shaderDefinitionNames = newAndOlds.Where(x => x.Item1.ObjectID == GuiObjectID.ShaderDefinition).Select(x => x.Item1.Name).ToArray();

                        // シェーダー側
                        var materials = (from material in DocumentManager.Materials
                                         where shaderDefinitionNames.Any(x => x == material.MaterialShaderAssign.ShaderDefinitionFileName) || models.Any(x => material.Referrers.Contains(x))
                                         let errorType = ShaderAssignUtility.IsConsistentWithDefinitionDetail(material)
                                         select new { material, errorType }).ToArray();

                        var errorMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.Critical).Select(x => x.material).ToArray();
                        if (errorMaterials.Any())
                        {
                            DocumentManager.AddMaterialFixCommand(commandSet, errorMaterials);
                        }
                        var disorderedMaterials = materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.Unimportant).Select(x => x.material).ToArray();
                        if (disorderedMaterials.Any())
                        {
                            commandSet.Add(ShaderAssignUtility.ExecuteFixParameters(disorderedMaterials, false, reload:false));
                        }

                        models = models.Union(materials.Where(x => x.errorType == ShaderAssignUtility.ErrorType.Critical || x.errorType == ShaderAssignUtility.ErrorType.Unimportant)
                            .SelectMany(x => x.material.Referrers).Distinct()).ToArray();
                        models = models.Union(DocumentManager.Models.Where(x => x.ReferenceDocuments.Intersect(referencedDocuments).Any())).ToArray();
                    }

                    // バインドの修正
                    if (commandSet.CommandCount > 0)
                    {
                        var command = DocumentManager.CreateAnimationUpdateBindCommand(DocumentManager.Animations).Execute();
                        if (command.IsValid())
                        {
                            commandSet.Add(command);
                        }
                    }
                }

                var animations = newAndOlds.Where(x => x.Item1 is AnimationDocument).Select(x => (AnimationDocument)x.Item1)
                    .Union(DocumentManager.Animations.Where(x => x.ReferenceDocuments.Intersect(referencedDocuments).Any())).ToArray();
                var shaderDefinitions = newAndOlds.Where(x => x.Item1 is ShaderDefinition).Select(x => (ShaderDefinition)x.Item1).ToArray();
                EditCommandSet commandSet2 = new EditCommandSet();
                commandSet.Reverse();
                commandSet2.Add(commandSet);
                EventHandler postEdit = (s, e) =>
                    {
                        //if (structureChangedFile == null)
                        {
                            bool unloaded = false;
                            foreach (var model in models)
                            {
                                // モデルアタッチ時に「アンロード/ロード」を行うとアタッチが継続されなくなるので
                                // アタッチ時には「アンロード/ロード」ではなく「リロード」を行う。
                                if (!model.IsSendAttached && (structureChangedFiles.Contains(model) || forceUnloadAndLoad))
                                {
                                    unloaded = true;
                                    Viewer.Close.Send(model);
                                    Viewer.LoadOrReloadModel.Send(model);
                                }
                                else
                                {
                                    Viewer.LoadOrReloadModel.Send(model);
                                }
                            }

                            if (unloaded)
                            {
                                foreach (var model in DocumentManager.BindSortedModels)
                                {
                                    model.SendEditBoneBind();
                                    model.SendEditModelLayout(structureChangedFiles.Contains(model) || forceUnloadAndLoad);
                                }
                            }

                            foreach (var doc in animations)
                            {
                                if (forceUnloadAndLoad)
                                {
                                    Viewer.Close.Send(doc);
                                    Viewer.LoadOrReloadAnimation.Send(doc);
                                }
                                else
                                {
                                    Viewer.LoadOrReloadAnimation.Send(doc);
                                }
                            }

                            foreach (var doc in shaderDefinitions)
                            {
                                string tempFolder = ShaderDifinitionUtility.GetShaderDefinitionTemporaryPath(doc);
                                Directory.CreateDirectory(tempFolder);
                                string fsdbFilePath = tempFolder +"\\" + doc.Name + ".fsdb";
                                (new DocumentSaver()).WriteDocument(doc, fsdbFilePath, false);

                                if (doc.IsAttached == true && doc.ShaderArchiveKey != 0 && !doc.FailedToBinarize)
                                {
                                    lock (doc)
                                    {
                                        if (!Viewer.Manager.IsDeviceTargeted && !doc.CompileTested)
                                        {
                                            // コンパイルテストを行う
                                            byte[] result;
                                            if (doc.FailedToBinarize || ShaderDifinitionUtility.RebuildShaderDefinition(doc, out result, true, Connecter.Platform) == null)
                                            {
                                                doc.FailedToBinarize = true;
                                            }

                                            doc.CompileTested = true;
                                        }
                                    }
                                }

                                // TODO: null でいいか
                                Viewer.LoadOptimizedShaderArchive.OnShaderDefinitionChanged(doc, null);
                            }

                            foreach (var model in DocumentManager.Documents.OfType<Model>())
                            {
                                model.SendBindAnimations();
                            }
                        }
                    };
                commandSet2.OnPostEdit += postEdit;
                using (new Viewer.ViewerDrawSuppressBlock())
                {
                    postEdit(null, null);
                }

                TheApp.CommandManager.Add(commandSet2);
            }
        }

        private static List<BridgeInfo.Result> Process_ReloadTemporaryFiles(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();

            var newAndOlds = new List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>>>();
            // TODO: 並列実行
            foreach (var fileName in bridgeInfo.Files)
            {
                var docs = DocumentManager.Documents.Where(x => x.FileName == fileName).OfType<IntermediateFileDocument>().ToArray();

                if (docs.Length > 1)
                {
                    // 重複
                    results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.FileDuplicated, Message = fileName });
                    continue;
                }

                var doc = docs.FirstOrDefault();
                List<G3dStream> streams = new List<G3dStream>();
                nw4f_3difType nw4f_3dif = null;
                if (doc != null)
                {
                    var filePath = Path.Combine(bridgeInfo.Path, fileName);
                    try
                    {
                        nw4f_3dif = nw.g3d.iflib.IfReadUtility.Read(streams, filePath, DocumentManager.XsdBasePath);
                        if (nw4f_3dif.Item.GetType() != doc.nw4f_3difItem.GetType())
                        {
                            throw new Exception(res.Strings.Bridge_ExtConflict);
                        }
                        newAndOlds.Add(new Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>>(doc, nw4f_3dif, streams));
                    }
                    catch (Exception e)
                    {
                        nw4f_3dif = null;
                        StringBuilder builder = new StringBuilder();
                        builder.Append(fileName);
                        while (e != null)
                        {
                            builder.Append(" " + e.Message);
                            e = e.InnerException;
                        }
                        results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.CannotRead, Message = builder.ToString()});
                    }
                }
                else
                {
                    // ファイルがない
                    results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.FileNotFoundIn3DEditor, Message = fileName });
                }
            }

            ExecuteReload(newAndOlds, bridgeInfo.ForceUnloadAndLoad);

            return results;
        }

        // ファイル読み込み
        private static List<BridgeInfo.Result> Process_LoadFiles(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();
            var paths = new List<string>();
            foreach (var file in bridgeInfo.Files)
            {
                string fullPath;
                try
                {
                    fullPath = Path.GetFullPath(Path.Combine(bridgeInfo.WorkingDirectory, file));
                    if (!File.Exists(fullPath) && !Directory.Exists(fullPath))
                    {
                        results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.CannotRead, Message = file });
                    }
                    else
                    {
                        paths.Add(fullPath);
                    }
                }
                catch
                {
                    results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.CannotRead, Message = file});
                }
            }
            DocumentManager.LoadFromFileOrDirectory(paths);

            return results;
        }

        // ファイルを閉じる
        private static List<BridgeInfo.Result> Process_CloseAll(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();
            var commandSet = new EditCommandSet();
            // 既存ファイルの変更をチェックします。
            DocumentManager.CloseAll(commandSet, true, true);

            if (commandSet.CommandCount > 0)
            {
                commandSet.Reverse();
                TheApp.CommandManager.Add(commandSet);
            }
            return results;
        }

        // ファイル保存
        private static List<BridgeInfo.Result> Process_SaveFiles(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();
            var documents = new List<Document>();
            foreach (var file in bridgeInfo.Files)
            {
                try
                {
                    Document doc;
                    if (Path.IsPathRooted(file))
                    {
                        var fullPath = Path.GetFullPath(file);
                        doc = DocumentManager.Documents.FirstOrDefault(y => string.Compare(y.FilePath, fullPath, true) == 0);
                    }
                    else
                    {
                        var name = Path.GetFileNameWithoutExtension(file);
                        var ext = Path.GetExtension(file);
                        var docs = DocumentManager.Documents.Where(x => x.Name == name && string.Compare(ext, x.FileDotExt, true) == 0).ToArray();
                        if (docs.Length > 1)
                        {
                            results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.FileDuplicated, Message = file });
                            continue;
                        }
                        doc = docs.FirstOrDefault();
                    }

                    // 編集されていない
                    if ((doc.IsModifiedObject == false) && (doc.ContentsModified == false))
                    {
                        // 保存対象でない
                        continue;
                    }

                    if (doc == null)
                    {
                        results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.FileNotFoundIn3DEditor, Message = file });
                    }
                    else
                    {
                        documents.Add(doc);
                    }
                }
                catch
                {
                    results.Add(new BridgeInfo.Result() { Type = BridgeInfo.Result.ResultType.FileNotFoundIn3DEditor, Message = file });
                }
            }

            new DocumentSaver(bridgeInfo.DisableFileInfo).SaveDocuments(documents);
            return results;
        }

        private static List<BridgeInfo.Result> Process_SaveAll(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();
            var saveDocuments = DocumentManager.GetAllSaveTargetDocuments();
            new DocumentSaver(bridgeInfo.DisableFileInfo).SaveDocuments(saveDocuments);
            return results;
        }

        private static List<BridgeInfo.Result> Process_OptimizeShader(BridgeInfo bridgeInfo)
        {
            var results = new List<BridgeInfo.Result>();

            DocumentManager.OptimizeShader = !bridgeInfo.DisableOptimizeShader;
            foreach (var model in DocumentManager.Models)
            {
                Viewer.LoadOptimizedShaderArchive.Send(model);
            }

            App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(DummyObject.TheDummyObject, null));

            return results;
        }

        public class DocumentResetArgs : DocumentPropertyChangedArgs
        {
            public DocumentResetArgs(GuiObject target, EditCommand command) : base(target, command){}
        }

        private static void ExecuteResetCommand(List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>>> list, EditCommandSet commandSet, HashSet<Document> structureChangedFiles)
        {
            foreach (var tuple in list)
            {
                if (tuple.Item1 is Model)
                {
                    bool isSame;
                    commandSet.Add(CreateModelResetCommand(tuple, out isSame).Execute());
                    if (!isSame)
                    {
                        structureChangedFiles.Add(tuple.Item1);
                    }
                }
                else if (tuple.Item1 is AnimationDocument)
                {
                    commandSet.Add(CreateAnimationResetCommand(tuple).Execute());
                }
                else if (tuple.Item1 is Texture)
                {
                    commandSet.Add(CreateTextureResetCommand(tuple).Execute());
                }
                else if (tuple.Item1 is ShaderDefinition)
                {
                    commandSet.Add(CreateShaderDefinitionResetCommand(tuple).Execute());
                }
            }
        }

        private static GroupEditCommand CreateModelResetCommand(Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>> tuple, out bool isSameStructure)
        {
            var model = (Model)tuple.Item1;
            var swapData = model.CreateSwapData((modelType)tuple.Item2.Item, tuple.Item3, out isSameStructure);
            return new GeneralGroupReferenceEditCommand<Model.SwapData>(
                new GuiObjectGroup(model),
                GuiObjectID.Model,
                Enumerable.Repeat(swapData, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var document = (Model)target;
                    swap = document.GetSwapData();
                    document.Swap((Model.SwapData)data, true);
                },
                createEventArgsDelegate: (target, command) => new DocumentResetArgs(target, command));
        }

        private static GroupEditCommand CreateAnimationResetCommand(Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>> tuple)
        {
            var animation = (AnimationDocument)tuple.Item1;
            var swapData = animation.CreateSwapData(tuple.Item2.Item, tuple.Item3);
            return new GeneralGroupReferenceEditCommand<AnimationDocument.SwapData>(
                new GuiObjectGroup(animation),
                null,
                Enumerable.Repeat(swapData, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var document = (AnimationDocument)target;
                    swap = document.GetSwapData();
                    document.Swap((AnimationDocument.SwapData)data, true);
                },
                createEventArgsDelegate: (target, command) => new DocumentResetArgs(target, command));
        }

        private static GroupEditCommand CreateTextureResetCommand(Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>> tuple)
        {
            var texture = (Texture)tuple.Item1;
            var swapData = texture.CreateSwapData((textureType)tuple.Item2.Item, tuple.Item3);
            return new GeneralGroupReferenceEditCommand<Texture.SwapData>(
                new GuiObjectGroup(texture),
                null,
                Enumerable.Repeat(swapData, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var document = (Texture)target;
                    swap = document.GetSwapData();
                    document.Swap((Texture.SwapData)data, true);
                },
                createEventArgsDelegate: (target, command) => new FileTreeView.TextureImageChangedArgs(target, command));
        }

        private static GroupEditCommand CreateShaderDefinitionResetCommand(Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>> tuple)
        {
            var shaderDefinition = (ShaderDefinition)tuple.Item1;
            var swapData = shaderDefinition.CreateSwapData((shader_definitionType)tuple.Item2.Item, tuple.Item3);
            return new GeneralGroupReferenceEditCommand<ShaderDefinition.SwapData>(
                new GuiObjectGroup(shaderDefinition),
                null,
                Enumerable.Repeat(swapData, 1),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var document = (ShaderDefinition)target;
                    swap = document.GetSwapData();
                    document.Swap((ShaderDefinition.SwapData)data, true);
                },
                createEventArgsDelegate: (target, command) => new DocumentResetArgs(target, command));
        }
    }
}
