﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include "msclr/marshal.h"
#include "msclr/marshal_windows.h"
#include "msclr/marshal_cppstd.h"
#include "msclr/marshal_atl.h"

#include "RenderInfoCommandMaker.h"
#include "CommandUtility.h"
#include "Utility.h"
#include "HIO.h"
#include <nw/g3d/res/g3d_ResUtility.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>
#include <nn/util/util_ResDic.h>
#include <nn/g3d/g3d_ResMaterial.h>

using namespace System;
using namespace msclr::interop;
using namespace nw::g3d::edit::detail;
using namespace NintendoWare::G3d::Edit;

namespace {

    u32 GetRenderInfoEditInfoSize()
    {
        return sizeof(RenderInfoEditInfo);
    }

    u32 GetRenderInfoUpdateSize()
    {
        return sizeof(RenderInfoUpdateBlock) - (sizeof(u32) * 4);
    }

    size_t GetRenderInfoRecvBlockSize()
    {
        switch (HIO::GetInstance()->TargetDeviceType)
        {
        case HIOBase::TargetType::Cafe:
        case HIOBase::TargetType::OldPc:
            {
                return sizeof(nw::g3d::edit::detail::RenderInfoRecvBlock);
            }
        case HIOBase::TargetType::Htc:
            {
                return sizeof(nn::g3d::viewer::detail::RenderInfoRecvBlock);
            }
            break;
        default:
            unexpected();
        }
    }

    size_t GetNnG3dRenderInfoValueArraySize(IRenderInfo^ renderInfo)
    {
        switch (renderInfo->Kind)
        {
        case RenderInfoKind::Float:
            {
                size_t renderInfoSize = sizeof(float) * renderInfo->ValueCount;
                return nw::g3d::tool::Align(renderInfoSize, NW_G3D_EDIT_ALIGNMENT);
            }
        case RenderInfoKind::Int:
            {
                size_t renderInfoSize = sizeof(int) * renderInfo->ValueCount;
                return nw::g3d::tool::Align(renderInfoSize, NW_G3D_EDIT_ALIGNMENT);
            }
        case RenderInfoKind::String:
            {
                size_t renderInfoSize = sizeof(nn::util::BinString) * renderInfo->ValueCount;
                return nw::g3d::tool::Align(renderInfoSize, NW_G3D_EDIT_ALIGNMENT);
            }
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        return 0;
    }

    size_t GetNnG3dRenderInfoSize()
    {
        return sizeof(nn::g3d::ResRenderInfoData);
    }

    size_t GetNnG3dRenderInfoArraySize(array<IRenderInfo^>^ renderInfoArray)
    {
        size_t size = 0;
        for (int i = 0; i < renderInfoArray->Length; ++i)
        {
            size += GetNnG3dRenderInfoSize();
        }

        return size;
    }

    size_t GetNnG3dRenderInfoValueArraySize(array<IRenderInfo^>^ renderInfoArray)
    {
        size_t size = 0;
        for (int i = 0; i < renderInfoArray->Length; ++i)
        {
            size += GetNnG3dRenderInfoValueArraySize(renderInfoArray[i]);
        }
        return size;
    }

    size_t GetBinStringDataSize(int stringLength)
    {
        return nw::g3d::tool::Align(sizeof(nn::util::BinString) + stringLength + 1, NW_G3D_EDIT_ALIGNMENT);
    }

    size_t GetNnG3dRenderInfoNameSize(IRenderInfo^ renderInfo)
    {
        return GetBinStringDataSize(renderInfo->Name->Length);
    }

    size_t GetNnG3dRenderInfoNamePoolSize(array<IRenderInfo^>^ renderInfoArray)
    {
        size_t size = GetBinStringDataSize(0); // ResDicData.entries[0]の空文字用
        for (int i = 0; i < renderInfoArray->Length; ++i)
        {
            size += GetNnG3dRenderInfoNameSize(renderInfoArray[i]);
        }

        return size;
    }

    size_t GetNnG3dRenderInfoStringValueSize(StringRenderInfo^ renderInfo)
    {
        size_t size = 0;
        for (int i = 0, end = renderInfo->Values->Count; i < end; ++i)
        {
            size += GetBinStringDataSize(renderInfo->Values[i]->EditedValue->Length);
        }

        return size;
    }

    size_t GetNnG3dRenderInfoStringValuePoolSize(array<IRenderInfo^>^ renderInfoArray)
    {
        size_t size = 0;
        for (int i = 0; i < renderInfoArray->Length; ++i)
        {
            StringRenderInfo^ stringRenderInfo = dynamic_cast<StringRenderInfo^>(renderInfoArray[i]);
            if (stringRenderInfo != nullptr)
            {
                size += GetNnG3dRenderInfoStringValueSize(stringRenderInfo);
            }
        }

        return size;
    }
}

namespace NintendoWare { namespace G3d { namespace Edit {

int RenderInfoCommandMaker::CalcSetupRenderInfoSize(array<IRenderInfo^>^ renderInfoArray)
{
    if ( renderInfoArray == nullptr )
    {
        return -1;
    }

    u32 renderInfoCount = renderInfoArray->Length;
    size_t renderInfoArraySize = nw::g3d::tool::Align(sizeof(SetupRenderInfoData) * renderInfoCount, NW_G3D_EDIT_ALIGNMENT);

    size_t namesSize = 0;
    for (u32 i = 0; i < renderInfoCount; ++i)
    {
        // 描画情報の名前が空の時点で不正なデータなので、-1 を返して終了
        if (renderInfoArray[i]->Name == String::Empty)
        {
            return -1;
        }

        namesSize += nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + renderInfoArray[i]->Name->Length + 1, NW_G3D_EDIT_ALIGNMENT);
    }

    size_t choiceDataSize = 0;
    size_t defaultDataSize = 0;
    size_t choiceSize = 0;
    size_t defaultSize = 0;
    for (u32 i = 0; i < renderInfoCount; ++i)
    {
        u32 choiceCount = 0;
        switch (renderInfoArray[i]->Kind)
        {
        case RenderInfoKind::Int:
        case RenderInfoKind::Float:
            choiceCount = 2;
            break;
        }

        choiceDataSize += nw::g3d::tool::Align(sizeof(SetupRenderInfoChoiceData) + sizeof(u32) * (renderInfoArray[i]->ItemCount + choiceCount), NW_G3D_EDIT_ALIGNMENT);
        defaultDataSize += nw::g3d::tool::Align(sizeof(SetupRenderInfoDefaultData) + sizeof(u32) * renderInfoArray[i]->ValueCount, NW_G3D_EDIT_ALIGNMENT);

        StringRenderInfo^ renderInfo = dynamic_cast<StringRenderInfo^>(renderInfoArray[i]);
        if (renderInfo == nullptr) // 文字列型ではない場合は処理をとばす
        {
            continue;
        }

        for (int j = 0; j < renderInfo->ItemCount; ++j)
        {
            StringRenderInfo::Item^ item = renderInfo->GetItem(j);
            if (item == nullptr)
            {
                return -1;
            }

            String^ stringValue = dynamic_cast<String^>(item->Choice->Clone());
            choiceSize +=
                nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + stringValue->Length + 1, NW_G3D_EDIT_ALIGNMENT);
        }

        for (int j = 0; j < renderInfo->ValueCount; ++j)
        {
            StringRenderInfo::Value^ value = renderInfo->GetValue(j);
            if (value == nullptr)
            {
                return -1;
            }

            String^ stringValue = dynamic_cast<String^>(value->DefaultValue->Clone());
            defaultSize +=
                nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + stringValue->Length + 1, NW_G3D_EDIT_ALIGNMENT);
        }
    }

    return static_cast<int>(renderInfoArraySize + namesSize + choiceDataSize + defaultDataSize + choiceSize + defaultSize);
}


int RenderInfoCommandMaker::CalcQueryRenderInfoPacketSize(array<INT32>^ indices, array<IRenderInfo^>^ renderInfoArray)
{
    if (socket == NULL || indices == nullptr)
    {
        return -1;
    }

    int setupRenderInfoSize = CalcSetupRenderInfoSize(renderInfoArray);
    if (setupRenderInfoSize < 0)
    {
        return -1;
    }

    u32 renderInfoCount = renderInfoArray->Length;
    size_t renderInfoPtrArraySize = nw::g3d::tool::Align(sizeof(u32) * renderInfoCount, NW_G3D_EDIT_ALIGNMENT);

    if (indices->Length <= 0)
    {
        return -1;
    }

    size_t indexByteSize = nw::g3d::tool::Align(indices->Length * sizeof(s32), NW_G3D_EDIT_ALIGNMENT);

    size_t totalSize = nw::g3d::tool::Align(
            CommandUtility::GetPacketHeaderSize() + GetRenderInfoRecvBlockSize() +
            renderInfoPtrArraySize + indexByteSize + setupRenderInfoSize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

int RenderInfoCommandMaker::CalcSelectRenderInfoItemPacketSize(String^ labelName)
{
    if (String::IsNullOrEmpty(labelName))
    {
        return -1;
    }

    int labelNameLength = labelName->Length;
    if (labelNameLength <= 0)
    {
        return -1;
    }

    size_t labelNameSize = labelNameLength + 1;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetRenderInfoEditInfoSize() + labelNameSize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

int RenderInfoCommandMaker::CalcRenderInfoDicSize(array<IRenderInfo^>^ renderInfoArray)
{
    if ( renderInfoArray == nullptr )
    {
        return -1;
    }

    int renderInfoCount = renderInfoArray->Length;
    size_t totalSize = 0;

    switch (HIO::GetInstance()->TargetDeviceType)
    {
    case HIOBase::TargetType::Cafe:
    case HIOBase::TargetType::OldPc:
        {
            size_t dicDataSize = sizeof(nw::g3d::res::ResDicPatriciaData);
            size_t dicNodeSize = sizeof(nw::g3d::res::ResDicPatriciaData::Node) * renderInfoCount;
            totalSize =
                nw::g3d::tool::Align(dicDataSize + dicNodeSize, NW_G3D_EDIT_ALIGNMENT);

            size_t renderInfoArraySize = 0;
            for (int i = 0; i < renderInfoCount; ++i)
            {
                size_t renderInfoSize = sizeof(nw::g3d::res::ResRenderInfoData) + sizeof(u32) * renderInfoArray[i]->ValueCount;
                renderInfoArraySize += nw::g3d::tool::Align(renderInfoSize, NW_G3D_EDIT_ALIGNMENT);
            }

            size_t namesSize = 0;
            for (int i = 0; i < renderInfoCount; ++i)
            {
                // 描画情報の名前が空の時点で不正なデータなので、-1 を返して終了
                if (renderInfoArray[i]->Name == String::Empty)
                {
                    return -1;
                }

                namesSize += nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + renderInfoArray[i]->Name->Length + 1, NW_G3D_EDIT_ALIGNMENT);
            }

            size_t stringTypeSize = 0;
            for (int i = 0; i < renderInfoCount; ++i)
            {
                StringRenderInfo^ renderInfo = dynamic_cast<StringRenderInfo^>(renderInfoArray[i]);
                if (renderInfo == nullptr) // 文字列型ではない場合は処理をとばす
                {
                    continue;
                }

                for (int j = 0; j < renderInfo->ValueCount; ++j)
                {
                    StringRenderInfo::Value^ value = renderInfo->GetValue(j);
                    if (value == nullptr)
                    {
                        return -1;
                    }

                    String^ stringValue = dynamic_cast<String^>(value->EditedValue->Clone());
                    stringTypeSize +=
                        nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + stringValue->Length + 1, NW_G3D_EDIT_ALIGNMENT);
                }
            }

            totalSize += renderInfoArraySize + namesSize + stringTypeSize;
            return static_cast<int>(totalSize);
        }
        break;
    case HIOBase::TargetType::Htc:
        {
            // ResRenderInfo 辞書、描画情報配列、描画情報名文字列データ、値配列データ、文字列値プールの順番に格納
            u32 renderInfoCount = renderInfoArray->Length;

            size_t dicDataSize = sizeof(nn::util::ResDicData);
            size_t dicNodeSize = sizeof(nn::util::ResDicData::Entry) * renderInfoCount;
            size_t dicTotalSize = nw::g3d::tool::Align(dicDataSize + dicNodeSize, NW_G3D_EDIT_ALIGNMENT);
            size_t renderInfoArraySize = GetNnG3dRenderInfoArraySize(renderInfoArray);
            size_t namePoolSize = GetNnG3dRenderInfoNamePoolSize(renderInfoArray);
            size_t valueArraySize = GetNnG3dRenderInfoValueArraySize(renderInfoArray);
            size_t stringValuePoolSize = GetNnG3dRenderInfoStringValuePoolSize(renderInfoArray);

            totalSize = dicTotalSize + renderInfoArraySize + namePoolSize + valueArraySize + stringValuePoolSize;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return static_cast<int>(totalSize);
}

int RenderInfoCommandMaker::CalcUpdateRenderInfoPacketSize(array<IRenderInfo^>^ renderInfoArray)
{
    size_t renderInfoDicSize = CalcRenderInfoDicSize(renderInfoArray);
    if (renderInfoDicSize <= 0)
    {
        return -1;
    }

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetRenderInfoUpdateSize() + renderInfoDicSize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

void MakeQueryRenderInfoPacketImplShare()
{

}

bool MakeQueryRenderInfoPacketImpl(
    array<Byte>^ packetBuffer,
    UINT32 key,
    array<INT32>^ indices,
    array<IRenderInfo^>^ renderInfoArray,
    TargetEndianKind endianKind)
{
    size_t totalSize = packetBuffer->Length;
    int renderInfoCount = renderInfoArray->Length;

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    switch (HIO::GetInstance()->TargetDeviceType)
    {
    case HIOBase::TargetType::Cafe:
    case HIOBase::TargetType::OldPc:
        {
            RenderInfoRecvPacket* packet = reinterpret_cast<RenderInfoRecvPacket*>(bufferPtr);
            memset(packet, 0, totalSize);

            packet->header = CommandUtility::GetPacketHeader();
            packet->header.command = EDIT_RECV_RENDER_INFO_COMMAND_FLAG;
            packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

            packet->block.modelKey = key;
            packet->block.numRenderInfo = static_cast<u16>(renderInfoCount);
            packet->block.numMaterialIndex = static_cast<u16>(indices->Length);

            u32* renderInfoPtrArray = nw::g3d::ut::AddOffset<u32>(packet, sizeof(RenderInfoRecvPacket));
            size_t renderInfoPtrArraySize = nw::g3d::ut::Align(sizeof(u32) * renderInfoCount, NW_G3D_EDIT_ALIGNMENT);

            s32* materialIndexArray = nw::g3d::ut::AddOffset<s32>(renderInfoPtrArray, renderInfoPtrArraySize);
            size_t materialIndexArraySize = nw::g3d::ut::Align(sizeof(s32) * indices->Length, NW_G3D_EDIT_ALIGNMENT);

            packet->block.ofsMaterialIndexArray.set_ptr(materialIndexArray);
            for (u16 i = 0; i < packet->block.numMaterialIndex; ++i)
            {
                materialIndexArray[i] = indices[i];
            }

            SetupRenderInfoData* firstRenderInfo = nw::g3d::ut::AddOffset<SetupRenderInfoData>(materialIndexArray, materialIndexArraySize);
            packet->block.ofsRenderInfoArray.set_ptr(firstRenderInfo);
            std::vector<SetupRenderInfoData*> renderInfos;
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                SetupRenderInfoData* renderInfo = nw::g3d::ut::AddOffset<SetupRenderInfoData>(firstRenderInfo, sizeof(SetupRenderInfoData) * i);
                renderInfos.push_back(renderInfo);
                switch (renderInfoArray[i]->Kind)
                {
                case RenderInfoKind::String:
                    renderInfo->type = nw::g3d::res::ResRenderInfo::STRING;
                    break;
                case RenderInfoKind::Int:
                    renderInfo->type = nw::g3d::res::ResRenderInfo::INT;
                    break;
                case RenderInfoKind::Float:
                    renderInfo->type = nw::g3d::res::ResRenderInfo::FLOAT;
                    break;
                }
            }

            SetupRenderInfoChoiceData* firstChoice = nw::g3d::ut::AddOffset<SetupRenderInfoChoiceData>(firstRenderInfo, sizeof(SetupRenderInfoData) * packet->block.numRenderInfo);
            size_t choiceDataSize = 0;
            std::vector<SetupRenderInfoChoiceData*> choices;
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                SetupRenderInfoData* renderInfoData = renderInfos[i];
                SetupRenderInfoChoiceData* choice = nw::g3d::ut::AddOffset<SetupRenderInfoChoiceData>(firstChoice, choiceDataSize);
                renderInfoData->ofsChoice.set_ptr(choice);
                choices.push_back(choice);
                choiceDataSize += nw::g3d::tool::Align(sizeof(SetupRenderInfoChoiceData) + sizeof(u32) * (renderInfoArray[i]->ItemCount + 2), NW_G3D_EDIT_ALIGNMENT);

                // Int型
                {
                    IntRenderInfo^ renderInfo = dynamic_cast<IntRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        if (renderInfo->HasMinMax)
                        {
                            choice->iValue[0] = renderInfo->Min;
                            choice->iValue[1] = renderInfo->Max;
                            choice->numChoice = 2;
                        }
                        else
                        {
                            choice->numChoice = 0;
                        }
                    }
                }

                // Float型
                {
                    FloatRenderInfo^ renderInfo = dynamic_cast<FloatRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        if (renderInfo->HasMinMax)
                        {
                            choice->fValue[0] = renderInfo->Min;
                            choice->fValue[1] = renderInfo->Max;
                            choice->numChoice = 2;
                        }
                        else
                        {
                            choice->numChoice = 0;
                        }
                    }
                }
            }

            bool setupResult = true;
            SetupRenderInfoDefaultData* firstDefault = nw::g3d::ut::AddOffset<SetupRenderInfoDefaultData>(firstChoice, choiceDataSize);
            size_t defaultDataSize = 0;
            std::vector<SetupRenderInfoDefaultData*> defaults;
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                SetupRenderInfoData* renderInfoData = renderInfos[i];
                SetupRenderInfoDefaultData* defaultValue = nw::g3d::ut::AddOffset<SetupRenderInfoDefaultData>(firstDefault, defaultDataSize);
                renderInfoData->ofsDefault.set_ptr(defaultValue);
                defaults.push_back(defaultValue);
                defaultDataSize += nw::g3d::tool::Align(sizeof(SetupRenderInfoDefaultData) + sizeof(u32) * renderInfoArray[i]->ValueCount, NW_G3D_EDIT_ALIGNMENT);

                // Int型
                {
                    IntRenderInfo^ renderInfo = dynamic_cast<IntRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        defaultValue->numDefault = static_cast<u32>(renderInfo->ValueCount);
                        for (u32 j = 0; j < defaultValue->numDefault; ++j)
                        {
                            IntRenderInfo::Value^ value = renderInfo->GetValue(j);
                            if ( value == nullptr )
                            {
                                setupResult = false;
                                break;
                            }
                            defaultValue->iValue[j] = value->DefaultValue;
                        }
                    }
                }

                // Float型
                {
                    FloatRenderInfo^ renderInfo = dynamic_cast<FloatRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        defaultValue->numDefault = static_cast<u32>(renderInfo->ValueCount);
                        for (u32 j = 0; j < defaultValue->numDefault; ++j)
                        {
                            FloatRenderInfo::Value^ value = renderInfo->GetValue(j);
                            if ( value == nullptr )
                            {
                                setupResult = false;
                                break;
                            }
                            defaultValue->fValue[j] = value->DefaultValue;
                        }
                    }
                }
            }

            if (!setupResult)
            {
                bufferPtr = nullptr;
                return false;
            }

            size_t nameDataSize = 0;
            nw::g3d::res::ResNameData* nameDataFirst = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(firstDefault, defaultDataSize);
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                SetupRenderInfoData* renderInfo = renderInfos[i];
                nw::g3d::res::ResNameData* nameData = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(nameDataFirst, nameDataSize);

                renderInfo->ofsName.set_ptr(nameData->str);
                String^ copyValue = dynamic_cast<String^>(renderInfoArray[i]->Name->Clone());
                nameDataSize += nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + copyValue->Length + 1, NW_G3D_EDIT_ALIGNMENT);
                std::string name = marshal_as<std::string>(copyValue);
                memcpy(nameData->str, name.c_str(), name.length());
                nameData->len = static_cast<u32>(name.length());
            }

            setupResult = true;
            size_t stringChoiceDataSize = 0;
            nw::g3d::res::ResNameData* stringChoiceDataFirst = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(nameDataFirst, nameDataSize);
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                SetupRenderInfoData* renderInfoData = renderInfos[i];
                // 文字列型
                {
                    StringRenderInfo^ renderInfo = dynamic_cast<StringRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        u16 end = static_cast<u16>(renderInfo->ItemCount);
                        for (u16 j = 0; j < end; ++j)
                        {
                            StringRenderInfo::Item^ item = renderInfo->GetItem(j);
                            if ( item == nullptr )
                            {
                                setupResult = false;
                                break;
                            }

                            nw::g3d::res::ResNameData* nameData = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(stringChoiceDataFirst, stringChoiceDataSize);
                            SetupRenderInfoChoiceData* choice = renderInfoData->ofsChoice.to_ptr<SetupRenderInfoChoiceData>();
                            choice->ofsString[j].set_ptr(nameData->str);
                            choice->numChoice = end;

                            String^ copyValue = dynamic_cast<String^>(item->Choice->Clone());
                            stringChoiceDataSize += nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + copyValue->Length + 1, NW_G3D_EDIT_ALIGNMENT);
                            std::string name = marshal_as<std::string>(copyValue);
                            memcpy(nameData->str, name.c_str(), name.length());
                            nameData->len = static_cast<u32>(name.length());
                        }
                    }
                }
            }

            if (!setupResult)
            {
                bufferPtr = nullptr;
                return false;
            }

            setupResult = true;
            size_t stringDefaultDataSize = 0;
            nw::g3d::res::ResNameData* stringDefaultDataFirst = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(stringChoiceDataFirst, stringChoiceDataSize);
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                SetupRenderInfoData* renderInfoData = renderInfos[i];
                // 文字列型
                {
                    StringRenderInfo^ renderInfo = dynamic_cast<StringRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        u16 end = static_cast<u16>(renderInfo->ValueCount);
                        for (u16 j = 0; j < end; ++j)
                        {
                            StringRenderInfo::Value^ value = renderInfo->GetValue(j);
                            if ( value == nullptr )
                            {
                                setupResult = false;
                                break;
                            }

                            nw::g3d::res::ResNameData* nameData = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(stringDefaultDataFirst, stringDefaultDataSize);
                            SetupRenderInfoDefaultData* defaultValue = renderInfoData->ofsDefault.to_ptr<SetupRenderInfoDefaultData>();
                            defaultValue->ofsString[j].set_ptr(nameData->str);
                            defaultValue->numDefault = end;

                            String^ copyValue = dynamic_cast<String^>(value->DefaultValue->Clone());
                            stringDefaultDataSize += nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + copyValue->Length + 1, NW_G3D_EDIT_ALIGNMENT);
                            std::string name = marshal_as<std::string>(copyValue);
                            memcpy(nameData->str, name.c_str(), name.length());
                            nameData->len = static_cast<u32>(name.length());
                        }
                    }
                }
            }

            if (!setupResult)
            {
                bufferPtr = nullptr;
                return false;
            }

            if (endianKind == TargetEndianKind::BigEndian)
            {
                nw::g3d::tool::Endian::Swap(packet);
            }
        }
        break;
    case HIOBase::TargetType::Htc:
        {
            nn::g3d::viewer::detail::RenderInfoRecvPacket* packet = reinterpret_cast<nn::g3d::viewer::detail::RenderInfoRecvPacket*>(bufferPtr);
            memset(packet, 0, totalSize);

            packet->header = CommandUtility::GetNnG3dViewerPacketHeader();
            packet->header.command = EDIT_RECV_RENDER_INFO_COMMAND_FLAG;
            packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

            packet->block.modelKey = key;
            packet->block.numRenderInfo = static_cast<u16>(renderInfoCount);
            packet->block.numMaterialIndex = static_cast<u16>(indices->Length);

            u32* renderInfoPtrArray = nw::g3d::ut::AddOffset<u32>(packet, sizeof(nn::g3d::viewer::detail::RenderInfoRecvPacket));
            size_t renderInfoPtrArraySize = nw::g3d::ut::Align(sizeof(u32) * renderInfoCount, NW_G3D_EDIT_ALIGNMENT);

            s32* materialIndexArray = nw::g3d::ut::AddOffset<s32>(renderInfoPtrArray, renderInfoPtrArraySize);
            size_t materialIndexArraySize = nw::g3d::ut::Align(sizeof(s32) * indices->Length, NW_G3D_EDIT_ALIGNMENT);

            ::SetOffset(&packet->block.ofsMaterialIndexArray, materialIndexArray);

            for (u16 i = 0; i < packet->block.numMaterialIndex; ++i)
            {
                materialIndexArray[i] = indices[i];
            }

            nn::g3d::viewer::detail::SetupRenderInfoData* firstRenderInfo = nw::g3d::ut::AddOffset<nn::g3d::viewer::detail::SetupRenderInfoData>(materialIndexArray, materialIndexArraySize);
            ::SetOffset(&packet->block.ofsRenderInfoArray, firstRenderInfo);

            std::vector<nn::g3d::viewer::detail::SetupRenderInfoData*> renderInfos;
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                nn::g3d::viewer::detail::SetupRenderInfoData* renderInfo = nw::g3d::ut::AddOffset<nn::g3d::viewer::detail::SetupRenderInfoData>(firstRenderInfo, sizeof(nn::g3d::viewer::detail::SetupRenderInfoData) * i);
                renderInfos.push_back(renderInfo);
                switch (renderInfoArray[i]->Kind)
                {
                case RenderInfoKind::String:
                    renderInfo->type = nn::g3d::ResRenderInfo::Type_String;
                    break;
                case RenderInfoKind::Int:
                    renderInfo->type = nn::g3d::ResRenderInfo::Type_Int;
                    break;
                case RenderInfoKind::Float:
                    renderInfo->type = nn::g3d::ResRenderInfo::Type_Float;
                    break;
                }
            }

            nn::g3d::viewer::detail::SetupRenderInfoChoiceData* firstChoice = nw::g3d::ut::AddOffset<nn::g3d::viewer::detail::SetupRenderInfoChoiceData>(firstRenderInfo, sizeof(nn::g3d::viewer::detail::SetupRenderInfoData) * packet->block.numRenderInfo);
            size_t choiceDataSize = 0;
            std::vector<nn::g3d::viewer::detail::SetupRenderInfoChoiceData*> choices;
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                nn::g3d::viewer::detail::SetupRenderInfoData* renderInfoData = renderInfos[i];
                nn::g3d::viewer::detail::SetupRenderInfoChoiceData* choice = nw::g3d::ut::AddOffset<nn::g3d::viewer::detail::SetupRenderInfoChoiceData>(firstChoice, choiceDataSize);
                ::SetOffset(&renderInfoData->ofsChoice, choice);
                choices.push_back(choice);
                choiceDataSize += nw::g3d::tool::Align(sizeof(nn::g3d::viewer::detail::SetupRenderInfoChoiceData) + sizeof(u32) * (renderInfoArray[i]->ItemCount + 2), NW_G3D_EDIT_ALIGNMENT);

                // Int型
                {
                    IntRenderInfo^ renderInfo = dynamic_cast<IntRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        if (renderInfo->HasMinMax)
                        {
                            choice->iValue[0] = renderInfo->Min;
                            choice->iValue[1] = renderInfo->Max;
                            choice->numChoice = 2;
                        }
                        else
                        {
                            choice->numChoice = 0;
                        }
                    }
                }

                // Float型
                {
                    FloatRenderInfo^ renderInfo = dynamic_cast<FloatRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        if (renderInfo->HasMinMax)
                        {
                            choice->fValue[0] = renderInfo->Min;
                            choice->fValue[1] = renderInfo->Max;
                            choice->numChoice = 2;
                        }
                        else
                        {
                            choice->numChoice = 0;
                        }
                    }
                }
            }

            bool setupResult = true;
            nn::g3d::viewer::detail::SetupRenderInfoDefaultData* firstDefault = nw::g3d::ut::AddOffset<nn::g3d::viewer::detail::SetupRenderInfoDefaultData>(firstChoice, choiceDataSize);
            size_t defaultDataSize = 0;
            std::vector<nn::g3d::viewer::detail::SetupRenderInfoDefaultData*> defaults;
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                nn::g3d::viewer::detail::SetupRenderInfoData* renderInfoData = renderInfos[i];
                nn::g3d::viewer::detail::SetupRenderInfoDefaultData* defaultValue = nw::g3d::ut::AddOffset<nn::g3d::viewer::detail::SetupRenderInfoDefaultData>(firstDefault, defaultDataSize);
                ::SetOffset(&renderInfoData->ofsDefault, defaultValue);
                defaults.push_back(defaultValue);
                defaultDataSize += nw::g3d::tool::Align(sizeof(nn::g3d::viewer::detail::SetupRenderInfoDefaultData) + sizeof(u32) * renderInfoArray[i]->ValueCount, NW_G3D_EDIT_ALIGNMENT);

                // Int型
                {
                    IntRenderInfo^ renderInfo = dynamic_cast<IntRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        defaultValue->numDefault = static_cast<u32>(renderInfo->ValueCount);
                        for (u32 j = 0; j < defaultValue->numDefault; ++j)
                        {
                            IntRenderInfo::Value^ value = renderInfo->GetValue(j);
                            if ( value == nullptr )
                            {
                                setupResult = false;
                                break;
                            }
                            defaultValue->iValue[j] = value->DefaultValue;
                        }
                    }
                }

                // Float型
                {
                    FloatRenderInfo^ renderInfo = dynamic_cast<FloatRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        defaultValue->numDefault = static_cast<u32>(renderInfo->ValueCount);
                        for (u32 j = 0; j < defaultValue->numDefault; ++j)
                        {
                            FloatRenderInfo::Value^ value = renderInfo->GetValue(j);
                            if ( value == nullptr )
                            {
                                setupResult = false;
                                break;
                            }
                            defaultValue->fValue[j] = value->DefaultValue;
                        }
                    }
                }
            }

            if (!setupResult)
            {
                bufferPtr = nullptr;
                return false;
            }

            size_t nameDataSize = 0;
            nw::g3d::res::ResNameData* nameDataFirst = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(firstDefault, defaultDataSize);
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                nn::g3d::viewer::detail::SetupRenderInfoData* renderInfo = renderInfos[i];
                nw::g3d::res::ResNameData* nameData = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(nameDataFirst, nameDataSize);

                renderInfo->ofsName.Initialize(nameData->str);
                String^ copyValue = dynamic_cast<String^>(renderInfoArray[i]->Name->Clone());
                nameDataSize += nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + copyValue->Length + 1, NW_G3D_EDIT_ALIGNMENT);
                std::string name = marshal_as<std::string>(copyValue);
                memcpy(nameData->str, name.c_str(), name.length());
                nameData->len = static_cast<u32>(name.length());
            }

            setupResult = true;
            size_t stringChoiceDataSize = 0;
            nw::g3d::res::ResNameData* stringChoiceDataFirst = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(nameDataFirst, nameDataSize);
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                nn::g3d::viewer::detail::SetupRenderInfoData* renderInfoData = renderInfos[i];
                // 文字列型
                {
                    StringRenderInfo^ renderInfo = dynamic_cast<StringRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        u16 end = static_cast<u16>(renderInfo->ItemCount);
                        for (u16 j = 0; j < end; ++j)
                        {
                            StringRenderInfo::Item^ item = renderInfo->GetItem(j);
                            if ( item == nullptr )
                            {
                                setupResult = false;
                                break;
                            }

                            nw::g3d::res::ResNameData* nameData = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(stringChoiceDataFirst, stringChoiceDataSize);
                            nn::g3d::viewer::detail::SetupRenderInfoChoiceData* choice = ::GetPtr<nn::g3d::viewer::detail::SetupRenderInfoChoiceData>(&renderInfoData->ofsChoice);

                            choice->ofsString[j].Initialize(nameData->str);
                            choice->numChoice = end;

                            String^ copyValue = dynamic_cast<String^>(item->Choice->Clone());
                            stringChoiceDataSize += nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + copyValue->Length + 1, NW_G3D_EDIT_ALIGNMENT);
                            std::string name = marshal_as<std::string>(copyValue);
                            memcpy(nameData->str, name.c_str(), name.length());
                            nameData->len = static_cast<u32>(name.length());
                        }
                    }
                }
            }

            if (!setupResult)
            {
                bufferPtr = nullptr;
                return false;
            }

            setupResult = true;
            size_t stringDefaultDataSize = 0;
            nw::g3d::res::ResNameData* stringDefaultDataFirst = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(stringChoiceDataFirst, stringChoiceDataSize);
            for (u16 i = 0; i < packet->block.numRenderInfo; ++i)
            {
                nn::g3d::viewer::detail::SetupRenderInfoData* renderInfoData = renderInfos[i];
                // 文字列型
                {
                    StringRenderInfo^ renderInfo = dynamic_cast<StringRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        u16 end = static_cast<u16>(renderInfo->ValueCount);
                        for (u16 j = 0; j < end; ++j)
                        {
                            StringRenderInfo::Value^ value = renderInfo->GetValue(j);
                            if ( value == nullptr )
                            {
                                setupResult = false;
                                break;
                            }

                            nw::g3d::res::ResNameData* nameData = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(stringDefaultDataFirst, stringDefaultDataSize);
                            nn::g3d::viewer::detail::SetupRenderInfoDefaultData* defaultValue = ::GetPtr<nn::g3d::viewer::detail::SetupRenderInfoDefaultData>(&renderInfoData->ofsDefault);
                            defaultValue->ofsString[j].Initialize(nameData->str);
                            defaultValue->numDefault = end;

                            String^ copyValue = dynamic_cast<String^>(value->DefaultValue->Clone());
                            stringDefaultDataSize += nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + copyValue->Length + 1, NW_G3D_EDIT_ALIGNMENT);
                            std::string name = marshal_as<std::string>(copyValue);
                            memcpy(nameData->str, name.c_str(), name.length());
                            nameData->len = static_cast<u32>(name.length());
                        }
                    }
                }
            }

            if (!setupResult)
            {
                bufferPtr = nullptr;
                return false;
            }

            if (endianKind == TargetEndianKind::BigEndian)
            {
                // TODO: とりえあずビッグエンディアンはサポート外
                //nw::g3d::tool::Endian::Swap(packet);
                unexpected();
            }
        }
        break;
    default:
        unexpected();
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

bool RenderInfoCommandMaker::MakeQueryRenderInfoPacket(
    array<Byte>^ packetBuffer,
    UINT32 key,
    array<INT32>^ indices,
    array<IRenderInfo^>^ renderInfoArray,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }

    if (renderInfoArray == nullptr)
    {
        return false;
    }

    if (indices == nullptr)
    {
        return false;
    }

    return MakeQueryRenderInfoPacketImpl(
        packetBuffer,
        key,
        indices,
        renderInfoArray,
        endianKind);
}

bool RenderInfoCommandMaker::MakeEditStringRenderInfoPacket(
    array<Byte>^ packetBuffer,
    UINT32 modelObjKey,
    INT32 materialIndex,
    String^ labelName,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }

    if (String::IsNullOrEmpty(labelName))
    {
        return false;
    }

    std::string editLabelName = marshal_as<std::string>(labelName);

    size_t labelNameSize = editLabelName.length() + 1;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetRenderInfoEditInfoSize() + labelNameSize, NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    RenderInfoEditPacket* packet = reinterpret_cast<RenderInfoEditPacket*>(bufferPtr);

    memset(packet, 0, totalSize);

    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_SELECT_EDIT_RENDER_INFO_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());
    packet->info.modelKey = modelObjKey;
    packet->info.materialIndex = materialIndex;
    packet->info.labelOffset = sizeof(RenderInfoEditPacket);
    packet->info.slotIndex = -1;
    packet->info.itemIndex = -1;
    packet->info.type = static_cast<u32>(nw::g3d::res::ResRenderInfo::STRING);

    char* labelNamePtr = nw::g3d::ut::AddOffset<char>(packet, sizeof(RenderInfoEditPacket));

    std::memcpy(labelNamePtr, editLabelName.c_str(), editLabelName.length());

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

bool RenderInfoCommandMaker::MakeEditIntRenderInfoPacket(
    array<Byte>^ packetBuffer,
    UINT32 modelObjKey,
    INT32 materialIndex,
    String^ labelName,
    INT32 slotIndex,
    INT32 value,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }

    if (String::IsNullOrEmpty(labelName))
    {
        return false;
    }

    std::string editLabelName = marshal_as<std::string>(labelName);

    size_t labelNameSize = editLabelName.length() + 1;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetRenderInfoEditInfoSize() + labelNameSize, NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    RenderInfoEditPacket* packet = reinterpret_cast<RenderInfoEditPacket*>(bufferPtr);

    memset(packet, 0, totalSize);

    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_SELECT_EDIT_RENDER_INFO_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());
    packet->info.modelKey = modelObjKey;
    packet->info.materialIndex = materialIndex;
    packet->info.labelOffset = sizeof(RenderInfoEditPacket);
    packet->info.slotIndex = slotIndex;
    packet->info.iValue = value;
    packet->info.type = static_cast<u32>(nw::g3d::res::ResRenderInfo::INT);

    char* labelNamePtr = nw::g3d::ut::AddOffset<char>(packet, sizeof(RenderInfoEditPacket));

    std::memcpy(labelNamePtr, editLabelName.c_str(), editLabelName.length());

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

bool RenderInfoCommandMaker::MakeEditFloatRenderInfoPacket(
    array<Byte>^ packetBuffer,
    UINT32 modelObjKey,
    INT32 materialIndex,
    String^ labelName,
    INT32 slotIndex,
    float value,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }

    if (String::IsNullOrEmpty(labelName))
    {
        return false;
    }

    std::string editLabelName = marshal_as<std::string>(labelName);

    size_t labelNameSize = editLabelName.length() + 1;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetRenderInfoEditInfoSize() + labelNameSize, NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    RenderInfoEditPacket* packet = reinterpret_cast<RenderInfoEditPacket*>(bufferPtr);

    memset(packet, 0, totalSize);

    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_SELECT_EDIT_RENDER_INFO_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());
    packet->info.modelKey = modelObjKey;
    packet->info.materialIndex = materialIndex;
    packet->info.labelOffset = sizeof(RenderInfoEditPacket);
    packet->info.slotIndex = slotIndex;
    packet->info.fValue = value;
    packet->info.type = static_cast<u32>(nw::g3d::res::ResRenderInfo::FLOAT);

    char* labelNamePtr = nw::g3d::ut::AddOffset<char>(packet, sizeof(RenderInfoEditPacket));

    std::memcpy(labelNamePtr, editLabelName.c_str(), editLabelName.length());

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

bool RenderInfoCommandMaker::MakeUpdateRenderInfoPacket(
    array<Byte>^ packetBuffer,
    UINT32 modelObjKey,
    INT32 materialIndex,
    array<IRenderInfo^>^ renderInfoArray,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }

    if ( renderInfoArray == nullptr )
    {
        return false;
    }

    u32 totalSize =
        CalcUpdateRenderInfoPacketSize(renderInfoArray);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    switch (HIO::GetInstance()->TargetDeviceType)
    {
    case HIOBase::TargetType::Cafe:
    case HIOBase::TargetType::OldPc:
        {
            RenderInfoUpdatePacket* packet = reinterpret_cast<RenderInfoUpdatePacket*>(bufferPtr);

            memset(packet, 0, totalSize);

            packet->header = CommandUtility::GetPacketHeader();
            packet->header.command = EDIT_UPDATE_RENDER_INFO_COMMAND_FLAG;
            packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

            packet->block.modelKey = modelObjKey;
            packet->block.materialIndex = materialIndex;
            packet->block.renderInfoDataSize = packet->header.dataSize - GetRenderInfoUpdateSize();

            // 辞書、文字列プールの順番に格納
            u32 renderInfoCount = renderInfoArray->Length;

            nw::g3d::res::ResDicPatriciaData* dicData = reinterpret_cast<nw::g3d::res::ResDicPatriciaData*>(&packet->block.renderInfoData[0]);
            // memset で 0 になっている前提で他は値を初期化しない。
            dicData->numData = renderInfoCount;

            u32 dicDataSize = sizeof(nw::g3d::res::ResDicPatriciaData);
            u32 dicNodeSize = sizeof(nw::g3d::res::ResDicPatriciaData::Node) * renderInfoCount;

            size_t dicTotalSize =
                nw::g3d::tool::Align(dicDataSize + dicNodeSize, NW_G3D_EDIT_ALIGNMENT);

            nw::g3d::res::ResRenderInfoData* renderInfoFirst = nw::g3d::ut::AddOffset<nw::g3d::res::ResRenderInfoData>(dicData, dicTotalSize);

            size_t renderInfoArraySize = 0;
            for (u32 i = 0; i < renderInfoCount; ++i)
            {
                nw::g3d::res::ResRenderInfoData* renderInfo = nw::g3d::ut::AddOffset<nw::g3d::res::ResRenderInfoData>(renderInfoFirst, renderInfoArraySize);
                dicData->node[i+1].ofsData.set_ptr(renderInfo);
                u32 renderInfoSize = sizeof(nw::g3d::res::ResRenderInfoData) + sizeof(u32) * renderInfoArray[i]->ValueCount;
                renderInfoArraySize += nw::g3d::tool::Align(renderInfoSize, NW_G3D_EDIT_ALIGNMENT);
            }

            char* nameOffsetFirst = nw::g3d::ut::AddOffset<char>(renderInfoFirst, renderInfoArraySize);
            size_t namesSize = 0;
            for (u32 i = 0; i < renderInfoCount; ++i)
            {
                // 描画情報の名前が空の時点で不正なデータなので失敗
                if (renderInfoArray[i]->Name == String::Empty)
                {
                    return false;
                }

                nw::g3d::res::ResNameData* nameOffset = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(nameOffsetFirst, namesSize);
                namesSize += nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + renderInfoArray[i]->Name->Length + 1, NW_G3D_EDIT_ALIGNMENT);

                dicData->node[i+1].ofsName.set_ptr(nameOffset->str);
                dicData->node[i+1].ofsData.to_ptr<nw::g3d::res::ResRenderInfoData>()->ofsName.set_ptr(nameOffset->str);

                String^ copyName = dynamic_cast<String^>(renderInfoArray[i]->Name->Clone());
                std::string name = marshal_as<std::string>(copyName);
                memcpy(nameOffset->str, name.c_str(), name.length());
                nameOffset->len = static_cast<u32>(name.length());
            }

            bool setupResult = true;
            char* valueOffsetFirst = nw::g3d::ut::AddOffset<char>(nameOffsetFirst, namesSize);
            size_t stringTypeSize = 0;
            for (u32 i = 0; i < renderInfoCount; ++i)
            {
                nw::g3d::res::ResRenderInfoData* renderInfoData = dicData->node[i+1].ofsData.to_ptr<nw::g3d::res::ResRenderInfoData>();

                // 文字列型
                {
                    StringRenderInfo^ renderInfo = dynamic_cast<StringRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        u16 end = static_cast<u16>(renderInfo->ValueCount);
                        for (u16 j = 0; j < end; ++j)
                        {
                            StringRenderInfo::Value^ value = renderInfo->GetValue(j);
                            if ( value == nullptr )
                            {
                                setupResult = false;
                                break;
                            }

                            nw::g3d::res::ResNameData* valueOffset = nw::g3d::ut::AddOffset<nw::g3d::res::ResNameData>(valueOffsetFirst, stringTypeSize);
                            renderInfoData->ofsString[j].set_ptr(valueOffset->str);

                            String^ copyValue = dynamic_cast<String^>(value->EditedValue->Clone());
                            stringTypeSize += nw::g3d::tool::Align(sizeof(nw::g3d::res::ResNameData::LengthType) + copyValue->Length + 1, NW_G3D_EDIT_ALIGNMENT);
                            std::string name = marshal_as<std::string>(copyValue);
                            memcpy(valueOffset->str, name.c_str(), name.length());
                            valueOffset->len = static_cast<u32>(name.length());
                        }
                        renderInfoData->type = nw::g3d::res::ResRenderInfo::STRING;
                        renderInfoData->arrayLength = end;
                    }
                }

                // Int型
                {
                    IntRenderInfo^ renderInfo = dynamic_cast<IntRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        u16 end = static_cast<u16>(renderInfo->ValueCount);
                        for (u16 j = 0; j < end; ++j)
                        {
                            IntRenderInfo::Value^ value = renderInfo->GetValue(j);
                            if ( value == nullptr )
                            {
                                setupResult = false;
                                break;
                            }
                            renderInfoData->iValue[j] = value->EditedValue;
                        }
                        renderInfoData->type = nw::g3d::res::ResRenderInfo::INT;
                        renderInfoData->arrayLength = end;
                    }
                }

                // Float型
                {
                    FloatRenderInfo^ renderInfo = dynamic_cast<FloatRenderInfo^>(renderInfoArray[i]);
                    if (renderInfo != nullptr)
                    {
                        u16 end = static_cast<u16>(renderInfo->ValueCount);
                        for (u16 j = 0; j < end; ++j)
                        {
                            FloatRenderInfo::Value^ value = renderInfo->GetValue(j);
                            if ( value == nullptr )
                            {
                                setupResult = false;
                                break;
                            }
                            renderInfoData->fValue[j] = value->EditedValue;
                        }
                        renderInfoData->type = nw::g3d::res::ResRenderInfo::FLOAT;
                        renderInfoData->arrayLength = end;
                    }
                }
            }

            // 算出したサイズと設定して得たサイズが違う場合は失敗
            if (packet->block.renderInfoDataSize != dicTotalSize + renderInfoArraySize + namesSize + stringTypeSize)
            {
                return false;
            }

            nw::g3d::res::ResDicPatricia* dic = nw::g3d::res::ResDicPatricia::ResCast(dicData);
            nw::g3d::res::ResDicPatricia::BuildResult result = dic->Build();
            if (result != nw::g3d::res::ResDicPatricia::SUCCESS)
            {
                return false;
            }


            if (endianKind == TargetEndianKind::BigEndian)
            {
                nw::g3d::tool::Endian::Swap(packet);
            }
        }
        break;
    case HIOBase::TargetType::Htc:
        {
            nn::g3d::viewer::detail::RenderInfoUpdatePacket* packet = reinterpret_cast<nn::g3d::viewer::detail::RenderInfoUpdatePacket*>(bufferPtr);

            memset(packet, 0, totalSize);

            packet->header = CommandUtility::GetNnG3dViewerPacketHeader();
            packet->header.command = nn::g3d::viewer::detail::EDIT_UPDATE_RENDER_INFO_COMMAND_FLAG;
            packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

            packet->block.modelKey = modelObjKey;
            packet->block.materialIndex = materialIndex;

            u32 renderInfoCount = renderInfoArray->Length;

            nn::util::ResDic* pResDic = reinterpret_cast<nn::util::ResDic*>(&packet->block.renderInfoData[0]);
            nn::util::ResDicData* dicData = &pResDic->ToData();

            // memset で 0 になっている前提で他は値を初期化しない。
            dicData->count = renderInfoCount;

            size_t dicDataSize = sizeof(nn::util::ResDicData);
            size_t dicNodeSize = sizeof(nn::util::ResDicData::Entry) * renderInfoCount;
            size_t dicTotalSize = nw::g3d::tool::Align(dicDataSize + dicNodeSize, NW_G3D_EDIT_ALIGNMENT);

            size_t arrayDataSize = sizeof(nn::g3d::ResRenderInfoData) * renderInfoCount;
            size_t namePoolSize = GetNnG3dRenderInfoNamePoolSize(renderInfoArray);
            size_t renderInfoArraySize = GetNnG3dRenderInfoArraySize(renderInfoArray);
            size_t valueArrayOffsetSize = GetNnG3dRenderInfoValueArraySize(renderInfoArray);

            packet->block.renderInfoDicDataSize = static_cast<uint32_t>(dicTotalSize);
            packet->block.renderInfoArrayDataSize = static_cast<uint32_t>(renderInfoArraySize);

            // ResRenderInfo 辞書、描画情報配列、描画情報名文字列データ、値配列データ、文字列値プールの順番に格納
            nn::g3d::ResRenderInfoData* renderInfoFirst = nw::g3d::ut::AddOffset<nn::g3d::ResRenderInfoData>(dicData, dicTotalSize);
            void* nameOffsetFirst = nw::g3d::ut::AddOffset<void>(renderInfoFirst, renderInfoArraySize);
            void* valueOffsetFirst = nw::g3d::ut::AddOffset<void>(nameOffsetFirst, namePoolSize);
            void* stringValuePoolFirst = nw::g3d::ut::AddOffset<void>(valueOffsetFirst, valueArrayOffsetSize);
            {
                size_t renderInfoArrayOffset = 0;
                size_t nameDataOffset = 0;
                size_t valueDataOffset = 0;
                size_t stringValuePoolOffset = 0;

                // nn::util::ResDic は仕様上、entries[0] に空文字を入れておく必要がある
                {
                    nn::util::BinString* nameData = nw::g3d::ut::AddOffset<nn::util::BinString>(nameOffsetFirst, nameDataOffset);
                    nameData->Initialize(nn::util::string_view(""));
                    nameDataOffset += GetBinStringDataSize(nameData->GetLength());

                    // 辞書のキー設定
                    dicData->entries[0].pKey.SetOffset(&dicData->entries[0].pKey, nameData);
                }

                for (u32 i = 0; i < renderInfoCount; ++i)
                {
                    IRenderInfo^ sourceRenderInfo = renderInfoArray[i];
                    // 描画情報の名前が空の時点で不正なデータなので失敗
                    if (sourceRenderInfo->Name == String::Empty)
                    {
                        return false;
                    }

                    // 名前データを名前プールにコピー
                    nn::util::BinString* nameData = nw::g3d::ut::AddOffset<nn::util::BinString>(nameOffsetFirst, nameDataOffset);
                    nameDataOffset += GetNnG3dRenderInfoNameSize(sourceRenderInfo);
                    String^ copyName = dynamic_cast<String^>(sourceRenderInfo->Name->Clone());
                    std::string name = marshal_as<std::string>(copyName);
                    nameData->Initialize(nn::util::string_view(name.data()));

                    // 辞書のキー設定
                    dicData->entries[i+1].pKey.SetOffset(&dicData->entries[i+1].pKey, nameData);

                    // 描画情報のメンバを設定
                    {
                        nn::g3d::ResRenderInfoData* renderInfoData =
                            nw::g3d::ut::AddOffset<nn::g3d::ResRenderInfoData>(renderInfoFirst, renderInfoArrayOffset);

                        // 名前設定
                        renderInfoData->pName.SetOffset(&renderInfoData->pName, nameData);
                        renderInfoArrayOffset += GetNnG3dRenderInfoSize();

                        // 文字列型
                        {
                            StringRenderInfo^ renderInfo = dynamic_cast<StringRenderInfo^>(sourceRenderInfo);
                            if (renderInfo != nullptr)
                            {
                                u16 end = static_cast<u16>(renderInfo->ValueCount);
                                nn::util::BinPtrToString* valueArray = nw::g3d::ut::AddOffset<nn::util::BinPtrToString>(valueOffsetFirst, valueDataOffset);
                                valueDataOffset += GetNnG3dRenderInfoValueArraySize(sourceRenderInfo);
                                for (u16 j = 0; j < end; ++j)
                                {
                                    StringRenderInfo::Value^ value = renderInfo->GetValue(j);
                                    if ( value == nullptr )
                                    {
                                        return false;
                                    }

                                    nn::util::BinString* stringValueData = nw::g3d::ut::AddOffset<nn::util::BinString>(stringValuePoolFirst, stringValuePoolOffset);
                                    String^ copyValue = dynamic_cast<String^>(value->EditedValue->Clone());
                                    stringValuePoolOffset += GetBinStringDataSize(copyValue->Length);
                                    std::string stringValueSource = marshal_as<std::string>(copyValue);
                                    stringValueData->Initialize(nn::util::string_view(stringValueSource.data()));

                                    valueArray[j].SetOffset(&valueArray[j], stringValueData);
                                }
                                renderInfoData->stringArray.SetOffset(&renderInfoData->stringArray, valueArray);
                                renderInfoData->type = nn::g3d::ResRenderInfo::Type_String;
                                renderInfoData->arrayLength = end;
                            }
                        }

                        // Int型
                        {
                            IntRenderInfo^ renderInfo = dynamic_cast<IntRenderInfo^>(sourceRenderInfo);
                            if (renderInfo != nullptr)
                            {
                                u16 end = static_cast<u16>(renderInfo->ValueCount);
                                int* valueArray = nw::g3d::ut::AddOffset<int>(valueOffsetFirst, valueDataOffset);
                                valueDataOffset += GetNnG3dRenderInfoValueArraySize(sourceRenderInfo);
                                for (u16 j = 0; j < end; ++j)
                                {
                                    IntRenderInfo::Value^ value = renderInfo->GetValue(j);
                                    if ( value == nullptr )
                                    {
                                        return false;
                                    }

                                    valueArray[j] = value->EditedValue;
                                }
                                renderInfoData->intValueArray.SetOffset(&renderInfoData->intValueArray, valueArray);
                                renderInfoData->type = nn::g3d::ResRenderInfo::Type_Int;
                                renderInfoData->arrayLength = end;
                            }
                        }

                        // Float型
                        {
                            FloatRenderInfo^ renderInfo = dynamic_cast<FloatRenderInfo^>(sourceRenderInfo);
                            if (renderInfo != nullptr)
                            {
                                u16 end = static_cast<u16>(renderInfo->ValueCount);
                                float* valueArray = nw::g3d::ut::AddOffset<float>(valueOffsetFirst, valueDataOffset);
                                valueDataOffset += GetNnG3dRenderInfoValueArraySize(sourceRenderInfo);
                                for (u16 j = 0; j < end; ++j)
                                {
                                    FloatRenderInfo::Value^ value = renderInfo->GetValue(j);
                                    if ( value == nullptr )
                                    {
                                        return false;
                                    }

                                    valueArray[j] = value->EditedValue;
                                }
                                renderInfoData->floatValueArray.SetOffset(&renderInfoData->floatValueArray, valueArray);
                                renderInfoData->type = nn::g3d::ResRenderInfo::Type_Float;
                                renderInfoData->arrayLength = end;
                            }
                        }
                    }
                }
            }

            // リロケート後にビルドしないといけないので、ビルドはランタイムで行う

            if (endianKind == TargetEndianKind::BigEndian)
            {
                NN_G3D_NOT_IMPLEMENTED();
                //nw::g3d::tool::Endian::Swap(packet);
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }


    return true;
}

}}} // namespace NintendoWare.G3d.Edit
