﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include "CommandUtility.h"
#include "PreviewCommandMaker.h"
#include "Utility.h"
#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>

using namespace nw::g3d::edit::detail;

namespace {

u32 GetModelLayoutRecvBlockSize()
{
    return sizeof(ModelLayoutRecvBlock);
}

u32 GetModelLayoutEditBlockSize()
{
    return sizeof(ModelLayoutEditBlock);
}

u32 GetBondBindEditBlockSize()
{
    return sizeof(BondBindEditBlock);
}

u32 GetShapeLodLevelEditBlockSize()
{
    return sizeof(ShapeLodLevelEditBlock);
}

}

namespace NintendoWare {
namespace G3d {
namespace Edit {

int PreviewCommandMaker::CalcQueryModelLayoutPacketSize()
{
    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetModelLayoutRecvBlockSize(), NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}
int PreviewCommandMaker::CalcEditModelLayoutPacketSize()
{
    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetModelLayoutEditBlockSize(), NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}
int PreviewCommandMaker::CalcEditBoneBindPacketSize()
{
    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetBondBindEditBlockSize(), NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}
int PreviewCommandMaker::CalcEditShapeLodLevelPacketSize()
{
    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + GetShapeLodLevelEditBlockSize(), NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

bool PreviewCommandMaker::MakeQueryModelLayoutPacket(
    array<System::Byte>^ packetBuffer,
    UINT32 modelObjKey,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    int totalSize = CalcQueryModelLayoutPacketSize();
    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    ModelLayoutRecvPacket* packet = reinterpret_cast<ModelLayoutRecvPacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_RECV_MODEL_LAYOUT_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.modelKey = modelObjKey;
    packet->block.padding = 0;

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }

    bufferPtr = nullptr;
    return true;
}

bool PreviewCommandMaker::MakeEditModelLayoutPacket(
    array<System::Byte>^ packetBuffer,
    UINT32 modelObjKey,
    bool isBind,
    Math::Vector3^ scale,
    Math::Vector3^ rotate,
    Math::Vector3^ translate,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    if (scale == nullptr || rotate == nullptr || translate == nullptr)
    {
        return false;
    }

    int totalSize = CalcEditModelLayoutPacketSize();
    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    ModelLayoutEditPacket* packet = reinterpret_cast<ModelLayoutEditPacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_MODEL_LAYOUT_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.modelKey = modelObjKey;
    packet->block.isBind = isBind ? 1 : 0;

    for (int i = 0; i < 3; ++i)
    {
        packet->block.scale.a[i] = scale[i];
        packet->block.rotate.a[i] = rotate[i];
        packet->block.translate.a[i] = translate[i];
    }

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }

    bufferPtr = nullptr;
    return true;
}

bool PreviewCommandMaker::MakeEditBoneBindPacket(
    array<System::Byte>^ packetBuffer,
    UINT32 modelObjKey,
    UINT32 parentModelObjKey,
    INT32 parentBoneIndex,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    int totalSize = CalcEditBoneBindPacketSize();
    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    BondBindEditPacket* packet = reinterpret_cast<BondBindEditPacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_MODEL_BONE_BIND_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.childModelKey = modelObjKey;
    packet->block.parentModelKey = parentModelObjKey;
    packet->block.parentBoneIndex = parentBoneIndex;

    packet->block.isBind = parentBoneIndex != -1 ? 1 : 0;

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }

    bufferPtr = nullptr;
    return true;
}

bool PreviewCommandMaker::MakeEditShapeLodLevelPacket(
    array<System::Byte>^ packetBuffer,
    UINT32 modelObjKey,
    INT32 lodLevel,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    int totalSize = CalcEditShapeLodLevelPacketSize();
    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    ShapeLodLevelEditPacket* packet = reinterpret_cast<ShapeLodLevelEditPacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command =
        (lodLevel >= 0) ? EDIT_SET_SHAPE_LOD_LEVEL_COMMAND_FLAG : EDIT_RESET_SHAPE_LOD_LEVEL_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.modelKey = modelObjKey;
    packet->block.lodLevel = lodLevel;

    if (endianKind == TargetEndianKind::BigEndian)
    {
        nw::g3d::tool::Endian::Swap(packet);
    }

    bufferPtr = nullptr;
    return true;
}

}}} // namespace NintendoWare.G3d.Edit
