﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include "msclr/marshal.h"
#include "msclr/marshal_windows.h"
#include "msclr/marshal_cppstd.h"
#include "msclr/marshal_atl.h"

#include "MaterialCommandMaker.h"
#include "CommandMaker.h"
#include "CommandUtility.h"
#include "Utility.h"
#include <nw/g3d/res/g3d_ResUtility.h>

using namespace System;
using namespace msclr::interop;
using namespace nw::g3d::edit::detail;

namespace
{
const int MATRIX_MIN_COUNT = 4;
const int MATRIX_MAX_COUNT = 16;

}

namespace NintendoWare { namespace G3d { namespace Edit {

int MaterialCommandMaker::CalcEditMaterialSamplerValuePacketSize(array<INT32>^ indices)
{
    if (indices == nullptr)
    {
        return -1;
    }

    if (indices->Length <= 0)
    {
        return -1;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditSamplerValue) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

int MaterialCommandMaker::CalcEditMaterialVector4ValuePacketSize(array<INT32>^ indices)
{
    if (indices == nullptr)
    {
        return -1;
    }

    if (indices->Length <= 0)
    {
        return -1;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditVector4Value) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

int MaterialCommandMaker::CalcEditMaterialShaderParamVectorValuePacketSize(array<INT32>^ indices)
{
    if (indices == nullptr)
    {
        return -1;
    }

    if (indices->Length <= 0)
    {
        return -1;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditShaderParamVectorValue) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

int MaterialCommandMaker::CalcEditMaterialShaderParamMatrixValuePacketSize(array<INT32>^ indices)
{
    if (indices == nullptr)
    {
        return -1;
    }

    if (indices->Length <= 0)
    {
        return -1;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditShaderParamMatrixValue) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);
    return static_cast<int>(totalSize);
}

generic<typename ValueType>
bool MaterialCommandMaker::MakeEditMaterialValuePacket(
    array<Byte>^ packetBuffer,
    EditTargetKind editTargetKind,
    UINT32 key,
    array<INT32>^ indices,
    ValueType value,
    TargetEndianKind endianKind)
{
    return CommandMaker::MakeEditValuePacket(
        packetBuffer,
        EDIT_MATERIAL_COMMAND_FLAG,
        editTargetKind,
        key,
        indices,
        value,
        endianKind);
}

bool MaterialCommandMaker::MakeEditMaterialValueU16Packet(
    array<Byte>^ packetBuffer,
    EditTargetKind editTargetKind,
    UINT32 key,
    array<INT32>^ indices,
    u16 highValue,
    u16 lowValue,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    if (indices == nullptr)
    {
        return false;
    }

    if (indices->Length <= 0)
    {
        return false;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditSimpleValue) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    EditValuePacket* packet = reinterpret_cast<EditValuePacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_MATERIAL_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.key = key;
    packet->block.editTargetKind = editTargetKind;
    packet->block.length = static_cast<u32>(packet->header.dataSize - CommandUtility::GetEditValueInfoBlockSize());
    packet->block.indexSize = indices->Length;

    EditValueBlock* valueBlock = reinterpret_cast<EditValueBlock*>(&packet[1]);

    for (u32 i = 0, end = packet->block.indexSize; i < end; ++i)
    {
        valueBlock->index[i] = indices[i];
    }

    packet->block.valueKind = U32_VALUE;
    valueBlock->value.uHigh = highValue;
    valueBlock->value.uLow = lowValue;

    if (endianKind == TargetEndianKind::BigEndian)
    {
        u32 indexSize = packet->block.indexSize;
        nw::g3d::tool::Endian::Swap(packet);

        nw::g3d::tool::Endian::Swap(&valueBlock->value.uHigh);
        nw::g3d::tool::Endian::Swap(&valueBlock->value.uLow);

        nw::g3d::tool::Endian::SwapIndices(valueBlock, indexSize);
    }

    bufferPtr = nullptr;
    return true;
}

generic<typename ValueType>
bool MaterialCommandMaker::MakeEditBoneValuePacket(
    array<Byte>^ packetBuffer,
    EditTargetKind editTargetKind,
    UINT32 key,
    array<INT32>^ indices,
    ValueType value,
    TargetEndianKind endianKind)
{
    return CommandMaker::MakeEditValuePacket(
        packetBuffer,
        EDIT_BONE_COMMAND_FLAG,
        editTargetKind,
        key,
        indices,
        value,
        endianKind);
}

generic<typename ValueType>
bool MaterialCommandMaker::MakeEditMaterialSamplerValuePacket(
    array<Byte>^ packetBuffer,
    EditTargetKind editTargetKind,
    UINT32 key,
    array<INT32>^ indices,
    INT32 samplerIndex,
    ValueType value,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    if (indices == nullptr)
    {
        return false;
    }

    if (indices->Length <= 0)
    {
        return false;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditSamplerValue) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    EditValuePacket* packet = reinterpret_cast<EditValuePacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_MATERIAL_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.key = key;
    packet->block.editTargetKind = editTargetKind;
    packet->block.length = static_cast<u32>(packet->header.dataSize - CommandUtility::GetEditValueInfoBlockSize());
    packet->block.indexSize = indices->Length;

    EditSamplerValueBlock* valueBlock = reinterpret_cast<EditSamplerValueBlock*>(&packet[1]);

    for (u32 i = 0, end = packet->block.indexSize; i < end; ++i)
    {
        valueBlock->index[i] = indices[i];
    }

    valueBlock->value.samplerIndex = samplerIndex;

    if (ValueType::typeid == bool::typeid)
    {
        packet->block.valueKind = BOOL_VALUE;
        bool bValue = Convert::ToBoolean(value);
        if (bValue)
        {
            valueBlock->value.uValue = 1;
        }
        else
        {
            valueBlock->value.uValue = 0;
        }
    }
    else if (ValueType::typeid == int::typeid)
    {
        packet->block.valueKind = S32_VALUE;
        valueBlock->value.sValue = Convert::ToInt32(value);
    }
    else if (ValueType::typeid == unsigned int::typeid)
    {
        packet->block.valueKind = U32_VALUE;
        valueBlock->value.uValue = Convert::ToUInt32(value);
    }
    else if (ValueType::typeid == float::typeid)
    {
        packet->block.valueKind = F32_VALUE;
        valueBlock->value.fValue = Convert::ToSingle(value);
    }
    else
    {
        // pin_ptr 解除
        bufferPtr = nullptr;
        // 対象外の型は失敗にする
        return false;
    }

    if (endianKind == TargetEndianKind::BigEndian)
    {
        u32 indexSize = packet->block.indexSize;
        nw::g3d::tool::Endian::Swap(packet);

        nw::g3d::tool::Endian::SwapValue(valueBlock);
        nw::g3d::tool::Endian::SwapIndices(valueBlock, indexSize);
    }
    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

bool MaterialCommandMaker::MakeEditMaterialVector4ValueFloatPacket(
    array<Byte>^ packetBuffer,
    EditTargetKind editTargetKind,
    UINT32 key,
    array<INT32>^ indices,
    array<float>^ value,
    TargetEndianKind endianKind)
{
    if (packetBuffer == nullptr)
    {
        return false;
    }
    if (packetBuffer->Length <= 0)
    {
        return false;
    }

    if (value == nullptr)
    {
        return false;
    }

    // 配列数が４未満なら失敗
    if (value->Length < 4)
    {
        return false;
    }

    if (indices == nullptr)
    {
        return false;
    }

    if (indices->Length <= 0)
    {
        return false;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditVector4Value) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);

    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    EditValuePacket* packet = reinterpret_cast<EditValuePacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_MATERIAL_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.key = key;
    packet->block.editTargetKind = editTargetKind;
    packet->block.length = static_cast<u32>(packet->header.dataSize - CommandUtility::GetEditValueInfoBlockSize());
    packet->block.indexSize = indices->Length;

    EditVector4ValueBlock* valueBlock = reinterpret_cast<EditVector4ValueBlock*>(&packet[1]);

    for (u32 i = 0, end = packet->block.indexSize; i < end; ++i)
    {
        valueBlock->index[i] = indices[i];
    }

    packet->block.valueKind = VECTOR4_VALUE;
    for (int i = 0; i < 4; ++i)
    {
        valueBlock->value.fValue[i] = value[i];
    }

    if (endianKind == TargetEndianKind::BigEndian)
    {
        u32 indexSize = packet->block.indexSize;
        nw::g3d::tool::Endian::Swap(packet);

        nw::g3d::tool::Endian::SwapValue(valueBlock);
        nw::g3d::tool::Endian::SwapIndices(valueBlock, indexSize);
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

generic<typename ValueType>
bool MaterialCommandMaker::MakeEditMaterialShaderParamVectorValuePacket(
    array<Byte>^ packetBuffer,
    EditTargetKind editTargetKind,
    UINT32 key,
    String^ paramName,
    array<INT32>^ indices,
    array<ValueType>^ value,
    TargetEndianKind endianKind)
{
    if (paramName == nullptr || indices == nullptr || value == nullptr)
    {
        return false;
    }

    if (indices->Length <= 0)
    {
        return false;
    }

    if (value->Length > 4 || value->Length <= 0)
    {
        return false;
    }

    std::string shaderParamName = marshal_as<std::string>(paramName);

    size_t paramNameSize = shaderParamName.length();
    if (paramNameSize >= NW_G3D_EDIT_FILENAME_MAX)
    {
        return false;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditShaderParamVectorValue) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);
    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    EditValuePacket* packet = reinterpret_cast<EditValuePacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_MATERIAL_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.key = key;
    packet->block.editTargetKind = editTargetKind;
    packet->block.length = static_cast<u32>(packet->header.dataSize - CommandUtility::GetEditValueInfoBlockSize());
    packet->block.indexSize = indices->Length;

    EditShaderParamVectorValueBlock* valueBlock = reinterpret_cast<EditShaderParamVectorValueBlock*>(&packet[1]);

    memset(&valueBlock->value.value, 0, sizeof(EditVector4Value));
    memset(&valueBlock->value.paramName, 0, NW_G3D_EDIT_FILENAME_MAX);
    std::memcpy(&valueBlock->value.paramName, shaderParamName.c_str(), shaderParamName.length());

    for (u32 i = 0, end = packet->block.indexSize; i < end; ++i)
    {
        valueBlock->index[i] = indices[i];
    }

    u32 dimention = value->Length;
    if (ValueType::typeid == bool::typeid)
    {
        packet->block.valueKind = BOOL_VALUE;

        for(u32 i = 0; i < dimention; ++i)
        {
            bool bValue = Convert::ToBoolean(value[i]);
            if (bValue)
            {
                valueBlock->value.value.uValue[i] = 1;
            }
            else
            {
                valueBlock->value.value.uValue[i] = 0;
            }
        }
    }
    else if (ValueType::typeid == int::typeid)
    {
        packet->block.valueKind = S32_VALUE;

        for (u32 i = 0; i < dimention; ++i)
        {
            valueBlock->value.value.sValue[i] = Convert::ToInt32(value[i]);
        }
    }
    else if (ValueType::typeid == unsigned int::typeid)
    {
        packet->block.valueKind = U32_VALUE;

        for (u32 i = 0; i < dimention; ++i)
        {
            valueBlock->value.value.uValue[i] = Convert::ToUInt32(value[i]);
        }
    }
    else if (ValueType::typeid == float::typeid)
    {
        packet->block.valueKind = F32_VALUE;

        for (u32 i = 0; i < dimention; ++i)
        {
            valueBlock->value.value.fValue[i] = Convert::ToSingle(value[i]);
        }
    }
    else
    {
        // pin_ptr 解除
        bufferPtr = nullptr;
        // 対象外の型は失敗にする
        return false;
    }

    if (endianKind == TargetEndianKind::BigEndian)
    {
        u32 indexSize = packet->block.indexSize;
        nw::g3d::tool::Endian::Swap(packet);

        if (ValueType::typeid == bool::typeid)
        {
            nw::g3d::tool::Endian::SwapValue(valueBlock, true);
        }
        else
        {
            nw::g3d::tool::Endian::SwapValue(valueBlock, false);
        }

        nw::g3d::tool::Endian::SwapIndices(valueBlock, indexSize);
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

bool MaterialCommandMaker::MakeEditMaterialShaderParamMatrixValuePacket(
    array<Byte>^ packetBuffer,
    EditTargetKind editTargetKind,
    UINT32 key,
    String^ paramName,
    array<INT32>^ indices,
    array<float>^ value,
    TargetEndianKind endianKind)
{
    if (paramName == nullptr || indices == nullptr || value == nullptr)
    {
        return false;
    }

    if (indices->Length <= 0)
    {
        return false;
    }

    if (value->Length < MATRIX_MIN_COUNT || value->Length > MATRIX_MAX_COUNT)
    {
        return false;
    }

    std::string shaderParamName = marshal_as<std::string>(paramName);

    u32 paramNameSize = static_cast<u32>(shaderParamName.length());
    if (paramNameSize >= NW_G3D_EDIT_FILENAME_MAX)
    {
        return false;
    }

    size_t indexByteSize = indices->Length * sizeof(s32);
    size_t valueBlockSize = sizeof(EditShaderParamMatrixValue) + indexByteSize;

    size_t totalSize =
        nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + CommandUtility::GetEditValueInfoBlockSize() + valueBlockSize, NW_G3D_EDIT_ALIGNMENT);
    if (packetBuffer->Length != totalSize)
    {
        return false;
    }

    pin_ptr<u8> bufferPtr = &packetBuffer[0];

    EditValuePacket* packet = reinterpret_cast<EditValuePacket*>(bufferPtr);
    packet->header = CommandUtility::GetPacketHeader();
    packet->header.command = EDIT_MATERIAL_COMMAND_FLAG;
    packet->header.dataSize = static_cast<s32>(totalSize - CommandUtility::GetPacketHeaderSize());

    packet->block.key = key;
    packet->block.editTargetKind = editTargetKind;
    packet->block.length = static_cast<u32>(packet->header.dataSize - CommandUtility::GetEditValueInfoBlockSize());
    packet->block.indexSize = indices->Length;

    EditShaderParamMatrixValueBlock* valueBlock = reinterpret_cast<EditShaderParamMatrixValueBlock*>(&packet[1]);

    memset(&valueBlock->value.value, 0, sizeof(EditMatrix44Value));
    memset(&valueBlock->value.paramName, 0, NW_G3D_EDIT_FILENAME_MAX);
    std::memcpy(&valueBlock->value.paramName, shaderParamName.c_str(), shaderParamName.length());

    for (u32 i = 0, end = packet->block.indexSize; i < end; ++i)
    {
        valueBlock->index[i] = indices[i];
    }

    packet->block.valueKind = F32_VALUE;
    for (int i = 0; i < value->Length; ++i)
    {
        valueBlock->value.value.a[i] = value[i];
    }

    if (endianKind == TargetEndianKind::BigEndian)
    {
        u32 indexSize = packet->block.indexSize;
        nw::g3d::tool::Endian::Swap(packet);
        nw::g3d::tool::Endian::SwapValue(valueBlock);
        nw::g3d::tool::Endian::SwapIndices(valueBlock, indexSize);
    }

    // pin_ptr 解除
    bufferPtr = nullptr;
    return true;
}

}}} // namespace NintendoWare.G3d.Edit
