﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// g3dhiolib.h

#pragma once

#include "CommDevice.h"
#include "RecvInfo.h"
#include "MaterialShaderInfo.h"
#include <nw/g3d/edit/g3d_EditDefs.h>

namespace NintendoWare {
namespace G3d {
namespace Edit {

public ref class HIO : public HIOBase
{
public:
    // TODO: HIO の API はほぼ全て bool 戻り値で、失敗原因をハンドリングできないので、いつか Result に置き換える
    enum class Result
    {
        Success,
        InvalidArgument,
        FailedQueueCommand,
        UnsupportedPlatform,
        InvalidFilePath,
    };

    enum class TextureClampKind : int
    {
        Repeat,
        Mirror,
        Clamp,
        MirrorOnce,
    };

    //! @deprecated GX2 依存なので廃止されます。TextureFilterKind を使用して下さい。
    enum class TexturFilterKind : int
    {
        Point =     GX2_TEX_XY_FILTER_POINT,
        Linear =    GX2_TEX_XY_FILTER_BILINEAR,
    };

    enum class TextureFilterKind : int
    {
        None, //!< GX2 のミップマップフィルタのみ設定可
        Point,
        Linear,
    };

    //! @deprecated GX2 依存なので廃止されます。TextureFilterKind を使用して下さい。
    enum class TextureMipFilterKind : int
    {
        None =      GX2_TEX_MIP_FILTER_NO_MIP,
        Point =     GX2_TEX_MIP_FILTER_POINT,
        Linear =    GX2_TEX_MIP_FILTER_LINEAR,
    };

    //! @deprecated GX2 依存なので廃止されます。int を使用して下さい。
    enum class TextureAnisoRatioKind : int
    {
        Aniso1 =     GX2_TEX_ANISO_1_TO_1,
        Aniso2 =     GX2_TEX_ANISO_2_TO_1,
        Aniso4 =     GX2_TEX_ANISO_4_TO_1,
        Aniso8 =     GX2_TEX_ANISO_8_TO_1,
        Aniso16 =    GX2_TEX_ANISO_16_TO_1,
    };

    enum class MaterialDisplayFaceKind : int
    {
        Both,
        Front,
        Back,
        None,
    };

    enum class RenderStateMode : int
    {
        Custom =        nw::g3d::res::ResRenderState::MODE_CUSTOM,
        Opaque =        nw::g3d::res::ResRenderState::MODE_OPAQUE,
        Mask =          nw::g3d::res::ResRenderState::MODE_ALPHAMASK,
        Translucent =   nw::g3d::res::ResRenderState::MODE_TRANSLUCENT,
    };

    enum class BlendMode : int
    {
        None =  nw::g3d::res::ResRenderState::BLEND_NONE,
        Color = nw::g3d::res::ResRenderState::BLEND_COLOR,
        Logic = nw::g3d::res::ResRenderState::BLEND_LOGICAL
    };

    enum class DepthTestFuncKind : int
    {
        Never =         GX2_COMPARE_NEVER,
        Less =          GX2_COMPARE_LESS,
        Equal =         GX2_COMPARE_EQUAL,
        LessEqual =     GX2_COMPARE_LEQUAL,
        Greater =       GX2_COMPARE_GREATER,
        NotEqual =      GX2_COMPARE_NOTEQUAL,
        GreaterEqual =  GX2_COMPARE_GEQUAL,
        Always =        GX2_COMPARE_ALWAYS,
    };

    enum class AlphaTestFuncKind : int
    {
        Never =         GX2_COMPARE_NEVER,
        Less =          GX2_COMPARE_LESS,
        Equal =         GX2_COMPARE_EQUAL,
        LessEqual =     GX2_COMPARE_LEQUAL,
        Greater =       GX2_COMPARE_GREATER,
        NotEqual =      GX2_COMPARE_NOTEQUAL,
        GreaterEqual =  GX2_COMPARE_GEQUAL,
        Always =        GX2_COMPARE_ALWAYS,
    };

    enum class LogicOpKind : int
    {
        Clear =     GX2_LOGIC_OP_CLEAR,
        Set =       GX2_LOGIC_OP_SET,
        Copy =      GX2_LOGIC_OP_COPY,
        InvCopy =   GX2_LOGIC_OP_INVCOPY,
        NoOp =      GX2_LOGIC_OP_NOOP,
        Inv =       GX2_LOGIC_OP_INV,
        And =       GX2_LOGIC_OP_AND,
        NotAnd =    GX2_LOGIC_OP_NAND,
        Or =        GX2_LOGIC_OP_OR,
        NotOr =     GX2_LOGIC_OP_NOR,
        XOr =       GX2_LOGIC_OP_XOR,
        Equiv =     GX2_LOGIC_OP_EQUIV,
        RevAnd =    GX2_LOGIC_OP_REVAND,
        InvAnd =    GX2_LOGIC_OP_INVAND,
        RevOr =     GX2_LOGIC_OP_REVOR,
        InvOr =     GX2_LOGIC_OP_INVOR,
    };

    enum class BlendCombineKind : int
    {
        Add =           GX2_BLEND_COMBINE_ADD,
        SrcMinusDst =   GX2_BLEND_COMBINE_SRC_MINUS_DST,
        Min =           GX2_BLEND_COMBINE_MIN,
        Max =           GX2_BLEND_COMBINE_MAX,
        DstMinusSrc =   GX2_BLEND_COMBINE_DST_MINUS_SRC,
    };

    enum class ColorBlendFuncKind : int
    {
        Zero =                  GX2_BLEND_ZERO,
        One =                   GX2_BLEND_ONE,
        SrcColor =              GX2_BLEND_SRC_COLOR,
        OneMinusSrcColor =      GX2_BLEND_ONE_MINUS_SRC_COLOR,
        SrcAlpha =              GX2_BLEND_SRC_ALPHA,
        OneMinusSrcAlpha =      GX2_BLEND_ONE_MINUS_SRC_ALPHA,
        DstAlpha =              GX2_BLEND_DST_ALPHA,
        OneMinusDstAlpha =      GX2_BLEND_ONE_MINUS_DST_ALPHA,
        DstColor =              GX2_BLEND_DST_COLOR,
        OneMinusDstColor =      GX2_BLEND_ONE_MINUS_DST_COLOR,
        SrcAlphaSaturate =      GX2_BLEND_SRC_ALPHA_SATURATE,
        ConstantColor =         GX2_BLEND_CONSTANT_COLOR,
        OneMinusConstantColor = GX2_BLEND_ONE_MINUS_CONSTANT_COLOR,
        Src1Color =             GX2_BLEND_SRC1_COLOR,
        OneMinusSrc1Color =     GX2_BLEND_ONE_MINUS_SRC1_COLOR,
        Src1Alpha =             GX2_BLEND_SRC1_ALPHA,
        OneMinusSrc1Alpha =     GX2_BLEND_ONE_MINUS_SRC1_ALPHA,
        ConstantAlpha =         GX2_BLEND_CONSTANT_ALPHA,
        OneMinusConstantAlpha = GX2_BLEND_ONE_MINUS_CONSTANT_ALPHA,
    };

    enum class AlphaBlendFuncKind : int
    {
        Zero =                  GX2_BLEND_ZERO,
        One =                   GX2_BLEND_ONE,
        SrcAlpha =              GX2_BLEND_SRC_ALPHA,
        OneMinusSrcAlpha =      GX2_BLEND_ONE_MINUS_SRC_ALPHA,
        DstAlpha =              GX2_BLEND_DST_ALPHA,
        OneMinusDstAlpha =      GX2_BLEND_ONE_MINUS_DST_ALPHA,
        SrcAlphaSaturate =      GX2_BLEND_SRC_ALPHA_SATURATE,
        Src1Alpha =             GX2_BLEND_SRC1_ALPHA,
        OneMinusSrc1Alpha =     GX2_BLEND_ONE_MINUS_SRC1_ALPHA,
        ConstantAlpha =         GX2_BLEND_CONSTANT_ALPHA,
        OneMinusConstantAlpha = GX2_BLEND_ONE_MINUS_CONSTANT_ALPHA,
    };

    property BYTE MajorVersion
    {
        BYTE get() { return NN_G3D_VERSION_EDITMAJOR; }
    }

    property BYTE MinorVersion
    {
        BYTE get() { return NN_G3D_VERSION_EDITMINOR; }
    }

    property BYTE MicroVersion
    {
        BYTE get() { return NN_G3D_VERSION_EDITMICRO; }
    }

    property BYTE BugFixVersion
    {
        BYTE get() { return NN_G3D_VERSION_EDITBUGFIX; }
    }

public:
    static bool InitializeHIO();
    static void FinalizeHIO();

    static HIO^ GetInstance() { return s_Instance; }

    HtcResult^ Connect();
    bool Disconnect();

    void Poll();

    bool ChangeCommDevice(HIOBase::TargetType targetType, System::String^ peerType);

    // システム
    bool ResetRuntimeState();

    //! @brief ランタイムのデバッグログを有効化するコマンドをエンキューします。
    Result EnableRuntimeDebugLog();

    //! @brief ランタイムのデバッグログを無効化するコマンドをエンキューします。
    Result DisableRuntimeDebugLog();

    // 全体をアンロード
    bool UnloadAll();

    // @param targetModel 編集対象のモデルです。
    // @param fileData ロードするファイルの情報です。fileNameはフルパスで指定する必要があります。
    bool LoadModel(IEditModelTarget^ targetModel, FileData^ fileData);

    bool UnloadModel(IEditModelTarget^ targetModel);
    bool SendAttachModel(IEditModelTarget^ targetModel, FileData^ fileData);
    bool CancelAttachModel(uint modelObjKey);

    //! @brief モデルをリロードします。
    //! @param targetModel リロードするモデルです。
    //! @param bfresFileData モデルのバイナリファイルデータです。
    Result ReloadModel(IEditModelTarget^ targetModel, FileData^ bfresFileData);

    //! @brief 各マテリアル毎のシェーダー更新と一緒にモデルをリロードします。
    //! @param targetModel リロードするモデルです。
    //! @param bfresFileData モデルのバイナリファイルデータです。
    //! @param bfshaFileNames モデルに必要なバリエーション分のシェーダーバイナリリソースのファイルパスです。
    //! @param materialShaderInfos　各マテリアルに対応するシェーダー情報の配列です。
    //!        bfshaFileNames のインデックス、マテリアルがシェーダー最適化の対象になっているか、
    //!        シェーダー最適化の対象になっている場合にシェーダー生成がスキップされたかを格納します。
    //!        配列の長さはマテリアル数になります。
    //!
    Result ReloadModel(
        IEditModelTarget^ targetModel,
        FileData^ bfresFileData,
        array<System::String^>^ bfshaFileNames,
        array<MaterialShaderInfo^>^ materialShaderInfos);

    // @param targetTexture 編集対象のテクスチャです。
    // @param fileData ロードするバイナリリソースファイルの情報です。フルパスで指定します。
    Result LoadTexture(IEditTextureTarget^ targetTexture, FileData^ fileData);

    // @param targetTexture 編集対象のテクスチャです。
    Result UnloadTexture(IEditTextureTarget^ targetTexture);

    // @param targetTexture 編集対象のテクスチャです。
    // @param fileData ロードするバイナリリソースファイルの情報です。フルパスで指定します。
    Result ReloadTexture(IEditTextureTarget^ targetTexture, FileData^ fileData);

    // @brief テクスチャのバインド情報をセットします。ランタイムではこの関数により送信されたバインド情報でテクスチャをバインドします。
    //        既にバインド情報が送信されていた場合、この関数で送信したバインド情報でランタイムのバインド情報は上書きされます。
    Result SetTextureBindingsForAnimation(IEditTarget^ bindTarget, array<IEditTextureTarget^>^ textures);
    Result SetTextureBindingsForModel(IEditModelTarget^ bindTarget, array<IEditTextureTarget^>^ textures);

    // シェーダー関連

    bool SendAttachShaderArchive(IEditShaderArchiveTarget^ targetShaderArchive);
    bool CancelAttachShaderArchive(uint shaderArchiveKey);

    bool UnloadShaderArchive(IEditShaderArchiveTarget^ targetShaderArchive);
    bool EditShadingModels(IEditShaderArchiveTarget^ targetShaderArchive, array<INT32>^ shadingModelIndices);

    bool ReloadShaderProgram(IEditShaderArchiveTarget^ targetShaderArchive, INT32 shadingModelIndex, INT32 shaderProgramIndex, System::String^ binaryFileName);

    // ここからリアルタイム編集関連

    bool EditMaterialVisibility(bool isVisible);
    bool EditBoneVisibility(bool isVisible);
    bool EditBoneBillboard(BillboardKind kind);


    //----------------------------------------
    // レンダーステート編集関連

    bool EditMaterialDisplayFace(MaterialDisplayFaceKind displayFaceKind);
    bool EditMaterialRenderStateMode(RenderStateMode mode);
    bool EditMaterialBlendMode(BlendMode mode);

    // DepthTest
    bool EditMaterialDepthTestEnable(bool enable);
    bool EditMaterialDepthTestWriteEnable(bool enable);
    bool EditMaterialDepthTestFunc(DepthTestFuncKind kind);

    // AlphaTest
    bool EditMaterialAlphaTestEnable(bool enable);
    bool EditMaterialAlphaTestFunc(AlphaTestFuncKind kind);
    bool EditMaterialAlphaTestValue(float value);

    // カラーブレンド
    bool EditMaterialColorCombine(BlendCombineKind kind);
    bool EditMaterialAlphaCombine(BlendCombineKind kind);

    bool EditMaterialColorSrcBlend(ColorBlendFuncKind kind);
    bool EditMaterialColorDstBlend(ColorBlendFuncKind kind);
    bool EditMaterialAlphaSrcBlend(AlphaBlendFuncKind kind);
    bool EditMaterialAlphaDstBlend(AlphaBlendFuncKind kind);

    bool EditMaterialConstantColor(float r, float g, float b, float a, Drawing::ColorChannelFlag useChannelFlag);

    // 論理ブレンド
    bool EditMaterialLogicOp(LogicOpKind kind);

    // サンプラ編集関連
    bool EditMaterialSamplerWrapU(INT32 samplerIndex, TextureClampKind kind);
    bool EditMaterialSamplerWrapV(INT32 samplerIndex, TextureClampKind kind);
    bool EditMaterialSamplerWrapW(INT32 samplerIndex, TextureClampKind kind);

    //! @deprecated GX2 依存なので廃止されます。EditMaterialSamplerFilterを使用して下さい。
    bool EditMaterialSamplerMagFilter(INT32 samplerIndex, TexturFilterKind kind);

    //! @deprecated GX2 依存なので廃止されます。EditMaterialSamplerFilterを使用して下さい。
    bool EditMaterialSamplerMinFilter(INT32 samplerIndex, TexturFilterKind kind);

    //! @deprecated GX2 依存なので廃止されます。EditMaterialSamplerFilterを使用して下さい。
    bool EditMaterialSamplerMipmapFilter(INT32 samplerIndex, TextureMipFilterKind kind);

    //! @deprecated GX2 依存なので廃止されます。EditMaterialSamplerFilterを使用して下さい。
    bool EditMaterialSamplerMaxAniso(INT32 samplerIndex, TextureAnisoRatioKind kind);

    bool EditMaterialSamplerMinLOD(INT32 samplerIndex, float minLOD);
    bool EditMaterialSamplerMaxLOD(INT32 samplerIndex, float maxLOD);
    bool EditMaterialSamplerLODBias(INT32 samplerIndex, float bias);

    //! @brief フィルターモードを設定します。
    //! @detail maxAnisotropy が 1 のときは異方性フィルタリングは無効になります。
    //!         maxAnisotropy が 1 より大きい場合は縮小、拡大、ミップフィルターはすべてリニアになります。
    Result EditMaterialSamplerFilter(
        int samplerIndex,
        TextureFilterKind minFilter,
        TextureFilterKind magFilter,
        TextureFilterKind mipmapFilter,
        int maxAnisotropy);


    //----------------------------------------
    // シェーダー割り当て関連

    bool EditMaterialShaderParameter(System::String^ parameterName, bool value);
    bool EditMaterialShaderParameter(System::String^ parameterName, int value);
    bool EditMaterialShaderParameter(System::String^ parameterName, uint value);
    bool EditMaterialShaderParameter(System::String^ parameterName, float value);

    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector2^ value, Math::VectorChannelFlag useChannelFlag);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector3^ value, Math::VectorChannelFlag useChannelFlag);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector4^ value, Math::VectorChannelFlag useChannelFlag);

    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector2i^ value, Math::VectorChannelFlag useChannelFlag);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector3i^ value, Math::VectorChannelFlag useChannelFlag);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector4i^ value, Math::VectorChannelFlag useChannelFlag);

    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector2u^ value, Math::VectorChannelFlag useChannelFlag);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector3u^ value, Math::VectorChannelFlag useChannelFlag);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector4u^ value, Math::VectorChannelFlag useChannelFlag);

    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector2b^ value, Math::VectorChannelFlag useChannelFlag);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector3b^ value, Math::VectorChannelFlag useChannelFlag);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Vector4b^ value, Math::VectorChannelFlag useChannelFlag);

    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Matrix22^ value);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Matrix23^ value);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Matrix24^ value);

    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Matrix32^ value);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Matrix33^ value);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Matrix34^ value);

    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Matrix42^ value);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Matrix43^ value);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::Matrix44^ value);

    bool EditMaterialShaderParameter(System::String^ parameterName, Math::TextureSRT2D^ value);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::TextureSRT3D^ value);
    bool EditMaterialShaderParameter(System::String^ parameterName, Math::TextureSRT2DEx^ value);


    //----------------------------------------
    // 描画情報要求・編集

    // 描画情報を要求します。
    bool QueryRenderInfo(array<IRenderInfo^>^ renderInfoArray);

    // 描画情報の更新
    bool UpdateRenderInfo(array<IRenderInfo^>^ renderInfoArray);
    // System::String型の描画情報編集
    // - 通常は UpdateRenderInfo を行った後に呼びます。
    bool EditStringRenderInfo(System::String^ renderInfoName);
    // Int型の描画情報編集
    bool EditIntRenderInfo(System::String^ renderInfoName, int slotNumber, int itemValue);
    // Float型の描画情報編集
    bool EditFloatRenderInfo(System::String^ renderInfoName, int slotNumber, float itemValue);

    //----------------------------------------
    // アニメーション関連

    bool SelectTargetAnimation(IEditTarget^ targetAnimation);

    //! @brief モデルのアニメーション単位でアニメーションの再生、停止を切り替えます。
    bool EditAnimationPause(IEditModelTarget^ targetModel, IEditTarget^ targetAnimation, float frame, bool enable);

    bool EditShaderParamCurve(UINT32 materialNameIndex, UINT32 paramAnimIndex, UINT32 componentIndex, int curveIndex, ICurve^ curve);

    bool EditTexPatternCurve(UINT32 materialNameIndex, int curveIndex, ICurve^ curve);
    bool EditMatVisibilityCurve(UINT32 animIndex, int curveIndex, ICurve^ curve);
    bool EditBoneVisibilityCurve(UINT32 animIndex, int curveIndex, ICurve^ curve);
    bool EditMaterialCurve(UINT32 materialNameIndex, UINT32 componentIndex, int curveIndex, ICurve^ curve);

    bool EditShapeCurve(UINT32 vertexShapeAnimIndex, int curveIndex, ICurve^ curve);

    bool FlushShaderParamCurves();

    bool BindAnimations(IEditModelTarget^ targetModel, array<IEditTarget^>^ bindAnimations);
    bool UnbindAnimations(IEditModelTarget^ targetModel, array<IEditTarget^>^ bindAnimations);
    bool UnbindAnimationAll(IEditModelTarget^ targetModel);

    bool LoadShaderParamAnimation(IEditTarget^ targetAnimation, FileData^ fileData);
    bool UnloadShaderParamAnimation(IEditTarget^ targetAnimation);
    bool ReloadShaderParamAnimation(IEditTarget^ targetAnimation, FileData^ fileData);

    bool LoadSkeletalAnimation(IEditTarget^ targetAnimation, FileData^ fileData);
    bool LoadSkeletalAnimation(IEditTarget^ targetAnimation, FileData^ fileData, System::String^ retargetingHostModelName);
    bool UnloadSkeletalAnimation(IEditTarget^ targetAnimation);
    bool ReloadSkeletalAnimation(IEditTarget^ targetAnimation, FileData^ fileData);

    //! @brief リターゲティングのホストモデルを設定します。設定を解除する場合は retargetHostModel に nullptr を渡します。
    void SetRetargetingHostModel(IEditTarget^ targetAnimation, IEditModelTarget^ retargetHostModel);
    void SetPlayMotionMirroringEnabled(IEditTarget^ targetAnimation, bool enabled);

    bool LoadColorAnimation(IEditTarget^ targetAnimation, FileData^ fileData);
    bool UnloadColorAnimation(IEditTarget^ targetAnimation);
    bool ReloadColorAnimation(IEditTarget^ targetAnimation, FileData^ fileData);

    bool LoadTextureSRTAnimation(IEditTarget^ targetAnimation, FileData^ fileData);
    bool UnloadTextureSRTAnimation(IEditTarget^ targetAnimation);
    bool ReloadTextureSRTAnimation(IEditTarget^ targetAnimation, FileData^ fileData);

    bool LoadBoneVisibilityAnimation(IEditTarget^ targetAnimation, FileData^ fileData);
    bool UnloadBoneVisibilityAnimation(IEditTarget^ targetAnimation);
    bool ReloadBoneVisibilityAnimation(IEditTarget^ targetAnimation, FileData^ fileData);

    bool LoadMaterialVisibilityAnimation(IEditTarget^ targetAnimation, FileData^ fileData);
    bool UnloadMaterialVisibilityAnimation(IEditTarget^ targetAnimation);
    bool ReloadMaterialVisibilityAnimation(IEditTarget^ targetAnimation, FileData^ fileData);

    bool LoadTexturePatternAnimation(IEditTarget^ targetAnimation, FileData^ fileData);
    bool UnloadTexturePatternAnimation(IEditTarget^ targetAnimation);
    bool ReloadTexturePatternAnimation(IEditTarget^ targetAnimation, FileData^ fileData);

    bool LoadMaterialAnimation(IEditTarget^ targetAnimation, FileData^ fileData);
    bool UnloadMaterialAnimation(IEditTarget^ targetAnimation);
    bool ReloadMaterialAnimation(IEditTarget^ targetAnimation, FileData^ fileData);

    bool LoadShapeAnimation(IEditTarget^ targetAnimation, FileData^ fileData);
    bool UnloadShapeAnimation(IEditTarget^ targetAnimation);
    bool ReloadShapeAnimation(IEditTarget^ targetAnimation, FileData^ fileData);

    // シーンアニメーション関連
    bool LoadSceneAnimation(IEditSceneAnimTarget^ targetAnimation, FileData^ fileData);
    bool UnloadSceneAnimation(IEditSceneAnimTarget^ targetAnimation);
    bool ReloadSceneAnimation(IEditSceneAnimTarget^ targetAnimation, FileData^ fileData);

    bool BindSceneAnimations(array<IEditSceneAnimTarget^>^ bindAnimations);
    bool UnbindSceneAnimations(array<IEditSceneAnimTarget^>^ bindAnimations);

    bool EditCameraAnimCurve(UINT32 cameraIndex, int targetIndex, int curveIndex, ICurve^ curve);
    bool EditLightAnimCurve(UINT32 lightIndex, int targetIndex, int curveIndex, ICurve^ curve);
    bool EditFogAnimCurve(UINT32 fogIndex, int targetIndex, int curveIndex, ICurve^ curve);


    //----------------------------------------
    // アニメーションフレームコントロール関連

    // プレイポリシーを設定します。
    bool SetPlayPolicy(PlayPolicy policy);

    // フレームコントロールを再生します。
    bool PlayFrameCtrl(float frame);

    // フレームコントロールを停止します。
    bool StopFrameCtrl(float frame);

    // フレームステップを設定します。
    bool SetFrameStep(float value);

    // フレームカウントを設定します。
    bool SetFrameCount(float value);

    // 開始フレームを設定します。
    bool SetStartFrame(float frame);


    //----------------------------------------
    // 配置情報の要求・編集

    // 配置情報を要求します。
    bool QueryModelLayout(bool isBind);

    bool EditModelLayout(bool isBind, Math::Vector3^ scale, Math::Vector3^ rotate, Math::Vector3^ translate);
    bool EditBoneBind(IEditModelTarget^ targetModel, int boneIndex);

    bool ClearBoneBind(array<IEditModelTarget^>^ targetModels);


    //----------------------------------------
    // その他編集関連

    // TODO: EditSelectTarget に置き換えられたら削除する。
    bool EditBlinkMaterials();

    //! @brief 選択を実行します。HIOBase.Selected～が選択対象に使われます。
    bool EditSelectTarget(SelectTargetKind selectTargetkind);

    bool EditSetShapeLodLevel(int level);
    bool EditResetShapeLodLevel();

    virtual ICommandTransaction^ BeginCommand() override;
    virtual ICommandTransaction^ BeginCommand(bool sync) override;

    //! @brief ユーザースクリプトをランタイムに送信します。
    void ExecuteModelUserScript(
        IEditModelTarget^ targetModel,
        System::String^ scriptText);

    // ここまでリアルタイム編集関連

private:
    bool EditMaterialIniternal();

    bool CheckFile(System::String^ fileName);

    generic<typename ValueType>
    bool EditMaterialShaderParameterVector_(System::String^ parameterName, array<ValueType>^ values, nw::g3d::edit::detail::EditTargetKind editTargetKind);

    bool EditMaterialShaderParameterMatrix_(System::String^ parameterName, array<float>^ values, nw::g3d::edit::detail::EditTargetKind editTargetKind);

    bool EditMaterialSamplerValueU32(INT32 samplerIndex, UINT32 value, nw::g3d::edit::detail::EditTargetKind editTargetKind);
    bool EditMaterialSamplerValueFloat(INT32 samplerIndex, float value, nw::g3d::edit::detail::EditTargetKind editTargetKind);

    bool LoadAnimation(nw::g3d::edit::detail::FileDataKind fileKind, IEditTarget^ targetAnimation, FileData^ fileData);
    bool UnloadAnimation(nw::g3d::edit::detail::FileDataKind fileKind, IEditTarget^ targetAnimation);
    bool ReloadAnimation(nw::g3d::edit::detail::FileDataKind fileKind, IEditTarget^ targetAnimation, FileData^ fileData);

    generic<typename ValueType>
    bool EditMaterialValue(ValueType value, nw::g3d::edit::detail::EditTargetKind editTargetKind);

    generic<typename ValueType>
    bool EditBoneValue(ValueType value, nw::g3d::edit::detail::EditTargetKind editTargetKind);

    TargetEndianKind GetTargetEndian()
    {
        TargetEndianKind kind = TargetEndianKind::LittleEndian;
        switch (TargetDeviceType)
        {
        case TargetType::Cafe:
            kind = TargetEndianKind::BigEndian;
            break;
        }
        return kind;
    }

    bool EditMaterialSamplerWrap(INT32 samplerIndex, HIO::TextureClampKind kind, nw::g3d::edit::detail::EditTargetKind target);

private:
    HIO();

    static HIO^     s_Instance = nullptr;

    IEditTarget^    m_TargetAnimation;

    NintendoWare::G3d::Edit::RecvInfo^       m_RecvInfo;

    System::String^ PeerType;
};

}}} // namespace NintendoWare.G3d.Edit
