﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// これは メイン DLL ファイルです。

#include "stdafx.h"

#include "HIO.h"
#include "CafeCommDevice.h"
#include "PCCommDevice.h"
#include "HtcCommDevice.h"

#include <nw/g3d/res/g3d_ResSkeleton.h>
#include "LoadModelCommand.h"
#include "TextureCommand.h"
#include "LoadAnimCommand.h"
#include "LoadShaderCommand.h"
#include "RenderInfoCommand.h"
#include "EditMaterialCommand.h"
#include "EditBoneCommand.h"
#include "EditAnimCommand.h"
#include "EditShaderCommand.h"
#include "PreviewCommand.h"
#include "EditModelAnimCommand.h"
#include "EditSceneAnimCommand.h"
#include "SystemCommand.h"
#include "ExecuteUserScriptCommand.h"

#include <nn/gfx/gfx_Enum.h>
#include <nn/diag.h>

using namespace System;
using namespace System::ComponentModel;
using namespace System::Diagnostics;
using namespace System::Collections::Generic;

using namespace nw::g3d::edit::detail;
using namespace NintendoWare::G3d::Edit;

namespace {
    ref class DummyTarget : public NintendoWare::G3d::Edit::IEditTarget
    {
    public:
        /// <summary>
        /// ターゲットを特定するためのキーを取得します。
        /// </summary>
        virtual property uint Key;

        /// <summary>
        /// 編集対象のResFileを特定するキーを取得します。
        /// </summary>
        virtual property uint ResFileKey;

        /// <summary>
        /// ターゲットがアタッチされているか状態かを判定します。
        /// </summary>
        virtual property bool IsAttached;

        /// <summary>
        /// ターゲットが削除されている状態かを判定します。
        /// </summary>
        virtual property bool IsDeleted;

        /// <summary>
        /// ランタイムで使用する情報をリセットします。
        /// </summary>
        virtual void ResetStatus()
        {
        }
    };

    // コマンド作成のための一時的な編集モデル
    ref class DummyModel : public DummyTarget, NintendoWare::G3d::Edit::IEditModelTarget
    {
    public:
        DummyModel()
            : textures(gcnew System::Collections::Generic::List<IEditTextureTarget^>())
        {
        }

        /// <summary>
        /// 編集対象のResModelを特定するキーを取得します。
        /// </summary>
        virtual property uint ResModelKey;

        /// <summary>
        /// 編集対象のModelObjを特定するキーを取得します。
        /// </summary>
        virtual property uint ModelObjKey;

        /// <summary>
        /// アタッチから実行されたモデルか判定します。
        /// </summary>
        virtual property bool IsSendAttached;

        /// <summary>
        /// バインドされたテクスチャのキーです。
        /// </summary>
        virtual property array<IEditTextureTarget^>^ BoundTextures
        {
            array<IEditTextureTarget^>^ get()
            {
                return textures->ToArray();
            }
        }

    private:
        System::Collections::Generic::List<IEditTextureTarget^>^ textures;
    };

    // コマンド作成のための一時的な編集シェーダーアーカイブ
    ref class DummyShaderArchive : public DummyTarget, NintendoWare::G3d::Edit::IEditShaderArchiveTarget
    {
    public:
        virtual property uint ShaderArchiveKey;
    };


    GX2TexAnisoRatio ConvertToGx2TexAnisoRatio(int maxAnisotropy)
    {
        switch (maxAnisotropy)
        {
        case 1:
            return GX2_TEX_ANISO_1_TO_1;
        case 2:
            return GX2_TEX_ANISO_2_TO_1;
        case 4:
            return GX2_TEX_ANISO_4_TO_1;
        case 8:
            return GX2_TEX_ANISO_8_TO_1;
        case 16:
            return GX2_TEX_ANISO_16_TO_1;
        default:
            return static_cast<GX2TexAnisoRatio>(-1);
        }
    }

    GX2TexClamp ConvertToGx2Clamp(HIO::TextureClampKind kind)
    {
        switch (kind)
        {
        case HIO::TextureClampKind::Repeat: return GX2_TEX_CLAMP_WRAP;
        case HIO::TextureClampKind::Mirror: return GX2_TEX_CLAMP_MIRROR;
        case HIO::TextureClampKind::Clamp: return GX2_TEX_CLAMP_CLAMP;
        case HIO::TextureClampKind::MirrorOnce: return GX2_TEX_CLAMP_MIRROR_ONCE;
        default:
            return static_cast<GX2TexClamp>(-1);
        }
    }

    nn::gfx::TextureAddressMode ConvertToGfxTextureAddress(HIO::TextureClampKind kind)
    {
        switch (kind)
        {
        case HIO::TextureClampKind::Repeat: return nn::gfx::TextureAddressMode_Repeat;
        case HIO::TextureClampKind::Mirror: return nn::gfx::TextureAddressMode_Mirror;
        case HIO::TextureClampKind::Clamp: return nn::gfx::TextureAddressMode_ClampToEdge;
        case HIO::TextureClampKind::MirrorOnce: return nn::gfx::TextureAddressMode_MirrorClampToEdge;
        default:
            return static_cast<nn::gfx::TextureAddressMode>(-1);
        }
    }

    GX2TexXYFilterType ConvertToGx2XyFilter(HIO::TextureFilterKind kind)
    {
        switch (kind)
        {
        case HIO::TextureFilterKind::Point: return GX2_TEX_XY_FILTER_POINT;
        case HIO::TextureFilterKind::Linear: return GX2_TEX_XY_FILTER_BILINEAR;
        case HIO::TextureFilterKind::None: // GX2 ではサポート外
        default:
            return static_cast<GX2TexXYFilterType>(-1);
        }
    }

    GX2TexMipFilterType ConvertToGx2MipFilter(HIO::TextureFilterKind kind)
    {
        switch (kind)
        {
        case HIO::TextureFilterKind::None: return GX2_TEX_MIP_FILTER_NO_MIP;
        case HIO::TextureFilterKind::Point: return GX2_TEX_MIP_FILTER_POINT;
        case HIO::TextureFilterKind::Linear: return GX2_TEX_MIP_FILTER_LINEAR;
        default:
            return static_cast<GX2TexMipFilterType>(-1);
        }
    }

    nn::gfx::detail::FilterModeBit ConvertToGfxFilterBit(HIO::TextureFilterKind kind)
    {
        switch (kind)
        {
        case HIO::TextureFilterKind::Point: return nn::gfx::detail::FilterModeBit_Point;
        case HIO::TextureFilterKind::Linear: return nn::gfx::detail::FilterModeBit_Linear;
        case HIO::TextureFilterKind::None:
            // gfx では今のところ 0 は公式ではサポート外だが nvn に設定はできる。DirectX とかだと問題になる可能性があるので注意
            return static_cast<nn::gfx::detail::FilterModeBit>(0);
        default:
            return static_cast<nn::gfx::detail::FilterModeBit>(-1);
        }
    }
}

namespace NintendoWare { namespace G3d { namespace Edit {

//-------------------------------------------------------------------------------------------------
bool HIO::InitializeHIO()
{
    WSADATA wsaData = { 0 };
    int result = WSAStartup(MAKEWORD(2,2), &wsaData);

    if (result == SOCKET_ERROR)
    {
        return false;
    }

    if (s_Instance != nullptr)
    {
        return false;
    }

    s_Instance = gcnew HIO();

    return true;
}

//-------------------------------------------------------------------------------------------------
void HIO::FinalizeHIO()
{
    if (CommDevice::m_ConnectionManager != nullptr)
    {
        //CommDevice::m_ConnectionManager->Stop();
        CommDevice::m_ConnectionManager = nullptr;
    }

    if (s_Instance != nullptr)
    {
        s_Instance->CloseAll();
        s_Instance = nullptr;
    }

    WSACleanup();
}

//-------------------------------------------------------------------------------------------------
HtcResult^ HIO::Connect()
{
    NintendoWare::G3d::Edit::HtcResult^ result = gcnew NintendoWare::G3d::Edit::HtcResult();
    if (Device == nullptr)
    {
        result->ResultCode = HtcResultCode::NullDevice;
        return result;
    }

    HtcResult^ connectResult = Device->Connect();
    if (!connectResult->IsSuccess)
    {
        // 接続できなかった場合の二重処理
        Device->CloseConnection();
        Device->OpenConnection();

        connectResult = Device->Connect();
        if (!connectResult->IsSuccess)
        {
            UpdateConnection();
            return connectResult;
        }
    }

    UpdateConnection();

    result->ResultCode = HtcResultCode::Success;
    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::Disconnect()
{
    this->SendCommandQueue->Clear();

    this->Close();
    this->Open();
    this->UpdateConnection();
    return !this->IsConnected;
}

//-------------------------------------------------------------------------------------------------
void HIO::Poll()
{
    if (Device == nullptr)
    {
        return;
    }

    Device->Poll();

    if (Device->GetRuntimeState() == RuntimeState::Closed)
    {
        Disconnect();
        return;
    }

    // 以下はinternalにフレンドアセンブリでアクセスしていて、
    // インテリセンスがそれをカバーできなくてエラー表示になっているだけ
    if (Device->IsBeginFreezeReceived)
    {
        m_RecvInfo->CallBeginFreezeReceivedEvent(gcnew BeginFreezeReceivedArgs());
        Device->ResetBeginFreezeFlag();
    }
    else if (Device->IsEndFreezeReceived)
    {
        m_RecvInfo->CallEndFreezeReceivedEvent(gcnew EndFreezeReceivedArgs());
        Device->ResetEndFreezeFlag();
    }
    else if (Device->IsAttachReceived)
    {
        switch(Device->AttachKind)
        {
        case nw::g3d::edit::detail::ATTACH_MODEL:
            {
                m_RecvInfo->CallAttachModelReceivedEvent(
                    gcnew AttachModelReceivedArgs(
                    Device->ToolKey, Device->ModelObjKey, Device->FileName, Device->AttachPathName));
            }
            break;
        case nw::g3d::edit::detail::ATTACH_SHADER_ARCHIVE:
            {
                m_RecvInfo->CallAttachShaderArchiveReceivedEvent(
                    gcnew AttachShaderArchiveReceivedArgs(
                    Device->ToolKey,
                    Device->ShaderArchiveKey,
                    Device->FileName,
                    Device->AttachPathName,
                    Device->IsAttachShaderArchiveBinary));
            }
            break;
        }
        Device->ResetAttachFlag();
    }
    else if (Device->IsDetachReceived)
    {
        switch(Device->AttachKind)
        {
        case nw::g3d::edit::detail::DETACH_MODEL:
            {
                m_RecvInfo->CallDetachModelReceivedEvent(
                    gcnew DetachModelReceivedArgs(Device->ToolKey, Device->ModelObjKey));
            }
            break;
        case nw::g3d::edit::detail::DETACH_SHADER_ARCHIVE:
            {
                m_RecvInfo->CallDetachShaderArchiveReceivedEvent(
                    gcnew DetachShaderArchiveReceivedArgs(Device->ToolKey, Device->ShaderArchiveKey));
            }
            break;
        }
        Device->ResetDetachFlag();
    }
    else if (Device->IsFileLoaded)
    {
        if (Device->IsModelFileLoaded)
        {
            m_RecvInfo->CallModelFileLoadedEvent(
                gcnew ModelFileLoadedEventArgs(
                Device->ToolKey, Device->ModelObjKey, Device->ResFileKey, Device->ResModelKey));
        }
        else
        {
            m_RecvInfo->CallFileLoadedEvent(
                gcnew FileLoadedEventArgs(
                Device->ToolKey, Device->ResFileKey));
        }
        Device->ResetFileLoadFlag();
    }
    else if (Device->IsFileReloaded)
    {
        if (Device->IsModelFileReloaded)
        {
            ModelFileReloadedEventArgs^ args = gcnew ModelFileReloadedEventArgs();
            args->ToolKey = Device->ToolKey;
            args->ResFileKey = Device->ResFileKey;
            args->ModelObjKey = Device->ModelObjKey;
            args->ResModelKey = Device->ResModelKey;
            m_RecvInfo->CallModelFileReloadedEvent(args);
        }
        else
        {
            m_RecvInfo->CallFileReloadedEvent(
                gcnew FileReloadedEventArgs(
                Device->NewResFileKey, Device->ResFileKey));

        }
        Device->ResetFileLoadFlag();
    }
    else if (Device->IsRenderInfoReceived)
    {
        Device->SetupRenderInfoPack();

        RenderInfoReceivedArgs^ args = gcnew RenderInfoReceivedArgs();
        args->RenderInfoPackQueue->AddRenderInfoPack(Device->RecvRenderInfoPack);
        args->SentRenderInfoPack = args->RenderInfoPackQueue->GetFirst();
        m_RecvInfo->CallRenderInfoReceivedEvent(args);

        Device->ResetRenderInfoFlag();
    }
    else if (Device->IsModelLayoutReceived)
    {
        ModelLayoutReceivedArgs^ args = gcnew ModelLayoutReceivedArgs();
        args->Scale = Device->GetModelLayoutScale();
        args->Rotate = Device->GetModelLayoutRotate();
        args->Translate = Device->GetModelLayoutTranslate();
        args->ModelObjKey = Device->ModelObjKey;
        m_RecvInfo->CallModelLayoutReceivedEvent(args);
        Device->ResetModelLayoutFlag();
    }
    else if (Device->IsModifiedShaderProgramReceived)
    {
        // TODO: ModifiedShaderProgramRecvDataをなくす
        ModifiedShaderProgramRecvData^ data = Device->CreateModifiedShaderProgramData();
        {
            UpdatedShaderProgramReceivedArgs^ args = gcnew UpdatedShaderProgramReceivedArgs();
            args->ShaderArchiveKey = data->ShaderArchiveKey;
            args->ShadingModelIndex = data->ShadingModelIndex;
            args->ShaderProgramIndex = data->ShaderProgramIndex;
            args->AddShaderCompileInfos(data->ShaderCompileInfoArray);

            m_RecvInfo->CallUpdatedShaderProgramReceivedEvent(args);
        }
        Device->ResetModifiedShaderProgramFlag();
    }
    else if (Device->IsPickupReceived)
    {
        // TODO: PickupRecvDataをなくす
        PickupRecvData^ data = Device->CreatePickupData();

        {
            SelectionTargetReceivedArgs^ arg = gcnew SelectionTargetReceivedArgs();
            if (data->IsClearMaterialSelection)
            {
                arg->SelectionTargetInfoArray->Add(gcnew ClearMaterialSelectionInfo());
            }
            else
            {
                for each (MaterialSelectionInfo^ value in data->MaterialSelectionTargetArray)
                {
                    arg->SelectionTargetInfoArray->Add(value);
                }
            }

            m_RecvInfo->CallSelectionTargetReceivedEvent(arg);
        }

        Device->ResetPickupFlag();
    }
    else if (Device->IsPlayFrameCtrlReceived)
    {
        PlayFrameCtrlReceivedArgs^ args = gcnew PlayFrameCtrlReceivedArgs();
        args->Frame = Device->Frame;
        m_RecvInfo->CallPlayFrameCtrlReceivedEvent(args);
        Device->ResetPlayFrameCtrlFlag();
    }
    else if (Device->IsStopFrameCtrlReceived)
    {
        StopFrameCtrlReceivedArgs^ args = gcnew StopFrameCtrlReceivedArgs();
        args->Frame = Device->Frame;
        m_RecvInfo->CallStopFrameCtrlReceivedEvent(args);
        Device->ResetStopFrameCtrlFlag();
    }
    else if (Device->IsSendFrameReceived)
    {
        SendFrameReceivedArgs^ args = gcnew SendFrameReceivedArgs();
        args->Frame = Device->Frame;
        m_RecvInfo->CallSendFrameReceivedEvent(args);
        Device->ResetSendFrameFlag();
    }
    else if (Device->IsSendFrameStepReceived)
    {
        SendFrameStepReceivedArgs^ args = gcnew SendFrameStepReceivedArgs();
        args->FrameStep = Device->FrameStep;
        m_RecvInfo->CallSendFrameStepReceivedEvent(args);
        Device->ResetSendFrameStepFlag();
    }
    else if (Device->IsSendModelNextAnimReceived)
    {
        SendModelNextAnimReceivedArgs^ args = gcnew SendModelNextAnimReceivedArgs();
        args->ModelObjKey = Device->ModelObjKey;
        m_RecvInfo->CallSendModelNextAnimReceivedEvent(args);
        Device->ResetSendModelNextAnimFlag();
    }
    else if (Device->IsSendModelPrevAnimReceived)
    {
        SendModelPrevAnimReceivedArgs^ args = gcnew SendModelPrevAnimReceivedArgs();
        args->ModelObjKey = Device->ModelObjKey;
        m_RecvInfo->CallSendModelPrevAnimReceivedEvent(args);
        Device->ResetSendModelPrevAnimFlag();
    }
    else if (Device->IsAbnormalPacketReceived)
    {
        m_RecvInfo->CallAbnormalPacketReceivedEvent(gcnew AbnormalPacketReceivedArgs());
        Device->ResetAbnormalPacketFlag();
    }
    else if (Device->IsIncorrectVersionReceived)
    {
        IncorrectVersionReceivedArgs^ arg = gcnew IncorrectVersionReceivedArgs(
            m_RecvInfo->MajorVersion, m_RecvInfo->MinorVersion, m_RecvInfo->MicroVersion, m_RecvInfo->BugFixVersion);
        m_RecvInfo->CallIncorrectVersionReceivedEvent(arg);
        Device->ResetIncorrectVersionFlag();
    }
    else if (Device->HasRuntimeErrorReceived)
    {
        m_RecvInfo->CallRuntimeErrorOccuredEvent(Device->GetRuntimeError());
        Device->ResetRuntimeErrorFlag();
    }
    else if (Device->IsShowMessageRequested)
    {
        ShowMessageRequestedArgs^ arg = gcnew ShowMessageRequestedArgs();
        arg->Message = Device->Message;
        arg->MessageType = Device->MessageType;
        arg->MessageDestination = Device->MessageDestination;
        m_RecvInfo->CallShowMessageRequestedEvent(arg);
        Device->ResetShowMessageRequestedFlag();
    }
    else if (Device->IsCodePageChanged)
    {
        CodePageUpdateReceivedArgs^ arg = gcnew CodePageUpdateReceivedArgs();
        arg->CodePage = Device->CodePage;
        m_RecvInfo->CallCodePageUpdateReceivedEvent(arg);
        Device->ResetCodePageChangedFlag();
    }

    ExecuteSendCommandQueue();
}

//-------------------------------------------------------------------------------------------------
bool HIO::ChangeCommDevice( HIOBase::TargetType targetType, String^ peerType)
{
    if (TargetDeviceType == targetType && PeerType == peerType)
    {
        return false;
    }

    if (Device == nullptr)
    {
        return false;
    }

    Device->CloseConnection();
    Device = nullptr;

    switch (targetType)
    {
    case HIOBase::TargetType::Cafe:
        Device = gcnew CafeCommDevice(m_RecvInfo);
        // ストリームの取得タイミングが早すぎるため、つながらないときがある
        //Device = gcnew HtcCommDevice(m_RecvInfo, gcnew String("HIO"), gcnew String("Cafe"));
        break;
    case HIOBase::TargetType::Htc:
        Device = gcnew HtcCommDevice(m_RecvInfo, peerType, gcnew String(""));
        break;
    case HIOBase::TargetType::OldPc:
        Device = gcnew PCCommDevice(m_RecvInfo);
        break;
    }

    Device->OpenConnection();
    TargetDeviceType = targetType;
    PeerType = peerType;

    return true;
}

//-------------------------------------------------------------------------------------------------
bool HIO::ResetRuntimeState()
{
    TargetEndianKind kind = GetTargetEndian();
    SystemResetRuntimeStateCommand^ command =
        gcnew SystemResetRuntimeStateCommand(kind);
    return this->SendCommandQueue->PushBack(command);
}

HIO::Result HIO::EnableRuntimeDebugLog()
{
    SystemRuntimeDebugLogCommand^ command = gcnew SystemRuntimeDebugLogCommand(GetTargetEndian(), true);
    bool success = this->SendCommandQueue->PushBack(command);
    if (!success)
    {
        return HIO::Result::FailedQueueCommand;
    }

    return HIO::Result::Success;
}

HIO::Result HIO::DisableRuntimeDebugLog()
{
    SystemRuntimeDebugLogCommand^ command = gcnew SystemRuntimeDebugLogCommand(GetTargetEndian(), false);
    bool success = this->SendCommandQueue->PushBack(command);
    if (!success)
    {
        return HIO::Result::FailedQueueCommand;
    }

    return HIO::Result::Success;
}

//-------------------------------------------------------------------------------------------------
bool HIO::UnloadAll()
{
    TargetEndianKind kind = GetTargetEndian();
    UnloadAllCommand^ command =
        gcnew UnloadAllCommand(kind);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------

bool HIO::LoadModel(IEditModelTarget^ targetModel, FileData^ fileData)
{
    G3D_HTC_REQUIRES_NOT_NULL(targetModel);
    G3D_HTC_REQUIRES_NOT_NULL(fileData);
    G3D_HTC_REQUIRES(fileData->Alignment != 0, "Invalid alignment");

    if (!CheckFile(fileData->FileName))
    {
        return false;
    }

    TargetEndianKind endian = GetTargetEndian();
    {
        LoadModelCommand^ command = gcnew LoadModelCommand(endian, targetModel, fileData);
        bool success = this->SendCommandQueue->PushBack(command);
        if (!success)
        {
            return false;
        }
    }

    return true;
}

//-------------------------------------------------------------------------------------------------
bool HIO::UnloadModel(IEditModelTarget^ targetModel)
{
    if (targetModel == nullptr)
    {
        return false;
    }

    if (targetModel->ModelObjKey == 0)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    UnloadModelCommand^ command =
        gcnew UnloadModelCommand(kind, targetModel);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
HIO::Result HIO::LoadTexture(IEditTextureTarget^ targetTexture, FileData^ fileData)
{
    // TODO: ResetStatus()がLoadTextureをコールするスレッドと別スレッドでコールされる関係で、
    //       アンロードされたテクスチャでもResFileKeyが0のままテクスチャが渡される場合があるので、
    //       一旦事前条件からは外す。必要であれば正式な対処ができたらアサートを復活させる
    //G3D_HTC_REQUIRES(targetTexture->ResFileKey == 0, "ResFileKey must be zero");

    G3D_HTC_REQUIRES_NOT_NULL(fileData);

    if (TargetDeviceType != TargetType::Htc)
    {
        return HIO::Result::UnsupportedPlatform;
    }

    LoadTextureCommand^ command = gcnew LoadTextureCommand(GetTargetEndian(), targetTexture, fileData);
    bool success = this->SendCommandQueue->PushBack(command);
    if (!success)
    {
        return HIO::Result::FailedQueueCommand;
    }

    return HIO::Result::Success;
}

//-------------------------------------------------------------------------------------------------
HIO::Result HIO::ReloadTexture(IEditTextureTarget^ targetTexture, FileData^ fileData)
{
    G3D_HTC_REQUIRES(targetTexture->ResFileKey != 0, "ResFileKey must be zero");
    G3D_HTC_REQUIRES_NOT_NULL(fileData);
    if (TargetDeviceType != TargetType::Htc)
    {
        return HIO::Result::UnsupportedPlatform;
    }

    ReloadTextureCommand^ command = gcnew ReloadTextureCommand(GetTargetEndian(), targetTexture, fileData);
    bool success = this->SendCommandQueue->PushBack(command);
    if (!success)
    {
        return HIO::Result::FailedQueueCommand;
    }

    return HIO::Result::Success;
}

//-------------------------------------------------------------------------------------------------
HIO::Result HIO::UnloadTexture(IEditTextureTarget^ targetTexture)
{
    G3D_HTC_REQUIRES_NOT_NULL(targetTexture);
    G3D_HTC_REQUIRES(targetTexture->Key != 0, "key must not be zero");
    if (TargetDeviceType != TargetType::Htc)
    {
        return HIO::Result::UnsupportedPlatform;
    }

    UnloadTextureCommand^ command = gcnew UnloadTextureCommand(GetTargetEndian(), targetTexture);
    bool success = this->SendCommandQueue->PushBack(command);
    if (!success)
    {
        return HIO::Result::FailedQueueCommand;
    }

    return HIO::Result::Success;
}

HIO::Result HIO::SetTextureBindingsForAnimation(IEditTarget^ bindTarget, array<IEditTextureTarget^>^ textures)
{
    G3D_HTC_REQUIRES_NOT_NULL(bindTarget);

    TargetEndianKind endian = GetTargetEndian();
    BindTextureCommand^ command = gcnew BindTextureCommand(endian, bindTarget, textures);
    bool success = this->SendCommandQueue->PushBack(command);
    if (!success)
    {
        return HIO::Result::FailedQueueCommand;
    }

    return HIO::Result::Success;
}

HIO::Result HIO::SetTextureBindingsForModel(IEditModelTarget^ bindTarget, array<IEditTextureTarget^>^ textures)
{
    G3D_HTC_REQUIRES_NOT_NULL(bindTarget);

    TargetEndianKind endian = GetTargetEndian();
    BindTextureCommand^ command = gcnew BindTextureCommand(endian, bindTarget, textures);
    bool success = this->SendCommandQueue->PushBack(command);
    if (!success)
    {
        return HIO::Result::FailedQueueCommand;
    }

    return HIO::Result::Success;
}

//-------------------------------------------------------------------------------------------------

bool HIO::SendAttachModel(IEditModelTarget^ targetModel, FileData^ fileData)
{
    if (targetModel == nullptr)
    {
        return false;
    }

    if (fileData->Alignment==0)
    {
        return false;
    }

    if (!CheckFile(fileData->FileName))
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    SendAttachModelCommand^ command =
        gcnew SendAttachModelCommand(kind, targetModel, fileData);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------

bool HIO::CancelAttachModel(UInt32 modelObjKey)
{
    DummyModel^ targetModel = gcnew DummyModel();
    targetModel->ModelObjKey = modelObjKey;
    return UnloadModel(targetModel);
}

//-------------------------------------------------------------------------------------------------
HIO::Result HIO::ReloadModel(
    IEditModelTarget^ targetModel, FileData^ bfresFileData)
{
    G3D_HTC_REQUIRES_NOT_NULL(targetModel);
    G3D_HTC_REQUIRES_NOT_NULL(bfresFileData);
    G3D_HTC_REQUIRES(CheckFile(bfresFileData->FileName), System::String::Format("Invalid bfres file: {0}", bfresFileData->FileName));

    TargetEndianKind endian = GetTargetEndian();
    ReloadModelCommand^ command =
        gcnew ReloadModelCommand(endian, targetModel, bfresFileData);
    bool success = this->SendCommandQueue->PushBack(command);
    if (!success)
    {
        return HIO::Result::FailedQueueCommand;
    }

    return HIO::Result::Success;
}

//-------------------------------------------------------------------------------------------------
HIO::Result HIO::ReloadModel(
    IEditModelTarget^ targetModel,
    FileData^ bfresFileData,
    array<String^>^ bfshaFileNames,
    array<MaterialShaderInfo^>^ materialShaderInfos)
{
    G3D_HTC_REQUIRES_NOT_NULL(targetModel);
    G3D_HTC_REQUIRES_NOT_NULL(bfresFileData);
    G3D_HTC_REQUIRES_NOT_NULL(materialShaderInfos);
    G3D_HTC_REQUIRES(CheckFile(bfresFileData->FileName), System::String::Format("Invalid bfres file: {0}", bfresFileData->FileName));
    G3D_HTC_REQUIRES(bfresFileData->Alignment != 0, "Invalid alignemnt");

    if (bfshaFileNames != nullptr)
    {
        for (int i = 0; i < bfshaFileNames->Length; ++i)
        {
            if (!CheckFile(bfshaFileNames[i]))
            {
                return HIO::Result::InvalidFilePath;
            }
        }
    }

    TargetEndianKind endian = GetTargetEndian();
    LoadModelShaderArchiveCommand^ command =
        gcnew LoadModelShaderArchiveCommand(
            endian,
            targetModel,
            bfresFileData,
            bfshaFileNames,
            materialShaderInfos);
    bool success = this->SendCommandQueue->PushBack(command);
    if (!success)
    {
        return HIO::Result::FailedQueueCommand;
    }

    return HIO::Result::Success;
}

//-------------------------------------------------------------------------------------------------
bool HIO::SendAttachShaderArchive(IEditShaderArchiveTarget^ targetShaderArchive)
{
    if (targetShaderArchive == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    SendAttachShaderDefineCommand^ command =
        gcnew SendAttachShaderDefineCommand(kind, targetShaderArchive);

    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::UnloadShaderArchive(IEditShaderArchiveTarget^ targetShaderArchive)
{
    if (targetShaderArchive == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    UnloadShaderArchiveCommand^ command =
        gcnew UnloadShaderArchiveCommand(kind, targetShaderArchive);

    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::CancelAttachShaderArchive(uint shaderArchiveKey)
{
    DummyShaderArchive^ targetShaderArchive = gcnew DummyShaderArchive();
    targetShaderArchive->ShaderArchiveKey = shaderArchiveKey;
    return UnloadShaderArchive(targetShaderArchive);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditShadingModels(IEditShaderArchiveTarget^ targetShaderArchive, array<INT32>^ shadingModelIndices)
{
    if (targetShaderArchive == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    EditShadingModelCommand^ command =
        gcnew EditShadingModelCommand(
        kind,
        targetShaderArchive,
        shadingModelIndices);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::ReloadShaderProgram(IEditShaderArchiveTarget^ targetShaderArchive, INT32 shadingModelIndex, INT32 shaderProgramIndex, String^ binaryFileName)
{
    if (targetShaderArchive == nullptr)
    {
        return false;
    }

    if (!CheckFile(binaryFileName))
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    ReloadShaderProgramCommand^ command =
        gcnew ReloadShaderProgramCommand(
            kind,
            targetShaderArchive,
            shadingModelIndex,
            shaderProgramIndex,
            binaryFileName);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialVisibility(bool isVisible)
{
    return EditMaterialValue<bool>(isVisible, EDIT_TARGET_MATERIAL_VISIBILITY);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditBoneVisibility(bool isVisible)
{
    return EditBoneValue<bool>(isVisible, EDIT_TARGET_BONE_VISIBILITY);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditBoneBillboard(BillboardKind kind)
{
    u32 value = 0;
    switch(kind)
    {
    case BillboardKind::World:
        value = nw::g3d::res::ResBone::BILLBOARD_WORLD_VIEWVECTOR;
        break;
    case BillboardKind::WorldViewpoint:
        value = nw::g3d::res::ResBone::BILLBOARD_WORLD_VIEWPOINT;
        break;
    case BillboardKind::Screen:
        value = nw::g3d::res::ResBone::BILLBOARD_SCREEN_VIEWVECTOR;
        break;
    case BillboardKind::ScreenViewpoint:
        value = nw::g3d::res::ResBone::BILLBOARD_SCREEN_VIEWPOINT;
        break;
    case BillboardKind::YAxis:
        value = nw::g3d::res::ResBone::BILLBOARD_YAXIS_VIEWVECTOR;
        break;
    case BillboardKind::YAxisViewpoint:
        value = nw::g3d::res::ResBone::BILLBOARD_YAXIS_VIEWPOINT;
        break;
    }
    return EditBoneValue<u32>(value, EDIT_TARGET_BONE_BILLBOARD);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialDisplayFace(MaterialDisplayFaceKind displayFaceKind)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    if (!EditMaterialIniternal())
    {
        return false;
    }

    bool result = false;
    EditSimpleValue value;
    value.uValue = 0;

    if (displayFaceKind == MaterialDisplayFaceKind::None ||
        displayFaceKind == MaterialDisplayFaceKind::Back)
    {
        value.uHigh = 1;
    }

    if (displayFaceKind == MaterialDisplayFaceKind::None ||
        displayFaceKind == MaterialDisplayFaceKind::Front)
    {
        value.uLow = 1;
    }

    TargetEndianKind kind = GetTargetEndian();

    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        EditMaterialValueU16Command^ command =
            gcnew EditMaterialValueU16Command(kind, key, indices, value.uHigh, value.uLow, EDIT_TARGET_MATERIAL_DISPLAYFACE);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialRenderStateMode(RenderStateMode mode)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(mode);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_RENDER_STATE_MODE);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialBlendMode(BlendMode mode)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(mode);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_RENDER_STATE_BLEND_MODE);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialDepthTestEnable(bool enable)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    return EditMaterialValue<bool>(enable, EDIT_TARGET_MATERIAL_DEPTHTEST_ENABLE);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialDepthTestWriteEnable(bool enable)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    return EditMaterialValue<bool>(enable, EDIT_TARGET_MATERIAL_DEPTHTEST_WRITE_ENABLE);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialDepthTestFunc(DepthTestFuncKind kind)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(kind);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_DEPTHTEST_FUNC);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialAlphaTestEnable(bool enable)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    return EditMaterialValue<bool>(enable, EDIT_TARGET_MATERIAL_ALPHATEST_ENABLE);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialAlphaTestFunc(AlphaTestFuncKind kind)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(kind);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_ALPHATEST_FUNC);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialAlphaTestValue(float value)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    return EditMaterialValue<float>(value, EDIT_TARGET_MATERIAL_ALPHATEST_VALUE);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialColorCombine(BlendCombineKind kind)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(kind);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_COLOR_COMBINE);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialAlphaCombine(BlendCombineKind kind)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(kind);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_ALPHA_COMBINE);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialColorSrcBlend(ColorBlendFuncKind kind)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(kind);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_COLOR_SRC_BLEND);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialColorDstBlend(ColorBlendFuncKind kind)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(kind);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_COLOR_DST_BLEND);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialAlphaSrcBlend(AlphaBlendFuncKind kind)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(kind);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_ALPHA_SRC_BLEND);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialAlphaDstBlend(AlphaBlendFuncKind kind)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(kind);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_ALPHA_DST_BLEND);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialConstantColor(float r, float g, float b, float a, Drawing::ColorChannelFlag useChannelFlag)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    array<float>^ values = { r, g, b, a };

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        EditMaterialVector4ValueFloatCommand^ command =
            gcnew EditMaterialVector4ValueFloatCommand(kind, key, indices, values, EDIT_TARGET_MATERIAL_CONSTANT_COLOR);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialLogicOp(LogicOpKind kind)
{
    if ((TargetDeviceType != TargetType::Cafe) && (TargetDeviceType != TargetType::OldPc))
    {
        return false;
    }

    u32 value = static_cast<u32>(kind);
    return EditMaterialValue<UINT32>(value, EDIT_TARGET_MATERIAL_LOGIC_OP);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerWrap(INT32 samplerIndex, HIO::TextureClampKind kind, EditTargetKind target)
{
    switch (TargetDeviceType)
    {
    case TargetType::Cafe:
    case TargetType::OldPc:
        {
            GX2TexClamp value = ConvertToGx2Clamp(kind);
            if (value == -1)
            {
                return false;
            }

            return EditMaterialSamplerValueU32(samplerIndex, value, target);
        }
    case TargetType::Htc:
        {
            nn::gfx::TextureAddressMode value = ConvertToGfxTextureAddress(kind);
            if (value == -1)
            {
                return false;
            }

            return EditMaterialSamplerValueU32(samplerIndex, value, target);
        }
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return true;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerWrapU(INT32 samplerIndex, TextureClampKind kind)
{
    return EditMaterialSamplerWrap(samplerIndex, kind, EDIT_TARGET_MATERIAL_SAMPLER_WRAP_U);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerWrapV(INT32 samplerIndex, TextureClampKind kind)
{
    return EditMaterialSamplerWrap(samplerIndex, kind, EDIT_TARGET_MATERIAL_SAMPLER_WRAP_V);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerWrapW(INT32 samplerIndex, TextureClampKind kind)
{
    return EditMaterialSamplerWrap(samplerIndex, kind, EDIT_TARGET_MATERIAL_SAMPLER_WRAP_W);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerMagFilter(INT32 samplerIndex, TexturFilterKind kind)
{
    u32 value = static_cast<u32>(kind);
    return EditMaterialSamplerValueU32(samplerIndex, value, EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerMinFilter(INT32 samplerIndex, TexturFilterKind kind)
{
    u32 value = static_cast<u32>(kind);
    return EditMaterialSamplerValueU32(samplerIndex, value, EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerMipmapFilter(INT32 samplerIndex, TextureMipFilterKind kind)
{
    u32 value = static_cast<u32>(kind);
    return EditMaterialSamplerValueU32(samplerIndex, value, EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerMaxAniso(INT32 samplerIndex, TextureAnisoRatioKind kind)
{
    u32 value = static_cast<u32>(kind);
    return EditMaterialSamplerValueU32(samplerIndex, value, EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerMinLOD(INT32 samplerIndex, float minLOD)
{
    return EditMaterialSamplerValueFloat(samplerIndex, minLOD, EDIT_TARGET_MATERIAL_SAMPLER_MIN_LOD);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerMaxLOD(INT32 samplerIndex, float maxLOD)
{
    return EditMaterialSamplerValueFloat(samplerIndex, maxLOD, EDIT_TARGET_MATERIAL_SAMPLER_MAX_LOD);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerLODBias(INT32 samplerIndex, float bias)
{
    return EditMaterialSamplerValueFloat(samplerIndex, bias, EDIT_TARGET_MATERIAL_SAMPLER_LOD_BIAS);
}

//-------------------------------------------------------------------------------------------------

HIO::Result HIO::EditMaterialSamplerFilter(
    int samplerIndex,
    TextureFilterKind minFilter,
    TextureFilterKind magFilter,
    TextureFilterKind mipmapFilter,
    int maxAnisotropy)
{
    switch (TargetDeviceType)
    {
    case TargetType::Cafe:
    case TargetType::OldPc:
        {
            bool success;
            GX2TexXYFilterType gx2MinFilter = ConvertToGx2XyFilter(minFilter);
            if (gx2MinFilter == -1)
            {
                return Result::InvalidArgument;
            }

            GX2TexXYFilterType gx2MagFilter = ConvertToGx2XyFilter(magFilter);
            if (gx2MagFilter == -1)
            {
                return Result::InvalidArgument;
            }

            GX2TexMipFilterType gx2mipmapFilter = ConvertToGx2MipFilter(mipmapFilter);
            if (gx2mipmapFilter == -1)
            {
                return Result::InvalidArgument;
            }

            GX2TexAnisoRatio gx2MaxAnisotropy = ConvertToGx2TexAnisoRatio(maxAnisotropy);
            if (gx2MaxAnisotropy == -1)
            {
                return Result::InvalidArgument;
            }

            success = EditMaterialSamplerValueU32(samplerIndex, gx2MinFilter, EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER);
            if (!success)
            {
                return Result::FailedQueueCommand;
            }

            success = EditMaterialSamplerValueU32(samplerIndex, gx2MagFilter, EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER);
            if (!success)
            {
                return Result::FailedQueueCommand;
            }

            success = EditMaterialSamplerValueU32(samplerIndex, gx2mipmapFilter, EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER);
            if (!success)
            {
                return Result::FailedQueueCommand;
            }

            success = EditMaterialSamplerValueU32(samplerIndex, gx2MaxAnisotropy, EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO);
            if (!success)
            {
                return Result::FailedQueueCommand;
            }
        }
        break;
    case TargetType::Htc:
        {
            // gfx は enum が縮小、拡大、ミップマップの組み合わせになっているので各フィルター毎に設定を区別しない
            // コマンドが MIN、MAX、MIP で分かれているのは GX2 がベースに作られたため
            // TODO: Cafe のサポートが切れたら、gfx ベースのコマンドに変える
            if (maxAnisotropy > 1)
            {
                // maxAnisotropy が 1 より大きいときは強制的に異方性フィルタリング
                nn::gfx::FilterMode filterMode = nn::gfx::FilterMode_Anisotropic;
                bool success = EditMaterialSamplerValueU32(samplerIndex, filterMode, EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER);
                if (!success)
                {
                    return Result::FailedQueueCommand;
                }
            }
            else
            {
                nn::gfx::detail::FilterModeBit gfxMinFilterBit = ConvertToGfxFilterBit(minFilter);
                if (gfxMinFilterBit == -1)
                {
                    return Result::InvalidArgument;
                }

                nn::gfx::detail::FilterModeBit gfxMagFilterBit = ConvertToGfxFilterBit(magFilter);
                if (gfxMagFilterBit == -1)
                {
                    return Result::InvalidArgument;
                }

                nn::gfx::detail::FilterModeBit gfxMipmapFilterBit = ConvertToGfxFilterBit(mipmapFilter);
                if (gfxMipmapFilterBit == -1)
                {
                    return Result::InvalidArgument;
                }

                nn::gfx::FilterMode filterMode = static_cast<nn::gfx::FilterMode>(
                    gfxMinFilterBit << nn::gfx::detail::FilterModeBit_MinFilterShift |
                    gfxMagFilterBit << nn::gfx::detail::FilterModeBit_MagFilterShift |
                    gfxMipmapFilterBit << nn::gfx::detail::FilterModeBit_MipFilterShift);
                bool success = EditMaterialSamplerValueU32(samplerIndex, filterMode, EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER);
                if (!success)
                {
                    return Result::FailedQueueCommand;
                }
            }

            {
                // フィルターモードが FilterMode_Anisotropic でない場合は、gfx でも maxAnisotropy が無視されるようになるが、
                // 念のためここでもセットしておく
                bool success = EditMaterialSamplerValueU32(samplerIndex, maxAnisotropy, EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO);
                if (!success)
                {
                    return Result::FailedQueueCommand;
                }
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return Result::Success;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, bool value)
{
    array<bool>^ values = { value };
    return EditMaterialShaderParameterVector_<bool>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, int value)
{
    array<int>^ values = { value };
    return EditMaterialShaderParameterVector_<int>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_INT);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, uint value)
{
    array<uint>^ values = { value };
    return EditMaterialShaderParameterVector_<uint>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, float value)
{
    array<float>^ values = { value };
    return EditMaterialShaderParameterVector_<float>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector2^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<float>^ values = { value->X, value->Y };
    return EditMaterialShaderParameterVector_<float>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector3^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<float>^ values = { value->X, value->Y, value->Z };
    return EditMaterialShaderParameterVector_<float>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector4^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<float>^ values = { value->X, value->Y, value->Z, value->W };
    return EditMaterialShaderParameterVector_<float>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector2i^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<int>^ values = { value->X, value->Y };
    return EditMaterialShaderParameterVector_<int>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_INT2);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector3i^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<int>^ values = { value->X, value->Y, value->Z };
    return EditMaterialShaderParameterVector_<int>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_INT3);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector4i^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<int>^ values = { value->X, value->Y, value->Z, value->W };
    return EditMaterialShaderParameterVector_<int>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_INT4);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector2u^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<uint>^ values = { value->X, value->Y };
    return EditMaterialShaderParameterVector_<uint>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT2);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector3u^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<uint>^ values = { value->X, value->Y, value->Z };
    return EditMaterialShaderParameterVector_<uint>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT3);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector4u^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<uint>^ values = { value->X, value->Y, value->Z, value->W };
    return EditMaterialShaderParameterVector_<uint>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT4);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector2b^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<bool>^ values = { value->X, value->Y };
    return EditMaterialShaderParameterVector_<bool>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL2);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector3b^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<bool>^ values = { value->X, value->Y, value->Z };
    return EditMaterialShaderParameterVector_<bool>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL3);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Vector4b^ value, Math::VectorChannelFlag useChannelFlag)
{
    array<bool>^ values = { value->X, value->Y, value->Z, value->W };
    return EditMaterialShaderParameterVector_<bool>(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL4);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Matrix22^ value)
{
    array<float>^ values = value->ToArray();
    return EditMaterialShaderParameterMatrix_(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x2);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Matrix23^ value)
{
    array<float>^ values = value->ToArray();
    return EditMaterialShaderParameterMatrix_(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x3);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Matrix24^ value)
{
    array<float>^ values = value->ToArray();
    return EditMaterialShaderParameterMatrix_(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x4);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Matrix32^ value)
{
    array<float>^ values = value->ToArray();
    return EditMaterialShaderParameterMatrix_(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x2);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Matrix33^ value)
{
    array<float>^ values = value->ToArray();
    return EditMaterialShaderParameterMatrix_(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x3);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Matrix34^ value)
{
    array<float>^ values = value->ToArray();
    return EditMaterialShaderParameterMatrix_(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x4);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Matrix42^ value)
{
    array<float>^ values = value->ToArray();
    return EditMaterialShaderParameterMatrix_(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x2);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Matrix43^ value)
{
    array<float>^ values = value->ToArray();
    return EditMaterialShaderParameterMatrix_(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x3);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::Matrix44^ value)
{
    array<float>^ values = value->ToArray();
    return EditMaterialShaderParameterMatrix_(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x4);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::TextureSRT2D^ value)
{
    array<float>^ values = value->ToArray();
    nw::g3d::edit::detail::EditTargetKind kind = EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT2D;
    switch(value->Kind)
    {
    case Math::TextureSRT2D::SRTKind::SRTMaya:
        kind = EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA;
        break;
    case Math::TextureSRT2D::SRTKind::SRT3DSMax:
        kind = EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX;
        break;
    case Math::TextureSRT2D::SRTKind::SRTSoftimage:
        kind = EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE;
        break;
    }
    return EditMaterialShaderParameterMatrix_(parameterName, values, kind);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::TextureSRT3D^ value)
{
    array<float>^ values = value->ToArray();
    return EditMaterialShaderParameterMatrix_(parameterName, values, EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT3D);
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameter(String^ parameterName, Math::TextureSRT2DEx^ value)
{
    array<float>^ values = value->ToArray();
    nw::g3d::edit::detail::EditTargetKind kind = EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT2D;
    switch(value->Kind)
    {
    case Math::TextureSRT2D::SRTKind::SRTMaya:
        kind = EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA;
        break;
    case Math::TextureSRT2D::SRTKind::SRT3DSMax:
        kind = EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX;
        break;
    case Math::TextureSRT2D::SRTKind::SRTSoftimage:
        kind = EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE;
        break;
    }
    return EditMaterialShaderParameterMatrix_(parameterName, values, kind);
}

//-------------------------------------------------------------------------------------------------
bool HIO::SelectTargetAnimation(IEditTarget^ targetAnimation)
{
    m_TargetAnimation = targetAnimation;
    return true;
}

//-------------------------------------------------------------------------------------------------

bool HIO::EditAnimationPause(IEditModelTarget^ targetModel, IEditTarget^ targetAnimation, float frame, bool enable)
{
    if (targetAnimation == nullptr)
    {
        return false;
    }
    if (targetAnimation->ResFileKey == 0)
    {
        return false;
    }
    if (targetModel == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    if (enable)
    {
        EditModelAnimStopCommand^ command =
            gcnew EditModelAnimStopCommand(kind, targetModel, targetAnimation, frame);
        result = this->SendCommandQueue->PushBack(command);
    }
    else
    {
        EditModelAnimPlayCommand^ command =
            gcnew EditModelAnimPlayCommand(kind, targetModel, targetAnimation, frame);
        result = this->SendCommandQueue->PushBack(command);
    }
    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditShaderParamCurve(UINT32 materialNameIndex, UINT32 paramAnimIndex, UINT32 componentIndex, int curveIndex, ICurve^ curve)
{
    if (m_TargetAnimation == nullptr)
    {
        return false;
    }

    if (m_TargetAnimation->ResFileKey == 0)
    {
        return false;
    }

    if (curve == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    EditShaderParamCurveCommand^ command =
        gcnew EditShaderParamCurveCommand(kind, m_TargetAnimation, curve, materialNameIndex, curveIndex, componentIndex);
    return this->SendCommandQueue->PushBack(command);
}
//-------------------------------------------------------------------------------------------------
bool HIO::EditTexPatternCurve(UINT32 materialNameIndex, int curveIndex, ICurve^ curve)
{
    if (m_TargetAnimation == nullptr)
    {
        return false;
    }

    if (m_TargetAnimation->ResFileKey == 0)
    {
        return false;
    }

    if (curve == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    EditTexPatternCurveCommand^ command =
        gcnew EditTexPatternCurveCommand(kind, m_TargetAnimation, curve, materialNameIndex, curveIndex);
    return this->SendCommandQueue->PushBack(command);
}
//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialCurve(UINT32 materialNameIndex, UINT32 componentIndex, int curveIndex, ICurve^ curve)
{
    if (m_TargetAnimation == nullptr)
    {
        return false;
    }

    if (m_TargetAnimation->ResFileKey == 0)
    {
        return false;
    }

    if (curve == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    EditMaterialCurveCommand^ command =
        gcnew EditMaterialCurveCommand(kind, m_TargetAnimation, curve, materialNameIndex, curveIndex, componentIndex);
    return this->SendCommandQueue->PushBack(command);
}
//-------------------------------------------------------------------------------------------------
bool HIO::EditMatVisibilityCurve(UINT32 animIndex, int curveIndex, ICurve^ curve)
{
    if (m_TargetAnimation == nullptr)
    {
        return false;
    }

    if (m_TargetAnimation->ResFileKey == 0)
    {
        return false;
    }

    if (curve == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    EditMatVisibilityCurveCommand^ command =
        gcnew EditMatVisibilityCurveCommand(kind, m_TargetAnimation, curve, curveIndex, animIndex);
    return this->SendCommandQueue->PushBack(command);
}
//-------------------------------------------------------------------------------------------------
bool HIO::EditBoneVisibilityCurve(UINT32 animIndex, int curveIndex, ICurve^ curve)
{
    if (m_TargetAnimation == nullptr)
    {
        return false;
    }

    if (m_TargetAnimation->ResFileKey == 0)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    EditBoneVisibilityCurveCommand^ command =
        gcnew EditBoneVisibilityCurveCommand(kind, m_TargetAnimation, curve, curveIndex, animIndex);
    return this->SendCommandQueue->PushBack(command);
}
//-------------------------------------------------------------------------------------------------
bool HIO::EditShapeCurve(UINT32 animIndex, int curveIndex, ICurve^ curve)
{
    if (m_TargetAnimation == nullptr)
    {
        return false;
    }

    if (m_TargetAnimation->ResFileKey == 0)
    {
        return false;
    }

    if (curve == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    EditShapeCurveCommand^ command =
        gcnew EditShapeCurveCommand(kind, m_TargetAnimation, curve, animIndex, curveIndex);
    return this->SendCommandQueue->PushBack(command);
}
//-------------------------------------------------------------------------------------------------
bool HIO::FlushShaderParamCurves()
{
    return false;
}

//-------------------------------------------------------------------------------------------------
bool HIO::BindAnimations(IEditModelTarget^ targetModel, array<IEditTarget^>^ bindAnimations)
{
    G3D_HTC_REQUIRES_NOT_NULL(targetModel);
    G3D_HTC_REQUIRES_NOT_NULL(bindAnimations);

    if (bindAnimations->Length <= 0)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    BindAnimsCommand^ command =
        gcnew BindAnimsCommand(kind, targetModel, bindAnimations);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::UnbindAnimations(IEditModelTarget^ targetModel, array<IEditTarget^>^ bindAnimations)
{
    G3D_HTC_REQUIRES_NOT_NULL(targetModel);
    G3D_HTC_REQUIRES_NOT_NULL(bindAnimations);

    if (bindAnimations->Length <= 0)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    UnbindAnimsCommand^ command =
        gcnew UnbindAnimsCommand(kind, targetModel, bindAnimations);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::UnbindAnimationAll(IEditModelTarget^ targetModel)
{
    return false;
}

//-------------------------------------------------------------------------------------------------
bool HIO::LoadShaderParamAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return LoadAnimation(nw::g3d::edit::detail::FILEDATA_SHADER_PARAM_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::UnloadShaderParamAnimation(IEditTarget^ targetAnimation)
{
    return UnloadAnimation(nw::g3d::edit::detail::FILEDATA_SHADER_PARAM_ANIM, targetAnimation);
}
//-------------------------------------------------------------------------------------------------
bool HIO::ReloadShaderParamAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return ReloadAnimation(nw::g3d::edit::detail::FILEDATA_SHADER_PARAM_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::LoadSkeletalAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return LoadAnimation(nw::g3d::edit::detail::FILEDATA_SKELETAL_ANIM, targetAnimation, fileData);
}

bool HIO::LoadSkeletalAnimation(IEditTarget^ targetAnimation, FileData^ fileData, String^ retargetingHostModelName)
{
    if (targetAnimation == nullptr)
    {
        return false;
    }

    if (!CheckFile(fileData->FileName))
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    LoadAnimCommand^ command = gcnew LoadAnimCommand(kind, targetAnimation, fileData, retargetingHostModelName);
    return this->SendCommandQueue->PushBack(command);
}

void HIO::SetRetargetingHostModel(IEditTarget^ targetAnimation, IEditModelTarget^ retargetHostModel)
{
    if (targetAnimation == nullptr)
    {
        gcnew ArgumentNullException("Target animation must not be null");
    }

    TargetEndianKind endian = GetTargetEndian();
    EditRetargetingHostModelCommand^ command = gcnew EditRetargetingHostModelCommand(endian, targetAnimation, retargetHostModel);

    this->SendCommandQueue->PushBack(command);
}

void HIO::SetPlayMotionMirroringEnabled(IEditTarget^ targetAnimation, bool isEnabled)
{
    if (targetAnimation == nullptr)
    {
        gcnew ArgumentNullException("Target animation must not be null");
    }

    TargetEndianKind endian = GetTargetEndian();
    EditPlayMotionMirroringCommand^ command = gcnew EditPlayMotionMirroringCommand(endian, targetAnimation, isEnabled);

    this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::UnloadSkeletalAnimation(IEditTarget^ targetAnimation)
{
    return UnloadAnimation(nw::g3d::edit::detail::FILEDATA_SKELETAL_ANIM, targetAnimation);
}
//-------------------------------------------------------------------------------------------------
bool HIO::ReloadSkeletalAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return ReloadAnimation(nw::g3d::edit::detail::FILEDATA_SKELETAL_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::LoadColorAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return LoadAnimation(nw::g3d::edit::detail::FILEDATA_MAT_COLOR_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::UnloadColorAnimation(IEditTarget^ targetAnimation)
{
    return UnloadAnimation(nw::g3d::edit::detail::FILEDATA_MAT_COLOR_ANIM, targetAnimation);
}
//-------------------------------------------------------------------------------------------------
bool HIO::ReloadColorAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return ReloadAnimation(nw::g3d::edit::detail::FILEDATA_MAT_COLOR_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::LoadTextureSRTAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return LoadAnimation(nw::g3d::edit::detail::FILEDATA_TEXTURE_SRT_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::UnloadTextureSRTAnimation(IEditTarget^ targetAnimation)
{
    return UnloadAnimation(nw::g3d::edit::detail::FILEDATA_TEXTURE_SRT_ANIM, targetAnimation);
}
//-------------------------------------------------------------------------------------------------
bool HIO::ReloadTextureSRTAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return ReloadAnimation(nw::g3d::edit::detail::FILEDATA_TEXTURE_SRT_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::LoadBoneVisibilityAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return LoadAnimation(nw::g3d::edit::detail::FILEDATA_BONE_VISIBILITY_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::UnloadBoneVisibilityAnimation(IEditTarget^ targetAnimation)
{
    return UnloadAnimation(nw::g3d::edit::detail::FILEDATA_BONE_VISIBILITY_ANIM, targetAnimation);
}
//-------------------------------------------------------------------------------------------------
bool HIO::ReloadBoneVisibilityAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return ReloadAnimation(nw::g3d::edit::detail::FILEDATA_BONE_VISIBILITY_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::LoadMaterialVisibilityAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return LoadAnimation(nw::g3d::edit::detail::FILEDATA_MAT_VISIBILITY_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::UnloadMaterialVisibilityAnimation(IEditTarget^ targetAnimation)
{
    return UnloadAnimation(nw::g3d::edit::detail::FILEDATA_MAT_VISIBILITY_ANIM, targetAnimation);
}
//-------------------------------------------------------------------------------------------------
bool HIO::ReloadMaterialVisibilityAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return ReloadAnimation(nw::g3d::edit::detail::FILEDATA_MAT_VISIBILITY_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::LoadTexturePatternAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return LoadAnimation(nw::g3d::edit::detail::FILEDATA_TEXTURE_PATTERN_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::UnloadTexturePatternAnimation(IEditTarget^ targetAnimation)
{
    return UnloadAnimation(nw::g3d::edit::detail::FILEDATA_TEXTURE_PATTERN_ANIM, targetAnimation);
}
//-------------------------------------------------------------------------------------------------
bool HIO::ReloadTexturePatternAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return ReloadAnimation(nw::g3d::edit::detail::FILEDATA_TEXTURE_PATTERN_ANIM, targetAnimation, fileData);
}

//-------------------------------------------------------------------------------------------------
bool HIO::LoadMaterialAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return LoadAnimation(nw::g3d::edit::detail::FILEDATA_MATERIAL_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::UnloadMaterialAnimation(IEditTarget^ targetAnimation)
{
    return UnloadAnimation(nw::g3d::edit::detail::FILEDATA_MATERIAL_ANIM, targetAnimation);
}
//-------------------------------------------------------------------------------------------------
bool HIO::ReloadMaterialAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return ReloadAnimation(nw::g3d::edit::detail::FILEDATA_MATERIAL_ANIM, targetAnimation, fileData);
}

//-------------------------------------------------------------------------------------------------
bool HIO::LoadShapeAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return LoadAnimation(nw::g3d::edit::detail::FILEDATA_SHAPE_ANIM, targetAnimation, fileData);
}
//-------------------------------------------------------------------------------------------------
bool HIO::UnloadShapeAnimation(IEditTarget^ targetAnimation)
{
    return UnloadAnimation(nw::g3d::edit::detail::FILEDATA_SHAPE_ANIM, targetAnimation);
}
//-------------------------------------------------------------------------------------------------
bool HIO::ReloadShapeAnimation(IEditTarget^ targetAnimation, FileData^ fileData)
{
    return ReloadAnimation(nw::g3d::edit::detail::FILEDATA_SHAPE_ANIM, targetAnimation, fileData);
}

//-------------------------------------------------------------------------------------------------
bool HIO::LoadSceneAnimation(IEditSceneAnimTarget^ targetAnimation, FileData^ fileData)
{
    return LoadAnimation(nw::g3d::edit::detail::FILEDATA_SCENE_ANIM, targetAnimation, fileData);
}
bool HIO::UnloadSceneAnimation(IEditSceneAnimTarget^ targetAnimation)
{
    return UnloadAnimation(nw::g3d::edit::detail::FILEDATA_SCENE_ANIM, targetAnimation);
}
bool HIO::ReloadSceneAnimation(IEditSceneAnimTarget^ targetAnimation, FileData^ fileData)
{
    return ReloadAnimation(nw::g3d::edit::detail::FILEDATA_SCENE_ANIM, targetAnimation, fileData);
}
bool HIO::BindSceneAnimations(array<IEditSceneAnimTarget^>^ bindAnimations)
{
    if (bindAnimations == nullptr)
    {
        return false;
    }

    if (bindAnimations->Length <= 0)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    BindSceneAnimsCommand^ command =
        gcnew BindSceneAnimsCommand(kind, bindAnimations);
    return this->SendCommandQueue->PushBack(command);
}
bool HIO::UnbindSceneAnimations(array<IEditSceneAnimTarget^>^ bindAnimations)
{
    if (bindAnimations == nullptr)
    {
        return false;
    }

    if (bindAnimations->Length <= 0)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    UnbindSceneAnimsCommand^ command =
        gcnew UnbindSceneAnimsCommand(kind, bindAnimations);
    return this->SendCommandQueue->PushBack(command);
}

bool HIO::EditCameraAnimCurve(UINT32 cameraIndex, int targetIndex, int curveIndex, ICurve^ curve)
{
    if (m_TargetAnimation == nullptr)
    {
        return false;
    }

    IEditSceneAnimTarget^ sceneAnimTarget = dynamic_cast<IEditSceneAnimTarget^>(m_TargetAnimation);
    if (sceneAnimTarget == nullptr)
    {
        return false;
    }

    if (m_TargetAnimation->ResFileKey == 0)
    {
        return false;
    }

    if (curve == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    EditCameraAnimCurveCommand^ command =
        gcnew EditCameraAnimCurveCommand(kind, sceneAnimTarget, curve, cameraIndex, targetIndex, curveIndex);
    return this->SendCommandQueue->PushBack(command);
}
bool HIO::EditLightAnimCurve(UINT32 lightIndex, int targetIndex, int curveIndex, ICurve^ curve)
{
    if (m_TargetAnimation == nullptr)
    {
        return false;
    }

    IEditSceneAnimTarget^ sceneAnimTarget = dynamic_cast<IEditSceneAnimTarget^>(m_TargetAnimation);
    if (sceneAnimTarget == nullptr)
    {
        return false;
    }

    if (m_TargetAnimation->ResFileKey == 0)
    {
        return false;
    }

    if (curve == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    EditLightAnimCurveCommand^ command =
        gcnew EditLightAnimCurveCommand(kind, sceneAnimTarget, curve, lightIndex, targetIndex, curveIndex);
    return this->SendCommandQueue->PushBack(command);
}
bool HIO::EditFogAnimCurve(UINT32 fogIndex, int targetIndex, int curveIndex, ICurve^ curve)
{
    if (m_TargetAnimation == nullptr)
    {
        return false;
    }

    IEditSceneAnimTarget^ sceneAnimTarget = dynamic_cast<IEditSceneAnimTarget^>(m_TargetAnimation);
    if (sceneAnimTarget == nullptr)
    {
        return false;
    }

    if (m_TargetAnimation->ResFileKey == 0)
    {
        return false;
    }

    if (curve == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    EditFogAnimCurveCommand^ command =
        gcnew EditFogAnimCurveCommand(kind, sceneAnimTarget, curve, fogIndex, targetIndex, curveIndex);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::SetPlayPolicy(PlayPolicy policy)
{
    nn::g3d::viewer::detail::EditPlayPolicyKind playPolicyKind = nn::g3d::viewer::detail::EDIT_PLAY_POLICY_AUTO;
    if (policy == PlayPolicy::Loop)
    {
        playPolicyKind = nn::g3d::viewer::detail::EDIT_PLAY_POLICY_LOOP;
    }
    else if (policy == PlayPolicy::Once)
    {
        playPolicyKind = nn::g3d::viewer::detail::EDIT_PLAY_POLICY_ONCE;
    }

    TargetEndianKind kind = GetTargetEndian();
    EditPlayPolicyCommand^ command =
        gcnew EditPlayPolicyCommand(kind, playPolicyKind);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::PlayFrameCtrl(float frame)
{
    TargetEndianKind kind = GetTargetEndian();
    EditPlayFrameCtrlCommand^ command =
        gcnew EditPlayFrameCtrlCommand(kind, frame);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::StopFrameCtrl(float frame)
{
    TargetEndianKind kind = GetTargetEndian();
    EditStopFrameCtrlCommand^ command =
        gcnew EditStopFrameCtrlCommand(kind, frame);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::SetFrameStep(float value)
{
    TargetEndianKind kind = GetTargetEndian();
    EditFrameStepCommand^ command =
        gcnew EditFrameStepCommand(kind, value);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::SetFrameCount(float value)
{
    TargetEndianKind kind = GetTargetEndian();
    EditFrameCountCommand^ command =
        gcnew EditFrameCountCommand(kind, value);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
bool HIO::SetStartFrame(float value)
{
    TargetEndianKind kind = GetTargetEndian();
    EditStartFrameCommand^ command =
        gcnew EditStartFrameCommand(kind, value);
    return this->SendCommandQueue->PushBack(command);
}


//-------------------------------------------------------------------------------------------------
bool HIO::QueryModelLayout(bool isBind)
{
    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ model in SelectedModels)
    {
        QueryModelLayoutCommand^ command =
            gcnew QueryModelLayoutCommand(kind, model);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditModelLayout(bool isBind, Math::Vector3^ scale, Math::Vector3^ rotate, Math::Vector3^ translate)
{
    // MEMO: 現状 isBind はランタイム側で利用されていない。

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ model in SelectedModels)
    {
        EditModelLayoutCommand^ command =
            gcnew EditModelLayoutCommand(kind, model, isBind, scale, rotate, translate);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditBoneBind(IEditModelTarget^ targetModel, int boneIndex)
{
    TargetEndianKind kind = GetTargetEndian();
    bool result = false;
    for each (IEditModelTarget^ model in SelectedModels)
    {
        // 同じものは親子関係にできないので、無視
        if (model == targetModel)
        {
            continue;
        }

        //TODO: isBind に値を渡すように修正すること
        EditBoneBindCommand^ command =
            gcnew EditBoneBindCommand(kind, model, targetModel, boneIndex);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::ClearBoneBind(array<IEditModelTarget^>^ targetModels)
{
    if (targetModels == nullptr)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    bool result = false;
    for each (IEditModelTarget^ model in targetModels)
    {
        // 同じものは親子関係にできないので、無視
        if (model == nullptr)
        {
            continue;
        }

        //TODO: isBind に値を渡すように修正すること
        EditBoneBindCommand^ command =
            gcnew EditBoneBindCommand(kind, model, nullptr, -1);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditBlinkMaterials()
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    bool result = false;

    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        EditBlinkMaterialsCommand^ command =
            gcnew EditBlinkMaterialsCommand(kind, key, indices);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditSelectTarget(SelectTargetKind selectTargetkind)
{
    ICollection<IEditModelTarget^>^ keys;
    IDictionary<IEditModelTarget^, HashSet<Int32>^>^ target;

    switch (selectTargetkind)
    {
    case NintendoWare::G3d::Edit::SelectTargetKind::Model:
        if (SelectedModels->Count <= 0)
        {
            return false;
        }
        keys = SelectedModels;
        break;
    case NintendoWare::G3d::Edit::SelectTargetKind::Material:
        if (SelectedMaterials->Count <= 0)
        {
            return false;
        }
        target = SelectedMaterials;
        keys = SelectedMaterials->Keys;
        break;
    case NintendoWare::G3d::Edit::SelectTargetKind::Bone:
        if (SelectedBones->Count <= 0)
        {
            return false;
        }
        target = SelectedBones;
        keys = SelectedBones->Keys;
        break;
    case NintendoWare::G3d::Edit::SelectTargetKind::Shape:
        if (SelectedShapes->Count <= 0)
        {
            return false;
        }
        target = SelectedShapes;
        keys = SelectedShapes->Keys;
        break;
    case NintendoWare::G3d::Edit::SelectTargetKind::Texture:
        return false;
    default:
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    bool result = false;

    for each (IEditModelTarget^ key in keys)
    {
        array<INT32>^ indices;

        if (selectTargetkind == NintendoWare::G3d::Edit::SelectTargetKind::Model)
        {
            if (key->ModelObjKey == 0)
            {
                continue;
            }

            // モデルにインデックスは無いのでダミーデータを付加する。
            indices = gcnew array<INT32>(1);
            indices[0] = 0;
        }
        else
        {
            HashSet<INT32>^ indexSet;
            if (!target->TryGetValue(key, indexSet))
            {
                continue;
            }

            if (key->ModelObjKey == 0)
            {
                continue;
            }

            if (indexSet->Count <= 0)
            {
                continue;
            }

            indices = gcnew array<INT32>(indexSet->Count);
            indexSet->CopyTo(indices);
        }

        SystemEditSelectTargetCommand^ command =
            gcnew SystemEditSelectTargetCommand(kind, key, selectTargetkind, indices);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditSetShapeLodLevel(int level)
{
    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ model in SelectedModels)
    {
        EditShapeLodLevelCommand^ command =
            gcnew EditShapeLodLevelCommand(kind, model, level);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

bool HIO::EditResetShapeLodLevel()
{
    return EditSetShapeLodLevel(-1);
}

//-------------------------------------------------------------------------------------------------
ICommandTransaction^ HIO::BeginCommand()
{
    TargetEndianKind kind = GetTargetEndian();
    SystemBeginFreezeCommand^ beginCommand = gcnew SystemBeginFreezeCommand(kind);
    SystemEndFreezeCommand^ endCommand = gcnew SystemEndFreezeCommand(kind);
    return this->BeginCommandInternal(beginCommand, endCommand);
}

ICommandTransaction^ HIO::BeginCommand(bool sync)
{
    TargetEndianKind kind = GetTargetEndian();

    if (sync)
    {
        SystemBeginFreezeCommand^ beginCommand = gcnew SystemBeginFreezeCommand(kind);
        SystemEndFreezeCommand^ endCommand = gcnew SystemEndFreezeCommand(kind);
        return this->BeginCommandInternal(beginCommand, endCommand);
    }
    else
    {
        SystemBeginFreezeNoSyncCommand^ beginCommand = gcnew SystemBeginFreezeNoSyncCommand(kind);
        SystemEndFreezeCommand^ endCommand = gcnew SystemEndFreezeCommand(kind);
        return this->BeginCommandInternal(beginCommand, endCommand);
    }
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialIniternal()
{
    if (SelectedMaterials->Count <= 0)
    {
        return false;
    }

    return true;
}

//-------------------------------------------------------------------------------------------------
bool HIO::CheckFile(String^ fileName)
{
    if (fileName == nullptr)
    {
        return false;
    }

    if (fileName == String::Empty)
    {
        return false;
    }

    if (!IO::File::Exists(fileName))
    {
        return false;
    }

    return true;
}

//-------------------------------------------------------------------------------------------------
generic<typename ValueType>
bool HIO::EditMaterialShaderParameterVector_(String^ parameterName, array<ValueType>^ values, nw::g3d::edit::detail::EditTargetKind editTargetKind)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        EditMaterialShaderParamVectorValueCommand<ValueType>^ command =
            gcnew EditMaterialShaderParamVectorValueCommand<ValueType>(kind, key, indices, parameterName, values, editTargetKind);
        result |= this->SendCommandQueue->PushBack(command);
    }
    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialShaderParameterMatrix_(String^ parameterName, array<float>^ values, nw::g3d::edit::detail::EditTargetKind editTargetKind)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        EditMaterialShaderParamMatrixValueCommand^ command =
            gcnew EditMaterialShaderParamMatrixValueCommand(kind, key, indices, parameterName, values, editTargetKind);
        result |= this->SendCommandQueue->PushBack(command);
    }
    return result;
}
//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerValueU32(INT32 samplerIndex, UINT32 value, nw::g3d::edit::detail::EditTargetKind editTargetKind)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        EditMaterialSamplerValueCommand<u32>^ command =
            gcnew EditMaterialSamplerValueCommand<u32>(kind, key, indices, samplerIndex, value, editTargetKind);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditMaterialSamplerValueFloat(INT32 samplerIndex, float value, nw::g3d::edit::detail::EditTargetKind editTargetKind)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        EditMaterialSamplerValueCommand<float>^ command =
            gcnew EditMaterialSamplerValueCommand<float>(kind, key, indices, samplerIndex, value, editTargetKind);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::QueryRenderInfo(array<IRenderInfo^>^ renderInfoArray)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        QueryRenderInfoCommand^ command =
            gcnew QueryRenderInfoCommand(kind, key, indices, renderInfoArray);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::UpdateRenderInfo(array<IRenderInfo^>^ renderInfoArray)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        for each(INT32 index in indices)
        {
            UpdateRenderInfoCommand^ command =
                gcnew UpdateRenderInfoCommand(kind, key, index, renderInfoArray);
            result |= this->SendCommandQueue->PushBack(command);
        }
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditStringRenderInfo(String^ renderInfoName)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        for each(INT32 index in indices)
        {
            EditStringRenderInfoCommand^ command =
                gcnew EditStringRenderInfoCommand(kind, key, index, renderInfoName);
            result |= this->SendCommandQueue->PushBack(command);
        }
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditIntRenderInfo(String^ renderInfoName, int slotNumber, int itemValue)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        for each(INT32 index in indices)
        {
            EditIntRenderInfoCommand^ command =
                gcnew EditIntRenderInfoCommand(kind, key, index, renderInfoName, slotNumber, itemValue);
            result |= this->SendCommandQueue->PushBack(command);
        }
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::EditFloatRenderInfo(String^ renderInfoName, int slotNumber, float itemValue)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        for each(INT32 index in indices)
        {
            EditFloatRenderInfoCommand^ command =
                gcnew EditFloatRenderInfoCommand(kind, key, index, renderInfoName, slotNumber, itemValue);
            result |= this->SendCommandQueue->PushBack(command);
        }
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
bool HIO::LoadAnimation(nw::g3d::edit::detail::FileDataKind fileKind, IEditTarget^ targetAnimation, FileData^ fileData)
{
    if (targetAnimation == nullptr)
    {
        return false;
    }

    if (!CheckFile(fileData->FileName))
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    LoadAnimCommand^ command =
        gcnew LoadAnimCommand(kind, targetAnimation, fileData, fileKind);
    return this->SendCommandQueue->PushBack(command);
}
//-------------------------------------------------------------------------------------------------
bool HIO::UnloadAnimation(nw::g3d::edit::detail::FileDataKind fileKind, IEditTarget^ targetAnimation)
{
    if (targetAnimation == nullptr)
    {
        return false;
    }

    if (targetAnimation->ResFileKey == 0)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    UnloadAnimCommand^ command =
        gcnew UnloadAnimCommand(kind, targetAnimation, fileKind);
    return this->SendCommandQueue->PushBack(command);
}
//-------------------------------------------------------------------------------------------------
bool HIO::ReloadAnimation(nw::g3d::edit::detail::FileDataKind fileKind, IEditTarget^ targetAnimation, FileData^ fileData)
{
    if (targetAnimation == nullptr)
    {
        return false;
    }

    if (!CheckFile(fileData->FileName))
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();
    ReloadAnimCommand^ command =
        gcnew ReloadAnimCommand(kind, targetAnimation, fileData, fileKind);
    return this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
generic<typename ValueType>
bool HIO::EditMaterialValue(ValueType value, nw::g3d::edit::detail::EditTargetKind editTargetKind)
{
    if (!EditMaterialIniternal())
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedMaterials->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedMaterials->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        EditMaterialValueCommand<ValueType>^ command =
            gcnew EditMaterialValueCommand<ValueType>(
            kind,
            key,
            indices,
            value,
            editTargetKind);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}

//-------------------------------------------------------------------------------------------------
generic<typename ValueType>
bool HIO::EditBoneValue(ValueType value, nw::g3d::edit::detail::EditTargetKind editTargetKind)
{
    if (SelectedBones->Count <= 0)
    {
        return false;
    }

    TargetEndianKind kind = GetTargetEndian();

    bool result = false;
    for each (IEditModelTarget^ key in SelectedBones->Keys)
    {
        HashSet<INT32>^ indexSet;
        if (!SelectedBones->TryGetValue(key, indexSet))
        {
            continue;
        }

        if (key->ModelObjKey == 0)
        {
            continue;
        }

        if (indexSet->Count <= 0)
        {
            continue;
        }

        array<INT32>^ indices = gcnew array<INT32>(indexSet->Count);
        indexSet->CopyTo(indices);

        EditBoneValueCommand<ValueType>^ command =
            gcnew EditBoneValueCommand<ValueType>(
            kind,
            key,
            indices,
            value,
            editTargetKind);
        result |= this->SendCommandQueue->PushBack(command);
    }

    return result;
}
//-------------------------------------------------------------------------------------------------
void HIO::ExecuteModelUserScript(
    IEditModelTarget^ targetModel,
    String^ scriptText)
{
    if (targetModel->ModelObjKey == 0)
    {
        throw gcnew ArgumentException("ModelObjKey must not be zero");
    }

    array<Int32>^ selectedBoneIndices = nullptr;
    array<Int32>^ selectedShapeIndices = nullptr;
    array<Int32>^ selectedMaterialIndices = nullptr;

    {
        HashSet<INT32>^ boneIndexSet;
        if (targetModel->ModelObjKey != 0 &&
            SelectedBones->TryGetValue(targetModel, boneIndexSet) &&
            boneIndexSet->Count > 0)
        {
            selectedBoneIndices = gcnew array<INT32>(boneIndexSet->Count);
            boneIndexSet->CopyTo(selectedBoneIndices);
        }
        else
        {
            selectedBoneIndices = gcnew array<INT32>(0);
        }
    }

    {
        HashSet<INT32>^ shapeIndexSet;
        if (targetModel->ModelObjKey != 0 &&
            SelectedShapes->TryGetValue(targetModel, shapeIndexSet) &&
            shapeIndexSet->Count > 0)
        {
            selectedShapeIndices = gcnew array<INT32>(shapeIndexSet->Count);
            shapeIndexSet->CopyTo(selectedShapeIndices);
        }
        else
        {
            selectedShapeIndices = gcnew array<INT32>(0);
        }
    }

    {
        HashSet<INT32>^ materialIndexSet;
        if (targetModel->ModelObjKey != 0 &&
            SelectedMaterials->TryGetValue(targetModel, materialIndexSet) &&
            materialIndexSet->Count > 0)
        {
            selectedMaterialIndices = gcnew array<INT32>(materialIndexSet->Count);
            materialIndexSet->CopyTo(selectedMaterialIndices);
        }
        else
        {
            selectedMaterialIndices = gcnew array<INT32>(0);
        }
    }

    ExecuteUserScriptCommand^ command = gcnew ExecuteUserScriptCommand(
        targetModel,
        scriptText,
        selectedBoneIndices,
        selectedShapeIndices,
        selectedMaterialIndices,
        GetTargetEndian());
    this->SendCommandQueue->PushBack(command);
}

//-------------------------------------------------------------------------------------------------
HIO::HIO() : HIOBase()
{
    m_TargetAnimation = nullptr;

    m_RecvInfo = gcnew NintendoWare::G3d::Edit::RecvInfo();

    Device = gcnew CafeCommDevice(m_RecvInfo);
    Device->OpenConnection();

    SetRecvInfo(m_RecvInfo);
}

}}} // namespace NintendoWare.G3d.Edit
