﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include "ExecuteUserScriptCommand.h"
#include "CommandUtility.h"
#include "Utility.h"

#include <nw/g3d/res/g3d_ResUtility.h>
#include <nw/g3d/ut/g3d_Inlines.h>

#include <msclr/marshal_cppstd.h>

using namespace NintendoWare::G3d::Edit;
using namespace nw::g3d::edit;
using namespace nw::g3d::edit::detail;
using namespace nw::g3d::tool;
using namespace msclr::interop;
using namespace System::Diagnostics;

ExecuteUserScriptCommand::ExecuteUserScriptCommand(
    IEditModelTarget^ targetModel,
    String^ scriptText,
    array<Int32>^ selectedBones,
    array<Int32>^ selectedShapes,
    array<Int32>^ selectedMaterials,
    TargetEndianKind targetEndian)
    : EditCommand(targetEndian)
    , m_TargetModel(targetModel)
    , m_SelectedBones(gcnew List<Int32>(selectedBones))
    , m_SelectedShapes(gcnew List<Int32>(selectedShapes))
    , m_SelectedMaterials(gcnew List<Int32>(selectedMaterials))
    , m_TargetKind(USER_SCRIPT_TARGET_KIND_MODEL)
{
    m_ScriptText = dynamic_cast<String^>(scriptText->Clone());
    m_TargetName = "test";
}

bool ExecuteUserScriptCommand::MakeCommandPacket()
{
    // ブロックサイズは可変長
    size_t blockSize = sizeof(UserScriptBlock);
    size_t dynamicDataSize = m_ScriptText->Length + 1
                           + m_TargetName->Length + 1
                           + m_SelectedBones->Count * sizeof(s32)
                           + m_SelectedShapes->Count * sizeof(s32)
                           + m_SelectedMaterials->Count * sizeof(s32);
    size_t packetSize = nw::g3d::tool::Align(CommandUtility::GetPacketHeaderSize() + static_cast<u32>(blockSize + dynamicDataSize), NW_G3D_EDIT_ALIGNMENT);

    this->AllocPacketBuffer(static_cast<int>(packetSize));
    if (PacketBuffer == nullptr)
    {
        throw gcnew InsufficientMemoryException("Failed to allocate packet buffer");
    }

    Debug::Assert(PacketBuffer->Length == packetSize);

    // パケットを作成
    {
        array<Byte>^ packetBuffer = PacketBuffer;
        pin_ptr<u8> bufferPtr = &packetBuffer[0];
        UserScriptPacket* packet = reinterpret_cast<UserScriptPacket*>(bufferPtr);
        packet->header = CommandUtility::GetPacketHeader();
        packet->header.command = OTHER_EXECUTE_USER_SCRIPT_FLAG;
        packet->header.dataSize = static_cast<s32>(packetSize - CommandUtility::GetPacketHeaderSize());

        s32* pSelectedBoneIndices = reinterpret_cast<s32*>(packet->block.dynamicData);
        s32* pSelectedShapeIndices = pSelectedBoneIndices + m_SelectedBones->Count;
        s32* pSelectedMaterialIndices = pSelectedShapeIndices + m_SelectedShapes->Count;

        s8* pScriptText = reinterpret_cast<s8*>(pSelectedMaterialIndices + m_SelectedMaterials->Count);
        s8* pTargetName = pScriptText + m_ScriptText->Length + 1;

        packet->block.targetModelObjKey = m_TargetModel->ModelObjKey;
        packet->block.scriptTextLength = m_ScriptText->Length;
        packet->block.targetNameLength = m_TargetName->Length;
        packet->block.targetKind = m_TargetKind;

        // 選択されているボーンをコピー
        {
            int selectedBoneCount = m_SelectedBones->Count;
            packet->block.selectedBoneCount = selectedBoneCount;
            for (int i = 0; i < selectedBoneCount; ++i, ++pSelectedBoneIndices)
            {
                *pSelectedBoneIndices =  m_SelectedBones[i];
            }
        }

        // 選択されているシェイプをコピー
        {
            int selectedShapeCount = m_SelectedShapes->Count;
            packet->block.selectedShapeCount = selectedShapeCount;
            for (int i = 0; i < selectedShapeCount; ++i, ++pSelectedShapeIndices)
            {
                *pSelectedShapeIndices =  m_SelectedShapes[i];
            }
        }

        // 選択されているマテリアルをコピー
        {
            int selectedMaterialCount = m_SelectedMaterials->Count;
            packet->block.selectedMaterialCount = selectedMaterialCount;
            for (int i = 0; i < selectedMaterialCount; ++i, ++pSelectedMaterialIndices)
            {
                *pSelectedMaterialIndices =  m_SelectedMaterials[i];
            }
        }

        StringUtility::CopyString(pScriptText, m_ScriptText, m_ScriptText->Length + 1);
        StringUtility::CopyString(pTargetName, m_TargetName, m_TargetName->Length + 1);

        if (TargetEndian == TargetEndianKind::BigEndian)
        {
            nw::g3d::tool::Endian::Swap(&packet->header);
            nw::g3d::tool::Endian::Swap(&packet->block.targetModelObjKey);
            nw::g3d::tool::Endian::Swap(&packet->block.scriptTextLength);
            nw::g3d::tool::Endian::Swap(&packet->block.targetNameLength);
            nw::g3d::tool::Endian::Swap(&packet->block.selectedBoneCount);
            nw::g3d::tool::Endian::Swap(&packet->block.selectedShapeCount);
            nw::g3d::tool::Endian::Swap(&packet->block.selectedMaterialCount);
            nw::g3d::tool::Endian::Swap(&packet->block.targetKind);
        }
    }

    return true;
}

