﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "EditBinaryBlock.h"

#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nw/g3d/res/g3d_ResAnimCurve.h>
#undef NOMINMAX

#include <nn/g3d/g3d_ResAnimCurve.h>

namespace nw {
namespace g3d {
namespace tool {

class EditBinAnimCurve : public EditBinaryBlock
{
public:
    EditBinAnimCurve()
        : EditBinaryBlock()
        , m_Type(FLOAT)
        , m_Scale(1.f)
        , m_Offset(0.f)
        , m_InterpolationType(HERMITE)
        , m_KeyTypeFlag(ResAnimCurve::KEY32)
        , m_FrameTypeFlag(ResAnimCurve::FRAME32)
        , m_KeyType(KEY32)
        , m_FrameType(FRAME32)
    {
        SetPreWrapType(Clamp);
        SetPostWrapType(Clamp);
    }

    void AddCurveValue(float frame, float value, float inSlope, float outSlope);

    virtual void Build();
    virtual void CalcSize();
    virtual void Convert(const Context& ctx);

    enum InterpolationType
    {
        HERMITE = 0,
        LINEAR = 1,
        STEP = 2
    };

    enum Type
    {
        FLOAT,
        INT,
        BOOL,
        NUM_TYPE
    };

    enum KeyType
    {
        KEY_NONE = 0,
        KEY32 = 1,
        KEY16 = 2,
        KEY8 = 3
    };

    enum FrameType
    {
        FRAME_NONE = 0,
        FRAME32 = 1,
        FRAME16 = 2,
        FRAME8 = 3
    };

    enum WrapType
    {
        Clamp,
        Repeat,
        Mirror,
        RelativeRepeat
    };

    void SetType(Type type) { m_Type = type; }
    Type GetType() const { return m_Type; }

    void SetTargetOffset(ptrdiff_t offset) { m_TargetOffset = offset; }
    void SetDegreeValue(bool degreeValue) { m_IsDegreeValue = degreeValue; }
    void SetScale(float scale) { m_Scale = scale; }
    void SetOffset(float offset) { m_Offset = offset; }
    void SetInterpolationType(InterpolationType type) { m_InterpolationType = type; }

    void SetKeyType(KeyType type)
    {
        switch(type)
        {
        case KEY32:
            m_KeyTypeFlag = ResAnimCurve::KEY32;
            break;
        case KEY16:
            m_KeyTypeFlag = ResAnimCurve::KEY16;
            break;
        case KEY8:
            m_KeyTypeFlag = ResAnimCurve::KEY8;
            break;
        }
        m_KeyType = type;
    }
    void SetFrameType(FrameType type)
    {
        switch(type)
        {
        case FRAME32:
            m_FrameTypeFlag = ResAnimCurve::FRAME32;
            break;
        case FRAME16:
            m_FrameTypeFlag = ResAnimCurve::FRAME16;
            break;
        case FRAME8:
            m_FrameTypeFlag = ResAnimCurve::FRAME8;
            break;
        }
        m_FrameType = type;
    }

    void SetPreWrapType(WrapType type)
    {
        switch(type)
        {
        case Clamp:
            m_PreWrapTypeFlag = ResAnimCurve::WRAP_CLAMP;
            break;
        case Repeat:
            m_PreWrapTypeFlag = ResAnimCurve::WRAP_REPEAT;
            break;
        case Mirror:
            m_PreWrapTypeFlag = ResAnimCurve::WRAP_MIRROR;
            break;
        case RelativeRepeat:
            m_PreWrapTypeFlag = ResAnimCurve::WRAP_RELATIVE_REPEAT;
            break;
        }
        m_PreWrapType = type;
    }

    void SetPostWrapType(WrapType type)
    {
        switch(type)
        {
        case Clamp:
            m_PostWrapTypeFlag = ResAnimCurve::WRAP_CLAMP;
            break;
        case Repeat:
            m_PostWrapTypeFlag = ResAnimCurve::WRAP_REPEAT;
            break;
        case Mirror:
            m_PostWrapTypeFlag = ResAnimCurve::WRAP_MIRROR;
            break;
        case RelativeRepeat:
            m_PostWrapTypeFlag = ResAnimCurve::WRAP_RELATIVE_REPEAT;
            break;
        }
        m_PostWrapType = type;
    }

private:
    void QuantizeFrame(const Context& ctx, ResAnimCurveData& curve);
    void QuantizeHermite(const Context& ctx, ResAnimCurveData& curve);
    void QuantizeLinear(const Context& ctx, ResAnimCurveData& curve);
    void QuantizeStepFloat(const Context& ctx, ResAnimCurveData& curve);
    void QuantizeStepInt(const Context& ctx, ResAnimCurveData& curve);
    void QuantizeStepBool(const Context& ctx, ResAnimCurveData& curve);

    void QuantizeFrame(const Context& ctx, nn::g3d::ResAnimCurveData& curve);
    void QuantizeHermite(const Context& ctx, nn::g3d::ResAnimCurveData& curve);
    void QuantizeLinear(const Context& ctx, nn::g3d::ResAnimCurveData& curve);
    void QuantizeStepFloat(const Context& ctx, nn::g3d::ResAnimCurveData& curve);
    void QuantizeStepInt(const Context& ctx, nn::g3d::ResAnimCurveData& curve);
    void QuantizeStepBool(const Context& ctx, nn::g3d::ResAnimCurveData& curve);


private:
    Type m_Type;
    ptrdiff_t m_TargetOffset;
    bool m_IsDegreeValue;
    static const int ALIGNMENT_FLAG = 32;
    float m_Scale;
    float m_Offset;
    InterpolationType m_InterpolationType;

    u32                 m_KeyTypeFlag;
    u32                 m_FrameTypeFlag;
    u32                 m_PreWrapTypeFlag;
    u32                 m_PostWrapTypeFlag;

    KeyType             m_KeyType;
    FrameType           m_FrameType;
    WrapType            m_PreWrapType;
    WrapType            m_PostWrapType;

    struct CurveValue
    {
        float frame;
        float value;
        float inSlope;
        float outSlope;
    };

    std::vector<CurveValue> m_CurveValueArray;

    struct FVSPair
    {
        union
        {
            float f[7];
            struct Hermite
            {
                float frame0, value0, in_slope0, out_slope0, frame1, value1, in_slope1;
            } hermite;
            struct Linear
            {
                float frame0, value0, frame1, value1;
            } linear;
            struct Step
            {
                float frame0, value0;
            } step;
        };
    };

    std::vector<FVSPair> m_FVSPairArray;

    enum
    {
        FRAME,
        KEY,
        NUM_CHUNK
    };

    Chunk m_Chunk[NUM_CHUNK];
};

}}} // namespace nw::g3d::tool
