﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "Socket.h"
#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>
#include <string>
#include <vector>
#include <nn/g3d/viewer/detail/g3d_ViewerPacketDefine.h>

namespace nw {
namespace g3d {
namespace tool {

class Client
{
public:
    enum RenderInfoType
    {
        RENDER_INFO_INT,
        RENDER_INFO_FLOAT,
        RENDER_INFO_STRING
    };

    struct RenderInfoItem
    {
        std::string choice;
        std::vector<u8> alias;
    };

    struct RenderInfoValue
    {
        union
        {
            s32 iDefault;
            f32 fDefault;
        };
        std::string sDefault;
    };

    struct RenderInfoLabel
    {
        std::string labelName;
        RenderInfoType type;
        std::vector<RenderInfoItem> items;
        std::vector<RenderInfoValue> values;

        union
        {
            s32 iMinValue;
            f32 fMinValue;
        };
        union
        {
            s32 iMaxValue;
            f32 fMaxValue;
        };
    };

    struct RenderInfo
    {
        u32 modelKey;
        s32 materialIndex;

        std::vector<RenderInfoLabel> labels;
    };

    struct ModelLayout
    {
        union
        {
            struct
            {
                f32 x, y, z;
            };
            f32 a[3];
        } scale;

        union
        {
            struct
            {
                f32 x, y, z;
            };
            f32 a[3];
        } rotate;

        union
        {
            struct
            {
                f32 x, y, z;
            };
            f32 a[3];
        } translate;
    };

    struct ModifiedShaderProgramOptionInfo
    {
        std::string option;
        std::string choice;
    };

    struct ModifiedShaderProgram
    {
        u32 shaderArchiveKey;
        s32 shadingModelIndex;
        s32 shaderProgramIndex;

        std::vector<ModifiedShaderProgramOptionInfo> optionInfos;
    };

    struct MaterialPickup
    {
        u32 modelObjKey;
        s32 materialIndex;
    };

    struct Pickup
    {
        std::vector<MaterialPickup> materialPickups;
    };

public:
    explicit Client(bool isSwap);
    ~Client();

    NintendoWare::G3d::Edit::HtcResult^ Open()
    {
        return m_Socket.Open();
    }

    void Close()
    {
        m_Socket.Close();
    }

    NintendoWare::G3d::Edit::HtcResult^ Connect(u16 port)
    {
        return m_Socket.Connect(port);
    }

    void Poll();

    NintendoWare::G3d::Edit::HtcResult^ OpenPing()
    {
        return m_PingSocket.Open();
    }

    void ClosePing()
    {
        ResetPing();
        m_PingSocket.Close();
    }

    NintendoWare::G3d::Edit::HtcResult^ ConnectPing(u16 pingPort)
    {
        return m_PingSocket.Connect(pingPort);
    }

    void ResetPing();

    void PollPing();

    Socket* GetSocket() { return &m_Socket; }

    bool IsPingConnected() const { return m_PingSocket.IsConnected(); }

    void ResetBeginFreezeFlag()
    {
        m_IsBeginFreezeReceived = false;
    }

    void ResetEndFreezeFlag()
    {
        m_IsEndFreezeReceived = false;
    }

    void ResetAttachFlag()
    {
        m_AttachKind = 0;
        m_ModelObjKey = 0;
        m_ShaderArchiveKey = 0;
        m_AttachPathName = "";
        m_IsAttachReceived = false;
        m_IsAttachShaderArchiveBinary = false;
    }

    void ResetDetachFlag()
    {
        m_AttachKind = 0;
        m_ModelObjKey = 0;
        m_ShaderArchiveKey = 0;
        m_AttachPathName = "";
        m_IsDetachReceived = false;
        m_IsAttachShaderArchiveBinary = false;
    }

    void ResetFileLoadFlag()
    {
        m_ToolKey = 0;
        m_ResFileKey = 0;
        m_ResModelKey = 0;
        m_ModelObjKey = 0;
        m_ShaderArchiveKey = 0;
        m_IsFileLoaded = false;
        m_IsModelFileLoaded = false;
        m_IsFileReloaded = false;
        m_IsModelFileReloaded = false;
        m_NewResFileKey = 0;
    }

    void ResetRenderInfoFlag()
    {
        m_IsRenderInfoReceived = false;
    }

    void ResetModelLayoutFlag()
    {
        m_IsModelLayoutReceived = false;
    }

    void ResetModifiedShaderProgramFlag()
    {
        m_ModifiedShaderProgram.shaderArchiveKey = 0;
        m_ModifiedShaderProgram.shadingModelIndex = -1;
        m_ModifiedShaderProgram.shaderProgramIndex = -1;
        m_ModifiedShaderProgram.optionInfos.clear();
        m_IsModifiedShaderProgramReceived = false;
    }

    void ResetPickupFlag()
    {
        m_Pickup.materialPickups.clear();
        m_IsPickupReceived = false;
    }

    void ResetPlayFrameCtrlFlag()
    {
        m_IsPlayFrameCtrlReceived = false;
    }

    void ResetStopFrameCtrlFlag()
    {
        m_IsStopFrameCtrlReceived = false;
    }

    void ResetSendFrameFlag()
    {
        m_IsSendFrameReceived = false;
    }

    void ResetSendFrameStepFlag()
    {
        m_IsSendFrameStepReceived = false;
    }

    void ResetSendModelNextAnimFlag()
    {
        m_IsSendModelNextAnimReceived = false;
    }

    void ResetSendModelPrevAnimFlag()
    {
        m_IsSendModelPrevAnimReceived = false;
    }

    void ResetAbnormalPacketFlag()
    {
        m_IsAbnormalPacketReceived = false;
    }

    void ResetIncorrectVersionFlag()
    {
        m_IsIncorrectVersionReceived = false;
    }

    void ResetRuntimeErrorFlag()
    {
        m_HasReceivedRuntimeError = false;
    }

    void ResetShowMessageRequestedFlag()
    {
        m_IsShowMessageRequested = false;
    }

    bool IsBeginFreezeReceived() const { return m_IsBeginFreezeReceived; }
    bool IsEndFreezeReceived() const { return m_IsEndFreezeReceived; }

    bool IsAttachReceived() const { return m_IsAttachReceived; }
    bool IsDetachReceived() const { return m_IsDetachReceived; }
    bool IsAttachShaderArchiveBinary() const { return m_IsAttachShaderArchiveBinary; }
    u32 GetAttachKind() const { return m_AttachKind; }
    u32 GetToolKey() const { return m_ToolKey; }
    u32 GetResFileKey() const { return m_ResFileKey; }
    u32 GetResModelKey() const { return m_ResModelKey; }
    u32 GetModelObjKey() const { return m_ModelObjKey; }
    u32 GetNewResFileKey() const { return m_NewResFileKey; }
    u32 GetShaderArchiveKey() const { return m_ShaderArchiveKey; }
    bool IsFileLoaded() const { return m_IsFileLoaded; }
    bool IsModelFileLoaded() const { return m_IsModelFileLoaded; }
    bool IsFileReloaded() const { return m_IsFileReloaded; }
    bool IsModelFileReloaded() const { return m_IsModelFileReloaded; }
    const char* GetFileName() const { return m_FileName.c_str(); }
    const char* GetAttachPathName() const { return m_AttachPathName.c_str(); }

    const RenderInfo* GetRenderInfo() const { return &m_RenderInfo; }
    bool IsRenderInfoReceived() const { return m_IsRenderInfoReceived; }

    const ModelLayout* GetModelLayout() const { return &m_ModelLayout; }
    bool IsModelLayoutReceived() const { return m_IsModelLayoutReceived; }

    const ModifiedShaderProgram* GetModifiedShaderProgram() const { return &m_ModifiedShaderProgram; }
    bool IsModifiedShaderProgramReceived() const { return m_IsModifiedShaderProgramReceived; }

    const Pickup* GetPickup() const { return &m_Pickup; }
    bool IsPickupReceived() const { return m_IsPickupReceived; }

    bool IsPlayFrameCtrlReceived() const { return m_IsPlayFrameCtrlReceived; }
    bool IsStopFrameCtrlReceived() const { return m_IsStopFrameCtrlReceived; }
    bool IsSendFrameReceived() const { return m_IsSendFrameReceived; }
    bool IsSendFrameStepReceived() const { return m_IsSendFrameStepReceived; }
    bool IsSendModelNextAnimReceived() const { return m_IsSendModelNextAnimReceived; }
    bool IsSendModelPrevAnimReceived() const { return m_IsSendModelPrevAnimReceived; }
    float GetFrame() const { return m_Frame; }
    float GetFrameStep() const { return m_FrameStep; }

    bool IsAbnormalPacketReceived() const { return m_IsAbnormalPacketReceived; }
    bool IsIncorrectVersionReceived() const { return m_IsIncorrectVersionReceived; }
    u32 GetLastIncorrectVersion() const { return m_LastIncorrectVersion; }

    bool IsPingReceived() const { return m_IsPingReceived; }
    u16 GetCodePage() const { return m_CodePage; }
    u32 GetPingState() const { return m_PingState; }
    u32 GetPingFlag() const { return m_PingFlag; }

    bool HasReceivedRuntimeError() const { return m_HasReceivedRuntimeError; }
    NintendoWare::G3d::Edit::RuntimeError GetRuntimeError() const { return m_RuntimeError; }

    bool IsShowMessageRequested() const { return m_IsShowMessageRequested; }
    NintendoWare::G3d::Edit::MessageType GetMessageType() const { return m_MessageType; }
    NintendoWare::G3d::Edit::MessageDestination GetMessageDestination() const { return m_MessageDestination; }
    int GetMessageCodePage() const { return m_MessageCodePage; }
    const char* GetMessage() const { return m_Message.c_str(); }

private:
    void AnalyzePacket();
    void AnalyzeCommand();

private:
    Socket m_Socket;
    Socket m_PingSocket;
    volatile bool m_IsReadStarted;
    volatile bool m_IsCommandAnalyzing;
    nw::g3d::edit::detail::PacketHeader m_PacketHeader;
    nn::g3d::viewer::detail::PingPacket m_PingPacket;
    std::vector<u32> m_WorkBuffer;
    std::string m_FileName;
    std::string m_AttachPathName;

    volatile bool m_IsBeginFreezeReceived;
    volatile bool m_IsEndFreezeReceived;

    volatile bool m_IsAttachReceived;
    volatile bool m_IsDetachReceived;
    volatile u32 m_AttachKind;
    volatile u32 m_ToolKey;
    volatile bool m_IsFileLoaded;

    volatile bool m_IsModelFileLoaded;
    volatile u32 m_ResFileKey;
    volatile u32 m_ResModelKey;
    volatile u32 m_ModelObjKey;
    volatile u32 m_ShaderArchiveKey;
    volatile bool m_IsAttachShaderArchiveBinary;

    volatile bool m_IsModelFileReloaded;
    volatile bool m_IsFileReloaded;
    volatile u32 m_NewResFileKey;

    volatile u16 m_CodePage;

    RenderInfo m_RenderInfo;
    volatile bool m_IsRenderInfoReceived;

    ModelLayout m_ModelLayout;
    volatile bool m_IsModelLayoutReceived;

    ModifiedShaderProgram m_ModifiedShaderProgram;
    volatile bool m_IsModifiedShaderProgramReceived;

    Pickup  m_Pickup;
    volatile bool m_IsPickupReceived;

    NintendoWare::G3d::Edit::RuntimeError m_RuntimeError;
    volatile bool m_HasReceivedRuntimeError;

    NintendoWare::G3d::Edit::MessageType m_MessageType;
    NintendoWare::G3d::Edit::MessageDestination m_MessageDestination;
    int m_MessageCodePage;
    std::string m_Message;
    volatile bool m_IsShowMessageRequested;

    volatile bool m_IsPlayFrameCtrlReceived;
    volatile bool m_IsStopFrameCtrlReceived;
    volatile bool m_IsSendFrameReceived;
    volatile bool m_IsSendFrameStepReceived;
    volatile bool m_IsSendModelNextAnimReceived;
    volatile bool m_IsSendModelPrevAnimReceived;
    volatile float m_Frame;
    volatile float m_FrameStep;

    volatile bool m_IsAbnormalPacketReceived;
    volatile bool m_IsIncorrectVersionReceived;
    volatile u32  m_LastIncorrectVersion;

    // Ping関連
    volatile bool m_IsPingSent;
    volatile bool m_IsPingReadStarted;
    volatile bool m_IsPingReceived;
    volatile u32 m_PingState;
    volatile u32 m_PingFlag;

    volatile bool m_IsSwap; // エンディアンスワップをするか
};

}}} // namespace nw::g3d::tool
