﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Client.h"
#include "Utility.h"

namespace nw {
namespace g3d {
namespace tool {

using namespace nw::g3d::edit::detail;

Client::Client(bool isSwap)
    : m_IsReadStarted(false)
    , m_IsCommandAnalyzing(false)
    , m_IsBeginFreezeReceived(false)
    , m_IsEndFreezeReceived(false)
    , m_IsAttachReceived(false)
    , m_IsDetachReceived(false)
    , m_ToolKey(0)
    , m_IsFileLoaded(false)
    , m_IsModelFileLoaded(false)
    , m_ResFileKey(0)
    , m_ResModelKey(0)
    , m_ModelObjKey(0)
    , m_ShaderArchiveKey(0)
    , m_IsFileReloaded(false)
    , m_IsModelFileReloaded(false)
    , m_NewResFileKey(0)

    , m_CodePage(0)

    , m_IsRenderInfoReceived(false)
    , m_IsModelLayoutReceived(false)
    , m_IsModifiedShaderProgramReceived(false)

    , m_IsPickupReceived(false)

    , m_IsPlayFrameCtrlReceived(false)
    , m_IsStopFrameCtrlReceived(false)
    , m_IsSendFrameReceived(false)
    , m_IsSendFrameStepReceived(false)
    , m_IsSendModelNextAnimReceived(false)
    , m_IsSendModelPrevAnimReceived(false)
    , m_Frame(0.f)
    , m_FrameStep(0.f)

    , m_IsAbnormalPacketReceived(false)
    , m_IsIncorrectVersionReceived(false)
    , m_LastIncorrectVersion(0)

    , m_RuntimeError(NintendoWare::G3d::Edit::RuntimeError::NoError)

    , m_IsPingSent(false)
    , m_IsPingReadStarted(false)
    , m_IsPingReceived(false)
    , m_PingState(PING_STATE_NORMAL)
    , m_PingFlag(0)

    , m_HasReceivedRuntimeError(false)
    , m_IsShowMessageRequested(false)

    , m_IsSwap(isSwap)
{
    memset(&m_PacketHeader, 0, sizeof(m_PacketHeader));
    memset(&m_PingPacket, 0, sizeof(m_PingPacket));
    m_Socket.SetDebugLogEnabled();
    m_PingSocket.SetDebugLogDisabled();
}

Client::~Client()
{
    m_Socket.Close();
    m_PingSocket.Close();
}

void Client::Poll()
{
    if (!m_Socket.IsConnected())
    {
        m_Socket.ResetReadFlag();
        m_IsReadStarted = false;
        m_IsCommandAnalyzing = false;
        return;
    }

    m_Socket.Poll();

    if (!m_Socket.IsConnected())
    {
        // 接続が切れたので、状態をリセットする。
        m_PingState = PING_STATE_NORMAL;
        m_PingFlag = 0;

        // m_Socket.Poll() でソケットが閉じられた場合にはここで return する。
        return;
    }

    if (!m_IsReadStarted && !m_IsCommandAnalyzing)
    {
        System::Diagnostics::Debug::WriteLine("Reading packet header");
        m_IsReadStarted = m_Socket.ReadASync(&m_PacketHeader, sizeof(m_PacketHeader));
    }

    if (m_IsReadStarted)
    {
        AnalyzePacket();
    }

    if (m_IsCommandAnalyzing)
    {
        AnalyzeCommand();
    }
}

void Client::ResetPing()
{
    m_PingSocket.ResetReadFlag();
    m_IsPingSent = false;
    m_IsPingReadStarted = false;
    m_IsPingReceived = false;
    m_PingState = PING_STATE_NORMAL;
    m_PingFlag = 0;
}

void Client::PollPing()
{
    if (!IsPingConnected())
    {
        m_PingSocket.ResetReadFlag();
        m_IsPingSent = false;
        m_IsPingReadStarted = false;
        m_IsPingReceived = false;
        return;
    }

    if (!m_IsPingSent)
    {
        m_PingPacket.header.magic = NN_G3D_EDIT_MAGIC;
        m_PingPacket.header.verWord = NN_G3D_EDIT_VERSION;
        m_PingPacket.header.dataSize = sizeof(PingBlock);
        m_PingPacket.header.command = SYSTEM_PING_RECV_COMMAND_FLAG;
        m_PingPacket.block.codePage = m_CodePage;

        if (m_IsSwap)
        {
            Endian::Swap(&m_PingPacket.header);
            Endian::Swap(&m_PingPacket.block.codePage);
            Endian::Swap(&m_PingPacket.block.uniqueID);
        }
        m_IsPingSent = m_PingSocket.WriteSync(&m_PingPacket, sizeof(m_PingPacket));
    }

    if (m_IsPingSent)
    {
        m_PingSocket.Poll();
        if (!m_IsPingReadStarted)
        {
            m_IsPingReadStarted = m_PingSocket.ReadASync(&m_PingPacket, sizeof(m_PingPacket));
        }

        if (m_IsPingReadStarted)
        {
            m_IsPingReceived = true;
            if (!m_PingSocket.IsReading())
            {
                m_IsPingReadStarted = false;
                m_IsPingSent = false;

                // 送信先とのエンディアンが違う場合はスワップ
                if (m_IsSwap)
                {
                    Endian::Swap(&m_PingPacket.header);
                    Endian::Swap(&m_PingPacket.block.codePage);
                    Endian::Swap(&m_PingPacket.block.uniqueID);
                    Endian::Swap(&m_PingPacket.block.state);
                }

                if (m_PingPacket.header.magic != NN_G3D_EDIT_MAGIC ||
                    m_PingPacket.header.verWord != NN_G3D_EDIT_VERSION)
                {
                    m_IsIncorrectVersionReceived = true;
                    m_LastIncorrectVersion = m_PingPacket.header.verWord;
                }

                if (m_PingPacket.header.command != SYSTEM_PING_SEND_COMMAND_FLAG)
                {
                    m_IsPingReceived = false;
                }
                else
                {
                    m_CodePage = static_cast<u16>(m_PingPacket.block.codePage);
                    m_PingState = m_PingPacket.block.state;
                    m_PingFlag = m_PingPacket.block.flag;
                }
                m_PingSocket.ResetReadFlag();
            }
        }
    }

    if (!IsPingConnected())
    {
        m_IsPingSent = false;
        m_IsPingReceived = false;
        m_PingState = PING_STATE_NORMAL;
        m_PingFlag = 0;
    }
}

void Client::AnalyzePacket()
{
    if (m_Socket.IsReading())
    {
        return;
    }

    System::Diagnostics::Debug::WriteLine("Complete reading packet header");

    // 送信先とのエンディアンが違う場合はスワップ
    if (m_IsSwap)
    {
        Endian::Swap(&m_PacketHeader);
    }

    bool isAbnormalPacket = false;
    if (m_PacketHeader.magic != NN_G3D_EDIT_MAGIC ||
        m_PacketHeader.verWord != NN_G3D_EDIT_VERSION)
    {
        System::Diagnostics::Debug::Write("Abnormal packet received: command = ");
        System::Diagnostics::Debug::WriteLine(m_PacketHeader.command);
        isAbnormalPacket = true;
    }

    m_Socket.ResetReadFlag();
    m_IsReadStarted = false;
    m_IsCommandAnalyzing = false;

    if (isAbnormalPacket)
    {
        m_IsAbnormalPacketReceived = true;
        return;
    }

    switch(m_PacketHeader.command)
    {
    case SYSTEM_BEGIN_FREEZE_COMMAND_FLAG:
        m_IsBeginFreezeReceived = true;
        break;
    case SYSTEM_END_FREEZE_COMMAND_FLAG:
        m_IsEndFreezeReceived = true;
        break;
    case EDIT_SEND_ATTACH_COMMAND_FLAG:
    case EDIT_SEND_DETACH_COMMAND_FLAG:
    case EDIT_FILE_LOADED_COMMAND_FLAG:
    case EDIT_FILE_RELOADED_COMMAND_FLAG:
    case EDIT_SEND_RENDER_INFO_COMMAND_FLAG:
    case EDIT_SEND_MODEL_LAYOUT_COMMAND_FLAG:
    case EDIT_SEND_MODIFIED_SHADER_COMMAND_FLAG:
    case PICK_TOOL_MATERIAL_COMMAND_FLAG:
    case SYSTEM_PLAY_FRAME_CTRL_COMMAND_FLAG:
    case SYSTEM_STOP_FRAME_CTRL_COMMAND_FLAG:
    case SYSTEM_SEND_FRAME_COMMAND_FLAG:
    case SYSTEM_SEND_FRAME_STEP_COMMAND_FLAG:
    case SYSTEM_SEND_MODEL_NEXT_ANIM_COMMAND_FLAG:
    case SYSTEM_SEND_MODEL_PREV_ANIM_COMMAND_FLAG:
    case SYSTEM_RUNTIME_ERROR_COMMAND_FLAG:
    case nn::g3d::viewer::detail::OTHER_USER_MESSAGE_COMMAND_FLAG:
        {
            if (m_PacketHeader.dataSize <= 0) // 今の所、dataSize が 0 以下の場合はパケット解析失敗
            {
                return;
            }
            m_WorkBuffer.resize(m_PacketHeader.dataSize);
            void* workBuffer = reinterpret_cast<void*>(&*m_WorkBuffer.begin());
            System::Diagnostics::Debug::WriteLine("Reading packet data");
            m_IsCommandAnalyzing = m_Socket.ReadASync(workBuffer, m_PacketHeader.dataSize);
        }
        break;
    default:
        break;
    }
}

void Client::AnalyzeCommand()
{
    if (m_Socket.IsReading())
    {
        return;
    }

    switch(m_PacketHeader.command)
    {
    case EDIT_SEND_ATTACH_COMMAND_FLAG:
        {
            AttachBlock* block = reinterpret_cast<AttachBlock*>(&*m_WorkBuffer.begin());

            // 送信先とのエンディアンが違う場合はスワップ
            if (m_IsSwap)
            {
                Endian::Swap(block);
            }

            char fileNameBuffer[nw::g3d::edit::detail::NW_G3D_EDIT_FILENAME_MAX];
            fileNameBuffer[nw::g3d::edit::detail::NW_G3D_EDIT_FILENAME_MAX - 1] = '\0';
            const char* fileName = reinterpret_cast<const char*>(block->fileName);
            memcpy(fileNameBuffer, fileName, nw::g3d::edit::detail::NW_G3D_EDIT_FILENAME_MAX - 1);

            char attachPathNameBuffer[nw::g3d::edit::detail::NW_G3D_EDIT_FILENAME_MAX];
            attachPathNameBuffer[nw::g3d::edit::detail::NW_G3D_EDIT_FILENAME_MAX - 1] = '\0';
            const char* attachPathName = reinterpret_cast<const char*>(block->attachFileName);
            memcpy(attachPathNameBuffer, attachPathName, nw::g3d::edit::detail::NW_G3D_EDIT_FILENAME_MAX - 1);

            m_IsAttachReceived = true;
            m_FileName = fileNameBuffer;
            m_AttachPathName = attachPathNameBuffer;
            m_AttachKind = block->attachKind;
            switch(m_AttachKind)
            {
            case nw::g3d::edit::detail::ATTACH_MODEL:
                m_ModelObjKey = block->attachedKey;
                break;
            case nw::g3d::edit::detail::ATTACH_SHADER_ARCHIVE:
                m_ShaderArchiveKey = block->attachedKey;
                m_IsAttachShaderArchiveBinary = block->flag & ATTACH_SHADER_ARCHIVE_IS_BINARY;
                break;
            }
        }
        break;
    case EDIT_SEND_DETACH_COMMAND_FLAG:
        {
            AttachBlock* block = reinterpret_cast<AttachBlock*>(&*m_WorkBuffer.begin());

            // 送信先とのエンディアンが違う場合はスワップ
            if (m_IsSwap)
            {
                Endian::Swap(block);
            }

            m_IsDetachReceived = true;
            m_AttachKind = block->attachKind;
            switch(m_AttachKind)
            {
            case nw::g3d::edit::detail::DETACH_MODEL:
                m_ModelObjKey = block->attachedKey;
                break;
            case nw::g3d::edit::detail::DETACH_SHADER_ARCHIVE:
                m_ShaderArchiveKey = block->attachedKey;
                break;
            }
        }
        break;
    case EDIT_FILE_LOADED_COMMAND_FLAG:
    case EDIT_FILE_RELOADED_COMMAND_FLAG:
        {
            FileLoadedBlock* block = reinterpret_cast<FileLoadedBlock*>(&*m_WorkBuffer.begin());

            // 送信先とのエンディアンが違う場合はスワップ
            if (m_IsSwap)
            {
                Endian::Swap(block);
            }

            m_ToolKey = block->toolKey;
            m_ResFileKey = block->resFileKey;
            m_NewResFileKey = block->newResFileKey;

            if (block->newResFileKey > 0)
            {
                m_IsFileReloaded = true;
                if (block->resModelKey > 0)
                {
                    m_ResModelKey = block->resModelKey;
                    m_ModelObjKey = block->modelObjKey;
                    m_IsModelFileReloaded = true;
                }
            }
            else
            {
                m_IsFileLoaded = true;
                if (block->resModelKey > 0)
                {
                    m_ResModelKey = block->resModelKey;
                    m_ModelObjKey = block->modelObjKey;
                    m_IsModelFileLoaded = true;
                }
            }
        }
        break;
    case EDIT_SEND_RENDER_INFO_COMMAND_FLAG:
        {
            // 新規の情報を登録するので、RenderInfo のラベルをクリア
            m_RenderInfo.labels.clear();

            size_t headerOffset = sizeof(nw::g3d::edit::detail::PacketHeader);
            RenderInfoSendInfo* info = reinterpret_cast<RenderInfoSendInfo*>(&*m_WorkBuffer.begin());

            // 送信先とのエンディアンが違う場合はスワップ
            if (m_IsSwap)
            {
                Endian::Swap(info);
            }

            m_RenderInfo.modelKey = info->modelKey;
            m_RenderInfo.materialIndex = info->materialIndex;

            u32 labelInfoSize = sizeof(RenderInfoLabelInfo);
            RenderInfoLabelInfo* firstLabelInfo = nw::g3d::ut::AddOffset<RenderInfoLabelInfo>(info, sizeof(RenderInfoSendInfo));
            for (u32 i = 0; i < info->labelInfoNum; ++i)
            {
                RenderInfoLabelInfo* labelInfo = nw::g3d::ut::AddOffset<RenderInfoLabelInfo>(firstLabelInfo, i * labelInfoSize);
                const char* labelName = nw::g3d::ut::AddOffset<const char>(info, labelInfo->labelOffset - headerOffset);

                RenderInfoLabel label;
                label.labelName = labelName;

                // Choice の設定
                switch(labelInfo->renderInfoType)
                {
                case ResRenderInfo::STRING:
                    {
                        label.type = RENDER_INFO_STRING;
                        RenderInfoChoiceInfo* itemOffset = nw::g3d::ut::AddOffset<RenderInfoChoiceInfo>(info, labelInfo->itemOffset - headerOffset);
                        for (u32 j = 0; j < labelInfo->itemNum; ++j)
                        {
                            const char* itemName = nw::g3d::ut::AddOffset<const char>(info, itemOffset[j].choiceOffset - headerOffset);
                            RenderInfoItem item;
                            item.choice = itemName;
                            if (itemOffset[j].aliasSize >0)
                            {
                                u8* aliasBuffer = nw::g3d::ut::AddOffset<u8>(info, itemOffset[j].aliasOffset - headerOffset);
                                item.alias.resize(itemOffset[j].aliasSize);
                                memcpy_s(&item.alias[0], itemOffset[j].aliasSize, aliasBuffer, itemOffset[j].aliasSize);
                            }
                            label.items.push_back(item);
                        }
                    }
                    break;
                case ResRenderInfo::INT:
                    label.type = RENDER_INFO_INT;
                    label.iMinValue = labelInfo->iMinValue;
                    label.iMaxValue = labelInfo->iMaxValue;
                    break;
                case ResRenderInfo::FLOAT:
                    label.type = RENDER_INFO_FLOAT;
                    label.fMinValue = labelInfo->fMinValue;
                    label.fMaxValue = labelInfo->fMaxValue;
                    break;
                }

                // Defaultが一つ以上存在すれば設定処理を行う。
                if (labelInfo->valueNum > 0)
                {
                    u32* valueOffsetArray = nw::g3d::ut::AddOffset<u32>(info, labelInfo->valueOffset - headerOffset);

                    // Defaultの設定
                    switch(labelInfo->renderInfoType)
                    {
                    case ResRenderInfo::STRING:
                        {
                            u32* valueOffsetArray = nw::g3d::ut::AddOffset<u32>(info, labelInfo->valueOffset - headerOffset);
                            for (u32 i = 0; i < labelInfo->valueNum; ++i)
                            {
                                RenderInfoValue value;
                                const char* defaultName = nw::g3d::ut::AddOffset<const char>(info, valueOffsetArray[i] - headerOffset);
                                value.sDefault.assign(defaultName);
                                label.values.push_back(value);
                            }
                        }
                        break;
                    case ResRenderInfo::INT:
                        {
                            s32* valueOffsetArray = nw::g3d::ut::AddOffset<s32>(info, labelInfo->valueOffset - headerOffset);
                            for (u32 i = 0; i < labelInfo->valueNum; ++i)
                            {
                                RenderInfoValue value;
                                value.iDefault = valueOffsetArray[i];
                                label.values.push_back(value);
                            }
                        }
                        break;
                    case ResRenderInfo::FLOAT:
                        {
                            f32* valueOffsetArray = nw::g3d::ut::AddOffset<f32>(info, labelInfo->valueOffset - headerOffset);
                            for (u32 i = 0; i < labelInfo->valueNum; ++i)
                            {
                                RenderInfoValue value;
                                value.fDefault = valueOffsetArray[i];
                                label.values.push_back(value);
                            }
                        }
                        break;
                    }
                }
                m_RenderInfo.labels.push_back(label);
            }
            m_IsRenderInfoReceived = true;
        }
        break;
    case EDIT_SEND_MODEL_LAYOUT_COMMAND_FLAG:
        {
            ModelLayoutEditBlock* block = reinterpret_cast<ModelLayoutEditBlock*>(&*m_WorkBuffer.begin());

            // 送信先とのエンディアンが違う場合はスワップ
            if (m_IsSwap)
            {
                Endian::Swap(block);
            }

            m_ModelObjKey = block->modelKey;

            for (int i = 0; i < 3; ++i)
            {
                m_ModelLayout.scale.a[i] = block->scale.a[i];
                m_ModelLayout.rotate.a[i] = block->rotate.a[i];
                m_ModelLayout.translate.a[i] = block->translate.a[i];
            }

            m_IsModelLayoutReceived = true;
        }
        break;
    case EDIT_SEND_MODIFIED_SHADER_COMMAND_FLAG:
        {
            // 新規の情報を登録するので、ModifiedShaderProgram のオプション情報をクリア
            m_ModifiedShaderProgram.optionInfos.clear();

            size_t headerOffset = sizeof(nw::g3d::edit::detail::PacketHeader);
            ShaderProgramSendInfo* sendInfo = reinterpret_cast<ShaderProgramSendInfo*>(&*m_WorkBuffer.begin());

            // 送信先とのエンディアンが違う場合はスワップ
            if (m_IsSwap)
            {
                Endian::Swap(sendInfo);
            }

            m_ModifiedShaderProgram.shaderArchiveKey = sendInfo->shaderArchiveKey;
            m_ModifiedShaderProgram.shadingModelIndex = sendInfo->shadingModelIndex;
            m_ModifiedShaderProgram.shaderProgramIndex = sendInfo->shaderProgramIndex;

            u32 optionInfoSize = sizeof(ShaderProgramOptionInfo);
            ShaderProgramOptionInfo* firstOptionInfo = nw::g3d::ut::AddOffset<ShaderProgramOptionInfo>(sendInfo, sizeof(ShaderProgramSendInfo));
            for (u32 i = 0; i < sendInfo->optionInfoNum; ++i)
            {
                ShaderProgramOptionInfo* optionInfo = nw::g3d::ut::AddOffset<ShaderProgramOptionInfo>(firstOptionInfo, i * optionInfoSize);
                const char* option = nw::g3d::ut::AddOffset<const char>(sendInfo, optionInfo->optionOffset - headerOffset);
                const char* choice = nw::g3d::ut::AddOffset<const char>(sendInfo, optionInfo->choiceOffset - headerOffset);
                ModifiedShaderProgramOptionInfo recvOptionInfo;
                recvOptionInfo.option = option;
                recvOptionInfo.choice = choice;
                m_ModifiedShaderProgram.optionInfos.push_back(recvOptionInfo);
            }
            m_IsModifiedShaderProgramReceived = true;
        }
        break;
    case PICK_TOOL_MATERIAL_COMMAND_FLAG:
        {
            m_Pickup.materialPickups.clear();

            size_t headerOffset = sizeof(nw::g3d::edit::detail::PacketHeader);
            PickupSendInfo* sendInfo = reinterpret_cast<PickupSendInfo*>(&*m_WorkBuffer.begin());

            // 送信先とのエンディアンが違う場合はスワップ
            if (m_IsSwap)
            {
                Endian::Swap(sendInfo);
            }

            u32 materialPickupInfoSize = sizeof(MaterialPickupInfo);
            MaterialPickupInfo* firstPickupInfo = nw::g3d::ut::AddOffset<MaterialPickupInfo>(sendInfo, sizeof(PickupSendInfo));
            for (u32 i = 0; i < sendInfo->materialPickupNum; ++i)
            {
                MaterialPickupInfo* pickupInfo = nw::g3d::ut::AddOffset<MaterialPickupInfo>(firstPickupInfo, i * materialPickupInfoSize);
                MaterialPickup pickup;
                pickup.modelObjKey = pickupInfo->modelObjKey;
                pickup.materialIndex = pickupInfo->materialIndex;
                m_Pickup.materialPickups.push_back(pickup);
            }
            if (m_Pickup.materialPickups.size() > 0)
            {
                m_IsPickupReceived = true;
            }
        }
        break;

    case SYSTEM_PLAY_FRAME_CTRL_COMMAND_FLAG:
    case SYSTEM_STOP_FRAME_CTRL_COMMAND_FLAG:
    case SYSTEM_SEND_FRAME_COMMAND_FLAG:
    case SYSTEM_SEND_FRAME_STEP_COMMAND_FLAG:
    case SYSTEM_SEND_MODEL_NEXT_ANIM_COMMAND_FLAG:
    case SYSTEM_SEND_MODEL_PREV_ANIM_COMMAND_FLAG:
        {
            FrameCtrlBlock* block = reinterpret_cast<FrameCtrlBlock*>(&*m_WorkBuffer.begin());

            // 送信先とのエンディアンが違う場合はスワップ
            if (m_IsSwap)
            {
                Endian::Swap(block);
            }

            if (m_PacketHeader.command == SYSTEM_PLAY_FRAME_CTRL_COMMAND_FLAG ||
                m_PacketHeader.command == SYSTEM_STOP_FRAME_CTRL_COMMAND_FLAG ||
                m_PacketHeader.command == SYSTEM_SEND_FRAME_COMMAND_FLAG)
            {
                m_Frame = block->frame;
            }
            else if (m_PacketHeader.command == SYSTEM_SEND_FRAME_STEP_COMMAND_FLAG)
            {
                m_FrameStep = block->frameStep;
            }
            else if (m_PacketHeader.command == SYSTEM_SEND_MODEL_NEXT_ANIM_COMMAND_FLAG ||
                     m_PacketHeader.command == SYSTEM_SEND_MODEL_PREV_ANIM_COMMAND_FLAG)
            {
                m_ModelObjKey = block->modelKey;
            }

            switch(m_PacketHeader.command)
            {
            case SYSTEM_PLAY_FRAME_CTRL_COMMAND_FLAG:
                m_IsPlayFrameCtrlReceived = true;
                break;
            case SYSTEM_STOP_FRAME_CTRL_COMMAND_FLAG:
                m_IsStopFrameCtrlReceived = true;
                break;
            case SYSTEM_SEND_FRAME_COMMAND_FLAG:
                m_IsSendFrameReceived = true;
                break;
            case SYSTEM_SEND_FRAME_STEP_COMMAND_FLAG:
                m_IsSendFrameStepReceived = true;
                break;
            case SYSTEM_SEND_MODEL_NEXT_ANIM_COMMAND_FLAG:
                m_IsSendModelNextAnimReceived = true;
                break;
            case SYSTEM_SEND_MODEL_PREV_ANIM_COMMAND_FLAG:
                m_IsSendModelPrevAnimReceived = true;
                break;
            }
        }
        break;
    case SYSTEM_RUNTIME_ERROR_COMMAND_FLAG:
        {
            RuntimeErrorNotificationInfo* block = reinterpret_cast<RuntimeErrorNotificationInfo*>(&*m_WorkBuffer.begin());

            // 送信先とのエンディアンが違う場合はスワップ
            if (m_IsSwap)
            {
                Endian::Swap(block);
            }

            m_HasReceivedRuntimeError = true;
            m_RuntimeError = static_cast<NintendoWare::G3d::Edit::RuntimeError>(block->runtimeErrorCode);
        }
        break;
    case nn::g3d::viewer::detail::OTHER_USER_MESSAGE_COMMAND_FLAG:
        {
            nn::g3d::viewer::detail::MessageNotificationInfo* block = reinterpret_cast<nn::g3d::viewer::detail::MessageNotificationInfo*>(&*m_WorkBuffer.begin());

            // 送信先とのエンディアンが違う場合はスワップ
            if (m_IsSwap)
            {
                Endian::Swap(block);
            }

            m_IsShowMessageRequested = true;
            m_MessageType = static_cast<NintendoWare::G3d::Edit::MessageType>(block->messageType);
            m_MessageDestination = static_cast<NintendoWare::G3d::Edit::MessageDestination>(block->messageDestination);
            m_MessageCodePage = static_cast<int>(block->messageCodePage);
            const char* messageData = reinterpret_cast<const char*>(block->message);
            System::Diagnostics::Debug::Assert(messageData[block->messageDataSize - 1] == '\0', "Invalid message data received.");
            m_Message = messageData;
        }
        break;
    default:
        break;
    }
    m_Socket.ResetReadFlag();
    m_IsCommandAnalyzing = false;
}

}}} // namespace nw::g3d::tool
