﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.ComponentModel;
using NintendoWare.G3d.Edit.Math;

namespace NintendoWare.G3d.Edit
{
    [Flags]
    public enum RuntimeState
    {
        Normal = 0,
        Keeping = 1 << 0,
        Lock = 1 << 1,
        Unlock = 1 << 2,
        Closed = 1 << 3,
    }

    /// <summary>
    /// 接続先のアドレス形式です。
    /// </summary>
    public enum PlatformAddressType
    {
        Adress32 = 0x0,
        Adress64 = 0x1,
        Unknown,
    }

    /// <summary>
    /// ランタイム側で発生したエラーの識別子です。
    /// g3d_EditDetailDefs.h のRuntimeErrorCodeと値を同期する必要があります。
    /// </summary>
    public enum RuntimeError
    {
        NoError = 0,

        /// <summary>
        /// モデルのマテリアル数がランタイム側のリソースと異なります。
        /// </summary>
        InvalidMaterialCount = 1,

        /// <summary>
        /// アタッチが中断されました。
        /// </summary>
        AttachCanceled = 2,

        /// <summary>
        /// 3DEditor から送られてきたファイルの読み込み処理に失敗しました。
        /// 接続先アプリケーションの実装の問題、もしくはメモリ不足などが原因の可能性があります。
        /// </summary>
        LoadFileFailed = 3,

        /// <summary>
        /// メモリ不足です。
        /// </summary>
        InsufficientMemory = 4,

        /// <summary>
        /// アニメーションのバインドに失敗しました。
        /// </summary>
        BindAnimFailed = 5,

        /// <summary>
        /// リターゲティングホストモデルがランタイム側に読み込まれていません。
        /// </summary>
        RetargetHostModelNotFound = 6,

        /// <summary>
        /// 重複したモデルキーが検知されました。
        /// 接続先アプリケーション、もしくはランタイムライブラリの問題です。
        /// </summary>
        DuplicateModelObjKey = 7,

        /// <summary>
        /// 不正なシェーダーが検出されました。
        /// アタッチしたシェーダーがアプリケーションでロードされているシェーダーと異る構造を持っている場合などに
        /// このエラーが発生します。
        /// アタッチしたシェーダーが間違っていないか、アプリケーションでロードしているシェーダーアーカイブと
        /// 3DEditor にロードしているシェーダー定義がどちらも最新であるかどうかを確認して下さい。
        /// </summary>
        InvalidShaderDetected = 8,

        /// <summary>
        /// アタッチしたモデル中間ファイルが正しくありません。
        /// アプリケーション側でアタッチ対象に登録したモデルが、3DEditor でアタッチしたモデル中間ファイルから
        /// 変換されたものであるかどうかを確認して下さい。
        /// </summary>
        InvalidModelAttached = 9,

        /// <summary>
        /// 3DEditor から指定されたキーに該当するモデルがランタイムライブラリに登録されていません。
        /// 3DEditor、もしくはランタイムライブラリの不具合です。
        /// </summary>
        TargetModelNotFound = 10,

        /// <summary>
        /// 3DEditor が作成したバイナリファイルのオープンに失敗しました。
        /// 詳細はアプリケーションのログに出力されているファイルオープンのリザルトコードを参照してください。
        /// </summary>
        OpenFileFailed = 11,

        /// <summary>
        /// 不明なエラーです。
        /// </summary>
        UnknownError = -1,
    }

    public interface ICommDevice : IDisposable, INotifyPropertyChanged
    {
        bool IsConnected
        {
            get;
            set;
        }

        bool IsPingConnected
        {
            get;
            set;
        }

        ushort Port
        {
            get;
            set;
        }

        bool IsBeginFreezeReceived
        {
            get;
        }

        bool IsEndFreezeReceived
        {
            get;
        }

        bool IsAttachReceived
        {
            get;
        }

        bool IsDetachReceived
        {
            get;
        }

        bool IsAttachShaderArchiveBinary
        {
            get;
        }

        uint AttachKind
        {
            get;
        }

        uint ToolKey
        {
            get;
        }

        uint ResFileKey
        {
            get;
        }

        uint ResModelKey
        {
            get;
        }

        uint ModelObjKey
        {
            get;
        }

        uint NewResFileKey
        {
            get;
        }

        uint ShaderArchiveKey
        {
            get;
        }

        bool IsFileLoaded
        {
            get;
        }

        bool IsModelFileLoaded
        {
            get;
        }

        bool IsFileReloaded
        {
            get;
        }

        bool IsModelFileReloaded
        {
            get;
        }

        string FileName
        {
            get;
        }

        string AttachPathName
        {
            get;
        }

        bool IsRenderInfoReceived
        {
            get;
        }

        RenderInfoPack RecvRenderInfoPack
        {
            get;
        }

        bool IsModelLayoutReceived
        {
            get;
        }

        bool IsModifiedShaderProgramReceived
        {
            get;
        }

        bool IsPickupReceived
        {
            get;
        }

        bool IsPlayFrameCtrlReceived
        {
            get;
        }

        bool IsStopFrameCtrlReceived
        {
            get;
        }

        bool IsSendFrameReceived
        {
            get;
        }

        bool IsSendFrameStepReceived
        {
            get;
        }

        bool IsSendModelNextAnimReceived
        {
            get;
        }

        bool IsSendModelPrevAnimReceived
        {
            get;
        }

        float Frame
        {
            get;
        }

        float FrameStep
        {
            get;
        }

        bool IsAbnormalPacketReceived
        {
            get;
        }

        bool IsIncorrectVersionReceived
        {
            get;
        }

        bool IsPingReceived
        {
            get;
        }

        bool HasRuntimeErrorReceived
        {
            get;
        }

        bool IsCodePageChanged
        {
            get;
        }

        RuntimeError GetRuntimeError();
        RuntimeState GetRuntimeState();
        PlatformAddressType GetRuntimeAddressType();

        bool IsShowMessageRequested
        {
            get;
        }

        MessageType MessageType
        {
            get;
        }

        MessageDestination MessageDestination
        {
            get;
        }

        string Message
        {
            get;
        }

        UInt16 CodePage
        {
            get;
        }

        bool SendData(byte[] buffer);

        void SetupRenderInfoPack();

        Vector3 GetModelLayoutScale();
        Vector3 GetModelLayoutRotate();
        Vector3 GetModelLayoutTranslate();

        ModifiedShaderProgramRecvData CreateModifiedShaderProgramData();
        PickupRecvData CreatePickupData();

        HtcResult OpenConnection();
        HtcResult Connect();
        void CloseConnection();
        void Poll();

        HtcResult OpenPingConnection();
        HtcResult ConnectPing();
        void ClosePingConnection();
        void PollPing();

        void ResetBeginFreezeFlag();
        void ResetEndFreezeFlag();

        void ResetAttachFlag();
        void ResetDetachFlag();
        void ResetFileLoadFlag();
        void ResetRenderInfoFlag();
        void ResetModelLayoutFlag();
        void ResetModifiedShaderProgramFlag();
        void ResetPickupFlag();
        void ResetPlayFrameCtrlFlag();
        void ResetStopFrameCtrlFlag();
        void ResetSendFrameFlag();
        void ResetSendFrameStepFlag();
        void ResetSendModelNextAnimFlag();
        void ResetSendModelPrevAnimFlag();
        void ResetAbnormalPacketFlag();
        void ResetIncorrectVersionFlag();
        void ResetRuntimeErrorFlag();
        void ResetShowMessageRequestedFlag();
        void ResetCodePageChangedFlag();
    }
}
