﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;

namespace NintendoWare.G3d.Edit
{
    public abstract class HIOBase : ObservableObject, IHIO
    {
        public enum TargetType
        {
            OldPc,
            Cafe,
            Htc,
        }

        private ICommDevice commDevice = null;
        private TargetType targetDeviceType = TargetType.Cafe;
        private bool isAutoConnection = false;

        private bool isPingReceived = false;

        private RecvInfoBase recvInfo = null;
        private readonly Dictionary<IEditModelTarget, HashSet<int>> selectedMaterials =
                new Dictionary<IEditModelTarget, HashSet<int>>();

        private readonly Dictionary<IEditModelTarget, HashSet<int>> selectedBones =
                new Dictionary<IEditModelTarget, HashSet<int>>();

        private readonly Dictionary<IEditModelTarget, HashSet<int>> selectedShapes =
                new Dictionary<IEditModelTarget, HashSet<int>>();

        private readonly HashSet<IEditModelTarget> selectedModels =
                new HashSet<IEditModelTarget>();

        private readonly SendCommandQueue sendCommandQueue = new SendCommandQueue();

        protected HIOBase()
        {
        }

        public TargetType TargetDeviceType
        {
            get{ return this.targetDeviceType; }
            protected set
            {
                if (this.commDevice == null)
                {
                    return;
                }
                if (this.targetDeviceType != value)
                {
                    this.targetDeviceType = value;
                    this.OnPropertyChanged("TargetDeviceType");
                }
            }
        }

        /// <summary>
        /// コマンド通信ソケットの接続が確立されているかどうかを返します。
        /// </summary>
        public bool IsConnected
        {
            get
            {
                if (this.commDevice == null)
                {
                    return false;
                }
                return this.commDevice.IsConnected;
            }
            set
            {
                if (this.commDevice == null)
                {
                    return;
                }
                if (this.commDevice.IsConnected != value)
                {
                    this.commDevice.IsConnected = value;
                    this.OnPropertyChanged("IsConnected");
                }
            }
        }

        /// <summary>
        /// Ping ソケットの接続が確立されているかどうかを返します。
        /// </summary>
        public bool IsPingConnected
        {
            get
            {
                if (this.commDevice == null)
                {
                    return false;
                }
                return this.commDevice.IsPingConnected;
            }
            set
            {
                if (this.commDevice == null)
                {
                    return;
                }
                if (this.commDevice.IsPingConnected != value)
                {
                    this.commDevice.IsPingConnected = value;
                    this.OnPropertyChanged("IsPingConnected");
                }
            }
        }

        public bool IsAutoConnection
        {
            get
            {
                if (this.commDevice == null)
                {
                    return false;
                }
                return this.isAutoConnection;
            }
            set
            {
                if (this.commDevice == null)
                {
                    return;
                }

                if (this.isAutoConnection != value)
                {
                    this.isAutoConnection = value;
                    this.OnPropertyChanged("IsAutoConnection");
                }
            }
        }

        public bool IsPingReceived
        {
            get
            {
                return this.isPingReceived;
            }
            protected set
            {
                if (this.isPingReceived != value)
                {
                    this.isPingReceived = value;
                    this.OnPropertyChanged("IsPingReceived");
                }
            }
        }

        public IRecvInfo RecvInfo
        {
            get { return this.recvInfo; }
        }

        public RuntimeState GetRuntimeState()
        {
            if (this.commDevice == null)
            {
                return RuntimeState.Normal;
            }
            return this.commDevice.GetRuntimeState();
        }

        /// <summary>
        /// ランタイムのアドレスの形式(32/64ビット)を取得します。
        /// </summary>
        /// <returns>ランタイムのアドレス形式の種類を返します。</returns>
        public PlatformAddressType GetRuntimeAddressType()
        {
            if (this.commDevice == null)
            {
                return PlatformAddressType.Unknown;
            }
            return this.commDevice.GetRuntimeAddressType();
        }

        public HtcResult Open()
        {
            if (this.commDevice == null)
            {
                return new HtcResult() { ResultCode = HtcResultCode.NullDevice };
            }
            return this.commDevice.OpenConnection();
        }

        public void Close()
        {
            if (this.commDevice == null)
            {
                return;
            }
            this.sendCommandQueue.ClearAll();
            this.commDevice.CloseConnection();
            UpdateConnection();
        }

        public void CloseAll()
        {
            if (this.commDevice == null)
            {
                return;
            }
            this.sendCommandQueue.ClearAll();
            this.commDevice.ClosePingConnection();
            this.commDevice.CloseConnection();
            UpdateConnection();
        }

        /// <summary>
        /// 通信接続確認用のPoll 処理
        /// </summary>
        public void PollPing()
        {
            if (this.commDevice == null)
            {
                return;
            }

            if (!this.commDevice.IsPingConnected)
            {
                HtcResult openResult = this.commDevice.OpenPingConnection();
                if (!openResult.IsSuccess)
                {
                    this.commDevice.CloseConnection();
                    IsConnected = false;
                    IsPingReceived = false;
                    return;
                }

                HtcResult connectResult = this.commDevice.ConnectPing();
                if (!connectResult.IsSuccess)
                {
                    this.commDevice.ClosePingConnection();
                    this.commDevice.CloseConnection();
                    IsConnected = false;
                    IsPingReceived = false;
                    return;
                }
            }

            this.commDevice.PollPing();

            // 自動接続が有効でない場合は処理をぬける
            if (!IsAutoConnection)
            {
                return;
            }

            IsPingReceived = this.commDevice.IsPingReceived;
        }

        public void SelectTargetMaterials(IEditModelTarget targetModel, int[] indices)
        {
            HashSet<int> materialIndices = null;

            if (this.selectedMaterials.TryGetValue(targetModel, out materialIndices))
            {
                foreach (var index in indices)
                {
                    materialIndices.Add(index);
                }
                return;
            }

            materialIndices = new HashSet<int>();
            foreach (var index in indices)
            {
                materialIndices.Add(index);
            }
            this.selectedMaterials.Add(targetModel, materialIndices);
        }

        public void SelectTargetBones(IEditModelTarget targetModel, int[] indices)
        {
            HashSet<int> boneIndices = null;

            if (this.selectedBones.TryGetValue(targetModel, out boneIndices))
            {
                foreach (var index in indices)
                {
                    boneIndices.Add(index);
                }
                return;
            }

            boneIndices = new HashSet<int>();
            foreach (var index in indices)
            {
                boneIndices.Add(index);
            }
            this.selectedBones.Add(targetModel, boneIndices);
        }

        public void SelectTargetShapes(IEditModelTarget targetModel, int[] indices)
        {
            HashSet<int> shapeIndices = null;

            if (this.selectedShapes.TryGetValue(targetModel, out shapeIndices))
            {
                foreach (var index in indices)
                {
                    shapeIndices.Add(index);
                }
                return;
            }

            shapeIndices = new HashSet<int>();
            foreach (var index in indices)
            {
                shapeIndices.Add(index);
            }
            this.selectedShapes.Add(targetModel, shapeIndices);
        }

        public void SelectTargetModel(IEditModelTarget targetModel)
        {
            this.selectedModels.Add(targetModel);
        }

        public void ClearSelectedMaterials()
        {
            foreach (var value in this.selectedMaterials.Values)
            {
                value.Clear();
            }
            this.selectedMaterials.Clear();
        }

        public void ClearSelectedBones()
        {
            foreach (var value in this.selectedBones.Values)
            {
                value.Clear();
            }
            this.selectedBones.Clear();
        }

        public void ClearSelectedShapes()
        {
            foreach (var value in this.selectedShapes.Values)
            {
                value.Clear();
            }
            this.selectedShapes.Clear();
        }

        public void ClearSelectedModels()
        {
            this.selectedModels.Clear();
        }

        public virtual ICommandTransaction BeginCommand()
        {
            return this.BeginCommandInternal(null, null);
        }

        public virtual ICommandTransaction BeginCommand(bool sync)
        {
            return this.BeginCommandInternal(null, null);
        }

        protected void SetRecvInfo(IRecvInfo recvInfo)
        {
            Debug.Assert(recvInfo != null, "Invalid argument.");
            this.recvInfo = recvInfo as RecvInfoBase;
            Debug.Assert(this.recvInfo != null, "Invalid argument.");
        }

        protected Dictionary<IEditModelTarget, HashSet<int>> SelectedMaterials
        {
            get { return this.selectedMaterials; }
        }

        protected Dictionary<IEditModelTarget, HashSet<int>> SelectedBones
        {
            get { return this.selectedBones; }
        }

        protected Dictionary<IEditModelTarget, HashSet<int>> SelectedShapes
        {
            get { return this.selectedShapes; }
        }

        protected HashSet<IEditModelTarget> SelectedModels
        {
            get { return this.selectedModels; }
        }

        protected SendCommandQueue SendCommandQueue
        {
            get { return this.sendCommandQueue; }
        }

        protected ICommDevice Device
        {
            get { return this.commDevice; }
            set { this.commDevice = value; }
        }

        protected void UpdateConnection()
        {
            if (this.commDevice == null)
            {
                return;
            }

            this.OnPropertyChanged("IsConnected");
        }

        protected ICommandTransaction BeginCommandInternal(ISendCommand beginCommand, ISendCommand endCommand)
        {
            return this.sendCommandQueue.CreateCommandTransaction(beginCommand, endCommand);
        }

        protected void ExecuteSendCommandQueue()
        {
            if (!this.IsConnected)
            {
                return;
            }
            ICommDevice device = this.Device;
            if (device == null)
            {
                return;
            }

            if (this.SendCommandQueue.IsCreatingTransaction)
            {
                return;
            }

            this.SendCommandQueue.ExecuteQueue();
            ISendCommand command = this.SendCommandQueue.GetValidQueue();
            if (command == null)
            {
                return;
            }

            if (command.IsSystemBeginCommand)
            {
                List<ISendCommand> commands = new List<ISendCommand>();
                commands.Add(command);
                this.SendCommandQueue.RemoveValidQueue();
                ISendCommand endCommand = null;

                while (true)
                {
                    this.SendCommandQueue.ExecuteQueue();
                    ISendCommand transactionCommand = this.SendCommandQueue.GetValidQueue();
                    if (transactionCommand == null)
                    {
                        if (endCommand == null)
                        {
                            // ここに来る場合には、IsSystemBeginCommand に対して IsSystemEndCommand が見つかっていません。
                            // または、IsSystemEndCommand の command の前に無効な command が含まれている可能性があります。
                            string message = "Commands:\n";
                            foreach (var sendCommand in this.sendCommandQueue.GetCopiedArrayFromCommandQueue())
                            {
                                message += string.Format("[{0}],\n", sendCommand.GetType().ToString());
                            }
                            Debug.Assert(endCommand != null, message);
                            return;
                        }

                        break;
                    }

                    if (transactionCommand.IsSystemEndCommand)
                    {
                        endCommand = transactionCommand;
                        this.SendCommandQueue.RemoveValidQueue();
                        break;
                    }

                    commands.Add(transactionCommand);
                    this.SendCommandQueue.RemoveValidQueue();
                }

                List<ISendCommand> sendCommands = new List<ISendCommand>();
                foreach (var transactionCommand in commands)
                {
                    if (transactionCommand.MakeCommandPacket())
                    {
                        sendCommands.Add(transactionCommand);
                    }
                }

                // 終了コマンドが作成できない場合は、キューに追加しなおして処理を抜ける。
                if (!endCommand.MakeCommandPacket())
                {
                    this.SendCommandQueue.PushFront(endCommand);
                    return;
                }

                foreach (var sendCommand in sendCommands)
                {
                    device.SendData(sendCommand.PacketBuffer);
                }

                // 終了コマンドが送信できない場合は、キューに追加しなおして処理を抜ける。
                if (!device.SendData(endCommand.PacketBuffer))
                {
                    this.SendCommandQueue.PushFront(endCommand);
                    return;
                }
            }
            else
            {
                if (!command.MakeCommandPacket())
                {
                    return;
                }
                if (device.SendData(command.PacketBuffer))
                {
                    this.SendCommandQueue.RemoveValidQueue();
                }
            }
        }
    }
}
