﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NintendoWare.G3d.Edit.Math
{
    /// <summary>
    /// 4 × 4 行列を表します。
    /// </summary>
    public sealed class Matrix44 : IMatrix
    {
        //-----------------------------------------------------------------
        private const int MatrixRowCount = 4;
        private const int MatrixColumnCount = 4;
        private float[,] m = new float[MatrixRowCount, MatrixColumnCount]
            {
                { 1f, 0f, 0f, 0f },
                { 0f, 1f, 0f, 0f },
                { 0f, 0f, 1f, 0f },
                { 0f, 0f, 0f, 1f },
            };

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public Matrix44()
        {
        }

        /// <summary>
        /// 行列の成分を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="m00">[0, 0]成分です。</param>
        /// <param name="m01">[0, 1]成分です。</param>
        /// <param name="m02">[0, 2]成分です。</param>
        /// <param name="m03">[0, 3]成分です。</param>
        /// <param name="m10">[1, 0]成分です。</param>
        /// <param name="m11">[1, 1]成分です。</param>
        /// <param name="m12">[1, 2]成分です。</param>
        /// <param name="m13">[1, 3]成分です。</param>
        /// <param name="m20">[2, 0]成分です。</param>
        /// <param name="m21">[2, 1]成分です。</param>
        /// <param name="m22">[2, 2]成分です。</param>
        /// <param name="m23">[2, 3]成分です。</param>
        /// <param name="m30">[3, 0]成分です。</param>
        /// <param name="m31">[3, 1]成分です。</param>
        /// <param name="m32">[3, 2]成分です。</param>
        /// <param name="m33">[3, 3]成分です。</param>
        public Matrix44(
            float m00,
            float m01,
            float m02,
            float m03,
            float m10,
            float m11,
            float m12,
            float m13,
            float m20,
            float m21,
            float m22,
            float m23,
            float m30,
            float m31,
            float m32,
            float m33)
        {
            this.Set(
                m00, m01, m02, m03, m10, m11, m12, m13, m20, m21, m22, m23, m30, m31, m32, m33);
        }

        /// <summary>
        /// 列数を取得します。
        /// </summary>
        public int ColumnCount
        {
            get
            {
                return MatrixColumnCount;
            }
        }

        /// <summary>
        /// 行数を取得します。
        /// </summary>
        public int RowCount
        {
            get
            {
                return MatrixRowCount;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M00
        {
            get
            {
                return this.m[0, 0];
            }

            set
            {
                this.m[0, 0] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M01
        {
            get
            {
                return this.m[0, 1];
            }

            set
            {
                this.m[0, 1] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M02
        {
            get
            {
                return this.m[0, 2];
            }

            set
            {
                this.m[0, 2] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M03
        {
            get
            {
                return this.m[0, 3];
            }

            set
            {
                this.m[0, 3] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M10
        {
            get
            {
                return this.m[1, 0];
            }

            set
            {
                this.m[1, 0] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M11
        {
            get
            {
                return this.m[1, 1];
            }

            set
            {
                this.m[1, 1] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M12
        {
            get
            {
                return this.m[1, 2];
            }

            set
            {
                this.m[1, 2] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M13
        {
            get
            {
                return this.m[1, 3];
            }

            set
            {
                this.m[1, 3] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M20
        {
            get
            {
                return this.m[2, 0];
            }

            set
            {
                this.m[2, 0] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M21
        {
            get
            {
                return this.m[2, 1];
            }

            set
            {
                this.m[2, 1] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M22
        {
            get
            {
                return this.m[2, 2];
            }

            set
            {
                this.m[2, 2] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M23
        {
            get
            {
                return this.m[2, 3];
            }

            set
            {
                this.m[2, 3] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M30
        {
            get
            {
                return this.m[3, 0];
            }

            set
            {
                this.m[3, 0] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M31
        {
            get
            {
                return this.m[3, 1];
            }

            set
            {
                this.m[3, 1] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M32
        {
            get
            {
                return this.m[3, 2];
            }

            set
            {
                this.m[3, 2] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M33
        {
            get
            {
                return this.m[3, 3];
            }

            set
            {
                this.m[3, 3] = value;
            }
        }

        /// <summary>
        /// 行成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Rows
        {
            get
            {
                IVector[] data =
                {
                     new Vector4(this[0, 0], this[0, 1], this[0, 2], this[0, 3]),
                     new Vector4(this[1, 0], this[1, 1], this[1, 2], this[1, 3]),
                     new Vector4(this[2, 0], this[2, 1], this[2, 2], this[2, 3]),
                     new Vector4(this[3, 0], this[3, 1], this[3, 2], this[3, 3])
                };

                return data;
            }
        }

        /// <summary>
        /// 列成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Columns
        {
            get
            {
                IVector[] data =
                {
                     new Vector4(this[0, 0], this[1, 0], this[2, 0], this[3, 0]),
                     new Vector4(this[0, 1], this[1, 1], this[2, 1], this[3, 1]),
                     new Vector4(this[0, 2], this[1, 2], this[2, 2], this[3, 2]),
                     new Vector4(this[0, 3], this[1, 3], this[2, 3], this[3, 3])
                };

                return data;
            }
        }

        /// <summary>
        /// 指定したインデックス位置の成分へのアクセスを行うインデクサです。
        /// </summary>
        /// <param name="row">アクセスする成分の行の 0 から始まるインデックスです。</param>
        /// <param name="column">アクセスする成分の列の 0 から始まるインデックスです。</param>
        /// <returns>指定したインデックス位置の成分です。</returns>
        public float this[int row, int column]
        {
            get
            {
                return this.m[row, column];
            }

            set
            {
                this.m[row, column] = value;
            }
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source as Matrix44);
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public void Set(Matrix44 source)
        {
            this.Set(
                source.M00, source.M01, source.M02, source.M03,
                source.M10, source.M11, source.M12, source.M13,
                source.M20, source.M21, source.M22, source.M23,
                source.M30, source.M31, source.M32, source.M33);
        }

        /// <summary>
        /// 行列の成分を指定の値に設定します。
        /// </summary>
        /// <param name="m00">[0, 0]成分です。</param>
        /// <param name="m01">[0, 1]成分です。</param>
        /// <param name="m02">[0, 2]成分です。</param>
        /// <param name="m03">[0, 3]成分です。</param>
        /// <param name="m10">[1, 0]成分です。</param>
        /// <param name="m11">[1, 1]成分です。</param>
        /// <param name="m12">[1, 2]成分です。</param>
        /// <param name="m13">[1, 3]成分です。</param>
        /// <param name="m20">[2, 0]成分です。</param>
        /// <param name="m21">[2, 1]成分です。</param>
        /// <param name="m22">[2, 2]成分です。</param>
        /// <param name="m23">[2, 3]成分です。</param>
        /// <param name="m30">[3, 0]成分です。</param>
        /// <param name="m31">[3, 1]成分です。</param>
        /// <param name="m32">[3, 2]成分です。</param>
        /// <param name="m33">[3, 3]成分です。</param>
        public void Set(
            float m00,
            float m01,
            float m02,
            float m03,
            float m10,
            float m11,
            float m12,
            float m13,
            float m20,
            float m21,
            float m22,
            float m23,
            float m30,
            float m31,
            float m32,
            float m33)
        {
            this[0, 0] = m00;
            this[0, 1] = m01;
            this[0, 2] = m02;
            this[0, 3] = m03;

            this[1, 0] = m10;
            this[1, 1] = m11;
            this[1, 2] = m12;
            this[1, 3] = m13;

            this[2, 0] = m20;
            this[2, 1] = m21;
            this[2, 2] = m22;
            this[2, 3] = m23;

            this[3, 0] = m30;
            this[3, 1] = m31;
            this[3, 2] = m32;
            this[3, 3] = m33;
        }

        /// <summary>
        /// 行列の成分を指定の値に設定します。
        /// </summary>
        /// <param name="row0">第０行の成分を格納したベクトルです。</param>
        /// <param name="row1">第１行の成分を格納したベクトルです。</param>
        /// <param name="row2">第２行の成分を格納したベクトルです。</param>
        /// <param name="row3">第３行の成分を格納したベクトルです。</param>
        public void Set(Vector4 row0, Vector4 row1, Vector4 row2, Vector4 row3)
        {
            this.M00 = row0.X;
            this.M01 = row0.Y;
            this.M02 = row0.Z;
            this.M03 = row0.W;
            this.M10 = row1.X;
            this.M11 = row1.Y;
            this.M12 = row1.Z;
            this.M13 = row1.W;
            this.M20 = row2.X;
            this.M21 = row2.Y;
            this.M22 = row2.Z;
            this.M23 = row2.W;
            this.M30 = row3.X;
            this.M31 = row3.Y;
            this.M32 = row3.Z;
            this.M33 = row3.W;
        }

        /// <summary>
        /// 要素を格納した配列を返します。
        /// </summary>
        /// <returns>要素の配列です。</returns>
        public float[] ToArray()
        {
            return new float[]
            {
                this.m[0, 0],
                this.m[0, 1],
                this.m[0, 2],
                this.m[0, 3],
                this.m[1, 0],
                this.m[1, 1],
                this.m[1, 2],
                this.m[1, 3],
                this.m[2, 0],
                this.m[2, 1],
                this.m[2, 2],
                this.m[2, 3],
                this.m[3, 0],
                this.m[3, 1],
                this.m[3, 2],
                this.m[3, 3]
            };
        }
    }
}
