﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NintendoWare.G3d.Edit
{
    public abstract class CommandQueue<T> where T : class, ICommand
    {
        private static readonly int MaxOfTrials = 100;
        private readonly SynchronizedCollection<T> commandQueue = new SynchronizedCollection<T>();
        private readonly object syncRoot = new object();
        private T validCommand = null;

        public int Count
        {
            get
            {
                lock (this.syncRoot)
                {
                    return this.commandQueue.Count;
                }
            }
        }

        internal bool Insert(int index, T command)
        {
            lock (this.syncRoot)
            {
                int count = this.commandQueue.Count;
                this.commandQueue.Insert(index, command);
                return (count + 1) == this.commandQueue.Count;
            }
        }

        public bool PushBack(T command)
        {
            lock (this.syncRoot)
            {
                int count = this.commandQueue.Count;
                this.commandQueue.Add(command);
                return (count + 1) == this.commandQueue.Count;
            }
        }

        public bool PushFront(T command)
        {
            lock (this.syncRoot)
            {
                return Insert(0, command);
            }
        }

        public T PopBack()
        {
            lock (this.syncRoot)
            {
                if (this.commandQueue.Count <= 0)
                {
                    return null;
                }

                int index = this.commandQueue.Count - 1;
                var value = this.commandQueue[index];
                this.commandQueue.RemoveAt(index);
                return value;
            }
        }

        public T PopFront()
        {
            lock (this.syncRoot)
            {
                if (this.commandQueue.Count <= 0)
                {
                    return null;
                }

                var value = this.commandQueue[0];
                this.commandQueue.RemoveAt(0);
                return value;
            }
        }

        public T GetFront()
        {
            lock (this.syncRoot)
            {
                if (this.commandQueue.Count <= 0)
                {
                    return null;
                }
                return this.commandQueue[0];
            }
        }

        public void Remove(T command)
        {
            lock (this.syncRoot)
            {
                if (this.commandQueue.Contains(command))
                {
                    this.commandQueue.Remove(command);
                }
            }
        }

        public void Clear()
        {
            lock (this.syncRoot)
            {
                this.commandQueue.Clear();
                this.validCommand = null;
            }
        }

        public void RemoveValidQueue()
        {
            lock (this.syncRoot)
            {
                if (this.validCommand != null && this.commandQueue.Contains(this.validCommand))
                {
                    this.commandQueue.Remove(this.validCommand);
                    this.validCommand.OnRemoved();
                    this.validCommand = null;
                }
            }
        }

        public void ExecuteQueue()
        {
            lock (this.syncRoot)
            {
                // 有効なコマンドが処理されていない場合は処理をスキップ
                if (this.validCommand != null)
                {
                    return;
                }

                var value = GetFront();
                if (value == null)
                {
                    return;
                }

                // 最大試行回数を超えていたら削除
                if (value.NumberOfTrials > MaxOfTrials)
                {
                    bool isInOrder = value.IsInOrder;
                    // この処理に入るのは、だいたいコマンドクラスの実装がまずい場合に起きます。

                    PopFront();

                    // コマンド順が強制されている場合は、失敗した場合は、積んでいるコマンドを破棄する。
                    if (isInOrder)
                    {
                        this.Clear();
                    }
                    return;
                }

                if (value.CanProcess())
                {
                    this.validCommand = value;
                }
            }
        }

        public T GetValidQueue()
        {
            lock (this.syncRoot)
            {
                return this.validCommand;
            }
        }

        internal IEnumerable<T> GetCopiedArrayFromCommandQueue()
        {
            lock (this.syncRoot)
            {
                T[] result = new T[this.commandQueue.Count];
                this.commandQueue.CopyTo(result, 0);
                return result;
            }
        }
    }
}
